<?php
/**
 * The settings-specific functionality of the plugin.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/admin
 */

/**
 * The settings-specific functionality of the plugin.
 */
class Audit_Export_Settings {

	/**
	 * The ID of this plugin.
	 *
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string $plugin_name The name of this plugin.
	 * @param string $version     The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	/**
	 * Register all settings.
	 */
	public function register_settings() {
		// Register core settings
		register_setting(
			'audit_export_settings_group',
			'audit_export_settings',
			array(
				'sanitize_callback' => array( $this, 'sanitize_core_settings' ),
				'default' => $this->get_default_core_settings(),
			)
		);

		// Register remote post settings
		register_setting(
			'audit_export_remote_post_settings_group',
			'audit_export_remote_post_settings',
			array(
				'sanitize_callback' => array( $this, 'sanitize_remote_post_settings' ),
				'default' => $this->get_default_remote_post_settings(),
			)
		);

		// Add settings sections
		$this->add_core_settings_sections();
		$this->add_remote_post_settings_sections();
	}

	/**
	 * Add core settings sections.
	 */
	private function add_core_settings_sections() {
		// File Export Settings Section
		add_settings_section(
			'audit_export_file_settings',
			__( 'File Export Settings', 'audit-export' ),
			array( $this, 'file_settings_section_callback' ),
			'audit-export-settings'
		);

		add_settings_field(
			'save_to_filesystem',
			__( 'Save to Filesystem', 'audit-export' ),
			array( $this, 'save_to_filesystem_callback' ),
			'audit-export-settings',
			'audit_export_file_settings'
		);

		add_settings_field(
			'filesystem_type',
			__( 'Filesystem Type', 'audit-export' ),
			array( $this, 'filesystem_type_callback' ),
			'audit-export-settings',
			'audit_export_file_settings'
		);

		add_settings_field(
			'filesystem_path',
			__( 'Export Path', 'audit-export' ),
			array( $this, 'filesystem_path_callback' ),
			'audit-export-settings',
			'audit_export_file_settings'
		);

		// Cron Settings Section
		add_settings_section(
			'audit_export_cron_settings',
			__( 'Cron Settings', 'audit-export' ),
			array( $this, 'cron_settings_section_callback' ),
			'audit-export-settings'
		);

		add_settings_field(
			'enable_cron',
			__( 'Enable Cron', 'audit-export' ),
			array( $this, 'enable_cron_callback' ),
			'audit-export-settings',
			'audit_export_cron_settings'
		);

		add_settings_field(
			'run_on_every_cron',
			__( 'Run on Every Cron', 'audit-export' ),
			array( $this, 'run_on_every_cron_callback' ),
			'audit-export-settings',
			'audit_export_cron_settings'
		);

		add_settings_field(
			'cron_frequency',
			__( 'Cron Frequency (minutes)', 'audit-export' ),
			array( $this, 'cron_frequency_callback' ),
			'audit-export-settings',
			'audit_export_cron_settings'
		);

		add_settings_field(
			'queue_timeout',
			__( 'Queue Timeout (seconds)', 'audit-export' ),
			array( $this, 'queue_timeout_callback' ),
			'audit-export-settings',
			'audit_export_cron_settings'
		);
	}

	/**
	 * Add remote post settings sections.
	 */
	private function add_remote_post_settings_sections() {
		// Remote Post Basic Settings
		add_settings_section(
			'audit_export_remote_post_basic',
			__( 'Remote Post Basic Settings', 'audit-export' ),
			array( $this, 'remote_post_basic_section_callback' ),
			'audit-export-remote-post-settings'
		);

		add_settings_field(
			'enable_remote_post',
			__( 'Enable Remote Post', 'audit-export' ),
			array( $this, 'enable_remote_post_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_basic'
		);

		add_settings_field(
			'remote_url',
			__( 'Remote URL', 'audit-export' ),
			array( $this, 'remote_url_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_basic'
		);

		add_settings_field(
			'site_name',
			__( 'Site Name', 'audit-export' ),
			array( $this, 'site_name_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_basic'
		);

		// Authentication Settings
		add_settings_section(
			'audit_export_remote_post_auth',
			__( 'Authentication Settings', 'audit-export' ),
			array( $this, 'remote_post_auth_section_callback' ),
			'audit-export-remote-post-settings'
		);

		add_settings_field(
			'authentication_type',
			__( 'Authentication Type', 'audit-export' ),
			array( $this, 'authentication_type_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_auth'
		);

		add_settings_field(
			'username',
			__( 'Username', 'audit-export' ),
			array( $this, 'username_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_auth'
		);

		add_settings_field(
			'password',
			__( 'Password', 'audit-export' ),
			array( $this, 'password_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_auth'
		);

		add_settings_field(
			'token',
			__( 'Bearer Token', 'audit-export' ),
			array( $this, 'token_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_auth'
		);

		// Advanced Settings
		add_settings_section(
			'audit_export_remote_post_advanced',
			__( 'Advanced Settings', 'audit-export' ),
			array( $this, 'remote_post_advanced_section_callback' ),
			'audit-export-remote-post-settings'
		);

		add_settings_field(
			'timeout',
			__( 'Timeout (seconds)', 'audit-export' ),
			array( $this, 'timeout_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_advanced'
		);

		add_settings_field(
			'verify_ssl',
			__( 'Verify SSL', 'audit-export' ),
			array( $this, 'verify_ssl_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_advanced'
		);

		add_settings_field(
			'debug_mode',
			__( 'Debug Mode', 'audit-export' ),
			array( $this, 'debug_mode_callback' ),
			'audit-export-remote-post-settings',
			'audit_export_remote_post_advanced'
		);
	}

	/**
	 * Get default core settings.
	 *
	 * @return array
	 */
	private function get_default_core_settings() {
		return array(
			'save_to_filesystem' => false,
			'filesystem_type' => 'uploads',
			'filesystem_path' => 'audit-export',
			'enable_cron' => false,
			'run_on_every_cron' => false,
			'cron_frequency' => 1440,
			'queue_timeout' => 120,
		);
	}

	/**
	 * Get default remote post settings.
	 *
	 * @return array
	 */
	private function get_default_remote_post_settings() {
		return array(
			'enable_remote_post' => false,
			'remote_url' => '',
			'site_name' => get_bloginfo( 'name' ),
			'authentication_type' => 'none',
			'username' => '',
			'password' => '',
			'token' => '',
			'timeout' => 300,
			'verify_ssl' => true,
			'debug_mode' => false,
		);
	}

	/**
	 * Sanitize core settings.
	 *
	 * @param array $input Raw input data.
	 * @return array Sanitized data.
	 */
	public function sanitize_core_settings( $input ) {
		$sanitized = array();
		$defaults = $this->get_default_core_settings();

		$sanitized['save_to_filesystem'] = isset( $input['save_to_filesystem'] ) ? (bool) $input['save_to_filesystem'] : $defaults['save_to_filesystem'];
		$sanitized['filesystem_type'] = isset( $input['filesystem_type'] ) ? sanitize_text_field( $input['filesystem_type'] ) : $defaults['filesystem_type'];
		$sanitized['filesystem_path'] = isset( $input['filesystem_path'] ) ? sanitize_text_field( $input['filesystem_path'] ) : $defaults['filesystem_path'];
		$sanitized['enable_cron'] = isset( $input['enable_cron'] ) ? (bool) $input['enable_cron'] : $defaults['enable_cron'];
		$sanitized['run_on_every_cron'] = isset( $input['run_on_every_cron'] ) ? (bool) $input['run_on_every_cron'] : $defaults['run_on_every_cron'];
		$sanitized['cron_frequency'] = isset( $input['cron_frequency'] ) ? max( 1, intval( $input['cron_frequency'] ) ) : $defaults['cron_frequency'];
		$sanitized['queue_timeout'] = isset( $input['queue_timeout'] ) ? max( 30, intval( $input['queue_timeout'] ) ) : $defaults['queue_timeout'];

		return $sanitized;
	}

	/**
	 * Sanitize remote post settings.
	 *
	 * @param array $input Raw input data.
	 * @return array Sanitized data.
	 */
	public function sanitize_remote_post_settings( $input ) {
		$sanitized = array();
		$defaults = $this->get_default_remote_post_settings();

		$sanitized['enable_remote_post'] = isset( $input['enable_remote_post'] ) ? (bool) $input['enable_remote_post'] : $defaults['enable_remote_post'];
		$sanitized['remote_url'] = isset( $input['remote_url'] ) ? esc_url_raw( $input['remote_url'] ) : $defaults['remote_url'];
		$sanitized['site_name'] = isset( $input['site_name'] ) ? sanitize_text_field( $input['site_name'] ) : $defaults['site_name'];
		$sanitized['authentication_type'] = isset( $input['authentication_type'] ) ? sanitize_text_field( $input['authentication_type'] ) : $defaults['authentication_type'];
		$sanitized['username'] = isset( $input['username'] ) ? sanitize_text_field( $input['username'] ) : $defaults['username'];
		$sanitized['token'] = isset( $input['token'] ) ? sanitize_text_field( $input['token'] ) : $defaults['token'];
		$sanitized['timeout'] = isset( $input['timeout'] ) ? max( 30, intval( $input['timeout'] ) ) : $defaults['timeout'];
		$sanitized['verify_ssl'] = isset( $input['verify_ssl'] ) ? (bool) $input['verify_ssl'] : $defaults['verify_ssl'];
		$sanitized['debug_mode'] = isset( $input['debug_mode'] ) ? (bool) $input['debug_mode'] : $defaults['debug_mode'];

		// Handle password separately to preserve existing value if not provided
		if ( isset( $input['password'] ) && ! empty( $input['password'] ) ) {
			$sanitized['password'] = sanitize_text_field( $input['password'] );
		} else {
			$existing = get_option( 'audit_export_remote_post_settings', $defaults );
			$sanitized['password'] = isset( $existing['password'] ) ? $existing['password'] : $defaults['password'];
		}

		return $sanitized;
	}

	// Section callbacks
	public function file_settings_section_callback() {
		echo '<p>' . esc_html( __( 'Configure how audit reports are exported and saved.', 'audit-export' ) ) . '</p>';
	}

	public function cron_settings_section_callback() {
		echo '<p>' . esc_html( __( 'Configure automated report processing via cron.', 'audit-export' ) ) . '</p>';
	}

	public function remote_post_basic_section_callback() {
		echo '<p>' . esc_html( __( 'Configure basic settings for posting audit data to remote endpoints.', 'audit-export' ) ) . '</p>';
	}

	public function remote_post_auth_section_callback() {
		echo '<p>' . esc_html( __( 'Configure authentication for remote endpoint access.', 'audit-export' ) ) . '</p>';
	}

	public function remote_post_advanced_section_callback() {
		echo '<p>' . esc_html( __( 'Advanced settings for remote posting.', 'audit-export' ) ) . '</p>';
	}

	// Field callbacks for core settings
	public function save_to_filesystem_callback() {
		$options = get_option( 'audit_export_settings', $this->get_default_core_settings() );
		echo '<input type="checkbox" id="save_to_filesystem" name="audit_export_settings[save_to_filesystem]" value="1" ' . checked( 1, $options['save_to_filesystem'], false ) . ' />';
		echo '<label for="save_to_filesystem">' . esc_html( __( 'Save export files to the filesystem', 'audit-export' ) ) . '</label>';
	}

	public function filesystem_type_callback() {
		$options = get_option( 'audit_export_settings', $this->get_default_core_settings() );
		$types = array(
			'uploads' => __( 'Uploads Directory', 'audit-export' ),
			'content' => __( 'Content Directory', 'audit-export' ),
		);
		
		foreach ( $types as $value => $label ) {
			echo '<label><input type="radio" name="audit_export_settings[filesystem_type]" value="' . esc_attr( $value ) . '" ' . checked( $value, $options['filesystem_type'], false ) . ' /> ' . esc_html( $label ) . '</label><br>';
		}
	}

	public function filesystem_path_callback() {
		$options = get_option( 'audit_export_settings', $this->get_default_core_settings() );
		echo '<input type="text" id="filesystem_path" name="audit_export_settings[filesystem_path]" value="' . esc_attr( $options['filesystem_path'] ) . '" class="regular-text" />';
		echo '<p class="description">' . esc_html( __( 'Path within the selected filesystem where exports will be saved.', 'audit-export' ) ) . '</p>';
	}

	public function enable_cron_callback() {
		$options = get_option( 'audit_export_settings', $this->get_default_core_settings() );
		echo '<input type="checkbox" id="enable_cron" name="audit_export_settings[enable_cron]" value="1" ' . checked( 1, $options['enable_cron'], false ) . ' />';
		echo '<label for="enable_cron">' . esc_html( __( 'Enable automated report processing via cron', 'audit-export' ) ) . '</label>';
	}

	public function run_on_every_cron_callback() {
		$options = get_option( 'audit_export_settings', $this->get_default_core_settings() );
		echo '<input type="checkbox" id="run_on_every_cron" name="audit_export_settings[run_on_every_cron]" value="1" ' . checked( 1, $options['run_on_every_cron'], false ) . ' />';
		echo '<label for="run_on_every_cron">' . esc_html( __( 'Run on every cron execution (ignores frequency setting)', 'audit-export' ) ) . '</label>';
	}

	public function cron_frequency_callback() {
		$options = get_option( 'audit_export_settings', $this->get_default_core_settings() );
		echo '<input type="number" id="cron_frequency" name="audit_export_settings[cron_frequency]" value="' . esc_attr( $options['cron_frequency'] ) . '" min="1" class="small-text" />';
		echo '<p class="description">' . esc_html( __( 'Frequency in minutes for running audits (only used if "Run on every cron" is disabled).', 'audit-export' ) ) . '</p>';
	}

	public function queue_timeout_callback() {
		$options = get_option( 'audit_export_settings', $this->get_default_core_settings() );
		echo '<input type="number" id="queue_timeout" name="audit_export_settings[queue_timeout]" value="' . esc_attr( $options['queue_timeout'] ) . '" min="30" class="small-text" />';
		echo '<p class="description">' . esc_html( __( 'Maximum time in seconds for processing each audit.', 'audit-export' ) ) . '</p>';
	}

	// Field callbacks for remote post settings
	public function enable_remote_post_callback() {
		$options = get_option( 'audit_export_remote_post_settings', $this->get_default_remote_post_settings() );
		echo '<input type="checkbox" id="enable_remote_post" name="audit_export_remote_post_settings[enable_remote_post]" value="1" ' . checked( 1, $options['enable_remote_post'], false ) . ' />';
		echo '<label for="enable_remote_post">' . esc_html( __( 'Enable posting audit data to remote endpoint', 'audit-export' ) ) . '</label>';
	}

	public function remote_url_callback() {
		$options = get_option( 'audit_export_remote_post_settings', $this->get_default_remote_post_settings() );
		echo '<input type="url" id="remote_url" name="audit_export_remote_post_settings[remote_url]" value="' . esc_attr( $options['remote_url'] ) . '" class="regular-text" />';
		echo '<p class="description">' . esc_html( __( 'The URL where audit data will be posted.', 'audit-export' ) ) . '</p>';
	}

	public function site_name_callback() {
		$options = get_option( 'audit_export_remote_post_settings', $this->get_default_remote_post_settings() );
		echo '<input type="text" id="site_name" name="audit_export_remote_post_settings[site_name]" value="' . esc_attr( $options['site_name'] ) . '" class="regular-text" />';
		echo '<p class="description">' . esc_html( __( 'Name of the site as it appears in posted data.', 'audit-export' ) ) . '</p>';
	}

	public function authentication_type_callback() {
		$options = get_option( 'audit_export_remote_post_settings', $this->get_default_remote_post_settings() );
		$types = array(
			'none' => __( 'None', 'audit-export' ),
			'basic' => __( 'Basic Authentication', 'audit-export' ),
			'bearer' => __( 'Bearer Token', 'audit-export' ),
		);
		
		foreach ( $types as $value => $label ) {
			echo '<label><input type="radio" name="audit_export_remote_post_settings[authentication_type]" value="' . esc_attr( $value ) . '" ' . checked( $value, $options['authentication_type'], false ) . ' /> ' . esc_html( $label ) . '</label><br>';
		}
	}

	public function username_callback() {
		$options = get_option( 'audit_export_remote_post_settings', $this->get_default_remote_post_settings() );
		echo '<input type="text" id="username" name="audit_export_remote_post_settings[username]" value="' . esc_attr( $options['username'] ) . '" class="regular-text" />';
		echo '<p class="description">' . esc_html( __( 'Username for Basic Authentication.', 'audit-export' ) ) . '</p>';
	}

	public function password_callback() {
		echo '<input type="password" id="password" name="audit_export_remote_post_settings[password]" value="" class="regular-text" />';
		echo '<p class="description">' . esc_html( __( 'Password for Basic Authentication. Leave empty to keep existing password.', 'audit-export' ) ) . '</p>';
	}

	public function token_callback() {
		$options = get_option( 'audit_export_remote_post_settings', $this->get_default_remote_post_settings() );
		echo '<input type="text" id="token" name="audit_export_remote_post_settings[token]" value="' . esc_attr( $options['token'] ) . '" class="regular-text" />';
		echo '<p class="description">' . esc_html( __( 'Bearer token for authentication.', 'audit-export' ) ) . '</p>';
	}

	public function timeout_callback() {
		$options = get_option( 'audit_export_remote_post_settings', $this->get_default_remote_post_settings() );
		echo '<input type="number" id="timeout" name="audit_export_remote_post_settings[timeout]" value="' . esc_attr( $options['timeout'] ) . '" min="30" max="900" class="small-text" />';
		echo '<p class="description">' . esc_html( __( 'Request timeout in seconds (30-900).', 'audit-export' ) ) . '</p>';
	}

	public function verify_ssl_callback() {
		$options = get_option( 'audit_export_remote_post_settings', $this->get_default_remote_post_settings() );
		echo '<input type="checkbox" id="verify_ssl" name="audit_export_remote_post_settings[verify_ssl]" value="1" ' . checked( 1, $options['verify_ssl'], false ) . ' />';
		echo '<label for="verify_ssl">' . esc_html( __( 'Verify SSL certificates (disable only for development)', 'audit-export' ) ) . '</label>';
	}

	public function debug_mode_callback() {
		$options = get_option( 'audit_export_remote_post_settings', $this->get_default_remote_post_settings() );
		echo '<input type="checkbox" id="debug_mode" name="audit_export_remote_post_settings[debug_mode]" value="1" ' . checked( 1, $options['debug_mode'], false ) . ' />';
		echo '<label for="debug_mode">' . esc_html( __( 'Enable debug logging for remote posts', 'audit-export' ) ) . '</label>';
	}
}