<?php
/**
 * Service class for handling queue functionality.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/services
 */

/**
 * Audit export queue service.
 */
class Audit_Export_Queue {

	/**
	 * The audit manager instance.
	 *
	 * @var Audit_Export_Audit_Manager
	 */
	private $audit_manager;

	/**
	 * Constructor.
	 *
	 * @param Audit_Export_Audit_Manager $audit_manager The audit manager instance.
	 */
	public function __construct( $audit_manager ) {
		$this->audit_manager = $audit_manager;
	}

	/**
	 * Process a queue item.
	 *
	 * @param string $audit_id The audit ID to process.
	 */
	public function process_queue_item( $audit_id ) {
		$audit = $this->audit_manager->get_audit( $audit_id );
		
		if ( ! $audit ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for queue debugging.
			error_log( "Audit Export Queue: Audit '{$audit_id}' not found." );
			return;
		}
		
		try {
			// Get report service
			$report_service = new Audit_Export_Report();
			
			// Process the audit
			$result = $report_service->process_audit( $audit_id, $this->audit_manager );
			
			if ( $result['success'] ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for queue debugging.
				error_log( "Audit Export Queue: Successfully processed audit '{$audit_id}'." );
			} else {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for queue debugging.
				error_log( "Audit Export Queue: Failed to process audit '{$audit_id}': " . $result['message'] );
			}
			
		} catch ( Exception $e ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for queue debugging.
			error_log( "Audit Export Queue: Error processing audit '{$audit_id}': " . $e->getMessage() );
		}
	}

	/**
	 * Add audit to queue.
	 *
	 * @param string $audit_id The audit ID.
	 * @param int    $delay    Delay in seconds before processing.
	 * @return bool True if scheduled successfully.
	 */
	public function add_to_queue( $audit_id, $delay = 0 ) {
		$time = time() + $delay;
		return wp_schedule_single_event( $time, 'audit_export_process_queue', array( $audit_id ) );
	}

	/**
	 * Clear all queued items.
	 */
	public function clear_queue() {
		wp_clear_scheduled_hook( 'audit_export_process_queue' );
	}

	/**
	 * Get queued items count.
	 *
	 * @return int Number of queued items.
	 */
	public function get_queue_count() {
		$cron_array = _get_cron_array();
		$count = 0;
		
		if ( is_array( $cron_array ) ) {
			foreach ( $cron_array as $timestamp => $hooks ) {
				if ( isset( $hooks['audit_export_process_queue'] ) ) {
					$count += count( $hooks['audit_export_process_queue'] );
				}
			}
		}
		
		return $count;
	}

	/**
	 * Get next queue item.
	 *
	 * @return array|false Next queue item data or false if none.
	 */
	public function get_next_queue_item() {
		$cron_array = _get_cron_array();
		
		if ( is_array( $cron_array ) ) {
			foreach ( $cron_array as $timestamp => $hooks ) {
				if ( isset( $hooks['audit_export_process_queue'] ) ) {
					$items = $hooks['audit_export_process_queue'];
					$first_item = reset( $items );
					return array(
						'timestamp' => $timestamp,
						'args' => $first_item['args'],
					);
				}
			}
		}
		
		return false;
	}

	/**
	 * Process all queued items immediately.
	 *
	 * @param int $limit Maximum number of items to process.
	 * @return int Number of items processed.
	 */
	public function process_all_immediate( $limit = 10 ) {
		$processed = 0;
		$cron_array = _get_cron_array();
		
		if ( is_array( $cron_array ) ) {
			foreach ( $cron_array as $timestamp => $hooks ) {
				if ( isset( $hooks['audit_export_process_queue'] ) && $processed < $limit ) {
					$items = $hooks['audit_export_process_queue'];
					
					foreach ( $items as $key => $item ) {
						if ( $processed >= $limit ) {
							break;
						}
						
						// Get audit ID from args
						$audit_id = isset( $item['args'][0] ) ? $item['args'][0] : '';
						
						if ( $audit_id ) {
							// Remove from cron
							wp_unschedule_event( $timestamp, 'audit_export_process_queue', $item['args'] );
							
							// Process immediately
							$this->process_queue_item( $audit_id );
							$processed++;
						}
					}
				}
			}
		}
		
		return $processed;
	}
}