<?php
/**
 * WP-CLI commands for Audit Export plugin.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/cli
 */

/**
 * Audit Export WP-CLI commands.
 */
class Audit_Export_CLI extends WP_CLI_Command {

	/**
	 * Lists all available audit reports.
	 *
	 * ## EXAMPLES
	 *
	 *     wp audit-export list
	 *
	 * @when after_wp_load
	 */
	public function list( $args, $assoc_args ) {
		$audit_manager = new Audit_Export_Audit_Manager();
		$audit_definitions = $audit_manager->get_audit_definitions();
		$report_service = new Audit_Export_Report();

		if ( empty( $audit_definitions ) ) {
			WP_CLI::warning( 'No audit reports are available.' );
			return;
		}

		$rows = array();

		foreach ( $audit_definitions as $audit_id => $definition ) {
			$last_processed = $report_service->get_last_processed_date( $audit_id );
			
			$rows[] = array(
				'Audit' => $definition['label'],
				'ID' => $audit_id,
				'Group' => $definition['group'],
				'Description' => $definition['description'],
				'Type' => $definition['data_type'],
				'Last Processed' => $last_processed ? $last_processed : 'Never',
			);
		}

		WP_CLI\Utils\format_items( 'table', $rows, array( 'Audit', 'ID', 'Group', 'Description', 'Type', 'Last Processed' ) );
	}

	/**
	 * Processes an audit report or all reports.
	 *
	 * ## OPTIONS
	 *
	 * [<audit_id>]
	 * : The ID of the audit to process. If not provided, an interactive choice will be displayed.
	 *
	 * [--all]
	 * : Process all audit reports.
	 *
	 * [--format=<format>]
	 * : Render output in a particular format.
	 * ---
	 * default: table
	 * options:
	 *   - table
	 *   - csv
	 *   - json
	 *   - yaml
	 * ---
	 *
	 * ## EXAMPLES
	 *
	 *     wp audit-export run
	 *     wp audit-export run --all
	 *     wp audit-export run site_report
	 *     wp audit-export run content_types --format=csv
	 *
	 * @when after_wp_load
	 */
	public function run( $args, $assoc_args ) {
		$audit_manager = new Audit_Export_Audit_Manager();
		$audit_definitions = $audit_manager->get_audit_definitions();
		$report_service = new Audit_Export_Report();

		if ( empty( $audit_definitions ) ) {
			WP_CLI::error( 'No audit reports are available.' );
		}

		$process_all = WP_CLI\Utils\get_flag_value( $assoc_args, 'all', false );
		$format = WP_CLI\Utils\get_flag_value( $assoc_args, 'format', 'table' );

		if ( $process_all ) {
			WP_CLI::log( 'Processing all audit reports...' );
			
			$progress = WP_CLI\Utils\make_progress_bar( 'Processing audits', count( $audit_definitions ) );
			$success_count = 0;
			$results = array();

			foreach ( $audit_definitions as $audit_id => $definition ) {
				$result = $report_service->process_audit( $audit_id, $audit_manager );
				$results[ $audit_id ] = $result;
				
				if ( $result['success'] ) {
					$success_count++;
				}
				
				$progress->tick();
			}

			$progress->finish();

			WP_CLI::success( sprintf( 'Processed %d of %d audits successfully.', $success_count, count( $audit_definitions ) ) );

			// Show failed audits
			foreach ( $results as $audit_id => $result ) {
				if ( ! $result['success'] ) {
					WP_CLI::warning( sprintf( 'Failed to process %s: %s', $audit_id, $result['message'] ) );
				}
			}

			return;
		}

		// Get audit ID from args or prompt user
		$audit_id = isset( $args[0] ) ? $args[0] : null;

		if ( empty( $audit_id ) ) {
			$choices = array();
			foreach ( $audit_definitions as $id => $definition ) {
				$choices[ $id ] = $definition['label'] . ' (' . $id . ')';
			}

			$audit_id = WP_CLI\Utils\prompt_choice( 'Which audit should be processed?', $choices );
		}

		// Validate audit ID
		if ( ! isset( $audit_definitions[ $audit_id ] ) ) {
			WP_CLI::error( sprintf( 'Audit report "%s" does not exist.', $audit_id ) );
		}

		// Process the audit
		WP_CLI::log( sprintf( 'Processing audit: %s', $audit_definitions[ $audit_id ]['label'] ) );
		
		$result = $report_service->process_audit( $audit_id, $audit_manager );

		if ( $result['success'] ) {
			WP_CLI::success( $result['message'] );

			// Display the results
			if ( isset( $result['data']['data'] ) && ! empty( $result['data']['data'] ) ) {
				$headers = isset( $result['data']['headers'] ) ? $result['data']['headers'] : array();
				$data = $result['data']['data'];

				// Convert data for WP-CLI formatting
				$formatted_data = array();
				foreach ( $data as $row ) {
					$formatted_row = array();
					foreach ( $headers as $index => $header ) {
						$formatted_row[ $header ] = isset( $row[ $index ] ) ? $row[ $index ] : '';
					}
					$formatted_data[] = $formatted_row;
				}

				WP_CLI\Utils\format_items( $format, $formatted_data, $headers );
			}
		} else {
			WP_CLI::error( $result['message'] );
		}
	}

	/**
	 * Exports an audit report to a file.
	 *
	 * ## OPTIONS
	 *
	 * <audit_id>
	 * : The ID of the audit to export.
	 *
	 * [--file=<file>]
	 * : The file path to export to. If not provided, will output to stdout.
	 *
	 * [--format=<format>]
	 * : Export format.
	 * ---
	 * default: csv
	 * options:
	 *   - csv
	 *   - json
	 * ---
	 *
	 * ## EXAMPLES
	 *
	 *     wp audit-export export site_report
	 *     wp audit-export export content_types --file=content_types.csv
	 *     wp audit-export export plugins --format=json --file=plugins.json
	 *
	 * @when after_wp_load
	 */
	public function export( $args, $assoc_args ) {
		if ( empty( $args[0] ) ) {
			WP_CLI::error( 'Please specify an audit ID to export.' );
		}

		$audit_id = $args[0];
		$file = WP_CLI\Utils\get_flag_value( $assoc_args, 'file', null );
		$format = WP_CLI\Utils\get_flag_value( $assoc_args, 'format', 'csv' );

		$audit_manager = new Audit_Export_Audit_Manager();
		$audit = $audit_manager->get_audit( $audit_id );
		$report_service = new Audit_Export_Report();

		if ( ! $audit ) {
			WP_CLI::error( sprintf( 'Audit "%s" not found.', $audit_id ) );
		}

		$report_data = $report_service->get_report_data( $audit_id );

		if ( empty( $report_data ) ) {
			WP_CLI::error( sprintf( 'No data found for audit "%s". Run the audit first.', $audit_id ) );
		}

		$headers = isset( $report_data['headers'] ) ? $report_data['headers'] : $audit->get_headers();
		$data = isset( $report_data['data'] ) ? $report_data['data'] : array();

		if ( $format === 'csv' ) {
			$content = $report_service->export_as_csv( $audit_id, $data, $headers );
		} elseif ( $format === 'json' ) {
			$export_data = array(
				'audit_id' => $audit_id,
				'audit_label' => $audit->get_label(),
				'exported_at' => current_time( 'mysql' ),
				'headers' => $headers,
				'data' => $data,
			);
			$content = wp_json_encode( $export_data, JSON_PRETTY_PRINT );
		} else {
			WP_CLI::error( sprintf( 'Unknown format: %s', $format ) );
		}

		if ( $file ) {
			// Export to file
			$result = file_put_contents( $file, $content );
			if ( $result === false ) {
				WP_CLI::error( sprintf( 'Failed to write to file: %s', $file ) );
			}
			WP_CLI::success( sprintf( 'Exported %s to %s (%d bytes)', $audit_id, $file, $result ) );
		} else {
			// Output to stdout
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Raw file content output for piping
			echo $content;
		}
	}

	/**
	 * Manages cron settings and operations.
	 *
	 * ## OPTIONS
	 *
	 * <action>
	 * : The action to perform.
	 * ---
	 * options:
	 *   - status
	 *   - enable
	 *   - disable
	 *   - run
	 *   - schedule
	 * ---
	 *
	 * ## EXAMPLES
	 *
	 *     wp audit-export cron status
	 *     wp audit-export cron enable
	 *     wp audit-export cron run
	 *
	 * @when after_wp_load
	 */
	public function cron( $args, $assoc_args ) {
		if ( empty( $args[0] ) ) {
			WP_CLI::error( 'Please specify an action: status, enable, disable, run, or schedule.' );
		}

		$action = $args[0];
		$audit_manager = new Audit_Export_Audit_Manager();
		$cron_service = new Audit_Export_Cron( $audit_manager );

		switch ( $action ) {
			case 'status':
				$settings = get_option( 'audit_export_settings', array() );
				$cron_enabled = ! empty( $settings['enable_cron'] );
				$next_run = $cron_service->get_next_scheduled_run();

				WP_CLI::log( sprintf( 'Cron Status: %s', $cron_enabled ? 'Enabled' : 'Disabled' ) );
				
				if ( $next_run ) {
					WP_CLI::log( sprintf( 'Next Run: %s', wp_date( 'Y-m-d H:i:s T', $next_run ) ) );
				} else {
					WP_CLI::log( 'Next Run: Not scheduled' );
				}
				break;

			case 'enable':
				$settings = get_option( 'audit_export_settings', array() );
				$settings['enable_cron'] = true;
				update_option( 'audit_export_settings', $settings );
				$cron_service->schedule_events();
				WP_CLI::success( 'Cron enabled and scheduled.' );
				break;

			case 'disable':
				$settings = get_option( 'audit_export_settings', array() );
				$settings['enable_cron'] = false;
				update_option( 'audit_export_settings', $settings );
				$cron_service->clear_scheduled_events();
				WP_CLI::success( 'Cron disabled and unscheduled.' );
				break;

			case 'run':
				WP_CLI::log( 'Running cron manually...' );
				$cron_service->force_run_now();
				WP_CLI::success( 'Cron run completed.' );
				break;

			case 'schedule':
				$cron_service->schedule_events();
				WP_CLI::success( 'Cron events scheduled.' );
				break;

			default:
				WP_CLI::error( sprintf( 'Unknown action: %s', $action ) );
		}
	}

	/**
	 * Tests remote post connection.
	 *
	 * ## EXAMPLES
	 *
	 *     wp audit-export test-connection
	 *
	 * @when after_wp_load
	 */
	public function test_connection( $args, $assoc_args ) {
		$remote_post_service = new Audit_Export_Remote_Post();

		if ( ! $remote_post_service->is_enabled() ) {
			WP_CLI::warning( 'Remote posting is not enabled.' );
			return;
		}

		WP_CLI::log( 'Testing remote connection...' );
		
		$result = $remote_post_service->test_connection();

		if ( $result['success'] ) {
			WP_CLI::success( $result['message'] );
		} else {
			WP_CLI::error( $result['message'] );
		}
	}

	/**
	 * Forces posting of an audit to the remote endpoint.
	 *
	 * ## OPTIONS
	 *
	 * <audit_id>
	 * : The ID of the audit to post.
	 *
	 * ## EXAMPLES
	 *
	 *     wp audit-export force-post site_report
	 *
	 * @when after_wp_load
	 */
	public function force_post( $args, $assoc_args ) {
		if ( empty( $args[0] ) ) {
			WP_CLI::error( 'Please specify an audit ID to post.' );
		}

		$audit_id = $args[0];
		$remote_post_service = new Audit_Export_Remote_Post();

		WP_CLI::log( sprintf( 'Force posting audit: %s', $audit_id ) );
		
		$result = $remote_post_service->force_post_audit( $audit_id );

		if ( $result['success'] ) {
			WP_CLI::success( $result['message'] );
		} else {
			WP_CLI::error( $result['message'] );
		}
	}

	/**
	 * Shows plugin information and statistics.
	 *
	 * ## EXAMPLES
	 *
	 *     wp audit-export info
	 *
	 * @when after_wp_load
	 */
	public function info( $args, $assoc_args ) {
		global $wpdb;

		$audit_manager = new Audit_Export_Audit_Manager();
		$audits = $audit_manager->get_audits();
		$report_service = new Audit_Export_Report();
		$available_reports = $report_service->get_available_reports();

		// Get database info
		$table_name = $wpdb->prefix . 'audit_export_reports';
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Necessary for checking table existence.
		$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) ) === $table_name;
		$row_count = 0;

		if ( $table_exists ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct query necessary for CLI tool.
			$row_count = $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(*) FROM %i', $table_name ) );
		}

		// Settings info
		$settings = get_option( 'audit_export_settings', array() );
		$remote_settings = get_option( 'audit_export_remote_post_settings', array() );

		$info = array(
			array(
				'Setting' => 'Plugin Version',
				'Value' => AUDIT_EXPORT_VERSION,
			),
			array(
				'Setting' => 'Available Audits',
				'Value' => count( $audits ),
			),
			array(
				'Setting' => 'Reports in Database',
				'Value' => $row_count,
			),
			array(
				'Setting' => 'Cron Enabled',
				'Value' => ! empty( $settings['enable_cron'] ) ? 'Yes' : 'No',
			),
			array(
				'Setting' => 'Remote Post Enabled',
				'Value' => ! empty( $remote_settings['enable_remote_post'] ) ? 'Yes' : 'No',
			),
			array(
				'Setting' => 'Database Table',
				'Value' => $table_exists ? 'Exists' : 'Missing',
			),
		);

		WP_CLI\Utils\format_items( 'table', $info, array( 'Setting', 'Value' ) );
	}
}