<?php
/**
 * Manages all audit types.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes
 */

/**
 * Audit manager class.
 */
class Audit_Export_Audit_Manager {

	/**
	 * Registered audits.
	 *
	 * @var array
	 */
	private $audits = array();

	/**
	 * Register an audit type.
	 *
	 * @param string $id    Unique identifier for the audit.
	 * @param string $class Class name for the audit.
	 */
	public function register_audit( $id, $class ) {
		if ( class_exists( $class ) && is_subclass_of( $class, 'Audit_Export_Audit' ) ) {
			$this->audits[ $id ] = $class;
		}
	}

	/**
	 * Get all registered audits.
	 *
	 * @return array Array of audit IDs and class names.
	 */
	public function get_audits() {
		return $this->audits;
	}

	/**
	 * Get a specific audit instance.
	 *
	 * @param string $id The audit ID.
	 * @return Audit_Export_Audit|null The audit instance or null if not found.
	 */
	public function get_audit( $id ) {
		if ( isset( $this->audits[ $id ] ) ) {
			$class = $this->audits[ $id ];
			return new $class();
		}
		return null;
	}

	/**
	 * Get audit definitions.
	 *
	 * @return array Array of audit definitions.
	 */
	public function get_audit_definitions() {
		$definitions = array();
		
		foreach ( $this->audits as $id => $class ) {
			$audit = $this->get_audit( $id );
			if ( $audit ) {
				$definitions[ $id ] = array(
					'id' => $id,
					'label' => $audit->get_label(),
					'description' => $audit->get_description(),
					'group' => $audit->get_group(),
					'data_type' => $audit->get_data_type(),
				);
			}
		}
		
		return $definitions;
	}

	/**
	 * Get audits by group.
	 *
	 * @param string $group The group name.
	 * @return array Array of audit IDs in the group.
	 */
	public function get_audits_by_group( $group ) {
		$grouped_audits = array();
		
		foreach ( $this->audits as $id => $class ) {
			$audit = $this->get_audit( $id );
			if ( $audit && $audit->get_group() === $group ) {
				$grouped_audits[ $id ] = $class;
			}
		}
		
		return $grouped_audits;
	}

	/**
	 * Get all groups.
	 *
	 * @return array Array of unique group names.
	 */
	public function get_groups() {
		$groups = array();
		
		foreach ( $this->audits as $id => $class ) {
			$audit = $this->get_audit( $id );
			if ( $audit ) {
				$group = $audit->get_group();
				if ( ! in_array( $group, $groups ) ) {
					$groups[] = $group;
				}
			}
		}
		
		return $groups;
	}
}