<?php
/**
 * Users and Roles audit implementation.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/audits
 */

/**
 * Users and Roles audit class.
 */
class Audit_Export_Users_Roles extends Audit_Export_Audit {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->label = __( 'Users and Roles', 'audit-export' );
		$this->description = __( 'Audit information about users and their roles.', 'audit-export' );
		$this->group = 'users';
		$this->data_type = 'flat';
		$this->identifier = 'role';
		$this->headers = array(
			__( 'Role Name', 'audit-export' ),
			__( 'Role Display Name', 'audit-export' ),
			__( 'User Count', 'audit-export' ),
			__( 'Capabilities Count', 'audit-export' ),
			__( 'Key Capabilities', 'audit-export' ),
		);
	}

	/**
	 * Prepare data for the audit.
	 *
	 * @return array
	 */
	public function prepare_data() {
		global $wp_roles;
		
		$roles = array();
		foreach ( $wp_roles->roles as $role_name => $role_info ) {
			$roles[] = array(
				'name' => $role_name,
				'info' => $role_info,
			);
		}
		
		// Add a summary row for users without roles
		$roles[] = array(
			'name' => 'none',
			'info' => array(
				'name' => __( 'No Role', 'audit-export' ),
				'capabilities' => array(),
			),
		);
		
		return $roles;
	}

	/**
	 * Process a single data item.
	 *
	 * @param array $role The role data.
	 * @return array
	 */
	public function process_data( $role ) {
		$role_name = $role['name'];
		$role_info = $role['info'];
		
		// Get user count for this role
		$user_count = count_users();
		$count = 0;
		
		if ( $role_name === 'none' ) {
			$count = isset( $user_count['avail_roles']['none'] ) ? $user_count['avail_roles']['none'] : 0;
		} else {
			$count = isset( $user_count['avail_roles'][ $role_name ] ) ? $user_count['avail_roles'][ $role_name ] : 0;
		}
		
		// Get key capabilities
		$capabilities = array_keys( array_filter( $role_info['capabilities'] ) );
		$cap_count = count( $capabilities );
		
		// Show first 5 most important capabilities
		$key_caps = array();
		$important_caps = array( 'manage_options', 'edit_posts', 'publish_posts', 'edit_pages', 'upload_files', 'moderate_comments', 'manage_categories' );
		
		foreach ( $important_caps as $cap ) {
			if ( in_array( $cap, $capabilities ) ) {
				$key_caps[] = $cap;
				if ( count( $key_caps ) >= 5 ) {
					break;
				}
			}
		}
		
		// If we don't have 5 yet, add more
		if ( count( $key_caps ) < 5 ) {
			foreach ( $capabilities as $cap ) {
				if ( ! in_array( $cap, $key_caps ) ) {
					$key_caps[] = $cap;
					if ( count( $key_caps ) >= 5 ) {
						break;
					}
				}
			}
		}

		return array(
			$role_name,
			$role_info['name'],
			$count,
			$cap_count,
			implode( ', ', $key_caps ),
		);
	}
}