<?php
/**
 * Taxonomies audit implementation.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/audits
 */

/**
 * Taxonomies audit class.
 */
class Audit_Export_Taxonomies extends Audit_Export_Audit {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->label = __( 'Taxonomies', 'audit-export' );
		$this->description = __( 'Audit information about taxonomies and their terms.', 'audit-export' );
		$this->group = 'content';
		$this->data_type = 'flat';
		$this->identifier = 'taxonomy';
		$this->headers = array(
			__( 'Taxonomy Name', 'audit-export' ),
			__( 'Taxonomy Label', 'audit-export' ),
			__( 'Term Count', 'audit-export' ),
			__( 'Post Types', 'audit-export' ),
			__( 'Hierarchical', 'audit-export' ),
			__( 'Public', 'audit-export' ),
		);
	}

	/**
	 * Prepare data for the audit.
	 *
	 * @return array
	 */
	public function prepare_data() {
		$taxonomies = get_taxonomies( array(), 'objects' );
		$taxonomies_to_audit = array();
		
		foreach ( $taxonomies as $taxonomy ) {
			// Skip built-in taxonomies we don't want to audit
			if ( in_array( $taxonomy->name, array( 'nav_menu', 'link_category', 'post_format', 'wp_theme', 'wp_template_part_area' ) ) ) {
				continue;
			}
			
			$taxonomies_to_audit[] = $taxonomy;
		}
		
		return $taxonomies_to_audit;
	}

	/**
	 * Process a single data item.
	 *
	 * @param WP_Taxonomy $taxonomy The taxonomy object.
	 * @return array
	 */
	public function process_data( $taxonomy ) {
		// Get term count
		$term_count = wp_count_terms( array(
			'taxonomy' => $taxonomy->name,
			'hide_empty' => false,
		) );
		
		if ( is_wp_error( $term_count ) ) {
			$term_count = 0;
		}
		
		// Get associated post types
		$post_types = is_array( $taxonomy->object_type ) ? implode( ', ', $taxonomy->object_type ) : __( 'None', 'audit-export' );
		
		return array(
			$taxonomy->name,
			$taxonomy->label,
			$term_count,
			$post_types,
			$taxonomy->hierarchical ? __( 'Yes', 'audit-export' ) : __( 'No', 'audit-export' ),
			$taxonomy->public ? __( 'Yes', 'audit-export' ) : __( 'No', 'audit-export' ),
		);
	}
}