<?php
/**
 * Plugins audit implementation.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/audits
 */

/**
 * Plugins audit class.
 */
class Audit_Export_Plugins extends Audit_Export_Audit {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->label = __( 'Plugins', 'audit-export' );
		$this->description = __( 'Audit information about installed plugins.', 'audit-export' );
		$this->group = 'system';
		$this->data_type = 'flat';
		$this->identifier = 'plugin_file';
		$this->headers = array(
			__( 'Plugin Name', 'audit-export' ),
			__( 'Version', 'audit-export' ),
			__( 'Author', 'audit-export' ),
			__( 'Status', 'audit-export' ),
			__( 'Network Active', 'audit-export' ),
			__( 'Update Available', 'audit-export' ),
		);
	}

	/**
	 * Prepare data for the audit.
	 *
	 * @return array
	 */
	public function prepare_data() {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		
		$all_plugins = get_plugins();
		$plugin_data = array();
		
		foreach ( $all_plugins as $plugin_file => $plugin_info ) {
			$plugin_data[] = array(
				'file' => $plugin_file,
				'info' => $plugin_info,
			);
		}
		
		return $plugin_data;
	}

	/**
	 * Process a single data item.
	 *
	 * @param array $plugin The plugin data.
	 * @return array
	 */
	public function process_data( $plugin ) {
		$plugin_file = $plugin['file'];
		$plugin_info = $plugin['info'];
		
		// Check if plugin is active
		$is_active = is_plugin_active( $plugin_file );
		$is_network_active = false;
		
		if ( is_multisite() ) {
			$is_network_active = is_plugin_active_for_network( $plugin_file );
		}
		
		// Determine status
		if ( $is_network_active ) {
			$status = __( 'Network Active', 'audit-export' );
		} elseif ( $is_active ) {
			$status = __( 'Active', 'audit-export' );
		} else {
			$status = __( 'Inactive', 'audit-export' );
		}
		
		// Check for updates
		$update_available = __( 'No', 'audit-export' );
		$update_plugins = get_site_transient( 'update_plugins' );
		
		if ( isset( $update_plugins->response[ $plugin_file ] ) ) {
			$update_available = __( 'Yes', 'audit-export' );
		}

		return array(
			$plugin_info['Name'],
			$plugin_info['Version'],
			$plugin_info['Author'],
			$status,
			$is_network_active ? __( 'Yes', 'audit-export' ) : __( 'No', 'audit-export' ),
			$update_available,
		);
	}
}