<?php
/**
 * Abstract base class for all audit types.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/audits
 */

/**
 * Abstract audit class.
 */
abstract class Audit_Export_Audit {

	/**
	 * The audit label.
	 *
	 * @var string
	 */
	protected $label = '';

	/**
	 * The audit description.
	 *
	 * @var string
	 */
	protected $description = '';

	/**
	 * The audit group.
	 *
	 * @var string
	 */
	protected $group = 'general';

	/**
	 * The data type (flat, process, cross).
	 *
	 * @var string
	 */
	protected $data_type = 'flat';

	/**
	 * The identifier field.
	 *
	 * @var string
	 */
	protected $identifier = 'id';

	/**
	 * The report headers.
	 *
	 * @var array
	 */
	protected $headers = array();

	/**
	 * Prepare data for the audit.
	 *
	 * @return array Array of items to process.
	 */
	abstract public function prepare_data();

	/**
	 * Process a single data item.
	 *
	 * @param mixed $item The item to process.
	 * @return array The processed data row.
	 */
	abstract public function process_data( $item );

	/**
	 * Get the audit label.
	 *
	 * @return string
	 */
	public function get_label() {
		return $this->label;
	}

	/**
	 * Get the audit description.
	 *
	 * @return string
	 */
	public function get_description() {
		return $this->description;
	}

	/**
	 * Get the audit group.
	 *
	 * @return string
	 */
	public function get_group() {
		return $this->group;
	}

	/**
	 * Get the data type.
	 *
	 * @return string
	 */
	public function get_data_type() {
		return $this->data_type;
	}

	/**
	 * Get the identifier field.
	 *
	 * @return string
	 */
	public function get_identifier() {
		return $this->identifier;
	}

	/**
	 * Get the report headers.
	 *
	 * @return array
	 */
	public function get_headers() {
		return $this->headers;
	}

	/**
	 * Set the report headers.
	 *
	 * @param array $headers The headers to set.
	 */
	public function set_headers( $headers ) {
		$this->headers = $headers;
	}

	/**
	 * Run the complete audit.
	 *
	 * @return array The complete audit data.
	 */
	public function run() {
		$data = array();
		$items = $this->prepare_data();
		
		if ( $this->data_type === 'process' ) {
			// For process type, items contain the data structure
			foreach ( $items as $item ) {
				$data[] = $this->process_data( $item );
			}
		} else {
			// For flat type, process each item
			foreach ( $items as $item ) {
				$data[] = $this->process_data( $item );
			}
		}
		
		return $data;
	}

	/**
	 * Get the audit data as an associative array.
	 *
	 * @return array
	 */
	public function get_audit_data() {
		return array(
			'label' => $this->get_label(),
			'description' => $this->get_description(),
			'group' => $this->get_group(),
			'data_type' => $this->get_data_type(),
			'identifier' => $this->get_identifier(),
			'headers' => $this->get_headers(),
			'data' => $this->run(),
		);
	}
}