/**
 * Admin JavaScript for Audit Export plugin.
 */

(function($) {
	'use strict';

	/**
	 * Audit Export Admin object
	 */
	var AuditExportAdmin = {
		
		/**
		 * Initialize the admin interface
		 */
		init: function() {
			this.bindEvents();
			this.initTooltips();
			this.initConditionalFields();
		},

		/**
		 * Bind event handlers
		 */
		bindEvents: function() {
			// Process audit buttons
			$(document).on('click', '.process-audit', this.processAudit);
			
			// Process all audits button
			$(document).on('click', '#process-all-audits', this.processAllAudits);
			
			// Test connection button
			$(document).on('click', '#test-connection', this.testConnection);
			
			// Form submission handlers
			$(document).on('submit', 'form[data-audit-export]', this.handleFormSubmit);
			
			// Settings form handlers
			$(document).on('change', 'input[name="audit_export_settings[enable_cron]"]', this.toggleCronSettings);
			$(document).on('change', 'input[name="audit_export_remote_post_settings[enable_remote_post]"]', this.toggleRemotePostSettings);
			$(document).on('change', 'input[name="audit_export_remote_post_settings[authentication_type]"]', this.toggleAuthFields);
		},

		/**
		 * Initialize tooltips
		 */
		initTooltips: function() {
			$('[data-tooltip]').each(function() {
				var $element = $(this);
				var tooltip = $element.data('tooltip');
				
				$element.attr('title', tooltip);
			});
		},

		/**
		 * Initialize conditional field display
		 */
		initConditionalFields: function() {
			// Trigger initial state
			this.toggleCronSettings();
			this.toggleRemotePostSettings();
			this.toggleAuthFields();
		},

		/**
		 * Process a single audit
		 */
		processAudit: function(e) {
			e.preventDefault();
			
			var $button = $(this);
			var auditId = $button.data('audit-id');
			var originalText = $button.text();
			
			if (!auditId) {
				AuditExportAdmin.showError('No audit ID specified.');
				return;
			}
			
			// Disable button and show processing state
			$button.prop('disabled', true)
				.text(audit_export_ajax.strings.processing)
				.addClass('processing');
			
			$.post(audit_export_ajax.ajax_url, {
				action: 'audit_export_process',
				audit_id: auditId,
				nonce: audit_export_ajax.nonce
			})
			.done(function(response) {
				if (response.success) {
					$button.text(audit_export_ajax.strings.completed)
						.removeClass('processing')
						.addClass('button-primary');
					
					AuditExportAdmin.showSuccess(response.data.message);
					
					// Reload page after a delay to show updated data
					setTimeout(function() {
						window.location.reload();
					}, 2000);
				} else {
					AuditExportAdmin.showError(response.data.message || 'Unknown error occurred.');
					AuditExportAdmin.resetButton($button, originalText);
				}
			})
			.fail(function(xhr, status, error) {
				AuditExportAdmin.showError('Request failed: ' + error);
				AuditExportAdmin.resetButton($button, originalText);
			});
		},

		/**
		 * Process all audits
		 */
		processAllAudits: function(e) {
			e.preventDefault();
			
			if (!confirm(audit_export_ajax.strings.confirm_process_all)) {
				return;
			}
			
			var $button = $(this);
			var originalText = $button.text();
			
			// Disable button and show processing state
			$button.prop('disabled', true)
				.text(audit_export_ajax.strings.processing)
				.addClass('processing');
			
			// Disable all individual audit buttons
			$('.process-audit').prop('disabled', true).addClass('processing');
			
			$.post(audit_export_ajax.ajax_url, {
				action: 'audit_export_process',
				process_all: 'true',
				nonce: audit_export_ajax.nonce
			})
			.done(function(response) {
				if (response.success) {
					$button.text(audit_export_ajax.strings.completed)
						.removeClass('processing')
						.addClass('button-primary');
					
					AuditExportAdmin.showSuccess(response.data.message);
					
					// Reload page after a delay
					setTimeout(function() {
						window.location.reload();
					}, 3000);
				} else {
					AuditExportAdmin.showError(response.data.message || 'Unknown error occurred.');
					AuditExportAdmin.resetButton($button, originalText);
					$('.process-audit').prop('disabled', false).removeClass('processing');
				}
			})
			.fail(function(xhr, status, error) {
				AuditExportAdmin.showError('Request failed: ' + error);
				AuditExportAdmin.resetButton($button, originalText);
				$('.process-audit').prop('disabled', false).removeClass('processing');
			});
		},

		/**
		 * Test remote connection
		 */
		testConnection: function(e) {
			e.preventDefault();
			
			var $button = $(this);
			var $result = $('#test-result');
			var originalText = $button.text();
			
			$button.prop('disabled', true).text('Testing...');
			$result.html('');
			
			$.post(audit_export_ajax.ajax_url, {
				action: 'audit_export_test_connection',
				nonce: audit_export_ajax.nonce
			})
			.done(function(response) {
				if (response.success) {
					$result.html('<div class="notice notice-success inline"><p>' + response.data.message + '</p></div>');
				} else {
					$result.html('<div class="notice notice-error inline"><p>' + response.data.message + '</p></div>');
				}
			})
			.fail(function() {
				$result.html('<div class="notice notice-error inline"><p>Connection test failed.</p></div>');
			})
			.always(function() {
				$button.prop('disabled', false).text(originalText);
			});
		},

		/**
		 * Handle form submissions
		 */
		handleFormSubmit: function(e) {
			var $form = $(this);
			var $submitButton = $form.find('input[type="submit"], button[type="submit"]');
			
			if ($submitButton.length) {
				$submitButton.prop('disabled', true).addClass('processing');
				
				// Re-enable after a delay to prevent double submission
				setTimeout(function() {
					$submitButton.prop('disabled', false).removeClass('processing');
				}, 2000);
			}
		},

		/**
		 * Toggle cron settings display
		 */
		toggleCronSettings: function() {
			var $checkbox = $('input[name="audit_export_settings[enable_cron]"]');
			var $cronFields = $('.cron-dependent');
			
			if ($checkbox.length) {
				var isEnabled = $checkbox.is(':checked');
				
				// Show/hide dependent fields
				$cronFields.closest('tr').toggle(isEnabled);
				
				// Handle run on every cron option
				var $runEveryCheckbox = $('input[name="audit_export_settings[run_on_every_cron]"]');
				var $frequencyField = $('input[name="audit_export_settings[cron_frequency]"]');
				
				if ($runEveryCheckbox.length && $frequencyField.length) {
					var showFrequency = isEnabled && !$runEveryCheckbox.is(':checked');
					$frequencyField.closest('tr').toggle(showFrequency);
				}
			}
		},

		/**
		 * Toggle remote post settings display
		 */
		toggleRemotePostSettings: function() {
			var $checkbox = $('input[name="audit_export_remote_post_settings[enable_remote_post]"]');
			var $remoteFields = $('.remote-post-dependent');
			
			if ($checkbox.length) {
				var isEnabled = $checkbox.is(':checked');
				$remoteFields.closest('tr').toggle(isEnabled);
			}
		},

		/**
		 * Toggle authentication fields
		 */
		toggleAuthFields: function() {
			var $authType = $('input[name="audit_export_remote_post_settings[authentication_type]"]:checked');
			
			if ($authType.length) {
				var authType = $authType.val();
				
				// Hide all auth fields first
				$('.auth-basic, .auth-bearer').closest('tr').hide();
				
				// Show relevant fields
				if (authType === 'basic') {
					$('.auth-basic').closest('tr').show();
				} else if (authType === 'bearer') {
					$('.auth-bearer').closest('tr').show();
				}
			}
		},

		/**
		 * Reset button to original state
		 */
		resetButton: function($button, originalText) {
			$button.prop('disabled', false)
				.text(originalText)
				.removeClass('processing button-primary');
		},

		/**
		 * Show success message
		 */
		showSuccess: function(message) {
			this.showNotice(message, 'success');
		},

		/**
		 * Show error message
		 */
		showError: function(message) {
			this.showNotice(message, 'error');
		},

		/**
		 * Show notice
		 */
		showNotice: function(message, type) {
			var $notice = $('<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p></div>');
			
			// Insert after h1 or at the top of .wrap
			var $target = $('.wrap h1').first();
			if ($target.length) {
				$target.after($notice);
			} else {
				$('.wrap').prepend($notice);
			}
			
			// Auto-dismiss after 5 seconds
			setTimeout(function() {
				$notice.fadeOut(function() {
					$(this).remove();
				});
			}, 5000);
			
			// Scroll to notice
			$('html, body').animate({
				scrollTop: $notice.offset().top - 100
			}, 300);
		},

		/**
		 * Show loading overlay
		 */
		showLoading: function($container) {
			if (!$container) {
				$container = $('.wrap').first();
			}
			
			var $overlay = $('<div class="audit-export-loading"><div class="spinner is-active"></div></div>');
			$container.css('position', 'relative').append($overlay);
		},

		/**
		 * Hide loading overlay
		 */
		hideLoading: function($container) {
			if (!$container) {
				$container = $('.wrap').first();
			}
			
			$container.find('.audit-export-loading').remove();
		}
	};

	/**
	 * Initialize when document is ready
	 */
	$(document).ready(function() {
		AuditExportAdmin.init();
	});

	/**
	 * Make AuditExportAdmin globally available
	 */
	window.AuditExportAdmin = AuditExportAdmin;

})(jQuery);