<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/admin
 */

/**
 * The admin-specific functionality of the plugin.
 */
class Audit_Export_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * The audit manager instance.
	 *
	 * @access   private
	 * @var      Audit_Export_Audit_Manager    $audit_manager    The audit manager.
	 */
	private $audit_manager;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string                         $plugin_name    The name of this plugin.
	 * @param string                         $version        The version of this plugin.
	 * @param Audit_Export_Audit_Manager     $audit_manager  The audit manager instance.
	 */
	public function __construct( $plugin_name, $version, $audit_manager ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->audit_manager = $audit_manager;
	}

	/**
	 * Register the stylesheets for the admin area.
	 */
	public function enqueue_styles() {
		$screen = get_current_screen();
		
		// Only load on our admin pages
		if ( strpos( $screen->id, 'audit-export' ) !== false ) {
			wp_enqueue_style( 
				$this->plugin_name, 
				AUDIT_EXPORT_PLUGIN_URL . 'admin/css/audit-export-admin.css', 
				array(), 
				$this->version, 
				'all' 
			);
		}
	}

	/**
	 * Register the JavaScript for the admin area.
	 */
	public function enqueue_scripts() {
		$screen = get_current_screen();
		
		// Only load on our admin pages
		if ( strpos( $screen->id, 'audit-export' ) !== false ) {
			wp_enqueue_script( 
				$this->plugin_name, 
				AUDIT_EXPORT_PLUGIN_URL . 'admin/js/audit-export-admin.js', 
				array( 'jquery' ), 
				$this->version, 
				true 
			);

			// Enqueue the inline scripts handler
			wp_enqueue_script(
				$this->plugin_name . '-inline',
				AUDIT_EXPORT_PLUGIN_URL . 'admin/js/audit-export-admin-inline.js',
				array( 'jquery', $this->plugin_name ),
				$this->version,
				true
			);
			
			// Localize script with all necessary data
			wp_localize_script( $this->plugin_name . '-inline', 'audit_export_ajax', array(
				'ajax_url'              => admin_url( 'admin-ajax.php' ),
				'nonce'                 => wp_create_nonce( 'audit_export_nonce' ),
				'test_connection_nonce' => wp_create_nonce( 'audit_export_test_connection' ),
				'strings'               => array(
					'processing'          => __( 'Processing...', 'audit-export' ),
					'completed'           => __( 'Completed', 'audit-export' ),
					'error'               => __( 'Error processing audit.', 'audit-export' ),
					'confirm_process_all' => __( 'Are you sure you want to process all audits? This may take some time.', 'audit-export' ),
					'testing'             => __( 'Testing...', 'audit-export' ),
					'connection_failed'   => __( 'Connection test failed.', 'audit-export' ),
				),
			) );
		}
	}

	/**
	 * Add admin menu items.
	 */
	public function add_admin_menu() {
		// Main menu page
		add_menu_page(
			__( 'Audit Export', 'audit-export' ),
			__( 'Audit Export', 'audit-export' ),
			'audit_export_view_reports',
			'audit-export',
			array( $this, 'display_reports_page' ),
			'dashicons-chart-bar',
			30
		);

		// Reports submenu (same as main page)
		add_submenu_page(
			'audit-export',
			__( 'Audit Reports', 'audit-export' ),
			__( 'Reports', 'audit-export' ),
			'audit_export_view_reports',
			'audit-export',
			array( $this, 'display_reports_page' )
		);

		// Settings submenu
		add_submenu_page(
			'audit-export',
			__( 'Audit Export Settings', 'audit-export' ),
			__( 'Settings', 'audit-export' ),
			'audit_export_manage_settings',
			'audit-export-settings',
			array( $this, 'display_settings_page' )
		);

		// Tools submenu
		add_submenu_page(
			'audit-export',
			__( 'Audit Export Tools', 'audit-export' ),
			__( 'Tools', 'audit-export' ),
			'audit_export_process_reports',
			'audit-export-tools',
			array( $this, 'display_tools_page' )
		);
	}

	/**
	 * Display the reports page.
	 */
	public function display_reports_page() {
		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- This is read-only display logic.
		$group = isset( $_GET['group'] ) ? sanitize_text_field( wp_unslash( $_GET['group'] ) ) : '';
		$audit_id = isset( $_GET['audit'] ) ? sanitize_text_field( wp_unslash( $_GET['audit'] ) ) : '';
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		if ( ! empty( $audit_id ) ) {
			$this->display_single_report( $audit_id );
		} elseif ( ! empty( $group ) ) {
			$this->display_group_reports( $group );
		} else {
			$this->display_reports_overview();
		}
	}

	/**
	 * Display reports overview.
	 */
	private function display_reports_overview() {
		$audit_definitions = $this->audit_manager->get_audit_definitions();
		$report_service = new Audit_Export_Report();
		$available_reports = $report_service->get_available_reports();
		
		// Group audits by group
		$grouped_audits = array();
		foreach ( $audit_definitions as $audit_id => $definition ) {
			$group = $definition['group'];
			if ( ! isset( $grouped_audits[ $group ] ) ) {
				$grouped_audits[ $group ] = array();
			}
			$grouped_audits[ $group ][ $audit_id ] = $definition;
		}

		include AUDIT_EXPORT_PLUGIN_DIR . 'admin/partials/audit-export-reports-overview.php';
	}

	/**
	 * Display group reports.
	 *
	 * @param string $group The group name.
	 */
	private function display_group_reports( $group ) {
		$audits = $this->audit_manager->get_audits_by_group( $group );
		$audit_definitions = $this->audit_manager->get_audit_definitions();
		$report_service = new Audit_Export_Report();

		include AUDIT_EXPORT_PLUGIN_DIR . 'admin/partials/audit-export-group-reports.php';
	}

	/**
	 * Display single report.
	 *
	 * @param string $audit_id The audit ID.
	 */
	private function display_single_report( $audit_id ) {
		$audit = $this->audit_manager->get_audit( $audit_id );
		$report_service = new Audit_Export_Report();
		$report_data = $report_service->get_report_data( $audit_id );
		$last_processed = $report_service->get_last_processed_date( $audit_id );

		if ( ! $audit ) {
			wp_die( esc_html__( 'Audit not found.', 'audit-export' ) );
		}

		include AUDIT_EXPORT_PLUGIN_DIR . 'admin/partials/audit-export-single-report.php';
	}

	/**
	 * Display the settings page.
	 */
	public function display_settings_page() {
		include AUDIT_EXPORT_PLUGIN_DIR . 'admin/partials/audit-export-settings.php';
	}

	/**
	 * Display the tools page.
	 */
	public function display_tools_page() {
		$cron_service = new Audit_Export_Cron( $this->audit_manager );
		$queue_service = new Audit_Export_Queue( $this->audit_manager );
		$remote_post_service = new Audit_Export_Remote_Post();
		
		// Handle tool actions
		if ( isset( $_POST['action'], $_POST['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'audit_export_tools' ) ) {
			$this->handle_tools_action( sanitize_text_field( wp_unslash( $_POST['action'] ) ), $cron_service, $queue_service, $remote_post_service );
		}

		include AUDIT_EXPORT_PLUGIN_DIR . 'admin/partials/audit-export-tools.php';
	}

	/**
	 * Handle tools page actions.
	 *
	 * @param string                       $action             The action to handle.
	 * @param Audit_Export_Cron            $cron_service       The cron service.
	 * @param Audit_Export_Queue           $queue_service      The queue service.
	 * @param Audit_Export_Remote_Post     $remote_post_service The remote post service.
	 */
	private function handle_tools_action( $action, $cron_service, $queue_service, $remote_post_service ) {
		switch ( $action ) {
			case 'force_cron':
				$cron_service->force_run_now();
				add_settings_error( 'audit_export_tools', 'cron_forced', esc_html__( 'Cron run forced successfully.', 'audit-export' ), 'updated' );
				break;

			case 'clear_queue':
				$queue_service->clear_queue();
				add_settings_error( 'audit_export_tools', 'queue_cleared', esc_html__( 'Queue cleared successfully.', 'audit-export' ), 'updated' );
				break;

			case 'process_queue':
				$processed = $queue_service->process_all_immediate( 5 );
				/* translators: %d: number of items processed */
				add_settings_error( 'audit_export_tools', 'queue_processed', sprintf( esc_html__( 'Processed %d queue items.', 'audit-export' ), $processed ), 'updated' );
				break;

			case 'test_connection':
				$result = $remote_post_service->test_connection();
				$type = $result['success'] ? 'updated' : 'error';
				add_settings_error( 'audit_export_tools', 'connection_tested', $result['message'], $type );
				break;
		}
	}

	/**
	 * AJAX handler for processing audits.
	 */
	public function ajax_process_audit() {
		// Verify nonce
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'audit_export_nonce' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'audit-export' ) );
		}

		// Check permissions
		if ( ! current_user_can( 'audit_export_process_reports' ) ) {
			wp_die( esc_html__( 'Insufficient permissions.', 'audit-export' ) );
		}

		$audit_id = isset( $_POST['audit_id'] ) ? sanitize_text_field( wp_unslash( $_POST['audit_id'] ) ) : '';
		$process_all = isset( $_POST['process_all'] ) && $_POST['process_all'] === 'true';

		$report_service = new Audit_Export_Report();

		if ( $process_all ) {
			$audits = $this->audit_manager->get_audits();
			$results = array();
			$success_count = 0;

			foreach ( $audits as $id => $class ) {
				$result = $report_service->process_audit( $id, $this->audit_manager );
				$results[ $id ] = $result;
				if ( $result['success'] ) {
					$success_count++;
				}
			}

			wp_send_json_success( array(
				'message' => sprintf(
					/* translators: 1: number of successful audits, 2: total number of audits */
					esc_html__( 'Processed %1$d of %2$d audits successfully.', 'audit-export' ),
					$success_count,
					count( $audits )
				),
				'results' => $results,
			) );
		} else {
			$result = $report_service->process_audit( $audit_id, $this->audit_manager );
			
			if ( $result['success'] ) {
				wp_send_json_success( $result );
			} else {
				wp_send_json_error( $result );
			}
		}
	}

	/**
	 * AJAX handler for downloading reports.
	 */
	public function ajax_download_report() {
		// Verify nonce
		if ( ! isset( $_GET['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['nonce'] ) ), 'audit_export_nonce' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'audit-export' ) );
		}

		// Check permissions
		if ( ! current_user_can( 'audit_export_view_reports' ) ) {
			wp_die( esc_html__( 'Insufficient permissions.', 'audit-export' ) );
		}

		$audit_id = isset( $_GET['audit_id'] ) ? sanitize_text_field( wp_unslash( $_GET['audit_id'] ) ) : '';
		$audit = $this->audit_manager->get_audit( $audit_id );
		$report_service = new Audit_Export_Report();
		$report_data = $report_service->get_report_data( $audit_id );

		if ( ! $audit || ! $report_data ) {
			wp_die( esc_html__( 'Report not found.', 'audit-export' ) );
		}

		// Generate CSV
		$headers = isset( $report_data['headers'] ) ? $report_data['headers'] : $audit->get_headers();
		$data = isset( $report_data['data'] ) ? $report_data['data'] : array();
		$csv_content = $report_service->export_as_csv( $audit_id, $data, $headers );

		// Set headers for download
		$filename = sanitize_file_name( $audit_id . '_' . gmdate( 'Y-m-d_H-i-s' ) . '.csv' );
		
		header( 'Content-Type: text/csv' );
		header( 'Content-Disposition: attachment; filename="' . $filename . '"' );
		header( 'Content-Length: ' . strlen( $csv_content ) );
		
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSV content is already properly formatted.
		echo $csv_content;
		exit;
	}
}