<?php

defined('WPINC') or die;

/**
 * Template loader for Atomic Events Plugin.
 *
 * Only need to specify class properties here.
 *
 */
class ATMCEVNTS_Template_Loader extends ATMCEVNTS_Base_Template_Loader {

    /**
     * Prefix for filter names.
     *
     * @since 1.0.0
     * @type string
     */
    protected $filter_prefix = 'atmcevnts';

    /**
     * Directory name where custom templates for this plugin should be found in the theme.
     *
     * @since 1.0.0
     * @type string
     */
    protected $theme_template_directory = 'atomic-events';

    /**
     * Reference to the root directory path of this plugin.
     *
     * @since 1.0.0
     * @type string
     */
    protected $plugin_directory = ATMCEVNTS_PLUGIN_DIR;

    /**
     * Return a list of paths to check for template locations with theme support.
     *
     * Updated to support the new theme-based structure:
     * 1. Child theme override: child-theme/atomic-events/
     * 2. Parent theme override: parent-theme/atomic-events/
     * 3. Child theme selected theme: child-theme/atomic-events/themes/{theme}/
     * 4. Parent theme selected theme: parent-theme/atomic-events/themes/{theme}/
     * 5. Plugin selected theme: plugin/templates/themes/{theme}/
     * 6. Plugin default theme: plugin/templates/themes/default/
     *
     * @since 1.0.0
     *
     * @return array
     */
    protected function get_template_paths() {
        $selected_theme = ATMCEVNTS_Theme_Manager::get_selected_theme();
        $theme_directory = trailingslashit($this->theme_template_directory);

        $file_paths = array(
            // 1. Child theme override (highest priority)
            1 => trailingslashit(get_stylesheet_directory()) . $theme_directory,
            
            // 2. Parent theme override
            10 => trailingslashit(get_template_directory()) . $theme_directory,
            
            // 3. Child theme selected theme
            20 => trailingslashit(get_stylesheet_directory()) . $theme_directory . 'themes/' . $selected_theme . '/',
            
            // 4. Parent theme selected theme
            30 => trailingslashit(get_template_directory()) . $theme_directory . 'themes/' . $selected_theme . '/',
            
            // 5. Plugin selected theme
            40 => trailingslashit($this->get_templates_dir()) . 'themes/' . $selected_theme . '/',
            
            // 6. Plugin default theme (fallback)
            100 => trailingslashit($this->get_templates_dir()) . 'themes/default/',
        );

        // Only add child theme paths if child theme is different from parent theme
        if (get_stylesheet_directory() === get_template_directory()) {
            unset($file_paths[1]); // Remove child theme override
            unset($file_paths[20]); // Remove child theme selected theme
        }

        /**
         * Allow ordered list of template paths to be amended.
         *
         * @since 1.0.0
         *
         * @param array $file_paths Default is directory in child theme at index 1, parent theme at 10, and plugin at 100.
         */
        $file_paths = apply_filters('atmcevnts_template_paths', $file_paths);

        // Sort the file paths based on priority.
        ksort($file_paths, SORT_NUMERIC);

        return array_map('trailingslashit', $file_paths);
    }
}
