<?php
defined('WPINC') or die;
class ATMCEVNTS_Shortcode {

    public function __construct() {
    }

    public function register_shortcodes() {
        add_shortcode('atomic-events', array($this, 'render_atomic_events_shortcode'));
        
        // Add cache invalidation hooks
        add_action('save_post', array($this, 'invalidate_shortcode_cache'), 10, 2);
        add_action('deleted_post', array($this, 'invalidate_shortcode_cache'), 10, 1);
        add_action('wp_update_nav_menu', array($this, 'invalidate_shortcode_cache_all'));
        add_action('switch_theme', array($this, 'invalidate_shortcode_cache_all'));
    }

    /**
     * Invalidate shortcode cache when an event is saved or deleted
     */
    public function invalidate_shortcode_cache($post_id, $post = null) {
        // Only invalidate cache for atomic-event post type
        if ($post && $post->post_type !== 'atomic-event') {
            return;
        }
        
        // If post is null (deleted), check if it was an atomic-event
        if (!$post) {
            $post_type = get_post_type($post_id);
            if ($post_type !== 'atomic-event') {
                return;
            }
        }
        
        $this->clear_all_shortcode_cache();
    }

    /**
     * Invalidate all shortcode cache (for theme switches, menu updates, etc.)
     */
    public function invalidate_shortcode_cache_all() {
        $this->clear_all_shortcode_cache();
    }

    /**
     * Clear all atomic events shortcode cache
     */
    private function clear_all_shortcode_cache() {
        // Use the EventQueryService to clear cache
        $event_query_service = new ATMCEVNTS_Event_Query_Service();
        $event_query_service->clear_cache();
    }

    public function render_atomic_events_shortcode($atts) {
        // Shortcode attributes
        $atts = shortcode_atts(array(
            'filter' => 'future', // Default filter
            'max' => 20, // Default maximum number of posts
            'columns' => 3, // Default number of columns
            'featured' => null, // Default to show all events
            'skip_first_x' => 0, // Default to not skip any events
            'skip_first_x_featured' => 0, // Default to not skip any featured events
            'order' => 'ASC' // Default order
        ), $atts);

        // Use the EventQueryService to get events
        $event_query_service = new ATMCEVNTS_Event_Query_Service();
        $result = $event_query_service->get_events($atts);
        $events = $result['events'];

        // Output buffer to capture template
        ob_start();

        // Check if we have events
        if (!empty($events)) {
            $template_loader = new ATMCEVNTS_Template_Loader;
            
            // Collect events for structured data
            $events_for_structured_data = [];

            // Determine if this is a featured-only query
            $featured_only = ($atts['featured'] !== null && filter_var($atts['featured'], FILTER_VALIDATE_BOOLEAN));

            /**
             * Load List Header
             */
            if ($featured_only) {
                $template_loader
                    ->set_template_data(['columns' => $atts['columns']], 'header_atts')
                    ->get_template_part('cards/card-list-header', 'featured');
            } else {
                $template_loader
                    ->set_template_data(['columns' => $atts['columns']], 'header_atts')
                    ->get_template_part('cards/card-list-header', 'default');
            }

            /**
             * Load List content
             */
            $counter = 0;
            foreach ($events as $event_model) {
                $counter++;
                
                // Add to structured data collection
                $events_for_structured_data[] = $event_model;

                // Load template part
                $tpl_suffix = $featured_only ? 'featured' : 'default';

                $template_loader
                    ->set_template_data($event_model, 'event')
                    ->set_template_data(['item_nr' => $counter], 'counter')
                    ->get_template_part('cards/card-item', $tpl_suffix);
            }

            /**
             * Load List Footer
             */
            if ($featured_only) {
                $template_loader->get_template_part('cards/card-list-footer', 'featured');
            } else {
                $template_loader->get_template_part('cards/card-list-footer', 'default');
            }

            // Add structured data for the events listing
            if (!empty($events_for_structured_data)) {
                $structured_data_service = new ATMCEVNTS_Structured_Data_Service();
                $structured_data_service->print_events_listing_json_ld($events_for_structured_data);
            }
            

        } else {
            echo 'No events found.';
        }

        // Return output
        return ob_get_clean();
    }
    

}
