<?php
/**
 * Structured Data Service
 * 
 * Handles generation of structured data (JSON-LD) for events
 * 
 * @package AtomicEvents
 * @since 1.0.0
 */

defined('WPINC') or die;

class ATMCEVNTS_Structured_Data_Service {
    
    /**
     * Generate JSON-LD structured data for an event
     * 
     * @param ATMCEVNTS_Event_Model $event The event model
     * @return array JSON-LD structured data array
     */
    public function generate_event_json_ld($event) {
        $data = [
            '@context' => 'https://schema.org',
            '@type' => 'Event',
            'name' => $event->get_title(),
            'description' => $event->get_excerpt() ?: $event->get_title(),
            'startDate' => $event->get_event_start_date(),
            'url' => $event->get_permalink(),
            'organizer' => [
                '@type' => 'Organization',
                'name' => get_bloginfo('name')
            ]
        ];
        
        // Add end date if available
        if (!empty($event->get_event_end_date())) {
            $data['endDate'] = $event->get_event_end_date();
        }
        
        // Add location if venue is available
        if (!empty($event->get_event_venue())) {
            $data['location'] = $this->build_location_data($event);
        }
        
        // Add image if available
        if (!empty($event->get_event_image_id())) {
            $data['image'] = wp_get_attachment_image_url($event->get_event_image_id(), 'full');
        }
        
        return $data;
    }
    
    /**
     * Build location data for structured data
     * 
     * @param ATMCEVNTS_Event_Model $event The event model
     * @return array Location data
     */
    private function build_location_data($event) {
        $location = [
            '@type' => 'Place',
            'name' => $event->get_event_venue(),
            'address' => [
                '@type' => 'PostalAddress'
            ]
        ];
        
        // Add address components if available
        if (!empty($event->get_event_street())) {
            $location['address']['streetAddress'] = $event->get_event_street();
        }
        
        if (!empty($event->get_event_location_city())) {
            $location['address']['addressLocality'] = $event->get_event_location_city();
        }
        
        if (!empty($event->get_event_location_region())) {
            $location['address']['addressRegion'] = $event->get_event_location_region();
        }
        
        if (!empty($event->get_event_postcode())) {
            $location['address']['postalCode'] = $event->get_event_postcode();
        }
        
        if (!empty($event->get_event_country())) {
            $location['address']['addressCountry'] = $event->get_event_country();
        }
        
        return $location;
    }
    
    /**
     * Print JSON-LD script tag for an event
     * 
     * @param ATMCEVNTS_Event_Model $event The event model
     */
    public function print_event_json_ld($event) {
        $data = $this->generate_event_json_ld($event);
        echo '<script type="application/ld+json">';
        echo wp_json_encode($data);
        echo '</script>';
    }
    
    /**
     * Generate JSON-LD for multiple events (for listing pages)
     * 
     * @param array $events Array of ATMCEVNTS_Event_Model objects
     * @return array JSON-LD structured data array
     */
    public function generate_events_listing_json_ld($events) {
        $data = [
            '@context' => 'https://schema.org',
            '@type' => 'ItemList',
            'itemListElement' => []
        ];
        
        foreach ($events as $index => $event) {
            $data['itemListElement'][] = [
                '@type' => 'ListItem',
                'position' => $index + 1,
                'item' => [
                    '@type' => 'Event',
                    'name' => $event->get_title(),
                    'description' => $event->get_excerpt() ?: $event->get_title(),
                    'startDate' => $event->get_event_start_date(),
                    'url' => $event->get_permalink()
                ]
            ];
        }
        
        return $data;
    }
    
    /**
     * Print JSON-LD script tag for events listing
     * 
     * @param array $events Array of ATMCEVNTS_Event_Model objects
     */
    public function print_events_listing_json_ld($events) {
        $data = $this->generate_events_listing_json_ld($events);
        echo '<script type="application/ld+json">';
        echo wp_json_encode($data);
        echo '</script>';
    }
}
