<?php
defined('WPINC') or die;
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://matyus.me
 * @since      1.0.0
 *
 * @package    Atomic_Events
 * @subpackage Atomic_Events/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Atomic_Events
 * @subpackage Atomic_Events/includes
 * @author     Lehel Matyus <contact@matyus.me>
 */
class ATMCEVNTS_Atomic_Events {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Atomic_Events_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if (defined('ATMCEVNTS_VERSION')) {
			$this->version = ATMCEVNTS_VERSION;
		} else {
			$this->version = ATMCEVNTS_VERSION;
		}
		$this->plugin_name = 'atomic-events-calendar';

		$this->load_dependencies();
		$this->define_admin_hooks();
		$this->define_public_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Atomic_Events_Loader. Orchestrates the hooks of the plugin.
	 * - Atomic_Events_i18n. Defines internationalization functionality.
	 * - Atomic_Events_Admin. Defines all hooks for the admin area.
	 * - Atomic_Events_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		// utils
		require ATMCEVNTS_PLUGIN_DIR . 'utils/class-template-loader.php';
		require ATMCEVNTS_PLUGIN_DIR . 'utils/class-atmc-template-loader.php';
		require ATMCEVNTS_PLUGIN_DIR . 'utils/class-atomic-event-date-utils.php';
		require ATMCEVNTS_PLUGIN_DIR . 'utils/class-atomic-template-utils.php';
		require ATMCEVNTS_PLUGIN_DIR . 'utils/class-atomic-member-utils.php';
		require ATMCEVNTS_PLUGIN_DIR . 'utils/class-atomic-events-image.php';
		require ATMCEVNTS_PLUGIN_DIR . 'utils/class-atomic-event-link.php';
		require ATMCEVNTS_PLUGIN_DIR . 'utils/class-atomic-image.php';

		// require_once plugin_dir_path(dirname(__FILE__)) . 'utils/atmc_vnts_utils.php';

		// models
		require_once plugin_dir_path(dirname(__FILE__)) . 'models/class-atomic-events-model.php';

		// ahortcodes
		require_once plugin_dir_path(dirname(__FILE__)) . 'shortcodes/atomic-events-shortcode.php';


		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/post-types/class-event-post-type.php';
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/post-types/class-event-taxonomy.php';
		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-atomic-events-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-atomic-events-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-atomic-events-admin.php';
		
		/**
		 * The class responsible for plugin settings with tabbed interface.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-atomic-events-settings.php';

		/**
		 * The class responsible for structured data generation.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'services/class-structured-data-service.php';
		
		/**
		 * The class responsible for theme management.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'services/class-theme-manager.php';
		
		/**
		 * The class responsible for event querying.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'services/class-event-query-service.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-atomic-events-public.php';

		$this->loader = new ATMCEVNTS_Loader();
	}



	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new ATMCEVNTS_Admin($this->get_plugin_name(), $this->get_version());

		$this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
		$this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
		$this->loader->add_action('admin_notices', $plugin_admin, 'activation_notice');
		
		// Settings page with tabbed interface
		$plugin_settings = new ATMCEVNTS_Settings($this->get_plugin_name(), $this->get_version());
		$this->loader->add_action('admin_menu', $plugin_settings, 'setup_plugin_options_menu');
		$this->loader->add_action('admin_init', $plugin_settings, 'initialize_settings');
		
		// Theme validation admin notice
		$this->loader->add_action('admin_notices', $this, 'theme_validation_notice');

		$evet_post_type = new ATMCEVNTS_Event_Post_Type($this->get_version());
		$this->loader->add_action('init', $evet_post_type, 'create_event_post_type');

		$evet_taxonomy = new ATMCEVNTS_Event_Taxonomy($this->get_version());
		$this->loader->add_action('init', $evet_taxonomy, 'register_taxonomy');

		/**
		 * Template
		 */
		$tpl_util = new ATMCEVNTS_Template_Utils();
		$this->loader->add_action('single_template', $tpl_util, 'load_custom_single_event_template');
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new ATMCEVNTS_Public($this->get_plugin_name(), $this->get_version());

		$this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
		$this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');


		$evet_post_type = new ATMCEVNTS_Event_Post_Type($this->get_version());
		$this->loader->add_action('add_meta_boxes', $evet_post_type, 'add_event_meta_boxes');
		$this->loader->add_action('save_post', $evet_post_type, 'save_event_details_meta_box');

		$shortcodes = new ATMCEVNTS_Shortcode();
		$this->loader->add_action('init', $shortcodes, 'register_shortcodes');
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    ATMCEVNTS_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}
	
	/**
	 * Display admin notice if selected theme doesn't exist
	 *
	 * @since     1.0.0
	 */
	public function theme_validation_notice() {
		$selected_theme = ATMCEVNTS_Theme_Manager::get_selected_theme();
		
		if (!ATMCEVNTS_Theme_Manager::theme_exists($selected_theme)) {
			?>
			<div class="notice notice-error">
				<p>
					<strong>Atomic Events Calendar:</strong> 
					The selected theme "<code><?php echo esc_html($selected_theme); ?></code>" does not exist. 
					Please check your <code>atomic-events-settings.yml</code> file or the plugin will use the default theme.
				</p>
			</div>
			<?php
		}
	}
}
