<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WPFAQCB_Form_Handler {

	public function __construct() {
		add_action( 'wp_ajax_wpfaqcb_submit_form', array( $this, 'handle_submit' ) );
		add_action( 'wp_ajax_nopriv_wpfaqcb_submit_form', array( $this, 'handle_submit' ) );
		add_action( 'wp_ajax_wpfaqcb_delete_log', array( $this, 'handle_delete' ) );
		add_action( 'wp_ajax_wpfaqcb_export_csv', array( $this, 'handle_export_csv' ) );
		add_action( 'wp_ajax_wpfaqcb_submit_rating', array( $this, 'handle_rating' ) );
		add_action( 'wp_ajax_nopriv_wpfaqcb_submit_rating', array( $this, 'handle_rating' ) );
	}

	public static function table_name() {
		global $wpdb;
		return $wpdb->prefix . 'wpfaqcb_submissions';
	}

	public static function ratings_table_name() {
		global $wpdb;
		return $wpdb->prefix . 'wpfaqcb_ratings';
	}

	public static function create_table() {
		global $wpdb;
		$table   = self::table_name();
		$charset = $wpdb->get_charset_collate();
		$sql     = "CREATE TABLE IF NOT EXISTS {$table} (
			id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
			form_data LONGTEXT NOT NULL,
			created_at DATETIME NOT NULL
		) {$charset};";
		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql );
	}

	public static function create_ratings_table() {
		global $wpdb;
		$table   = self::ratings_table_name();
		$charset = $wpdb->get_charset_collate();
		$sql     = "CREATE TABLE IF NOT EXISTS {$table} (
			id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
			section_title VARCHAR(255) NOT NULL,
			question_text VARCHAR(500) NOT NULL,
			rating TINYINT NOT NULL DEFAULT 0,
			visitor_ip VARCHAR(45) DEFAULT '',
			created_at DATETIME NOT NULL
		) {$charset};";
		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql );
	}

	public function handle_submit() {
		check_ajax_referer( 'wpfaqcb_form_nonce', 'nonce' );

		$settings = WPFAQCB_Settings_Manager::get_all();

		/* Honeypot check */
		if ( '1' === $settings['antispam_honeypot'] && ! empty( $_POST['wpfaqcb_website_url'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above via check_ajax_referer.
			wp_send_json_success( $settings['form_success_msg'] );
			return;
		}

		/* Rate limiting */
		if ( '1' === $settings['antispam_rate_limit'] ) {
			$ip            = self::get_visitor_ip();
			$transient_key = 'wpfaqcb_rl_' . md5( $ip );
			$count         = (int) get_transient( $transient_key );
			if ( $count >= 3 ) {
				wp_send_json_error( 'Too many submissions. Please wait a moment.' );
				return;
			}
			set_transient( $transient_key, $count + 1, 60 );
		}

		if ( empty( $_POST['fields'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above.
			wp_send_json_error( 'No data.' );
		}
		$raw = json_decode( sanitize_text_field( wp_unslash( $_POST['fields'] ) ), true ); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above.
		if ( ! is_array( $raw ) ) {
			wp_send_json_error( 'Invalid data.' );
		}
		$clean = array();
		foreach ( $raw as $field ) {
			if ( ! is_array( $field ) || ! isset( $field['label'], $field['value'] ) ) {
				continue;
			}
			$clean[] = array(
				'label' => sanitize_text_field( $field['label'] ),
				'value' => sanitize_textarea_field( $field['value'] ),
			);
		}
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom plugin table with no WP API equivalent.
		$result = $wpdb->insert(
			self::table_name(),
			array(
				'form_data'  => wp_json_encode( $clean, JSON_UNESCAPED_UNICODE ),
				'created_at' => current_time( 'mysql' ),
			),
			array( '%s', '%s' )
		);
		if ( false === $result ) {
			wp_send_json_error( 'Save failed.' );
		}
		wp_cache_delete( 'wpfaqcb_logs_count', 'wpfaqcb' );

		if ( '1' === $settings['form_email_notify'] && ! empty( $settings['form_email_to'] ) ) {
			$body = '';
			foreach ( $clean as $f ) {
				$body .= $f['label'] . ': ' . $f['value'] . "\n";
			}
			wp_mail( $settings['form_email_to'], sprintf( '[%s] New chatbot form submission', get_bloginfo( 'name' ) ), $body );
		}
		wp_send_json_success( $settings['form_success_msg'] );
	}

	public function handle_rating() {
		check_ajax_referer( 'wpfaqcb_form_nonce', 'nonce' );
		$section  = isset( $_POST['section'] ) ? sanitize_text_field( wp_unslash( $_POST['section'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above.
		$question = isset( $_POST['question'] ) ? sanitize_text_field( wp_unslash( $_POST['question'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above.
		$rating   = isset( $_POST['rating'] ) ? intval( $_POST['rating'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above.
		if ( ! in_array( $rating, array( 1, -1 ), true ) ) {
			wp_send_json_error( 'Invalid rating.' );
		}
		if ( empty( $question ) ) {
			wp_send_json_error( 'No question.' );
		}

		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom plugin table with no WP API equivalent.
		$wpdb->insert(
			self::ratings_table_name(),
			array(
				'section_title' => $section,
				'question_text' => $question,
				'rating'        => $rating,
				'visitor_ip'    => self::get_visitor_ip(),
				'created_at'    => current_time( 'mysql' ),
			),
			array( '%s', '%s', '%d', '%s', '%s' )
		);
		wp_send_json_success();
	}

	public function handle_delete() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Unauthorized' );
		}
		check_ajax_referer( 'wpfaqcb_logs_nonce', 'nonce' );
		$id = isset( $_POST['id'] ) ? absint( $_POST['id'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above.
		if ( ! $id ) {
			wp_send_json_error( 'Invalid ID.' );
		}
		global $wpdb;
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table; cache cleared below.
		$wpdb->delete( self::table_name(), array( 'id' => $id ), array( '%d' ) );
		wp_cache_delete( 'wpfaqcb_logs_count', 'wpfaqcb' );
		wp_send_json_success();
	}

	public function handle_export_csv() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'Unauthorized' );
		}
		check_admin_referer( 'wpfaqcb_export_csv' );

		$logs = self::get_logs( 10000 );
		if ( empty( $logs ) ) {
			wp_die( 'No data to export.' );
		}

		$filename = 'askova-form-logs-' . gmdate( 'Y-m-d' ) . '.csv';
		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=' . $filename );

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Writing CSV to php://output stream; WP_Filesystem is not applicable here.
		$output = fopen( 'php://output', 'w' );
		/* BOM for Excel UTF-8 */
		fprintf( $output, chr( 0xEF ) . chr( 0xBB ) . chr( 0xBF ) );

		/* Build header from first row */
		$first   = json_decode( $logs[0]->form_data, true );
		$headers = array( '#', 'Date' );
		if ( is_array( $first ) ) {
			foreach ( $first as $cell ) {
				$headers[] = $cell['label'];
			}
		}
		fputcsv( $output, $headers );

		foreach ( $logs as $log ) {
			$data = json_decode( $log->form_data, true );
			if ( ! is_array( $data ) ) {
				continue;
			}
			$row = array( $log->id, $log->created_at );
			foreach ( $data as $cell ) {
				$row[] = $cell['value'];
			}
			fputcsv( $output, $row );
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing php://output stream; WP_Filesystem is not applicable here.
		fclose( $output );
		exit;
	}

	public static function get_logs( $limit = 100 ) {
		global $wpdb;
		$table = self::table_name();
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table name from trusted prefix; not user input.
		return $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `{$table}` ORDER BY created_at DESC LIMIT %d", $limit ) );
	}

	public static function count_logs() {
		global $wpdb;
		$table  = self::table_name();
		$cached = wp_cache_get( 'wpfaqcb_logs_count', 'wpfaqcb' );
		if ( false !== $cached ) {
			return (int) $cached;
		}
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table name from trusted prefix; not user input.
		$count = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `{$table}` WHERE 1=%d", 1 ) );
		wp_cache_set( 'wpfaqcb_logs_count', $count, 'wpfaqcb', 300 );
		return $count;
	}

	private static function get_visitor_ip() {
		$ip = '';
		if ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$ips = explode( ',', sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) );
			$ip  = trim( $ips[0] );
		} elseif ( ! empty( $_SERVER['REMOTE_ADDR'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
		}
		return $ip;
	}
}
