<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class WPFAQCB_Admin_Settings {

	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_menu' ) );
		add_action( 'wp_ajax_wpfaqcb_save_settings', array( $this, 'ajax_save' ) );
		add_action( 'wp_ajax_wpfaqcb_switch_lang', array( $this, 'ajax_switch_lang' ) );
	}

	public function add_menu() {
		add_menu_page( 'Askova – Smart FAQ Chat', 'Askova FAQ Chat', 'manage_options', 'askova', array( $this, 'render_page' ), 'dashicons-format-chat', 80 );
	}

	public function ajax_save() {
		if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );
		check_ajax_referer( 'wpfaqcb_save_settings', 'nonce' );
		$raw = ! empty( $_POST['settings'] ) ? json_decode( sanitize_text_field( wp_unslash( $_POST['settings'] ) ), true ) : array();
		if ( ! is_array( $raw ) ) wp_send_json_error( 'Invalid' );
		WPFAQCB_Settings_Manager::save( WPFAQCB_Settings_Manager::sanitize( $raw ) );
		wp_send_json_success( WPFAQCB_I18n::t( 'saved' ) );
	}

	public function ajax_switch_lang() {
		if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error();
		check_ajax_referer( 'wpfaqcb_save_settings', 'nonce' );

		$new_lang = isset( $_POST['lang'] ) && 'ar' === sanitize_text_field( wp_unslash( $_POST['lang'] ) ) ? 'ar' : 'en';
		$old_lang = WPFAQCB_I18n::get_lang();

		/* Update the language option */
		update_option( WPFAQCB_I18n::LANG_OPTION, $new_lang );

		/* Swap content defaults if language actually changed */
		if ( $old_lang && $old_lang !== $new_lang ) {
			WPFAQCB_Settings_Manager::swap_content_lang( $old_lang, $new_lang );
		}
		/* First time setup (from welcome): set defaults for chosen language */
		if ( ! $old_lang ) {
			$s = WPFAQCB_Settings_Manager::get_all();
			$cd = WPFAQCB_Settings_Manager::content_defaults();
			foreach ( $cd as $k => $pair ) {
				$s[ $k ] = isset( $pair[ $new_lang ] ) ? $pair[ $new_lang ] : $pair['ar'];
			}
			if ( 'en' === $new_lang ) {
				$s['chat_direction'] = 'ltr';
			}
			WPFAQCB_Settings_Manager::save( $s );
		}

		wp_send_json_success();
	}

	public function render_page() {
		if ( ! current_user_can( 'manage_options' ) ) return;

		if ( ! WPFAQCB_I18n::has_lang() || ( isset( $_GET['welcome'] ) && ! WPFAQCB_I18n::has_lang() ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- No form data processed; only checking URL param presence for welcome redirect.
			$this->render_welcome();
			return;
		}

		$s    = WPFAQCB_Settings_Manager::get_all();
		$rtl  = WPFAQCB_I18n::is_rtl();
		$lang = WPFAQCB_I18n::get_lang();
		$dir  = $rtl ? 'rtl' : 'ltr';
		$tabs = array( 'general', 'content', 'faq', 'helpme', 'form', 'design', 'advanced', 'docs' );
		$fonts = WPFAQCB_I18n::google_fonts();
		$colors = array( 'color_primary','color_header','color_header_text','color_chat_bg','color_button','color_button_text','color_text','color_accent' );
		?>
		<div class="wrap wpfaqcb-wrap" dir="<?php echo esc_attr( $dir ); ?>">
			<div class="wpfaqcb-header-bar">
				<h1><svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="#1a5276" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/></svg> Askova <small>v<?php echo esc_html( WPFAQCB_VERSION ); ?></small></h1>
				<div class="wpfaqcb-lang-switch">
					<button type="button" class="button <?php echo 'en' === $lang ? 'button-primary' : ''; ?>" data-lang="en">English</button>
					<button type="button" class="button <?php echo 'ar' === $lang ? 'button-primary' : ''; ?>" data-lang="ar">العربية</button>
				</div>
			</div>
			<div id="wpfaqcb-notices"></div>
			<div class="wpfaqcb-tabs">
				<nav class="wpfaqcb-tab-nav">
					<?php foreach ( $tabs as $idx => $tab ) : ?>
						<button type="button" class="wpfaqcb-tab-btn<?php echo 0 === $idx ? ' active' : ''; ?>" data-tab="<?php echo esc_attr( $tab ); ?>"><?php WPFAQCB_I18n::e( 'tab_' . $tab ); ?></button>
					<?php endforeach; ?>
				</nav>
				<form id="wpfaqcb-form">

					<!-- GENERAL -->
					<div class="wpfaqcb-panel active" data-panel="general">
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'enable_chatbot' ); ?></th><td><label><input type="checkbox" name="chat_enabled" value="1" <?php checked( $s['chat_enabled'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'show_frontend' ); ?></label></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'position' ); ?></th><td><select name="chat_position"><option value="left" <?php selected( $s['chat_position'], 'left' ); ?>><?php WPFAQCB_I18n::e( 'left' ); ?></option><option value="right" <?php selected( $s['chat_position'], 'right' ); ?>><?php WPFAQCB_I18n::e( 'right' ); ?></option></select></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'direction' ); ?></th><td><select name="chat_direction"><option value="rtl" <?php selected( $s['chat_direction'], 'rtl' ); ?>><?php WPFAQCB_I18n::e( 'rtl_label' ); ?></option><option value="ltr" <?php selected( $s['chat_direction'], 'ltr' ); ?>><?php WPFAQCB_I18n::e( 'ltr_label' ); ?></option></select></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'show_popup' ); ?></th><td><label><input type="checkbox" name="popup_enabled" value="1" <?php checked( $s['popup_enabled'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'popup_auto' ); ?></label></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'popup_delay' ); ?></th><td><input type="number" name="popup_delay" value="<?php echo esc_attr( $s['popup_delay'] ); ?>" min="0" max="60" class="small-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'popup_duration' ); ?></th><td><input type="number" name="popup_duration" value="<?php echo esc_attr( $s['popup_duration'] ); ?>" min="1" max="60" class="small-text" /></td></tr>
						</table>
					</div>

					<!-- CONTENT -->
					<div class="wpfaqcb-panel" data-panel="content">
						<table class="form-table">
							<?php
							$cfs = array( 'header_title','header_subtitle','welcome_title','popup_text','input_placeholder','back_main_label','back_section_label','answer_label','contact_btn_label','request_btn_label','rating_question','rating_thanks','quick_reply_label','copy_success' );
							foreach ( $cfs as $f ) :
								$lbl = 'answer_label' === $f ? 'answer_lbl' : ( 'rating_question' === $f ? 'rating_question_lbl' : ( 'rating_thanks' === $f ? 'rating_thanks_lbl' : ( 'quick_reply_label' === $f ? 'quick_reply_label_lbl' : ( 'copy_success' === $f ? 'copy_success_lbl' : $f ) ) ) );
							?>
								<tr><th><?php WPFAQCB_I18n::e( $lbl ); ?></th><td><input type="text" name="<?php echo esc_attr( $f ); ?>" value="<?php echo esc_attr( $s[ $f ] ); ?>" class="regular-text" /></td></tr>
							<?php endforeach; ?>
							<tr><th><?php WPFAQCB_I18n::e( 'welcome_text' ); ?></th><td><textarea name="welcome_text" rows="3" class="large-text"><?php echo esc_textarea( $s['welcome_text'] ); ?></textarea></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'welcome_variations_lbl' ); ?></th><td><textarea name="welcome_variations" rows="4" class="large-text"><?php echo esc_textarea( $s['welcome_variations'] ); ?></textarea><p class="description"><?php WPFAQCB_I18n::e( 'welcome_variations_desc' ); ?></p></td></tr>
						</table>
						<hr />
						<h3><?php WPFAQCB_I18n::e( 'contact_info' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'enable_contact' ); ?></th><td><label><input type="checkbox" name="contact_enabled" value="1" <?php checked( $s['contact_enabled'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'show_contact_chat' ); ?></label></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'phone' ); ?></th><td><input type="text" name="contact_phone" value="<?php echo esc_attr( $s['contact_phone'] ); ?>" class="regular-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'email_addr' ); ?></th><td><input type="email" name="contact_email" value="<?php echo esc_attr( $s['contact_email'] ); ?>" class="regular-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'whatsapp' ); ?></th><td><input type="text" name="contact_whatsapp" value="<?php echo esc_attr( $s['contact_whatsapp'] ); ?>" class="regular-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'website_url' ); ?></th><td><input type="url" name="contact_website" value="<?php echo esc_url( $s['contact_website'] ); ?>" class="regular-text" /></td></tr>
						</table>
					</div>

					<!-- FAQ SECTIONS -->
					<div class="wpfaqcb-panel" data-panel="faq">
						<p class="description"><?php WPFAQCB_I18n::e( 'faq_desc' ); ?></p>
						<div id="wpfaqcb-faq-sections">
							<?php if ( ! empty( $s['faq_sections'] ) ) : foreach ( $s['faq_sections'] as $sec ) : self::render_section( $sec ); endforeach; endif; ?>
						</div>
						<button type="button" class="button button-primary" id="wpfaqcb-add-section"><?php WPFAQCB_I18n::e( 'add_section' ); ?></button>
					</div>

					<!-- HELP ME -->
					<div class="wpfaqcb-panel" data-panel="helpme">
						<p class="description"><?php WPFAQCB_I18n::e( 'helpme_desc' ); ?></p>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'helpme_enabled' ); ?></th><td><input type="checkbox" name="helpme_enabled" value="1" <?php checked( $s['helpme_enabled'], '1' ); ?> /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'helpme_title' ); ?></th><td><input type="text" name="helpme_title" value="<?php echo esc_attr( $s['helpme_title'] ); ?>" class="regular-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'helpme_question' ); ?></th><td><input type="text" name="helpme_question" value="<?php echo esc_attr( $s['helpme_question'] ); ?>" class="regular-text" /></td></tr>
						</table>
						<div id="wpfaqcb-helpme-options">
							<?php if ( ! empty( $s['helpme_options'] ) ) : foreach ( $s['helpme_options'] as $opt ) : self::render_helpme_option( $opt, $s['faq_sections'] ); endforeach; endif; ?>
						</div>
						<button type="button" class="button" id="wpfaqcb-add-helpme-option"><?php WPFAQCB_I18n::e( 'add_option' ); ?></button>
					</div>

					<!-- FORM -->
					<div class="wpfaqcb-panel" data-panel="form">
						<p class="description"><?php WPFAQCB_I18n::e( 'form_desc' ); ?></p>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'form_enabled' ); ?></th><td><label><input type="checkbox" name="form_enabled" value="1" <?php checked( $s['form_enabled'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'form_show' ); ?></label></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'form_title' ); ?></th><td><input type="text" name="form_title" value="<?php echo esc_attr( $s['form_title'] ); ?>" class="regular-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'form_subtitle' ); ?></th><td><input type="text" name="form_subtitle" value="<?php echo esc_attr( $s['form_subtitle'] ); ?>" class="regular-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'form_submit_label' ); ?></th><td><input type="text" name="form_submit_label" value="<?php echo esc_attr( $s['form_submit_label'] ); ?>" class="regular-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'form_success_msg' ); ?></th><td><textarea name="form_success_msg" rows="2" class="large-text"><?php echo esc_textarea( $s['form_success_msg'] ); ?></textarea></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'form_email_notify' ); ?></th><td><label><input type="checkbox" name="form_email_notify" value="1" <?php checked( $s['form_email_notify'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'form_email_desc' ); ?></label></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'form_email_to' ); ?></th><td><input type="email" name="form_email_to" value="<?php echo esc_attr( $s['form_email_to'] ); ?>" class="regular-text" /></td></tr>
						</table>
						<h3><?php WPFAQCB_I18n::e( 'quick_add' ); ?></h3>
						<div class="wpfaqcb-tpl-grid" id="wpfaqcb-field-tpls"></div>
						<h3><?php WPFAQCB_I18n::e( 'form_fields' ); ?></h3>
						<div id="wpfaqcb-form-fields">
							<?php if ( ! empty( $s['form_fields'] ) ) : foreach ( $s['form_fields'] as $fld ) : self::render_form_field( $fld ); endforeach; endif; ?>
						</div>
						<button type="button" class="button" id="wpfaqcb-add-field"><?php WPFAQCB_I18n::e( 'add_field' ); ?></button>
					</div>

					<!-- DESIGN -->
					<div class="wpfaqcb-panel" data-panel="design">
						<table class="form-table">
							<?php foreach ( $colors as $ck ) : ?>
								<tr><th><?php WPFAQCB_I18n::e( $ck ); ?></th><td><input type="text" name="<?php echo esc_attr( $ck ); ?>" value="<?php echo esc_attr( $s[ $ck ] ); ?>" class="wpfaqcb-color" /><p class="description"><?php WPFAQCB_I18n::e( $ck . '_d' ); ?></p></td></tr>
							<?php endforeach; ?>
							<tr><th><?php WPFAQCB_I18n::e( 'chat_width' ); ?></th><td><input type="number" name="chat_width" value="<?php echo esc_attr( $s['chat_width'] ); ?>" min="280" max="600" class="small-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'chat_max_height' ); ?></th><td><input type="number" name="chat_max_height" value="<?php echo esc_attr( $s['chat_max_height'] ); ?>" min="300" max="900" class="small-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'border_radius' ); ?></th><td><input type="number" name="border_radius" value="<?php echo esc_attr( $s['border_radius'] ); ?>" min="0" max="30" class="small-text" /></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'google_font' ); ?></th><td><select name="google_font" class="regular-text"><?php foreach ( $fonts as $val => $label ) : ?><option value="<?php echo esc_attr( $val ); ?>" <?php selected( $s['google_font'], $val ); ?>><?php echo esc_html( $label ); ?></option><?php endforeach; ?></select><p class="description"><?php WPFAQCB_I18n::e( 'google_font_d' ); ?></p></td></tr>
						</table>
					</div>

					<!-- ADVANCED -->
					<div class="wpfaqcb-panel" data-panel="advanced">
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'load_on_pages' ); ?></th><td><input type="text" name="load_on_pages" value="<?php echo esc_attr( $s['load_on_pages'] ); ?>" class="regular-text" /><p class="description"><?php WPFAQCB_I18n::e( 'load_pages_desc' ); ?></p></td></tr>
						</table>
						<hr />
						<h3>🔗 <?php WPFAQCB_I18n::e( 'quick_reply_enabled' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'quick_reply_enabled' ); ?></th><td><label><input type="checkbox" name="quick_reply_enabled" value="1" <?php checked( $s['quick_reply_enabled'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'quick_reply_enabled_d' ); ?></label></td></tr>
						</table>
						<hr />
						<h3>📋 <?php WPFAQCB_I18n::e( 'copy_answer_enabled' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'copy_answer_enabled' ); ?></th><td><label><input type="checkbox" name="copy_answer_enabled" value="1" <?php checked( $s['copy_answer_enabled'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'copy_answer_enabled_d' ); ?></label></td></tr>
						</table>
						<hr />
						<h3>🖨️ <?php WPFAQCB_I18n::e( 'print_answer_enabled' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'print_answer_enabled' ); ?></th><td><label><input type="checkbox" name="print_answer_enabled" value="1" <?php checked( $s['print_answer_enabled'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'print_answer_enabled_d' ); ?></label></td></tr>
						</table>
						<hr />
						<h3>⌨️ <?php WPFAQCB_I18n::e( 'typing_animation' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'typing_animation' ); ?></th><td><label><input type="checkbox" name="typing_animation" value="1" <?php checked( $s['typing_animation'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'typing_animation_d' ); ?></label></td></tr>
						</table>
						<hr />
						<h3>⭐ <?php WPFAQCB_I18n::e( 'rating_enabled' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'rating_enabled' ); ?></th><td><label><input type="checkbox" name="rating_enabled" value="1" <?php checked( $s['rating_enabled'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'rating_enabled_d' ); ?></label></td></tr>
						</table>
						<hr />
						<h3>💾 <?php WPFAQCB_I18n::e( 'session_memory' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'session_memory' ); ?></th><td><label><input type="checkbox" name="session_memory" value="1" <?php checked( $s['session_memory'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'session_memory_d' ); ?></label></td></tr>
						</table>
						<hr />
						<h3>⚡ <?php WPFAQCB_I18n::e( 'lazy_load' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'lazy_load' ); ?></th><td><label><input type="checkbox" name="lazy_load" value="1" <?php checked( $s['lazy_load'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'lazy_load_d' ); ?></label></td></tr>
						</table>
						<hr />
						<h3>🛡️ <?php WPFAQCB_I18n::e( 'antispam_label' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'honeypot_enabled' ); ?></th><td><label><input type="checkbox" name="antispam_honeypot" value="1" <?php checked( $s['antispam_honeypot'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'honeypot_d' ); ?></label></td></tr>
							<tr><th><?php WPFAQCB_I18n::e( 'rate_limit_enabled' ); ?></th><td><label><input type="checkbox" name="antispam_rate_limit" value="1" <?php checked( $s['antispam_rate_limit'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'rate_limit_d' ); ?></label></td></tr>
						</table>
						<hr />
						<h3>📋 <?php WPFAQCB_I18n::e( 'shortcode_label' ); ?></h3>
						<table class="form-table">
							<tr><th><?php WPFAQCB_I18n::e( 'shortcode_label' ); ?></th><td><code>[askova]</code> — <?php WPFAQCB_I18n::e( 'shortcode_desc' ); ?><br /><br /><code>[askova section="1"]</code> — Show specific section</td></tr>
						</table>
					</div>

					<!-- DOCS -->
					<div class="wpfaqcb-panel" data-panel="docs"><?php self::render_docs( $lang ); ?></div>

					<div class="wpfaqcb-save-bar">
						<button type="submit" class="button button-primary button-large"><?php WPFAQCB_I18n::e( 'save_settings' ); ?></button>
						<span class="spinner" id="wpfaqcb-spinner"></span>
					</div>
				</form>
			</div>
		</div>
		<?php
	}

	/* ============ WELCOME SCREEN ============ */
	private function render_welcome() {
		$nonce = wp_create_nonce( 'wpfaqcb_save_settings' );
		$ajax  = admin_url( 'admin-ajax.php' );
		$redir = admin_url( 'admin.php?page=askova' );
		?>
		<div class="wrap">
			<div class="wpfaqcb-wlc-wrap">
				<div class="wpfaqcb-wlc-card">
					<svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="#1a5276" stroke-width="1.8" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15a2 2 0 0 1-2 2H7l-4 4V5a2 2 0 0 1 2-2h14a2 2 0 0 1 2 2z"/></svg>
					<h1>Welcome to Askova!</h1>
					<h2 style="direction:rtl;">!Askova مرحباً بك في</h2>
					<p class="wpfaqcb-wlc-desc">Choose your admin panel language<br/>اختر لغة لوحة التحكم</p>
					<div class="wpfaqcb-wlc-btns">
						<button type="button" class="wpfaqcb-wlc-btn" data-lang="en"><strong style="font-size:20px;">English</strong><span>Admin panel in English</span></button>
						<button type="button" class="wpfaqcb-wlc-btn" data-lang="ar"><strong style="font-size:20px;">العربية</strong><span>لوحة التحكم بالعربية</span></button>
					</div>
				</div>
			</div>
		</div>
		<?php
		wp_enqueue_script( 'wpfaqcb-welcome', WPFAQCB_URL . 'assets/js/welcome.js', array(), WPFAQCB_VERSION, true );
		wp_localize_script( 'wpfaqcb-welcome', 'wpfaqcbWelcome', array(
			'nonce'    => $nonce,
			'ajaxurl'  => $ajax,
			'redirect' => $redir,
		) );
	}

	/* ============ RENDERERS ============ */
	private static function allowed_svg() {
		return array(
			'svg'    => array( 'width' => array(), 'height' => array(), 'viewbox' => array(), 'fill' => array(), 'xmlns' => array() ),
			'circle' => array( 'cx' => array(), 'cy' => array(), 'r' => array() ),
			'path'   => array( 'd' => array(), 'stroke' => array(), 'stroke-width' => array(), 'stroke-linecap' => array(), 'stroke-linejoin' => array() ),
		);
	}
	private static function drag_svg() {
		return '<svg width="16" height="16" viewBox="0 0 16 16" fill="#bbb"><circle cx="5" cy="3" r="1.5"/><circle cx="11" cy="3" r="1.5"/><circle cx="5" cy="8" r="1.5"/><circle cx="11" cy="8" r="1.5"/><circle cx="5" cy="13" r="1.5"/><circle cx="11" cy="13" r="1.5"/></svg>';
	}
	private static function chev_svg() {
		return '<svg width="20" height="20" viewBox="0 0 20 20" fill="none"><path d="M5 8l5 5 5-5" stroke="#555" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>';
	}

	private static function render_section( $sec ) {
		$hi = ! empty( $sec['icon_url'] );
		$sl = isset( $sec['lang'] ) ? $sec['lang'] : 'all';
		?>
		<div class="wpfaqcb-sec-block">
			<div class="wpfaqcb-sec-head">
				<span class="wpfaqcb-drag"><?php echo wp_kses( self::drag_svg(), self::allowed_svg() ); ?></span>
				<span class="sec-icon-wrap"><?php if ( $hi ) : ?><img src="<?php echo esc_url( $sec['icon_url'] ); ?>" class="sec-icon-preview" /><?php endif; ?><input type="hidden" class="sec-icon-url" value="<?php echo esc_url( $sec['icon_url'] ?? '' ); ?>" /><button type="button" class="button button-small wpfaqcb-media-btn"><?php echo $hi ? esc_html( WPFAQCB_I18n::t( 'change' ) ) : esc_html( WPFAQCB_I18n::t( 'choose_image' ) ); ?></button></span>
				<input type="text" class="sec-title" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'section_title' ) ); ?>" value="<?php echo esc_attr( $sec['title'] ); ?>" />
				<select class="sec-lang"><option value="all" <?php selected( $sl, 'all' ); ?>><?php echo esc_html( WPFAQCB_I18n::t( 'lang_all' ) ); ?></option><option value="ar" <?php selected( $sl, 'ar' ); ?>>العربية</option><option value="en" <?php selected( $sl, 'en' ); ?>>English</option></select>
				<label class="sec-has-subs-label"><input type="checkbox" class="sec-has-subs" <?php checked( $sec['has_subs'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'has_subs' ); ?></label>
				<button type="button" class="wpfaqcb-chevron"><?php echo wp_kses( self::chev_svg(), self::allowed_svg() ); ?></button>
				<button type="button" class="button wpfaqcb-remove-sec">&times;</button>
			</div>
			<div class="wpfaqcb-sec-body" style="display:none;">
				<div class="wpfaqcb-subs-area" style="<?php echo '1' !== $sec['has_subs'] ? 'display:none;' : ''; ?>"><input type="text" class="sec-sub-label" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'sub_label' ) ); ?>" value="<?php echo esc_attr( $sec['sub_label'] ); ?>" style="width:100%;margin-bottom:8px;" /><div class="wpfaqcb-subs-list"><?php if ( ! empty( $sec['subs'] ) ) : foreach ( $sec['subs'] as $sub ) : self::render_sub( $sub ); endforeach; endif; ?></div><button type="button" class="button wpfaqcb-add-sub"><?php WPFAQCB_I18n::e( 'add_sub' ); ?></button></div>
				<div class="wpfaqcb-direct-area" style="<?php echo '1' === $sec['has_subs'] ? 'display:none;' : ''; ?>"><p class="description"><?php WPFAQCB_I18n::e( 'direct_questions' ); ?></p><div class="wpfaqcb-items-list"><?php if ( ! empty( $sec['items'] ) ) : foreach ( $sec['items'] as $it ) : self::render_item( $it ); endforeach; endif; ?></div><button type="button" class="button wpfaqcb-add-item"><?php WPFAQCB_I18n::e( 'add_question' ); ?></button></div>
			</div>
		</div>
		<?php
	}

	private static function render_sub( $sub ) {
		$hi = ! empty( $sub['icon_url'] );
		?>
		<div class="wpfaqcb-sub-block">
			<div class="wpfaqcb-sub-head">
				<span class="wpfaqcb-drag"><?php echo wp_kses( self::drag_svg(), self::allowed_svg() ); ?></span>
				<span class="sub-icon-wrap"><?php if ( $hi ) : ?><img src="<?php echo esc_url( $sub['icon_url'] ); ?>" class="sub-icon-preview" /><?php endif; ?><input type="hidden" class="sub-icon-url" value="<?php echo esc_url( $sub['icon_url'] ?? '' ); ?>" /><button type="button" class="button button-small wpfaqcb-media-btn"><?php echo $hi ? esc_html( WPFAQCB_I18n::t( 'change' ) ) : esc_html( WPFAQCB_I18n::t( 'choose_image' ) ); ?></button></span>
				<input type="text" class="sub-title" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'sub_title' ) ); ?>" value="<?php echo esc_attr( $sub['title'] ); ?>" />
				<button type="button" class="wpfaqcb-chevron"><?php echo wp_kses( self::chev_svg(), self::allowed_svg() ); ?></button>
				<button type="button" class="button wpfaqcb-remove-sub">&times;</button>
			</div>
			<div class="wpfaqcb-sub-body" style="display:none;"><div class="wpfaqcb-items-list"><?php if ( ! empty( $sub['items'] ) ) : foreach ( $sub['items'] as $it ) : self::render_item( $it ); endforeach; endif; ?></div><button type="button" class="button wpfaqcb-add-item"><?php WPFAQCB_I18n::e( 'add_question' ); ?></button></div>
		</div>
		<?php
	}

	private static function render_item( $it ) {
		?>
		<div class="wpfaqcb-item">
			<span class="wpfaqcb-drag"><?php echo wp_kses( self::drag_svg(), self::allowed_svg() ); ?></span>
			<div class="wpfaqcb-item-fields"><input type="text" class="item-q" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'question' ) ); ?>" value="<?php echo esc_attr( $it['question'] ); ?>" /><textarea class="item-a" rows="2" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'answer' ) ); ?>"><?php echo esc_textarea( $it['answer'] ); ?></textarea></div>
			<button type="button" class="button wpfaqcb-remove-item">&times;</button>
		</div>
		<?php
	}

	private static function render_helpme_option( $opt, $sections ) {
		$ol = isset( $opt['lang'] ) ? $opt['lang'] : 'all';
		?>
		<div class="wpfaqcb-helpme-row">
			<span class="wpfaqcb-drag"><?php echo wp_kses( self::drag_svg(), self::allowed_svg() ); ?></span>
			<input type="text" class="hm-text" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'option_text' ) ); ?>" value="<?php echo esc_attr( $opt['text'] ); ?>" />
			<input type="text" class="hm-suggest" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'option_suggest' ) ); ?>" value="<?php echo esc_attr( $opt['suggest'] ); ?>" />
			<select class="hm-section"><option value="0"><?php WPFAQCB_I18n::e( 'select_section' ); ?></option><?php if ( ! empty( $sections ) ) : foreach ( $sections as $si => $sec ) : ?><option value="<?php echo esc_attr( $si + 1 ); ?>" <?php selected( $opt['section_id'], $si + 1 ); ?>><?php echo esc_html( ( $si + 1 ) . '. ' . $sec['title'] ); ?></option><?php endforeach; endif; ?></select>
			<select class="hm-lang"><option value="all" <?php selected( $ol, 'all' ); ?>><?php echo esc_html( WPFAQCB_I18n::t( 'lang_all' ) ); ?></option><option value="ar" <?php selected( $ol, 'ar' ); ?>>العربية</option><option value="en" <?php selected( $ol, 'en' ); ?>>English</option></select>
			<button type="button" class="button wpfaqcb-remove-helpme">&times;</button>
		</div>
		<?php
	}

	private static function render_form_field( $f ) {
		$fl = isset( $f['lang'] ) ? $f['lang'] : 'all';
		?>
		<div class="wpfaqcb-field-row">
			<span class="wpfaqcb-drag"><?php echo wp_kses( self::drag_svg(), self::allowed_svg() ); ?></span>
			<input type="text" class="ff-label" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'field_label' ) ); ?>" value="<?php echo esc_attr( $f['label'] ); ?>" />
			<select class="ff-type"><?php foreach ( array( 'text','email','tel','select','textarea','checkbox','radio','date' ) as $t ) : ?><option value="<?php echo esc_attr( $t ); ?>" <?php selected( $f['type'], $t ); ?>><?php echo esc_html( $t ); ?></option><?php endforeach; ?></select>
			<label class="ff-req-label"><input type="checkbox" class="ff-required" <?php checked( $f['required'], '1' ); ?> /> <?php WPFAQCB_I18n::e( 'field_required' ); ?></label>
			<input type="text" class="ff-placeholder" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'field_placeholder' ) ); ?>" value="<?php echo esc_attr( $f['placeholder'] ); ?>" />
			<input type="text" class="ff-options" placeholder="<?php echo esc_attr( WPFAQCB_I18n::t( 'field_options' ) ); ?>" value="<?php echo esc_attr( $f['options'] ); ?>" style="<?php echo ! in_array( $f['type'], array( 'select', 'checkbox', 'radio' ), true ) ? 'display:none;' : ''; ?>" />
			<select class="ff-lang"><option value="all" <?php selected( $fl, 'all' ); ?>><?php echo esc_html( WPFAQCB_I18n::t( 'lang_all' ) ); ?></option><option value="ar" <?php selected( $fl, 'ar' ); ?>>العربية</option><option value="en" <?php selected( $fl, 'en' ); ?>>English</option></select>
			<button type="button" class="button wpfaqcb-remove-field">&times;</button>
		</div>
		<?php
	}

	private static function render_docs( $lang ) {
		if ( 'ar' === $lang ) { ?>
		<div class="wpfaqcb-docs">
			<h2>📖 دليل استخدام Askova</h2>
			<div class="wpfaqcb-doc-section"><h3>🏗️ البدء السريع</h3><ol><li>فعّل الشات بوت من تاب <strong>"عام"</strong></li><li>أضف الأقسام والأسئلة في <strong>"أقسام الأسئلة"</strong></li><li>خصّص الألوان والتصميم من <strong>"التصميم"</strong></li><li>اضغط <strong>"حفظ الإعدادات"</strong></li></ol></div>
			<div class="wpfaqcb-doc-section"><h3>📂 أقسام الأسئلة</h3><p>كل قسم يمثل خدمة. أضف أيقونة من مكتبة الميديا. فعّل <strong>"يحتوي أقسام فرعية"</strong> لهيكل متعدد. استخدم السهم ▼ لفتح وإغلاق.</p></div>
			<div class="wpfaqcb-doc-section"><h3>🤔 ساعدني</h3><p>يظهر في القائمة الرئيسية. اختر القسم المرتبط من القائمة المنسدلة.</p></div>
			<div class="wpfaqcb-doc-section"><h3>📝 نموذج الطلبات</h3><p>استخدم <strong>"الإضافة السريعة"</strong> لحقول جاهزة. الطلبات في <strong>"سجل الطلبات"</strong>.</p></div>
			<div class="wpfaqcb-doc-section"><h3>🎨 التصميم</h3><p>غيّر الألوان. اختر خط جوجل من القائمة المنسدلة.</p></div>
		</div>
		<?php } else { ?>
		<div class="wpfaqcb-docs">
			<h2>📖 Askova User Guide</h2>
			<div class="wpfaqcb-doc-section"><h3>🏗️ Quick Start</h3><ol><li>Enable chatbot from <strong>"General"</strong></li><li>Add sections in <strong>"FAQ Sections"</strong></li><li>Customize in <strong>"Design"</strong></li><li>Click <strong>"Save Settings"</strong></li></ol></div>
			<div class="wpfaqcb-doc-section"><h3>📂 FAQ Sections</h3><p>Each section = a service. Upload icon from Media Library. Enable <strong>"Has Sub-Sections"</strong> for multi-level. Use ▼ to expand/collapse.</p></div>
			<div class="wpfaqcb-doc-section"><h3>🤔 Help Me</h3><p>Appears in main menu. Select linked section from dropdown.</p></div>
			<div class="wpfaqcb-doc-section"><h3>📝 Request Form</h3><p>Use <strong>"Quick Add"</strong> for pre-built fields. Submissions in <strong>"Form Logs"</strong>.</p></div>
			<div class="wpfaqcb-doc-section"><h3>🎨 Design</h3><p>Customize colors and select Google Font from dropdown.</p></div>
		</div>
		<?php }
	}
}
