<?php
/**
 * Plugin Name: Cubicsofts Phone Order Tracker for Asaan Retail
 * Plugin URI: https://cubicsofts.com
 * Description: Sync WooCommerce orders with Asaan Retail delivery status and allow customers to track orders using phone number.
 * Version: 1.1.1
 * Author: Talha Tabani
 * Author URI: https://cubicsofts.com/about-us/
 * Company: Cubicsofts
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: asaan-retail-phone-order-tracker

 */

if (!defined('ABSPATH')) exit;

/* ======================================================
   HELPERS
====================================================== */
function cubicsofts_asa_table_name() {
    global $wpdb;
    return $wpdb->prefix . 'asaan_wc_orders';
}

define('CUBICSOFTS_ASAAN_VERSION', '1.1.2');

function cubicsofts_asa_normalize_phone($phone) {
    if (!$phone) return '';
    $phone = preg_replace('/[^0-9+]/', '', $phone);

    if (strpos($phone, '+92') === 0) {
        $phone = '0' . substr($phone, 3);
    }
    if (strpos($phone, '92') === 0 && strlen($phone) === 12) {
        $phone = '0' . substr($phone, 2);
    }
    if (strpos($phone, '0') === 0 && strlen($phone) === 11) {
        return $phone;
    }
    return '';
}

function cubicsofts_asa_status_hash($order) {
    return md5(($order['delivery_status'] ?? '') . ($order['fulfillment_status'] ?? ''));
}

/* ======================================================
   ACTIVATION: TABLE + CRON
====================================================== */
register_activation_hook(__FILE__, function () {
    global $wpdb;

    $table = cubicsofts_asa_table_name();
    $charset = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE {$table} (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        order_id BIGINT UNSIGNED NOT NULL,
        phone VARCHAR(20) NOT NULL,
        customer_name VARCHAR(255),
        order_data LONGTEXT NOT NULL,
        delivery_status VARCHAR(100),
        status_hash CHAR(32),
        last_synced DATETIME,
        PRIMARY KEY (id),
        UNIQUE KEY order_id (order_id),
        KEY phone (phone)
    ) {$charset};";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);

    if (!wp_next_scheduled('cubicsofts_asa_sync_cron')) {
        wp_schedule_event(time(), 'quarter_hour', 'cubicsofts_asa_sync_cron');
    }
});

/* ======================================================
   CRON INTERVAL
====================================================== */
add_filter('cron_schedules', function ($schedules) {
    $schedules['quarter_hour'] = [
        'interval' => 900,
        'display'  => __('Every 15 Minutes', 'asaan-retail-phone-order-tracker')
    ];
    return $schedules;
});

/* ======================================================
   ADMIN SETTINGS
====================================================== */
add_action('admin_menu', function () {
    add_menu_page(
        __('Asaan Tracker', 'asaan-retail-phone-order-tracker'),
        __('Asaan Tracker', 'asaan-retail-phone-order-tracker'),
        'manage_options',
        'asaan-retail-phone-order-tracker',
        'cubicsofts_asa_settings_page'
    );

    add_submenu_page(
        'asaan-retail-phone-order-tracker',
        __('Synced Orders', 'asaan-retail-phone-order-tracker'),
        __('Synced Orders', 'asaan-retail-phone-order-tracker'),
        'manage_options',
        'asaan-orders',
        'cubicsofts_asa_admin_orders_page'
    );
});

add_action('admin_init', function () {

    register_setting('cubicsofts_asa_settings', 'cubicsofts_asa_api_base', [
        'sanitize_callback' => 'esc_url_raw'
    ]);

    register_setting('cubicsofts_asa_settings', 'cubicsofts_asa_api_key', [
        'sanitize_callback' => 'sanitize_text_field'
    ]);

    register_setting('cubicsofts_asa_settings', 'cubicsofts_asa_secret_token', [
        'sanitize_callback' => 'sanitize_text_field'
    ]);

    register_setting('cubicsofts_asa_settings', 'cubicsofts_asa_email', [
        'sanitize_callback' => 'sanitize_email'
    ]);

    register_setting('cubicsofts_asa_settings', 'cubicsofts_asa_password', [
        'sanitize_callback' => 'sanitize_text_field'
    ]);

});

add_action('wp_enqueue_scripts', function () {

    wp_register_style(
        'cubicsofts-asaaan-style',
        false,
        [],
        CUBICSOFTS_ASAAN_VERSION
    );
    wp_enqueue_style('cubicsofts-asaaan-style');

    wp_add_inline_style('cubicsofts-asaaan-style', '
        .asaan-card{border-left:4px solid #4caf50;padding:12px;margin:12px 0;background:#f9f9f9}
        .asaan-status{font-weight:bold}
    ');

    wp_register_script(
        'cubicsofts-asaaan-script',
        false,
        ['jquery'],
        CUBICSOFTS_ASAAN_VERSION,
        true
    );
    wp_enqueue_script('cubicsofts-asaaan-script');

    wp_add_inline_script('cubicsofts-asaaan-script', '
        jQuery(function($){
            $("#asaan-track").on("submit", function(e){
                e.preventDefault();
                $("#asaan-out").html("Loading...");
                $.post(cubicsoftsAsa.ajaxurl,{
                    action:"cubicsofts_asa_phone_lookup",
                    phone:$("#asaan-phone").val(),
                    nonce:cubicsoftsAsa.nonce
                },function(res){
                    $("#asaan-out").html(res);
                });
            });
        });
    ');

    wp_localize_script('cubicsofts-asaaan-script', 'cubicsoftsAsa', [
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce'   => wp_create_nonce('asaan_track_nonce')
    ]);
});



function cubicsofts_asa_settings_page() { ?>
<div class="wrap">
    <h1><?php esc_html_e('Asaan Retail Settings', 'asaan-retail-phone-order-tracker'); ?></h1>

    <form method="post" action="options.php">
        <?php settings_fields('cubicsofts_asa_settings'); ?>
        <table class="form-table">
            <tr><th>API Base</th><td><input name="cubicsofts_asa_api_base" value="<?php echo esc_attr(get_option('cubicsofts_asa_api_base')); ?>" class="regular-text"></td></tr>
            <tr><th>API Key</th><td><input name="cubicsofts_asa_api_key" value="<?php echo esc_attr(get_option('cubicsofts_asa_api_key')); ?>" class="regular-text"></td></tr>
            <tr><th>Secret Token</th><td><input name="cubicsofts_asa_secret_token" value="<?php echo esc_attr(get_option('cubicsofts_asa_secret_token')); ?>" class="regular-text"></td></tr>
            <tr><th>Email</th><td><input name="cubicsofts_asa_email" value="<?php echo esc_attr(get_option('cubicsofts_asa_email')); ?>" class="regular-text"></td></tr>
            <tr><th>Password</th><td><input type="password" name="cubicsofts_asa_password" value="<?php echo esc_attr(get_option('cubicsofts_asa_password')); ?>" class="regular-text"></td></tr>
        </table>
        <?php submit_button(); ?>
    </form>

    <form method="post">
        <?php wp_nonce_field('asaan_manual_sync', 'asaan_manual_sync_nonce'); ?>
        <?php submit_button(__('Sync Now', 'asaan-retail-phone-order-tracker'), 'secondary', 'asaan_manual_sync'); ?>
    </form>
</div>
<?php }

/* ======================================================
   SYNC ENGINE
====================================================== */
add_action('admin_init', function () {

    if ( ! isset($_POST['asaan_manual_sync']) ) {
        return;
    }

    if ( ! current_user_can('manage_options') ) {
        return;
    }

    $nonce = isset($_POST['asaan_manual_sync_nonce'])
        ? sanitize_text_field( wp_unslash( $_POST['asaan_manual_sync_nonce'] ) )
        : '';

    if ( ! wp_verify_nonce( $nonce, 'asaan_manual_sync' ) ) {
        return;
    }

    cubicsofts_asa_sync_orders();
});


add_action('cubicsofts_asa_sync_cron', 'cubicsofts_asa_sync_orders');

function cubicsofts_asa_sync_orders() {
    if (!class_exists('WooCommerce')) return;

    $base = rtrim(get_option('cubicsofts_asa_api_base'), '/');
    $key  = get_option('cubicsofts_asa_api_key');
    $sec  = get_option('cubicsofts_asa_secret_token');

    if (!$base || !$key || !$sec) return;

    $login = wp_remote_post($base . '/login/', [
        'headers' => [
            'api-key' => $key,
            'secret-token' => $sec
        ],
        'body' => [
            'email' => get_option('cubicsofts_asa_email'),
            'password' => get_option('cubicsofts_asa_password')
        ],
        'timeout' => 20
    ]);

    $token = json_decode(wp_remote_retrieve_body($login), true)['token'] ?? '';
    if (!$token) return;

    global $wpdb;
    $page = 1;
    $limit = 50;

    do {
        $orders = wc_get_orders([
            'limit' => $limit,
            'page'  => $page,
            'orderby' => 'date',
            'order' => 'DESC',
            'status' => ['processing', 'completed']
        ]);

        foreach ($orders as $wc) {
            $order_id = $wc->get_id();
            $phone = cubicsofts_asa_normalize_phone($wc->get_billing_phone());
            if (!$phone) continue;

            $res = wp_remote_get($base . '/orders/' . $order_id, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $token,
                    'api-key' => $key,
                    'secret-token' => $sec,
                    'Accept' => 'application/json'
                ],
                'timeout' => 20
            ]);

            $body = json_decode(wp_remote_retrieve_body($res), true);
            if (empty($body['order_details'])) continue;

            $o = $body['order_details'];
            $hash = cubicsofts_asa_status_hash($o);

            $table = esc_sql( cubicsofts_asa_table_name() );

            $existing = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT status_hash FROM {$table} WHERE order_id = %d",
                    $order_id
                )
            );

            if ($existing === $hash) continue;

            $wpdb->replace(cubicsofts_asa_table_name(), [
                'order_id'       => $order_id,
                'phone'          => $phone,
                'customer_name'  => $wc->get_formatted_billing_full_name(),
                'order_data'     => wp_json_encode($o),
                'delivery_status'=> sanitize_text_field($o['delivery_status'] ?? ''),
                'status_hash'    => $hash,
                'last_synced'    => current_time('mysql')
            ]);
        }

        $page++;
    } while (count($orders) === $limit);
}

/* ======================================================
   FRONTEND TRACKING SHORTCODE
====================================================== */
add_shortcode('asaan_phone_tracker', function () {
    ob_start(); ?>
<form id="asaan-track">
    <input placeholder="03XXXXXXXXX" id="asaan-phone" required>
    <button><?php esc_html_e('Track', 'asaan-retail-phone-order-tracker'); ?></button>
</form>
<div id="asaan-out"></div>

<?php return ob_get_clean();
});

/* ======================================================
   AJAX HANDLER
====================================================== */
add_action('wp_ajax_cubicsofts_asa_phone_lookup','cubicsofts_asa_phone_lookup');
add_action('wp_ajax_nopriv_cubicsofts_asa_phone_lookup','cubicsofts_asa_phone_lookup');

function cubicsofts_asa_phone_lookup() {
    $nonce = isset($_POST['nonce'])
        ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) )
        : '';

    if ( ! wp_verify_nonce( $nonce, 'asaan_track_nonce' ) ) {
        wp_die( esc_html__( 'Security check failed', 'asaan-retail-phone-order-tracker' ) );
    }

    global $wpdb;
    $phone = cubicsofts_asa_normalize_phone(sanitize_text_field($_POST['phone'] ?? ''));

    if (!$phone) wp_die(__('Invalid phone number', 'asaan-retail-phone-order-tracker'));

    $table = esc_sql( cubicsofts_asa_table_name() );

    $rows = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM {$table} WHERE phone = %s",
            $phone
        )
    );

    if (!$rows) wp_die(__('No orders found', 'asaan-retail-phone-order-tracker'));

    foreach ($rows as $r) {
        echo "<div class='asaan-card'>
            <div><strong>Order #" . esc_html($r->order_id) . "</strong></div>
            <div class='asaan-status'>" . esc_html($r->delivery_status) . "</div>
            <div>Last update: " . esc_html($r->last_synced) . "</div>
        </div>";
    }
    wp_die();
}

/* ======================================================
   ADMIN LIST TABLE
====================================================== */
function cubicsofts_asa_admin_orders_page() {
    global $wpdb;
    $table = esc_sql( cubicsofts_asa_table_name() );

    $rows = $wpdb->get_results(
        "SELECT * FROM {$table} ORDER BY last_synced DESC LIMIT 200"
    );

    echo "<div class='wrap'><h1>Synced Orders</h1>
    <table class='widefat striped'>
    <tr><th>Order</th><th>Phone</th><th>Status</th><th>Last Sync</th></tr>";

    foreach ($rows as $r) {
        echo "<tr>
            <td>" . esc_html($r->order_id) . "</td>
            <td>" . esc_html($r->phone) . "</td>
            <td>" . esc_html($r->delivery_status) . "</td>
            <td>" . esc_html($r->last_synced) . "</td>
        </tr>";
    }

    echo "</table></div>";
}
