<?php
/*
Plugin Name: Arunstheme Editorial Notes
Description: Add private editorial notes with professional status UI. Visible only to admins and editors.
Version: 1.1.1
Author: Arun Paul
License: GPLv2 or later
Text Domain: arunstheme-editorial-notes
*/

if (! defined('ABSPATH')) {
    exit;
}

/*----------------------------------------------
# Enqueue Admin Styles
----------------------------------------------*/
function arunstheme_en_admin_styles($hook)
{

    if ('post.php' !== $hook && 'post-new.php' !== $hook && 'edit.php' !== $hook) {
        return;
    }

    wp_enqueue_style(
        'arunstheme-en-admin-style',
        plugin_dir_url(__FILE__) . 'admin-style.css',
        array(),
        '1.1.1'
    );
}
add_action('admin_enqueue_scripts', 'arunstheme_en_admin_styles');


/*----------------------------------------------
# Capability
----------------------------------------------*/
function arunstheme_en_can_view_notes()
{
    return current_user_can('edit_others_posts');
}


/*----------------------------------------------
# Add Meta Box
----------------------------------------------*/
function arunstheme_en_add_meta_box()
{

    if (! arunstheme_en_can_view_notes()) {
        return;
    }

    add_meta_box(
        'arunstheme_editorial_notes',
        __('Editorial Notes', 'arunstheme-editorial-notes'),
        'arunstheme_en_meta_box_callback',
        array('post', 'page'),
        'side',
        'high'
    );
}
add_action('add_meta_boxes', 'arunstheme_en_add_meta_box');


/*----------------------------------------------
# Meta Box UI
----------------------------------------------*/
function arunstheme_en_meta_box_callback($post)
{

    wp_nonce_field('arunstheme_en_save_notes', 'arunstheme_en_notes_nonce');

    $note   = get_post_meta($post->ID, '_arunstheme_editorial_note', true);
    $status = get_post_meta($post->ID, '_arunstheme_editorial_status', true);

    if (empty($status)) {
        $status = 'not_started';
    }
?>

    <div class="arun-box">

        <strong><?php esc_html_e('Editorial Status', 'arunstheme-editorial-notes'); ?></strong>

        <div class="arun-status-wrap">

            <?php
            $statuses = array(
                'not_started' => 'Not Started',
                'in_progress' => 'In Progress',
                'done'        => '✔ Done',
            );

            foreach ($statuses as $key => $label) :
            ?>
                <label>
                    <input type="radio"
                        name="arunstheme_editorial_status"
                        value="<?php echo esc_attr($key); ?>"
                        <?php checked($status, $key); ?>>
                    <span class="arun-badge arun-<?php echo esc_attr(str_replace('_', '-', $key)); ?>">
                        <?php echo esc_html($label); ?>
                    </span>
                </label>
            <?php endforeach; ?>

        </div>

        <textarea name="arunstheme_editorial_note"><?php echo esc_textarea($note); ?></textarea>

        <div class="arun-info">
            <?php esc_html_e('Visible only to Admins and Editors.', 'arunstheme-editorial-notes'); ?>
        </div>

    </div>

<?php
}


/*----------------------------------------------
# Save Data (Fully Secure)
----------------------------------------------*/
function arunstheme_en_save_notes($post_id)
{

    if (
        ! isset($_POST['arunstheme_en_notes_nonce']) ||
        ! wp_verify_nonce(
            sanitize_text_field(
                wp_unslash($_POST['arunstheme_en_notes_nonce'])
            ),
            'arunstheme_en_save_notes'
        )
    ) {
        return;
    }

    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }

    if (! current_user_can('edit_post', $post_id)) {
        return;
    }

    if (isset($_POST['arunstheme_editorial_note'])) {

        $note = sanitize_textarea_field(
            wp_unslash($_POST['arunstheme_editorial_note'])
        );

        update_post_meta(
            $post_id,
            '_arunstheme_editorial_note',
            $note
        );
    }

    if (isset($_POST['arunstheme_editorial_status'])) {

        $allowed_status = array('not_started', 'in_progress', 'done');

        $status = sanitize_text_field(
            wp_unslash($_POST['arunstheme_editorial_status'])
        );

        if (in_array($status, $allowed_status, true)) {
            update_post_meta(
                $post_id,
                '_arunstheme_editorial_status',
                $status
            );
        }
    }
}
add_action('save_post', 'arunstheme_en_save_notes');


/*----------------------------------------------
# Status Filter Dropdown
----------------------------------------------*/
function arunstheme_editorial_status_filter_dropdown()
{

    global $typenow;

    if ($typenow !== 'post') {
        return;
    }

?>
    <select name="editorial_status_filter">
        <option value="">All Editorial Status</option>
        <option value="not_started">Not Started</option>
        <option value="in_progress">In Progress</option>
        <option value="done">Done</option>
    </select>

    <?php wp_nonce_field('arun_filter_action', 'arun_filter_nonce'); ?>

<?php
}
add_action('restrict_manage_posts', 'arunstheme_editorial_status_filter_dropdown');


/*----------------------------------------------
# Filter Query
----------------------------------------------*/
function arunstheme_en_filter_posts_by_status($query)
{

    global $pagenow;

    if (
        ! is_admin() ||
        'edit.php' !== $pagenow ||
        ! $query->is_main_query()
    ) {
        return;
    }

    if (
        isset($_GET['arun_filter_nonce']) &&
        wp_verify_nonce(
            sanitize_text_field(
                wp_unslash($_GET['arun_filter_nonce'])
            ),
            'arun_filter_action'
        )
    ) {

        if (isset($_GET['editorial_status_filter'])) {

            $allowed_status = array('not_started', 'in_progress', 'done');

            $status = sanitize_text_field(
                wp_unslash($_GET['editorial_status_filter'])
            );

            if (in_array($status, $allowed_status, true)) {

                $query->set(
                    'meta_query',
                    array(
                        array(
                            'key'     => '_arunstheme_editorial_status',
                            'value'   => $status,
                            'compare' => '=',
                        ),
                    )
                );
            }
        }
    }
}
add_action('pre_get_posts', 'arunstheme_en_filter_posts_by_status');

/*----------------------------------------------
# Add Admin Columns
----------------------------------------------*/
function arunstheme_en_add_columns($columns)
{

    $columns['editorial_status'] = esc_html__('Editorial Status', 'arunstheme-editorial-notes');
    $columns['editorial_notes']  = esc_html__('Editorial Notes', 'arunstheme-editorial-notes');

    return $columns;
}
add_filter('manage_post_posts_columns', 'arunstheme_en_add_columns');
add_filter('manage_page_posts_columns', 'arunstheme_en_add_columns');

function arunstheme_en_show_columns($column, $post_id)
{

    if ('editorial_status' === $column) {

        $status = get_post_meta($post_id, '_arunstheme_editorial_status', true);

        if ('done' === $status) {
            echo '<span class="arun-column-badge arun-column-done">✔ Done</span>';
        } elseif ('in_progress' === $status) {
            echo '<span class="arun-column-badge arun-column-in-progress">In Progress</span>';
        } else {
            echo '<span class="arun-column-badge arun-column-not-started">Not Started</span>';
        }
    }

    if ('editorial_notes' === $column) {

        $note = get_post_meta($post_id, '_arunstheme_editorial_note', true);

        if (! empty($note)) {
            echo esc_html(wp_trim_words($note, 10, '...'));
        } else {
            echo esc_html__('—', 'arunstheme-editorial-notes');
        }
    }
}
add_action('manage_post_posts_custom_column', 'arunstheme_en_show_columns', 10, 2);
add_action('manage_page_posts_custom_column', 'arunstheme_en_show_columns', 10, 2);
