<?php
/**
 * Plugin Name: Articla Lite Bulk Publisher
 * Description: One-click AI bulk generator for drafts with citations, categories, blacklist, and compliance checks.
 * Version: 0.7.8
 * Author: articla.ai
 * Author URI: https://www.articla.ai
 * License: GPL-2.0+
 * Text Domain: articla-lite-bulk-publisher
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

define( 'ALBP_VER',  '0.7.8' );
define( 'ALBP_SLUG', 'articla-lite-bulk-publisher' );
define( 'ALBP_URL',  plugin_dir_url( __FILE__ ) );
define( 'ALBP_DIR',  plugin_dir_path( __FILE__ ) );

add_filter(
    'pre_update_option_albp_openai_model',
    function( $new, $old ) { return 'o4-mini'; },
    10, 2
);

function albp_on_activate() {
    if ( version_compare( PHP_VERSION, '7.4', '<' ) ) {
        deactivate_plugins( plugin_basename( __FILE__ ) );
        wp_die( 'Articla Lite requires PHP 7.4 or higher.' );
    }
    $defaults = array(
        'albp_openai_model'    => 'o4-mini',
        'albp_default_status'  => 'draft',
        'albp_min_words'       => 400,
        'albp_max_words'       => 1200,
        'albp_min_headings'    => 3,
        'albp_min_citations'   => 3,
        'albp_language'        => 'en',
        'albp_compliance_mode' => 1,
        'albp_blacklist'       => '',

    );
    foreach ( $defaults as $k => $v ) {
        if ( get_option( $k, null ) === null ) {
            add_option( $k, $v );
        }
    }
}

register_activation_hook( __FILE__, 'albp_on_activate' );

require_once ALBP_DIR . 'includes/class-albp-generator.php';
require_once ALBP_DIR . 'includes/class-albp-wxr.php';

function albp_license_status( $force = false ) {
// Lite edition: no remote verification; always treat as not-pro
    return array(
        'valid'      => false,
        'plan'       => null,
        'reason'     => 'lite',
        'next_due_at'=> null,
        'checked_at' => time(),
    );
}

function albp_is_pro( $force = false ) {
return false;
}



class ALBP_Plugin {
    public function __construct() {
        add_action( 'admin_menu', array( $this, 'admin_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
        add_action( 'wp_ajax_albp_start_processing', array( $this, 'ajax_start_processing' ) );
        add_action( 'wp_ajax_albp_process_next', array( $this, 'ajax_process_next' ) );
        add_action( 'init', array( $this, 'register_meta' ) );
        add_action( 'add_meta_boxes', array( $this, 'add_qc_metabox' ) );
        add_action( 'save_post', array( $this, 'save_qc_metabox' ) );
        add_action( 'wp_head', array( $this, 'output_noindex_meta' ) );
        add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 2 );
}


    public function admin_menu() {
        add_menu_page(
            __( 'Articla Bulk', 'articla-lite-bulk-publisher' ),
            __( 'Articla Bulk', 'articla-lite-bulk-publisher' ),
            'manage_options',
            ALBP_SLUG,
            array( $this, 'render_settings' ),
            'dashicons-media-text',
            59
        );

        // Rename the first auto submenu to “Settings”
        add_submenu_page(
            ALBP_SLUG,
            __( 'Settings', 'articla-lite-bulk-publisher' ),
            __( 'Settings', 'articla-lite-bulk-publisher' ),
            'manage_options',
            ALBP_SLUG,
            array( $this, 'render_settings' )
        );

        add_submenu_page(
            ALBP_SLUG,
            __( 'Bulk Generate', 'articla-lite-bulk-publisher' ),
            __( 'Bulk Generate', 'articla-lite-bulk-publisher' ),
            'manage_options',
            ALBP_SLUG . '-bulk',
            array( $this, 'render_bulk' )
        );
        add_submenu_page(
            ALBP_SLUG,
            __( 'Export WXR', 'articla-lite-bulk-publisher' ),
            __( 'Export WXR', 'articla-lite-bulk-publisher' ),
            'manage_options',
            ALBP_SLUG . '-wxr',
            array( $this, 'render_wxr' )
        );
    }

    public function register_settings() {
        // Use a single args array per register_setting; defaults are already seeded on activation
        $args_text = array( 'sanitize_callback' => 'sanitize_text_field' );

        register_setting( 'albp_settings', 'albp_openai_key',        $args_text );
        register_setting( 'albp_settings', 'albp_openai_model',      $args_text );
        register_setting( 'albp_settings', 'albp_pexels_key',        $args_text );
        register_setting( 'albp_settings', 'albp_default_status',    $args_text );
        register_setting( 'albp_settings', 'albp_min_words',         $args_text );
        register_setting( 'albp_settings', 'albp_max_words',         $args_text );
        register_setting( 'albp_settings', 'albp_min_headings',      $args_text );
        register_setting( 'albp_settings', 'albp_min_citations',     $args_text );
        register_setting( 'albp_settings', 'albp_language',          $args_text );
        register_setting( 'albp_settings', 'albp_compliance_mode',   $args_text );
        register_setting( 'albp_settings', 'albp_blacklist',         $args_text );

    }

    public function enqueue_assets( $hook ) {
        if ( strpos( $hook, ALBP_SLUG ) === false ) return;
        wp_enqueue_style( 'dashicons' );
        wp_enqueue_style( 'albp-admin', ALBP_URL . 'assets/css/admin.css', array(), ALBP_VER );
        wp_enqueue_script( 'albp-admin', ALBP_URL . 'assets/js/admin.js', array( 'jquery' ), ALBP_VER, true );
        wp_localize_script(
            'albp-admin',
            'ALBP',
            array(
                'ajaxurl' => admin_url( 'admin-ajax.php' ),
                'nonce'   => wp_create_nonce( 'albp-ajax' ),
            )
        );
    }

    public function register_meta() {
        register_post_meta( 'post', '_albp_qc',           array( 'single' => true, 'type' => 'string',  'show_in_rest' => true, 'auth_callback' => function(){ return current_user_can( 'edit_posts' ); } ) );
        register_post_meta( 'post', '_albp_noindex',      array( 'single' => true, 'type' => 'boolean', 'show_in_rest' => true, 'auth_callback' => function(){ return current_user_can( 'edit_posts' ); } ) );
        register_post_meta( 'post', '_albp_generated_by', array( 'single' => true, 'type' => 'string',  'show_in_rest' => true, 'auth_callback' => function(){ return current_user_can( 'edit_posts' ); } ) );
    }

    private function header_logo() {
        echo '<div class="albp-header"><h1>Articla Lite Bulk Publisher</h1></div>';
    }

    private function top_links_card() {
        // Raw URLs, escape on output
        $review    = 'https://wordpress.org/support/plugin/articla-lite-bulk-publisher/reviews/';
        $readme    = 'https://articla.ai/readme.txt';
        $terms     = 'https://articla.ai/terms.txt';
        $changelog = 'https://articla.ai/changelog.txt';


        $badge_text  = 'FREE VERSION';
        $badge_class =  'Free';
        $logo_url = ALBP_URL . 'assets/logomain.png';
        echo '<img src=' . esc_url( $logo_url ) . ' alt="Articla logo" height="80">&nbsp;&nbsp;<div class="albp-version"><span class="albp-badge '
           . esc_attr( $badge_class )
           . '">'
           . esc_html( $badge_text )
           . '</span> <a target=_blank href=https://articla.ai/ class=button button-secondary style=margin-left:8px;>Upgrade to Pro</a></div><br><br>';

        echo '<div class="albp-topcard"><div><strong>⭐ Love Articla Lite?</strong> '
           . '<a target="_blank" href="' . esc_url( $review ) . '">Add a review</a>'
           . '</div>';

        echo '<div>Our plugin is strongly coded to avoid spam content and aligned with Google policy for AI content (2025).</div>';

        echo '<div>'
           . '<a target="_blank" href="' . esc_url( $terms ) . '">Terms and conditions</a>'
           . '  &nbsp;&nbsp;&nbsp;&nbsp;  '
           . '<a target="_blank" href="' . esc_url( $readme ) . '">Readme</a>'
           . '  &nbsp;&nbsp;&nbsp;&nbsp;  '
           . '<a target="_blank" href="' . esc_url( $changelog ) . '">Changelog</a>'
           . '</div><br>';
    }

    public function render_settings() {
        if ( ! current_user_can( 'manage_options' ) ) return;

        $def             = get_option( 'albp_default_status', 'draft' );
        $publish_checked = ( $def === 'publish_if_qc' ) ? 'checked' : '';
        $current_model   = get_option( 'albp_openai_model', 'o4-mini' );
        $is_pro          = false; // Lite edition



        echo '<div class="wrap albp-wrap">'; $this->header_logo(); $this->top_links_card();

        echo '<p><a href="' . esc_url( admin_url( 'admin.php?page=' . ALBP_SLUG . '-bulk' ) ) . '" class="button button-primary">Go to Bulk Generator</a></p>';


        echo '<form method="post" action="options.php">';
        settings_fields( 'albp_settings' );
        do_settings_sections( 'albp_settings' );



        echo '<h2>API Keys</h2><table class="form-table">';

echo '<tr><th>OpenAI API Key</th><td>'
   . '<input id="albp_openai_key" type="password" name="albp_openai_key" value="' . esc_attr( get_option( 'albp_openai_key', '' ) ) . '" class="regular-text" /> '
   . '<span class="dashicons dashicons-visibility albp-eye" data-for="albp_openai_key" title="' . esc_attr__( 'Show/Hide', 'articla-lite-bulk-publisher' ) . '"></span>'
   . '&nbsp;&nbsp;<a class="albp-help-link" href="https://www.articla.ai/apis.php" target="_blank" rel="noopener" aria-label="' . esc_attr__( 'Readme: API key help', 'articla-lite-bulk-publisher' ) . '"><span class="dashicons dashicons-editor-help" aria-hidden="true"></span></a>'
   . '</td></tr>';

        echo '<tr><th>OpenAI Model</th><td>';
        if ( ! $is_pro ) {
            echo '<select disabled><option selected>o4-mini</option></select><input type="hidden" name="albp_openai_model" value="o4-mini" /> <a target="_blank" href="https://articla.ai/" class="button button-secondary" style="margin-left:8px;">Upgrade to Pro</a><p class="description">Free version is locked to <code>o4-mini</code>. Upgrade to unlock more models.</p>';
        } else {
            $opts = array( 'o4-mini', 'gpt-4o-mini', 'gpt-4o', 'o4', 'gpt-4.1-mini' );
            echo '<select name="albp_openai_model">';
            foreach ( $opts as $opt ) {
                $sel = selected( $current_model, $opt, false );
                // Protect $sel to satisfy scanners
                echo '<option value="' . esc_attr( $opt ) . '" ' . esc_attr( $sel ) . '>' . esc_html( $opt ) . '</option>';
            }
            echo '</select>';
        }
        echo '</td></tr>';

        echo '<tr><th>Pexels API Key</th><td>'
   . '<input id="albp_pexels_key" type="password" name="albp_pexels_key" value="' . esc_attr( get_option( 'albp_pexels_key', '' ) ) . '" class="regular-text" /> '
   . '<span class="dashicons dashicons-visibility albp-eye" data-for="albp_pexels_key" title="' . esc_attr__( 'Show/Hide', 'articla-lite-bulk-publisher' ) . '"></span>'
   . '&nbsp;&nbsp;<a class="albp-help-link" href="https://www.articla.ai/apis.php" target="_blank" rel="noopener" aria-label="' . esc_attr__( 'Readme: API key help', 'articla-lite-bulk-publisher' ) . '"><span class="dashicons dashicons-editor-help" aria-hidden="true"></span></a>'
   . '</td></tr>';

        echo '</table>';

        echo '<h2>Quality & Compliance</h2><table class="form-table">';
        echo '<tr><th>Publish when QC passes</th><td><label><input id="albp_publish_if_qc" type="checkbox" ' . ( $publish_checked ? 'checked' : '' ) . ' /> Enable</label><input type="hidden" id="albp_default_status" name="albp_default_status" value="' . esc_attr( $def ) . '" /></td></tr>';
        echo '<tr><th>Language</th><td><input type="text" name="albp_language" value="' . esc_attr( get_option( 'albp_language', 'en' ) ) . '" /></td></tr>';
        echo '<tr><th>Min post words</th><td><input type="number" name="albp_min_words" value="' . esc_attr( get_option( 'albp_min_words', 400 ) ) . '" style="width:110px;" /></td></tr>';
        echo '<tr><th>Max post words</th><td><input type="number" name="albp_max_words" value="' . esc_attr( get_option( 'albp_max_words', 1200 ) ) . '" style="width:110px;" /></td></tr>';
        echo '<tr><th>Min H2/H3</th><td><input type="number" name="albp_min_headings" value="' . esc_attr( get_option( 'albp_min_headings', 3 ) ) . '" style="width:110px;" /></td></tr>';
        echo '<tr><th>Min Citations</th><td><input type="number" name="albp_min_citations" value="' . esc_attr( get_option( 'albp_min_citations', 3 ) ) . '" style="width:110px;" /></td></tr>';
        echo '<tr><th>Compliance Mode</th><td>'; $cm = get_option( 'albp_compliance_mode', 1 ); echo '<label><input type="checkbox" name="albp_compliance_mode" value="1" ' . checked( $cm, 1, false ) . ' /> Add temporary <code>noindex</code> when QC fails, removed upon approval.</label></td></tr>';
        echo '<tr><th>Blacklist (comma-separated)</th><td><textarea name="albp_blacklist" rows="3" class="large-text" placeholder="e.g., casino, pills, adult">' . esc_textarea( get_option( 'albp_blacklist', '' ) ) . '</textarea></td></tr>';
        echo '</table>';


        submit_button();
        echo '</form></div>';
    }


    public function render_bulk() {
        if ( ! current_user_can( 'manage_options' ) ) return;
        $prefill = '3-Day Marrakech Itinerary: Food Souks & Desert Day Trip,marrakech itinerary;medina;souk;jemaa el-fna;atlas mountains,https://visitmorocco.com/en;https://whc.unesco.org/en/list/331/;https://www.lonelyplanet.com/morocco/marrakesh,Before You Go: Safety Money & Dress|Day 1: Medina Highlights & Souks|Day 2: Gardens Museums & Hammam|Day 3: Atlas or Agafay Day Trip|Where to Eat & Bargaining Etiquette|Transport & Budget Tips|Cultural Etiquette & Friday Hours|Conclusion,Marrakech medina';
        echo '<div class="wrap albp-wrap">'; $this->header_logo(); $this->top_links_card();
        echo '<h2>Bulk Generate</h2><p style="color:#46b450">Paste CSV rows. <strong>Columns:</strong> title,keywords(optional),sources(semicolon-separated URLs),outline(H2s separated by |),image_query(optional)</p>';
        if ( ! albp_is_pro( false ) ) echo '<p><em>Free version is limited to 5 posts per batch. <a target=_blank href=https://articla.ai/subscribe.php class=button button-secondary style=margin-left:8px;> Upgrade to Pro </a> for unlimited posts.</em></p>';
        echo '<textarea id="albp_rows" rows="6" style="width:100%;" placeholder="' . esc_attr( $prefill ) . '"></textarea>';
        echo '<p><strong>Categories:</strong></p><div class="albp-cats">';
        $cats = get_categories( array( 'hide_empty' => 0 ) );
        foreach ( $cats as $c ) {
            echo '<label style="margin-right:10px;"><input type="checkbox" class="albp-cat" value="' . intval( $c->term_id ) . '" /> ' . esc_html( $c->name ) . '</label>';
        }
        echo '</div>';
        echo '<p>Interval between post processing (ms): <input type="number" id="albp_delay_ms" value="8000" style="width:110px;" ' . ( albp_is_pro( false ) ? '' : 'disabled' ) . ' /> ' . ( ! albp_is_pro( false ) ? '<em>(available only in PRO version)</em>' : '' ) . '</p>';
        echo '<div id="albp-queue-controls"><button id="albp-start" class="button button-primary">Start Processing</button> <button id="albp-stop" class="button">Stop</button><div id="albp-progress" class="albp-progress"><span></span></div><pre id="albp-log" class="albp-log"></pre></div>';
        echo '</div>';
    }

    public function render_wxr() {
        if ( ! current_user_can( 'manage_options' ) ) return;
        if ( ! empty( $_POST['albp_export_now'] ) && check_admin_referer( 'albp-wxr' ) ) {
            $wxr = new ALBP_WXR();
            $xml = $wxr->export_albp_posts();

            header( 'Content-Description: File Transfer' );
            header( 'Content-Disposition: attachment; filename=albp-export-' . gmdate( 'Ymd-His' ) . '.xml' );
            header( 'Content-Type: application/xml; charset=utf-8' );

            echo esc_xml( $xml );
            exit;
        }
        echo '<div class="wrap albp-wrap">'; $this->header_logo();
        echo '<h2>Export WXR</h2><p>Export posts generated by Articla Lite (recognized via post meta) to a portable WXR file.</p>';
        echo '<form method="post">'; wp_nonce_field( 'albp-wxr' ); echo '<p><button class="button button-primary" name="albp_export_now" value="1">Download WXR</button></p></form></div>';
    }

    public function add_qc_metabox() {
        add_meta_box( 'albp-qc', __( 'Articla Quality Checklist', 'articla-lite-bulk-publisher' ), array( $this, 'render_qc_metabox' ), 'post', 'side', 'high' );
    }

    public function render_qc_metabox( $post ) {
        $qc_json = get_post_meta( $post->ID, '_albp_qc', true );
        $qc      = $qc_json ? json_decode( $qc_json, true ) : array( 'words' => 0, 'headings' => 0, 'citations' => 0, 'pass' => false );
        $noindex = (bool) get_post_meta( $post->ID, '_albp_noindex', true );

        wp_nonce_field( 'albp-qc-save', 'albp_qc_nonce' );

        echo '<p><strong>Words:</strong> ' . intval( $qc['words'] ) . '<br/>';
        echo '<strong>Headings (H2/H3):</strong> ' . intval( $qc['headings'] ) . '<br/>';
        echo '<strong>Citations:</strong> ' . intval( $qc['citations'] ) . '<br/>';
        echo '<strong>Status:</strong> ' . ( $qc['pass'] ? '<span style="color:green">Pass</span>' : '<span style="color:#cc0000">Needs review</span>' ) . '</p>';
        echo '<label><input type="checkbox" name="albp_approve" value="1" ' . checked( ! $noindex, true, false ) . '/> Approve & allow indexing</label>';
    }

    public function save_qc_metabox( $post_id ) {
        if (
    ! isset( $_POST['albp_qc_nonce'] )
    || ! wp_verify_nonce(
        sanitize_text_field( wp_unslash( $_POST['albp_qc_nonce'] ) ),
        'albp-qc-save'
    )
) {
    return;
}
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;
        if ( ! current_user_can( 'edit_post', $post_id ) ) return;

        $approve = ! empty( $_POST['albp_approve'] ) ? (bool) intval( wp_unslash( $_POST['albp_approve'] ) ) : false;
        update_post_meta( $post_id, '_albp_noindex', $approve ? 0 : 1 );
    }

    public function output_noindex_meta() {
        if ( ! is_singular( 'post' ) ) return;
        global $post; if ( ! $post ) return;
        $noindex = (bool) get_post_meta( $post->ID, '_albp_noindex', true );
        if ( $noindex ) {
            echo '<meta name="robots" content="noindex,nofollow" />' . "\n";
        }
    }

    public function plugin_row_meta( $links, $file ) {
        if ( $file !== plugin_basename( __FILE__ ) ) return $links;
        $settings_url = admin_url( 'admin.php?page=' . ALBP_SLUG );
        $links[]      = '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', 'articla-lite-bulk-publisher' ) . '</a>';
        return $links;
    }

    public function ajax_start_processing() {
        check_ajax_referer( 'albp-ajax', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( array( 'msg' => 'forbidden' ), 403 );

        $is_pro   = albp_is_pro( true );
        $rows_raw = isset( $_POST['rows'] ) ? wp_kses_post( wp_unslash( $_POST['rows'] ) ) : '';
        $delay_ms = isset( $_POST['delay_ms'] ) ? intval( wp_unslash( $_POST['delay_ms'] ) ) : 8000;

$cats = array();
if ( isset( $_POST['cats'] ) && is_array( $_POST['cats'] ) ) {
    $cats_post = array_map( 'sanitize_text_field', wp_unslash( $_POST['cats'] ) );
    $cats      = array_map( 'intval', $cats_post );
}

        if ( ! $rows_raw ) wp_send_json_error( array( 'msg' => 'No rows provided. Paste CSV lines first.' ), 400 );
        if ( $delay_ms < 0 || $delay_ms > 120000 ) wp_send_json_error( array( 'msg' => 'Interval ms must be between 0 and 120000.' ), 400 );
        if ( ! $is_pro ) { $delay_ms = 8000; }

        $items = $this->parse_lines( $rows_raw );
        if ( empty( $items ) ) wp_send_json_error( array( 'msg' => 'No valid rows found. Ensure each line starts with a title.' ), 400 );

        $runtime = array( 'items' => $items, 'delay_ms' => $delay_ms, 'cats' => $cats, 'cursor' => 0 );
        update_option( 'albp_runtime', $runtime, false );

        $note = $is_pro ? 'Pro: unlimited batch size & custom delay enabled.' : 'Free: limited to 5 posts per batch, fixed delay 8000ms.';
        wp_send_json_success( array( 'total' => count( $items ), 'msg' => 'Ready to process ' . count( $items ) . ' items. ' . $note, 'delay_ms' => $delay_ms, 'pro' => $is_pro ) );
    }

    public function ajax_process_next() {
        check_ajax_referer( 'albp-ajax', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( array( 'msg' => 'forbidden' ), 403 );

        $runtime = get_option( 'albp_runtime', null );
        if ( ! $runtime || empty( $runtime['items'] ) ) wp_send_json_error( array( 'msg' => 'Nothing to process. Click Start Processing after pasting rows.' ), 400 );

        $cursor          = intval( $runtime['cursor'] );
        $processed_so_far = $cursor;
        $is_pro          = albp_is_pro( false );

        if ( ! $is_pro && $processed_so_far >= 5 ) {
            delete_option( 'albp_runtime' );
            wp_send_json_success( array( 'done' => true, 'remaining' => 0, 'result' => array( 'ok' => true, 'msg' => 'Free version is limited to 5 posts per batch. Upgrade to Pro for unlimited posts.' ) ) );
        }

        if ( $cursor >= count( $runtime['items'] ) ) {
            delete_option( 'albp_runtime' );
            wp_send_json_success( array( 'done' => true, 'msg' => 'All items processed.' ) );
        }

        $gen   = new ALBP_Generator();
        $item  = $runtime['items'][ $cursor ];
        $title = isset( $item['title'] ) ? sanitize_text_field( $item['title'] ) : '';

        $item['cats'] = $runtime['cats'];
        try {
            $result = $gen->generate_and_insert( $item );
        } catch ( Throwable $e ) {
            $result = array( 'ok' => false, 'msg' => 'Unhandled error: ' . $e->getMessage() );
        }

        $runtime['cursor'] = $cursor + 1;
        update_option( 'albp_runtime', $runtime, false );




        $delay = intval( $runtime['delay_ms'] );
        if ( $delay > 0 ) usleep( $delay * 1000 );

        $remaining = count( $runtime['items'] ) - $runtime['cursor'];

        if ( ! $is_pro && $runtime['cursor'] >= 5 ) {
            delete_option( 'albp_runtime' );
            wp_send_json_success( array( 'done' => true, 'remaining' => 0, 'result' => $result, 'delay_ms' => $delay, 'pro_note' => 'Free version is limited to 5 posts per batch. Upgrade to Pro for unlimited posts.' ) );
        }

        wp_send_json_success(
            array(
                'done'      => ( $remaining === 0 ),
                'remaining' => $remaining,
                'result'    => $result,
                'delay_ms'  => $delay,
                'title'     => $title, // <-- keep this
            )
        );
    }

    private function parse_lines( $rows_raw ) {
        $rows_raw = trim( str_replace( "\r", "", $rows_raw ) );
        $lines    = array_filter( array_map( 'trim', explode( "\n", $rows_raw ) ) );
        $items    = array();

        foreach ( $lines as $line ) {
            $parts     = array_map( 'trim', str_getcsv( $line ) );
            $title     = $parts[0] ?? '';
            if ( ! $title ) continue;

            $keywords  = $parts[1] ?? '';
            $sources   = isset( $parts[2] ) ? array_filter( array_map( 'trim', explode( ';', $parts[2] ) ) ) : array();
            $sources   = array_values( array_filter( $sources, function( $u ){ return filter_var( $u, FILTER_VALIDATE_URL ); } ) );
            $outline   = isset( $parts[3] ) ? array_filter( array_map( 'trim', explode( '|', $parts[3] ) ) ) : array();
            $image_query = $parts[4] ?? $title;

            $items[] = compact( 'title', 'keywords', 'sources', 'outline', 'image_query' );
        }
        return $items;
    }
}

new ALBP_Plugin();
