<?php
/**
 * Plugin Name: Armory Atlas for WooCommerce
 * Description: Adds an "Add to Armory Atlas" button on WooCommerce product pages (Firearms, Ammo, Accessories) and lets site owners map product fields to Armory Atlas fields. Deep links to the mobile app with web fallback for import.
 * Version: 1.0.4
 * Author: Armory Atlas
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: armory-atlas
 * Domain Path: /languages
 * 
 * Privacy Policy: This plugin does not collect, store, or transmit personal user data. 
 * Product information is only passed to Armory Atlas when users explicitly click the button.
 * No tracking, analytics, or personal data collection occurs.
 */

if (!defined('ABSPATH')) { exit; }

class Armory_Atlas_Woo {
    const OPT_KEY = 'armory_atlas_settings';
    const NONCE   = 'armory_atlas_nonce';

    private static $instance = null;
    private $rendered_once = false;
    private $settings_page_hook = '';

    public static function init() {
        if (self::$instance === null) { self::$instance = new self(); }
        return self::$instance;
    }

    private function __construct() {
        // Defaults on activation
        register_activation_hook(__FILE__, [__CLASS__, 'activate']);

        // Admin UI
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_post_armory_atlas_apply_recommended', [$this, 'apply_recommended_mapping']);
    add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);

        // Front-end hooks
        add_action('woocommerce_single_product_summary', [$this, 'render_button_hook'], 31);
        add_action('woocommerce_after_add_to_cart_form', [$this, 'render_button_hook_compat'], 10);
        // Additional hooks to improve compatibility across themes/layouts
        add_action('woocommerce_product_meta_end', [$this, 'render_button_hook_compat'], 50);
        add_action('woocommerce_after_single_product_summary', [$this, 'render_button_hook_compat'], 12);
        add_shortcode('armory_atlas_button', [$this, 'render_button_shortcode']);

        // Enqueue front-end JS (for deep-link open + fallback)
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
    }


    public static function activate() {
        $defaults = [
            'enabled'            => 'yes',
            'trigger_categories' => 'guns-firearms,ammunition,gun-parts,magazines,scopes-sights-optics', // category slugs (comma-separated)
            'button_text'        => 'Add to Armory Atlas',
            'button_position'    => 'auto', // auto or shortcode-only
            'button_style'       => 'primary', // primary | secondary | link
            'button_margin_left' => 8,
            'show_on_all_products' => 'no',
            'app_scheme'         => 'ArmoryAtlas://import',
            'web_fallback'       => 'https://www.armoryatlas.com/import',
            'api_base'           => 'https://www.armoryatlas.com/api',
            // Mapping placeholders: {name}, {sku}, {price}, {short_description}, {description}, {image}, {permalink}, {meta:KEY}, {attr:ATTRIBUTE_SLUG}
            'mapping' => [
                'type'        => '{product_category}',                    // derive firearm | ammo | accessory from categories
                'make'        => '{attr:pa_manufacturer}',                // Manufacturer attribute
                'model'       => '{attr:pa_model}',                       // Model attribute
                'caliber'     => '{attr:pa_caliber-gauge}',               // Caliber/Gauge attribute
                'upc'         => '{attr:pa_upc}',                        // UPC attribute
                'sku'         => '{sku}',                                 // Built-in SKU
                // Ammo-specific attributes
                'ammo_type'            => '{attr:pa_product-type}',       // Product Type attribute
                'bullet_weight_grains' => '{attr:pa_bullet-weight-grains}', // Bullet Weight (Grains) attribute
                'units_per_box'        => '{attr:pa_units-per-box}',      // Units per Box attribute
                'cost_per_round'       => '{attr:pa_cost-per-round}',     // Cost per Round attribute
                'case_material'        => '{attr:pa_case-material}',     // Case Material attribute
                'mpn'                  => '{attr:pa_manufacturer-part-number}', // Manufacturer Part Number attribute
                'length'               => '',                             // Cartridge Length / OAL (no default attribute)
                // Category convenience (primary category values)
                'category'      => '{primary_category}',                  // Primary Category (name)
                'category_slug' => '{primary_category_slug}',             // Primary Category (slug)
                'title'       => '{name}',                                // Built-in Name/Title
                'description' => '{description}',                         // Built-in Full Description
                'price'       => '{price}',                               // Built-in Price
                'image_url'   => '{image}',                               // Built-in Main Image URL
                'product_url' => '{permalink}',                          // Built-in Product URL
            ],
        ];

        // Default category type mappings based on your category structure
        $defaults['category_type_map'] = [
            // Ammunition categories
            'ammunition' => 'ammo',
            'handgun-ammunition' => 'ammo',
            'other-ammunition' => 'ammo',
            'rifle-ammunition' => 'ammo',
            'rimfire-ammunition' => 'ammo',
            'shotgun-shells' => 'ammo',
            
            // Firearm categories
            'guns-firearms' => 'firearm',
            'combination-guns' => 'firearm',
            'handguns' => 'firearm',
            'ak-pistols' => 'firearm',
            'ar-pistols' => 'firearm',
            'derringers' => 'firearm',
            'other-handguns' => 'firearm',
            'revolvers' => 'firearm',
            'semi-auto-handguns' => 'firearm',
            'single-shot-handguns' => 'firearm',
            'rifles' => 'firearm',
            'ak-rifles' => 'firearm',
            'ar-rifles' => 'firearm',
            'bolt-action-rifles' => 'firearm',
            'lever-action-rifles' => 'firearm',
            'other-rifles' => 'firearm',
            'pump-action-rifles' => 'firearm',
            'semi-auto-rifles' => 'firearm',
            'single-shot-rifles' => 'firearm',
            'shotguns' => 'firearm',
            'bolt-action-shotguns' => 'firearm',
            'lever-action-shotguns' => 'firearm',
            'other-shotguns' => 'firearm',
            'over-under-shotguns' => 'firearm',
            'pump-action-shotguns' => 'firearm',
            'semi-auto-shotguns' => 'firearm',
            'side-by-side-shotguns' => 'firearm',
            'single-shot-shotguns' => 'firearm',
            
            // Accessory categories
            'gun-parts' => 'accessory',
            'weapon-lights' => 'accessory',
            'magazines' => 'accessory',
            'ak-magazines' => 'accessory',
            'ar-magazines' => 'accessory',
            'handgun-magazines' => 'accessory',
            'rifle-magazines' => 'accessory',
            'shotgun-magazines' => 'accessory',
            'scopes-sights-optics' => 'accessory',
            'binoculars' => 'accessory',
            'gun-scopes' => 'accessory',
            'gun-sights' => 'accessory',
            'laser-sights' => 'accessory',
            'night-vision' => 'accessory',
            'range-finders' => 'accessory',
            'red-dots-sights' => 'accessory',
            'spotting-scopes' => 'accessory',
            'thermal-sights' => 'accessory',
        ];

        $current = get_option(self::OPT_KEY);
        if (!$current) { update_option(self::OPT_KEY, $defaults); }
    }

    public function add_settings_page() {
        $this->settings_page_hook = add_menu_page(
            __('Armory Atlas', 'armory-atlas'),
            __('Armory Atlas', 'armory-atlas'),
            'manage_woocommerce',
            'armory-atlas',
            [$this, 'settings_page_html'],
            'dashicons-admin-generic',
            56
        );
    }

    public function enqueue_admin_assets($hook) {
        if (empty($this->settings_page_hook) || $hook !== $this->settings_page_hook) {
            return;
        }

        $asset_path = plugin_dir_path(__FILE__) . 'admin/js/armory-atlas-admin.js';
        $asset_url  = plugins_url('admin/js/armory-atlas-admin.js', __FILE__);

        if (file_exists($asset_path)) {
            wp_register_script(
                'armory-atlas-admin',
                $asset_url,
                [],
                '1.0.4',
                true
            );
            wp_enqueue_script('armory-atlas-admin');
        }
    }

    public function register_settings() {
        register_setting('armory_atlas_group', self::OPT_KEY, [
            'type'              => 'array',
            'sanitize_callback' => [$this, 'sanitize_settings'],
            'default'           => [],
        ]);

        add_settings_section('armory_atlas_main', __('General Settings', 'armory-atlas'), function(){
            echo '<p>'.esc_html__('Configure deep-link behavior and where the button shows up.', 'armory-atlas').'</p>';
        }, 'armory_atlas');

        add_settings_field('enabled', __('Enable', 'armory-atlas'), [$this, 'field_enabled'], 'armory_atlas', 'armory_atlas_main');
    add_settings_field('trigger_categories', __('Trigger Categories', 'armory-atlas'), [$this, 'field_trigger_categories'], 'armory_atlas', 'armory_atlas_main');
        add_settings_field('button_text', __('Button Text', 'armory-atlas'), [$this, 'field_button_text'], 'armory_atlas', 'armory_atlas_main');
    add_settings_field('button_position', __('Button Position', 'armory-atlas'), [$this, 'field_button_position'], 'armory_atlas', 'armory_atlas_main');
    add_settings_field('button_style', __('Button Style', 'armory-atlas'), [$this, 'field_button_style'], 'armory_atlas', 'armory_atlas_main');
    add_settings_field('button_margin_left', __('Button Left Margin (px)', 'armory-atlas'), [$this, 'field_button_margin_left'], 'armory_atlas', 'armory_atlas_main');
    add_settings_field('show_on_all_products', __('Show on All Products', 'armory-atlas'), [$this, 'field_show_on_all_products'], 'armory_atlas', 'armory_atlas_main');
        add_settings_field('app_scheme', __('App Deep Link Scheme', 'armory-atlas'), [$this, 'field_app_scheme'], 'armory_atlas', 'armory_atlas_main');
        add_settings_field('web_fallback', __('Web Fallback URL', 'armory-atlas'), [$this, 'field_web_fallback'], 'armory_atlas', 'armory_atlas_main');
        

        add_settings_section('armory_atlas_mapping', __('Field Mapping', 'armory-atlas'), function(){
            echo '<p>'.esc_html__('Map WooCommerce product fields/meta to Armory Atlas fields. Choose from built-in fields or product attributes, or use a meta/custom value.', 'armory-atlas').'</p>';
        }, 'armory_atlas');

                foreach ($this->get_armory_fields() as $key => $label) {
                        // translators: %s is the field label (e.g., "Make", "Model", "Caliber")
                        add_settings_field('map_'.$key, sprintf(__('Map: %s', 'armory-atlas'), $label), function() use ($key, $label){
                $opts = get_option(self::OPT_KEY, []);
                $current = isset($opts['mapping'][$key]) ? (string)$opts['mapping'][$key] : '';
                $parsed = $this->parse_current_mapping_value($current);
                $select_val = $parsed['select'];
                $meta_val   = $parsed['meta'];
                $custom_val = $parsed['custom'];

                $options = $this->get_mapping_select_options($key);
                                echo '<div class="aa-map-row" data-map-key="'.esc_attr($key).'" data-label="'.esc_attr(wp_strip_all_tags($label)).'" style="display:flex; gap:8px; align-items:center; flex-wrap:wrap;">';
                echo '<select name="'.esc_attr(self::OPT_KEY).'[mapping_select]['.esc_attr($key).']" style="min-width:260px;">';
                foreach ($options as $group_label => $group) {
                    echo '<optgroup label="'.esc_attr($group_label).'">';
                    foreach ($group as $val => $label) {
                        $sel = selected($select_val, $val, false);
                        echo '<option value="'.esc_attr($val).'" '.esc_attr($sel).'>'.esc_html($label).'</option>';
                    }
                    echo '</optgroup>';
                }
                echo '</select>';
                $meta_placeholder = $this->get_meta_placeholder($key);
                $custom_placeholder = $this->get_custom_placeholder($key);
                echo '<input type="text" placeholder="'.esc_attr($meta_placeholder).'" name="'.esc_attr(self::OPT_KEY).'[mapping_meta]['.esc_attr($key).']" value="'.esc_attr($meta_val).'" class="regular-text aa-map-meta" style="max-width:220px;" />';
                echo '<input type="text" placeholder="'.esc_attr($custom_placeholder).'" name="'.esc_attr(self::OPT_KEY).'[mapping_custom]['.esc_attr($key).']" value="'.esc_attr($custom_val).'" class="regular-text aa-map-custom" style="max-width:320px;" />';
                                echo '<button type="button" class="button button-small aa-suggest">'.esc_html__('Suggest', 'armory-atlas').'</button>';
                echo '</div>';
                echo '<p class="description">'.esc_html__('Pick a source. If you choose Meta, provide the meta key. For Custom, enter a literal or placeholders (e.g., {meta:field_name}, {attr:pa_attribute}).', 'armory-atlas').'</p>';
            }, 'armory_atlas', 'armory_atlas_mapping');
        }

        // Admin enhancements handled via enqueued script.

        // Category → Type mapping
        add_settings_section('armory_atlas_cat_types', __('Category Type Mapping', 'armory-atlas'), function(){
            echo '<p>'.esc_html__('Assign a type (firearm, ammo, accessory) to your product categories. This mapping takes priority when determining the product type.', 'armory-atlas').'</p>';
        }, 'armory_atlas');
        add_settings_field('category_type_map', __('Category → Type', 'armory-atlas'), [$this, 'field_category_type_mapping'], 'armory_atlas', 'armory_atlas_cat_types');
    }

    private function get_armory_fields() : array {
        return [
            'type'        => __('Type (firearm/ammo/accessory)', 'armory-atlas'),
            'make'        => __('Make', 'armory-atlas'),
            'model'       => __('Model', 'armory-atlas'),
            'caliber'     => __('Caliber', 'armory-atlas'),
            'upc'         => __('UPC', 'armory-atlas'),
            'sku'         => __('SKU', 'armory-atlas'),
            // Ammo-specific
            'ammo_type'            => __('Ammo Type (FMJ/HP/etc.)', 'armory-atlas'),
            'bullet_weight_grains' => __('Bullet Weight (grains)', 'armory-atlas'),
            'units_per_box'        => __('Units per Box', 'armory-atlas'),
            'cost_per_round'       => __('Cost per Round', 'armory-atlas'),
            'case_material'        => __('Case Material', 'armory-atlas'),
            'mpn'                  => __('Manufacturer Part Number (MPN)', 'armory-atlas'),
            'length'               => __('Cartridge Length / OAL', 'armory-atlas'),
            // Category convenience fields
            'category'      => __('Primary Category (name)', 'armory-atlas'),
            'category_slug' => __('Primary Category (slug)', 'armory-atlas'),
            'title'       => __('Title', 'armory-atlas'),
            'description' => __('Description', 'armory-atlas'),
            'price'       => __('Price', 'armory-atlas'),
            'image_url'   => __('Image URL', 'armory-atlas'),
            'product_url' => __('Product URL', 'armory-atlas'),
        ];
    }

    public function sanitize_settings($input) {
        $out = get_option(self::OPT_KEY, []);
        $out['enabled']            = (isset($input['enabled']) && $input['enabled'] === 'yes') ? 'yes' : 'no';
        // Accept both selected IDs and slug string for backward compatibility
        $ids = [];
        if (isset($input['trigger_categories_ids']) && is_array($input['trigger_categories_ids'])) {
            foreach ($input['trigger_categories_ids'] as $id) {
                $id = absint($id);
                if ($id > 0) { $ids[] = $id; }
            }
        }
        $out['trigger_categories_ids'] = $ids;
        $out['trigger_categories'] = isset($input['trigger_categories']) ? sanitize_text_field($input['trigger_categories']) : '';
        $out['button_text']        = isset($input['button_text']) ? sanitize_text_field($input['button_text']) : '';
    $out['button_position']    = (isset($input['button_position']) && in_array($input['button_position'], ['auto','shortcode-only'], true)) ? $input['button_position'] : 'auto';
    $out['button_style']       = (isset($input['button_style']) && in_array($input['button_style'], ['primary','secondary','link'], true)) ? $input['button_style'] : 'primary';
    $out['button_margin_left'] = isset($input['button_margin_left']) ? max(0, (int)$input['button_margin_left']) : 8;
    $out['show_on_all_products'] = (isset($input['show_on_all_products']) && $input['show_on_all_products'] === 'yes') ? 'yes' : 'no';
        $out['app_scheme']         = isset($input['app_scheme']) ? esc_url_raw($input['app_scheme']) : '';
        $out['web_fallback']       = isset($input['web_fallback']) ? esc_url_raw($input['web_fallback']) : '';
        $out['api_base']           = isset($input['api_base']) ? esc_url_raw($input['api_base']) : '';

        // Mapping: build from select/meta/custom triples
        $mapping = [];
        $sel = isset($input['mapping_select']) && is_array($input['mapping_select']) ? $input['mapping_select'] : [];
        $meta = isset($input['mapping_meta']) && is_array($input['mapping_meta']) ? $input['mapping_meta'] : [];
        $cus = isset($input['mapping_custom']) && is_array($input['mapping_custom']) ? $input['mapping_custom'] : [];
        foreach (array_keys($this->get_armory_fields()) as $k) {
            $choice = isset($sel[$k]) ? sanitize_text_field($sel[$k]) : '';
            if ($choice === 'meta') {
                $mk = isset($meta[$k]) ? sanitize_key($meta[$k]) : '';
                $mapping[$k] = $mk ? '{meta:'.$mk.'}' : '';
            } elseif ($choice === 'product_category' && $k === 'type') {
                $mapping[$k] = '{product_category}';
            } elseif ($choice === 'primary_category' && $k === 'category') {
                $mapping[$k] = '{primary_category}';
            } elseif ($choice === 'primary_category_slug' && $k === 'category_slug') {
                $mapping[$k] = '{primary_category_slug}';
            } elseif ($choice === 'custom') {
                $cv = isset($cus[$k]) ? sanitize_text_field($cus[$k]) : '';
                $mapping[$k] = $cv;
            } elseif ($choice) {
                // built-ins or attributes
                if (strpos($choice, 'attr:') === 0) {
                    $a = substr($choice, 5);
                    $a = wc_sanitize_taxonomy_name($a);
                    $mapping[$k] = '{attr:'.$a.'}';
                } else {
                    $mapping[$k] = '{'.$choice.'}';
                }
            } else {
                $mapping[$k] = '';
            }
        }
        $out['mapping'] = $mapping;

        // Category → Type map
        $cat_map = [];
        if (isset($input['category_type_map']) && is_array($input['category_type_map'])) {
            foreach ($input['category_type_map'] as $term_id => $v) {
                $tid = absint($term_id);
                $val = is_string($v) ? strtolower(trim($v)) : '';
                if ($tid > 0 && in_array($val, ['firearm','ammo','accessory',''], true)) {
                    if ($val) { $cat_map[$tid] = $val; }
                }
            }
        }
        $out['category_type_map'] = $cat_map;

        return $out;
    }

    public function settings_page_html() {
        if (!current_user_can('manage_woocommerce')) return;
        $opts = get_option(self::OPT_KEY, []);
        echo '<div class="wrap">';
        echo '<h1>'.esc_html__('Armory Atlas for WooCommerce', 'armory-atlas').'</h1>';
        if (isset($_GET['aa_recommended']) && $_GET['aa_recommended'] === '1' && wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'] ?? '')), 'armory_atlas_apply_recommended')) {
            echo '<div class="notice notice-success is-dismissible"><p>'.esc_html__('Recommended category mapping applied. Review and click "Save Changes".', 'armory-atlas').'</p></div>';
        }
        echo '<form method="post" action="options.php">';
        settings_fields('armory_atlas_group');
        do_settings_sections('armory_atlas');
        submit_button();
        echo '</form>';

        echo '<hr/>'; 
    echo '<h2>'.esc_html__('Usage', 'armory-atlas').'</h2>';
    echo '<ol>';
    echo '<li>'.esc_html__('Configure your deep link and fallback URL.', 'armory-atlas').'</li>';
    echo '<li>'.esc_html__('Map fields using placeholders like {sku}, {name}, {meta:upc}, {attr:pa_caliber}.', 'armory-atlas').'</li>';
    echo '<li>'.esc_html__('Ensure your Firearm/Ammo/Accessories categories use the slugs listed in Trigger Categories.', 'armory-atlas').'</li>';
    echo '<li>'.esc_html__('Button is auto-inserted on product pages (position after cart). Or use the shortcode anywhere:', 'armory-atlas').'<br />';
    echo '<code>[armory_atlas_button]</code> '.esc_html__('(uses current product in loops or single product pages)', 'armory-atlas').'<br />';
    echo '<code>[armory_atlas_button text="Add to Armory Atlas"]</code> '.esc_html__('(override button text)', 'armory-atlas').'<br />';
    echo '<code>[armory_atlas_button id="123"]</code> '.esc_html__('(render for a specific product ID)', 'armory-atlas').'<br />';
    echo '<code>[armory_atlas_button sku="ABC-123"]</code> '.esc_html__('(render for a product by SKU)', 'armory-atlas').'</li>';
    echo '</ol>';
        echo '</div>';
    }

    public function field_enabled() {
        $opts = get_option(self::OPT_KEY, []);
        $val = isset($opts['enabled']) ? $opts['enabled'] : 'yes';
        echo '<label><input type="checkbox" name="'.esc_attr(self::OPT_KEY).'[enabled]" value="yes" '.checked('yes', $val, false).' /> '.esc_html__('Enable plugin features', 'armory-atlas').'</label>';
    }

    public function field_trigger_categories() {
        $opts = get_option(self::OPT_KEY, []);
        $selected_ids = isset($opts['trigger_categories_ids']) && is_array($opts['trigger_categories_ids'])
            ? array_map('intval', $opts['trigger_categories_ids']) : [];
        $slug_val = isset($opts['trigger_categories']) ? $opts['trigger_categories'] : '';

        echo '<div style="max-height:240px; overflow:auto; padding:8px; border:1px solid #ccd0d4; border-radius:4px; background:#fff;">';
        $this->render_category_checklist($selected_ids, 0, 0);
        echo '</div>';
        echo '<p class="description" style="margin-top:6px;">'.esc_html__('Select product categories to auto-show the button.', 'armory-atlas').'</p>';

        echo '<hr style="margin:12px 0;" />';
        echo '<label style="display:block; margin-bottom:4px;"><strong>'.esc_html__('Advanced (optional):', 'armory-atlas').'</strong> '.esc_html__('Comma-separated WooCommerce category slugs', 'armory-atlas').'</label>';
        echo '<input type="text" class="regular-text" name="'.esc_attr(self::OPT_KEY).'[trigger_categories]" value="'.esc_attr($slug_val).'" placeholder="firearm,accessories,ammo" />';
        echo '<p class="description">'.esc_html__('Slug list is kept for backward compatibility. IDs above take precedence.', 'armory-atlas').'</p>';

    }

    private function render_category_checklist(array $selected_ids, int $parent = 0, int $depth = 0) : void {
        $terms = get_terms([
            'taxonomy'   => 'product_cat',
            'hide_empty' => false,
            'parent'     => $parent,
            'orderby'    => 'name',
            'order'      => 'ASC',
        ]);
        if (is_wp_error($terms) || empty($terms)) return;
        foreach ($terms as $t) {
            $checked = in_array((int)$t->term_id, $selected_ids, true) ? 'checked' : '';
            $pad = (int) ($depth * 16);
            printf('<label style="display:block; margin-left:%dpx">', esc_attr($pad));
            printf('<input type="checkbox" name="%s[trigger_categories_ids][]" value="%d" %s data-term-id="%d" data-parent-id="%d" /> ', esc_attr(self::OPT_KEY), (int)$t->term_id, esc_attr($checked), (int)$t->term_id, (int)$parent);
            printf('%s <span style="opacity:0.6">(%s)</span>', esc_html($t->name), esc_html($t->slug));
            echo '</label>';
            $this->render_category_checklist($selected_ids, (int)$t->term_id, $depth + 1);
        }
    }

    public function field_category_type_mapping() {
        $opts = get_option(self::OPT_KEY, []);
        $map = isset($opts['category_type_map']) && is_array($opts['category_type_map']) ? $opts['category_type_map'] : [];
        echo '<div style="max-height:320px; overflow:auto; border:1px solid #ccd0d4; border-radius:4px; background:#fff;">';
        echo '<div style="padding:8px;">';
        $this->render_category_type_rows($map, 0, 0);
        echo '</div>';
        echo '</div>';
                echo '<p class="description">'.esc_html__('Leave as “Auto” to let the system infer from names. Child categories inherit if not set. Selecting a parent will auto-fill children currently set to Auto.', 'armory-atlas').'</p>';
        $recommend_url = wp_nonce_url(admin_url('admin-post.php?action=armory_atlas_apply_recommended'), self::NONCE . '_apply_recommended');
        echo '<p style="margin-top:10px;">';
        echo '<a href="'.esc_url($recommend_url).'" class="button button-secondary">'.esc_html__('Apply recommended mapping', 'armory-atlas').'</a> ';
        echo '<span class="description" style="vertical-align:middle;">'.esc_html__('Prefills Ammo/Firearm/Accessory for common roots like “Ammunition”, “Guns & Firearms”, and typical accessories. Review before saving.', 'armory-atlas').'</span>';
        echo '</p>';
    }

    private function render_category_type_rows(array $map, int $parent = 0, int $depth = 0) : void {
        $terms = get_terms([
            'taxonomy'   => 'product_cat',
            'hide_empty' => false,
            'parent'     => $parent,
            'orderby'    => 'name',
            'order'      => 'ASC',
        ]);
        if (is_wp_error($terms) || empty($terms)) return;
        foreach ($terms as $t) {
            $sel = isset($map[$t->term_id]) ? $map[$t->term_id] : '';
            $pad = (int) ($depth * 16);
            printf('<div style="display:flex; align-items:center; gap:8px; margin:4px 0; margin-left:%dpx">', esc_attr($pad));
            printf('<span style="min-width:220px; display:inline-block;">%s <span style="opacity:0.6">(%s)</span></span>', esc_html($t->name), esc_html($t->slug));
            printf('<select name="%s[category_type_map][%d]" style="min-width:180px;" data-term-id="%d" data-parent-id="%d">', esc_attr(self::OPT_KEY), (int)$t->term_id, (int)$t->term_id, (int)$parent);
            printf('<option value="" %s>%s</option>', selected($sel, '', false), esc_html__('Auto', 'armory-atlas'));
            printf('<option value="firearm" %s>%s</option>', selected($sel, 'firearm', false), esc_html__('Firearm', 'armory-atlas'));
            printf('<option value="ammo" %s>%s</option>', selected($sel, 'ammo', false), esc_html__('Ammo', 'armory-atlas'));
            printf('<option value="accessory" %s>%s</option>', selected($sel, 'accessory', false), esc_html__('Accessory', 'armory-atlas'));
            echo '</select>';
            printf('<button type="button" class="button button-small aa-reset-children" data-term-id="%d" title="%s" style="margin-left:6px;">%s</button>', (int)$t->term_id, esc_attr__('Reset children to Auto','armory-atlas'), esc_html__('Reset children','armory-atlas'));
            echo '</div>';
            $this->render_category_type_rows($map, (int)$t->term_id, $depth + 1);
        }
    }

    public function field_button_text() {
        $opts = get_option(self::OPT_KEY, []);
        $val = isset($opts['button_text']) ? $opts['button_text'] : '';
        echo '<input type="text" class="regular-text" name="'.esc_attr(self::OPT_KEY).'[button_text]" value="'.esc_attr($val).'" placeholder="Add to Armory Atlas" />';
    }

    public function field_button_position() {
        $opts = get_option(self::OPT_KEY, []);
        $val = isset($opts['button_position']) ? $opts['button_position'] : 'auto';
        echo '<select name="'.esc_attr(self::OPT_KEY).'[button_position]">';
        echo '<option value="auto" '.selected('auto', $val, false).'>'.esc_html__('Auto (below Add to Cart)', 'armory-atlas').'</option>';
        echo '<option value="shortcode-only" '.selected('shortcode-only', $val, false).'>'.esc_html__('Shortcode only', 'armory-atlas').'</option>';
        echo '</select>';
    }

    public function field_button_style() {
        $opts = get_option(self::OPT_KEY, []);
        $val = isset($opts['button_style']) ? $opts['button_style'] : 'primary';
        echo '<select name="'.esc_attr(self::OPT_KEY).'[button_style]">';
        echo '<option value="primary" '.selected('primary', $val, false).'>'.esc_html__('Primary (solid)', 'armory-atlas').'</option>';
        echo '<option value="secondary" '.selected('secondary', $val, false).'>'.esc_html__('Secondary (outline)', 'armory-atlas').'</option>';
        echo '<option value="link" '.selected('link', $val, false).'>'.esc_html__('Link (text only)', 'armory-atlas').'</option>';
        echo '</select>';
    }

    public function field_button_margin_left() {
        $opts = get_option(self::OPT_KEY, []);
        $val = isset($opts['button_margin_left']) ? (int)$opts['button_margin_left'] : 8;
        echo '<input type="number" min="0" step="1" class="small-text" name="'.esc_attr(self::OPT_KEY).'[button_margin_left]" value="'.esc_attr((string)$val).'" />';
        echo '<span class="description" style="margin-left:6px;">'.esc_html__('Space to the left so it doesn’t crowd other buttons.', 'armory-atlas').'</span>';
    }

    public function field_show_on_all_products() {
        $opts = get_option(self::OPT_KEY, []);
        $val = isset($opts['show_on_all_products']) ? $opts['show_on_all_products'] : 'no';
        echo '<label><input type="checkbox" name="'.esc_attr(self::OPT_KEY).'[show_on_all_products]" value="yes" '.checked('yes', $val, false).' /> '.esc_html__('Ignore Trigger Categories and show on every product page', 'armory-atlas').'</label>';
    }

    public function field_app_scheme() {
        $opts = get_option(self::OPT_KEY, []);
        $val = isset($opts['app_scheme']) ? $opts['app_scheme'] : '';
    echo '<input type="url" class="regular-text code" name="'.esc_attr(self::OPT_KEY).'[app_scheme]" value="'.esc_attr($val).'" placeholder="ArmoryAtlas://import" />';
    echo '<p class="description">'.esc_html__('Custom URL scheme to open the Armory Atlas app (e.g., ArmoryAtlas://import).', 'armory-atlas').'</p>';
    }

    public function field_web_fallback() {
        $opts = get_option(self::OPT_KEY, []);
        $val = isset($opts['web_fallback']) ? $opts['web_fallback'] : '';
    echo '<input type="url" class="regular-text code" name="'.esc_attr(self::OPT_KEY).'[web_fallback]" value="'.esc_attr($val).'" placeholder="https://www.armoryatlas.com/import" />';
        echo '<p class="description">'.esc_html__('Fallback web URL if the app is not installed. Payload is passed via ?data=...', 'armory-atlas').'</p>';
    }


    public function enqueue_assets() {
        $opts = get_option(self::OPT_KEY, []);
        if (empty($opts['enabled']) || $opts['enabled'] !== 'yes') return;
        if (!(function_exists('is_product') && is_product())) return;

        $scheme   = !empty($opts['app_scheme']) ? $opts['app_scheme'] : 'ArmoryAtlas://import';
        $fallback = !empty($opts['web_fallback']) ? $opts['web_fallback'] : 'https://www.armoryatlas.com/import';

        // Enqueue our sleek button styles
    wp_register_style('armory-atlas-style', plugins_url('assets/armory-atlas.css', __FILE__), [], '1.0.4');
        wp_enqueue_style('armory-atlas-style');
        // Optional: if deep-link helper script exists, enqueue with config
        $dl_handle = 'armory-atlas-dl';
        $dl_src = plugins_url('public/deep-link.js', __FILE__);
        // Only enqueue if file exists in plugin dir (avoid 404s)
        if (file_exists(plugin_dir_path(__FILE__) . 'public/deep-link.js')) {
            wp_register_script($dl_handle, $dl_src, [], '1.0.4', true);
            wp_enqueue_script($dl_handle);
            wp_add_inline_script($dl_handle,
                'window.ArmoryAtlasConfig = { scheme: ' . wp_json_encode($scheme) . ', fallback: ' . wp_json_encode($fallback) . ' };',
                'before'
            );
        }
    }

    public function render_button_hook() {
        $opts = get_option(self::OPT_KEY, []);
        if (empty($opts['enabled']) || $opts['enabled'] !== 'yes') return;
        if (isset($opts['button_position']) && $opts['button_position'] === 'shortcode-only') return;

        global $product;
        if (!$product instanceof WC_Product) return;
        if (!$this->should_show_for_product($product)) return;
        if ($this->rendered_once) return; // avoid duplicate renders across hooks

        echo wp_kses_post($this->button_html($product));
        $this->rendered_once = true;
    }

    public function render_button_hook_compat() {
        // Fallback location for themes not using the default summary hook
        $opts = get_option(self::OPT_KEY, []);
        if (empty($opts['enabled']) || $opts['enabled'] !== 'yes') return;
        if (isset($opts['button_position']) && $opts['button_position'] === 'shortcode-only') return;
        if ($this->rendered_once) return;
        global $product;
        if (!$product instanceof WC_Product) return;
        if (!$this->should_show_for_product($product)) return;
        echo wp_kses_post($this->button_html($product));
        $this->rendered_once = true;
    }

    public function render_button_shortcode($atts = []) {
        $atts = shortcode_atts([
            'id'   => '',
            'sku'  => '',
            'text' => '',
        ], $atts, 'armory_atlas_button');

        $prod = null;
        if (!empty($atts['id'])) {
            $prod = wc_get_product(absint($atts['id']));
        } elseif (!empty($atts['sku'])) {
            $pid = wc_get_product_id_by_sku(sanitize_text_field($atts['sku']));
            if ($pid) { $prod = wc_get_product((int)$pid); }
        } else {
            global $product;
            if ($product instanceof WC_Product) { $prod = $product; }
        }

        if (!$prod instanceof WC_Product) return '';
        $text_override = is_string($atts['text']) ? trim($atts['text']) : '';
        return $this->button_html($prod, $text_override);
    }

    private function should_show_for_product(WC_Product $product) : bool {
        $opts = get_option(self::OPT_KEY, []);
        if (!empty($opts['show_on_all_products']) && $opts['show_on_all_products'] === 'yes') {
            return true;
        }
        // Prefer ID-based selection if available
        $trigger_ids = isset($opts['trigger_categories_ids']) && is_array($opts['trigger_categories_ids'])
            ? array_filter(array_map('intval', $opts['trigger_categories_ids'])) : [];
        if (!empty($trigger_ids)) {
            $product_ids = wp_get_post_terms($product->get_id(), 'product_cat', ['fields' => 'ids']);
            if (is_wp_error($product_ids)) return false;
            if (empty($product_ids)) return false;
            // Include ancestors of product terms to allow parent-category triggers to match children
            $product_with_ancestors = [];
            foreach ($product_ids as $pid) {
                $product_with_ancestors[] = (int)$pid;
                $anc = get_ancestors((int)$pid, 'product_cat', 'taxonomy');
                if (!is_wp_error($anc) && !empty($anc)) {
                    foreach ($anc as $aid) { $product_with_ancestors[] = (int)$aid; }
                }
            }
            $product_with_ancestors = array_unique($product_with_ancestors);
            return (bool) array_intersect($trigger_ids, $product_with_ancestors);
        }

        // Fallback to slug-based matching (legacy option)
        $triggers = isset($opts['trigger_categories']) ? array_filter(array_map('trim', explode(',', strtolower($opts['trigger_categories'])))) : [];
        if (empty($triggers)) return true; // show if not configured
        $terms = get_the_terms($product->get_id(), 'product_cat');
        if (empty($terms) || is_wp_error($terms)) return false;
        $slugs = array_map(function($t){ return strtolower($t->slug); }, $terms);
        return (bool) array_intersect($triggers, $slugs);
    }

    private function button_html(WC_Product $product, ?string $override_text = null) : string {
        $opts     = get_option(self::OPT_KEY, []);
        $btn_txt  = ($override_text !== null && $override_text !== '') ? $override_text : (!empty($opts['button_text']) ? $opts['button_text'] : __('Add to Armory Atlas', 'armory-atlas'));
        $payload  = $this->build_payload($product);
        $encoded  = $this->base64url(json_encode($payload));
        $scheme   = !empty($opts['app_scheme']) ? $opts['app_scheme'] : 'ArmoryAtlas://import';
        $fallback = !empty($opts['web_fallback']) ? $opts['web_fallback'] : 'https://www.armoryatlas.com/import';

        $type_hint = '';
        if (!empty($payload['fields']['type'])) {
            $type_hint = strtolower((string)$payload['fields']['type']);
        }

        $utm_params = $this->build_utm_params($product, $type_hint);
        if (!empty($utm_params)) {
            $payload['utm'] = $utm_params;
            $encoded = $this->base64url(json_encode($payload));
        }

        $link_query = [ 'data' => $encoded ];
        if (!empty($type_hint)) {
            $link_query['type'] = $type_hint;
        }
        foreach ($utm_params as $key => $value) {
            if ($value !== '' && $value !== null) {
                $link_query[$key] = $value;
            }
        }

        $scheme_url   = $this->build_url_with_query($scheme, $link_query);
        $fallback_url = $this->build_url_with_query($fallback, $link_query);

    $template = $this->get_template_path('button.php');
        $icon_url = plugins_url('assets/icons/icon.png', __FILE__);
        // Compute classes and spacing based on settings
        $style = isset($opts['button_style']) ? $opts['button_style'] : 'primary';
        $margin_left = isset($opts['button_margin_left']) ? (int)$opts['button_margin_left'] : 8;
        $btn_class = 'armoryatlas-link armory-atlas-button';
        if ($style === 'primary') { $btn_class .= ' button alt'; }
        elseif ($style === 'secondary') { $btn_class .= ' button'; }
        // link style keeps base class only
        $wrapper_style = 'margin-left:'.max(0,$margin_left).'px;';
        $wrapper_class = 'armoryatlas-cta armory-atlas-cta';
        $vars = [
            'btn_text'     => $btn_txt,
            'scheme_url'   => $scheme_url,
            'fallback_url' => $fallback_url,
            'icon_url'     => $icon_url,
            'btn_class'    => $btn_class,
            'wrapper_class'=> $wrapper_class,
            'wrapper_style'=> $wrapper_style,
        ];
        ob_start();
        $btn_text = $vars['btn_text'];
        $scheme_url = $vars['scheme_url'];
        $fallback_url = $vars['fallback_url'];
        $icon_url = $vars['icon_url'];
        $btn_class = $vars['btn_class'];
    $wrapper_class = $vars['wrapper_class'];
        $wrapper_style = $vars['wrapper_style'];
        include $template;
        return (string) ob_get_clean();
    }

    private function build_url_with_query(string $base, array $params) : string {
        if (function_exists('wp_parse_url')) {
            $parts = wp_parse_url($base);
        } else {
            $parts = parse_url($base);
        }
        if ($parts === false) {
            return esc_url($base);
        }

        $query = [];
        if (!empty($parts['query'])) {
            parse_str($parts['query'], $query);
        }
        foreach ($params as $key => $value) {
            if ($value === '' || $value === null) {
                continue;
            }
            $query[$key] = (string)$value;
        }

        $built_query = http_build_query($query, '', '&', PHP_QUERY_RFC3986);

        $scheme   = isset($parts['scheme']) ? $parts['scheme'] . '://' : '';
        $host     = isset($parts['host']) ? $parts['host'] : '';
        $port     = isset($parts['port']) ? ':' . $parts['port'] : '';
        $path     = isset($parts['path']) ? $parts['path'] : '';
        $fragment = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';

        $url = $scheme;
        if (!empty($host)) {
            $url .= $host;
        }
        $url .= $port . $path;
        if ($built_query !== '') {
            $url .= '?' . $built_query;
        }
        $url .= $fragment;

        return esc_url($url);
    }

    private function build_utm_params(WC_Product $product, string $type_hint = '') : array {
        $base = [
            'utm_source'   => 'woocommerce',
            'utm_medium'   => 'integration',
            'utm_campaign' => 'armory_atlas_plugin',
        ];

        $site_host = wp_parse_url(home_url('/'), PHP_URL_HOST);
        if (!empty($site_host)) {
            $base['utm_content'] = sanitize_text_field($site_host);
        }

        $slug = $product->get_slug();
        if (!empty($slug)) {
            $base['utm_term'] = sanitize_title($slug);
        }

        $sku = $product->get_sku();
        if (!empty($sku)) {
            $base['utm_id'] = sanitize_text_field($sku);
        }

        if (!empty($type_hint)) {
            $base['utm_campaign'] .= '_' . sanitize_key($type_hint);
        }

        /**
         * Filter the UTM parameters added to scheme/fallback URLs and payloads.
         *
         * @param array       $base    Associative array of utm_* parameters.
         * @param WC_Product  $product Product in context.
         * @param string      $type    Lowercase type hint (firearm/ammo/accessory).
         */
        return apply_filters('armoryatlas_utm_parameters', $base, $product, $type_hint);
    }

    private function build_payload(WC_Product $product) : array {
        $opts    = get_option(self::OPT_KEY, []);
        $mapping = isset($opts['mapping']) && is_array($opts['mapping']) ? $opts['mapping'] : [];

        $data = [
            'source'      => 'woocommerce',
            'site'        => home_url('/'),
            'product_id'  => $product->get_id(),
            'timestamp'   => time(),
            'fields'      => [],
        ];

        foreach ($this->get_armory_fields() as $key => $label) {
            $tpl = isset($mapping[$key]) ? $mapping[$key] : '';
            $val = $this->resolve_placeholder($tpl, $product);
            if ($val !== '' && $val !== null) {
                $data['fields'][$key] = $val;
            }
        }

        // Ensure type is filled for prefill (firearm/ammo/accessory)
        if (empty($data['fields']['type'])) {
            $detected = $this->detect_product_type($product);
            if (!empty($detected)) {
                $data['fields']['type'] = $detected;
            }
        }

        // Include an explicit prefill hint for the web importer
        if (!empty($data['fields']['type'])) {
            $data['prefill'] = [ 'type' => $data['fields']['type'] ];
        }

        // Harmonize ammo-specific keys to common names the app expects
        if (!empty($data['fields'])) {
            // Brand/make: app prefers `make` for brand; we also keep `brand` copy
            if (!empty($data['fields']['make']) && empty($data['fields']['brand'])) {
                $data['fields']['brand'] = $data['fields']['make'];
            }
            // Ammo type
            if (!empty($data['fields']['ammo_type']) && empty($data['fields']['type_detail'])) {
                $data['fields']['type_detail'] = $data['fields']['ammo_type'];
            }
            // Bullet weight (grains)
            if (!empty($data['fields']['bullet_weight_grains']) && empty($data['fields']['grains'])) {
                $data['fields']['grains'] = $data['fields']['bullet_weight_grains'];
            }
            // Units per box
            if (!empty($data['fields']['units_per_box']) && empty($data['fields']['rounds'])) {
                $data['fields']['rounds'] = $data['fields']['units_per_box'];
            }
            // Cost per round
            if (!empty($data['fields']['cost_per_round']) && empty($data['fields']['ppr'])) {
                $data['fields']['ppr'] = $data['fields']['cost_per_round'];
            }
            // Case material
            if (!empty($data['fields']['case_material']) && empty($data['fields']['case'])) {
                $data['fields']['case'] = $data['fields']['case_material'];
            }
            // Manufacturer Part Number -> sku if sku is empty
            if (!empty($data['fields']['mpn']) && empty($data['fields']['sku'])) {
                $data['fields']['sku'] = $data['fields']['mpn'];
            }
            // Length passthrough
            if (!empty($data['fields']['length']) && empty($data['fields']['oal'])) {
                $data['fields']['oal'] = $data['fields']['length'];
            }
        }

        // Attach primary category and category context for better mapping on importer side
        $cat_ctx = $this->build_category_context($product);
        if (!empty($cat_ctx)) {
            $data['category'] = $cat_ctx; // rich context: primary, ancestors, all
            // Provide simple fallbacks on fields for convenience
            if (!empty($cat_ctx['primary']['name']) && empty($data['fields']['category'])) {
                $data['fields']['category'] = (string)$cat_ctx['primary']['name'];
            }
            if (!empty($cat_ctx['primary']['slug']) && empty($data['fields']['category_slug'])) {
                $data['fields']['category_slug'] = (string)$cat_ctx['primary']['slug'];
            }
        }

        $type_hint = '';
        if (!empty($data['fields']['type'])) {
            $type_hint = (string)$data['fields']['type'];
        }

        $utm = $this->build_utm_params($product, $type_hint);
        if (!empty($utm)) {
            $data['utm'] = $utm;
        }

        return $data;
    }

    /**
     * Build detailed category context for a product:
     * - primary: deepest category by hierarchy (id, name, slug, depth)
     * - ancestors: lineage from root to parent of primary (array of term objects simplified)
     * - all: flat list of all assigned categories (id, name, slug, parent)
     */
    private function build_category_context(WC_Product $product) : array {
        $terms = get_the_terms($product->get_id(), 'product_cat');
        if (empty($terms) || is_wp_error($terms)) {
            return [];
        }

        // Normalize terms into a simple array
        $all = [];
        foreach ($terms as $t) {
            if (!is_object($t)) { continue; }
            $all[] = [
                'id'     => (int)$t->term_id,
                'name'   => (string)$t->name,
                'slug'   => (string)$t->slug,
                'parent' => (int)$t->parent,
            ];
        }

        // Pick the deepest category as primary
        $primary = null;
        $maxDepth = -1;
        foreach ($terms as $t) {
            if (!is_object($t)) { continue; }
            $depth = $this->get_term_depth((int)$t->term_id, 'product_cat');
            if ($depth > $maxDepth) {
                $maxDepth = $depth;
                $primary = $t;
            }
        }

        if (!$primary) {
            return [ 'all' => $all ];
        }

        // Build ancestor chain (root -> ... -> parent of primary)
        $ancestor_ids = get_ancestors((int)$primary->term_id, 'product_cat', 'taxonomy');
        $ancestor_ids = array_reverse(array_map('intval', $ancestor_ids));
        $ancestors = [];
        foreach ($ancestor_ids as $aid) {
            $term = get_term($aid, 'product_cat');
            if (is_wp_error($term) || !$term) { continue; }
            $ancestors[] = [
                'id'     => (int)$term->term_id,
                'name'   => (string)$term->name,
                'slug'   => (string)$term->slug,
                'parent' => (int)$term->parent,
            ];
        }

        return [
            'primary' => [
                'id'     => (int)$primary->term_id,
                'name'   => (string)$primary->name,
                'slug'   => (string)$primary->slug,
                'parent' => (int)$primary->parent,
                'depth'  => $maxDepth,
            ],
            'ancestors' => $ancestors,
            'all'       => $all,
        ];
    }

    /**
     * Compute taxonomy depth for a term: 0 for root, 1 for direct child, etc.
     */
    private function get_term_depth(int $term_id, string $taxonomy) : int {
        $anc = get_ancestors($term_id, $taxonomy, 'taxonomy');
        if (is_wp_error($anc) || empty($anc)) { return 0; }
        return count($anc);
    }

    private function detect_product_type(WC_Product $product) : string {
        $opts = get_option(self::OPT_KEY, []);
        $map = isset($opts['category_type_map']) && is_array($opts['category_type_map']) ? $opts['category_type_map'] : [];
        $terms = get_the_terms($product->get_id(), 'product_cat');
        if (!empty($terms) && !is_wp_error($terms)) {
            // 1) Explicit mapping by category or its ancestors
            foreach ($terms as $t) {
                $type = $this->resolve_type_from_category_hierarchy((int)$t->term_id, $map);
                if ($type) return $type;
            }
            // 2) Heuristics by slug keywords (enhanced for your specific categories)
            $names = array_map(function($t){ return strtolower($t->name); }, $terms);
            $slugs = array_map(function($t){ return strtolower($t->slug); }, $terms);
            $hay = implode(' ', array_merge($slugs, $names));
            
            // Firearms (match your specific category structure)
            if (preg_match('/\b(guns[-\s]?firearms|handguns?|ak[-\s]?pistols?|ar[-\s]?pistols?|derringers?|revolvers?|semi[-\s]?auto[-\s]?handguns?|single[-\s]?shot[-\s]?handguns?|rifles?|ak[-\s]?rifles?|ar[-\s]?rifles?|bolt[-\s]?action[-\s]?rifles?|lever[-\s]?action[-\s]?rifles?|pump[-\s]?action[-\s]?rifles?|semi[-\s]?auto[-\s]?rifles?|single[-\s]?shot[-\s]?rifles?|shotguns?|bolt[-\s]?action[-\s]?shotguns?|lever[-\s]?action[-\s]?shotguns?|over[-\s]?under[-\s]?shotguns?|pump[-\s]?action[-\s]?shotguns?|semi[-\s]?auto[-\s]?shotguns?|side[-\s]?by[-\s]?side[-\s]?shotguns?|single[-\s]?shot[-\s]?shotguns?|combination[-\s]?guns)\b/i', $hay)) {
                return 'firearm';
            }
            // Ammo (match your specific ammunition categories)
            if (preg_match('/\b(ammunition|handgun[-\s]?ammunition|rifle[-\s]?ammunition|rimfire[-\s]?ammunition|shotgun[-\s]?shells|other[-\s]?ammunition)\b/i', $hay)) {
                return 'ammo';
            }
            // Accessories (match your specific accessory categories)
            if (preg_match('/\b(gun[-\s]?parts|weapon[-\s]?lights?|magazines?|ak[-\s]?magazines?|ar[-\s]?magazines?|handgun[-\s]?magazines?|rifle[-\s]?magazines?|shotgun[-\s]?magazines?|scopes?[-\s]?sights?[-\s]?optics?|binoculars?|gun[-\s]?scopes?|gun[-\s]?sights?|laser[-\s]?sights?|night[-\s]?vision|range[-\s]?finders?|red[-\s]?dots?[-\s]?sights?|spotting[-\s]?scopes?|thermal[-\s]?sights?)\b/i', $hay)) {
                return 'accessory';
            }
        }
        // Try attribute fallback
        $attr = $this->get_product_attribute($product, 'product_type');
        $attr = strtolower(trim($attr));
        if (in_array($attr, ['firearm','ammo','ammunition','accessory','accessories'], true)) {
            if ($attr === 'ammunition') return 'ammo';
            if ($attr === 'accessories') return 'accessory';
            return $attr;
        }
        return '';
    }

    private function resolve_type_from_category_hierarchy(int $term_id, array $map) : string {
        $seen = [];
        $current = $term_id;
        while ($current && !in_array($current, $seen, true)) {
            $seen[] = $current;
            if (isset($map[$current]) && in_array($map[$current], ['firearm','ammo','accessory'], true)) {
                return $map[$current];
            }
            $term = get_term($current, 'product_cat');
            if (is_wp_error($term) || !$term || !$term->parent) break;
            $current = (int)$term->parent;
        }
        return '';
    }

    private function resolve_placeholder(string $tpl = null, WC_Product $product) {
        if (empty($tpl)) return '';
        $tpl = trim($tpl);

        // Simple literal (no braces) — return as-is
        if (strpos($tpl, '{') === false) { return $tpl; }

        // Replace supported placeholders within the template string
        $replacements = [
            '{name}'              => $product->get_name(),
            '{sku}'               => $product->get_sku(),
            '{price}'             => wc_get_price_to_display($product),
            '{short_description}' => wp_strip_all_tags($product->get_short_description()),
            '{description}'       => wp_strip_all_tags($product->get_description()),
            '{permalink}'         => get_permalink($product->get_id()),
            '{image}'             => $this->get_product_image_url($product),
        ];

        $out = strtr($tpl, $replacements);

        // Product category -> map to detected type (firearm/ammo/accessory)
        if (strpos($out, '{product_category}') !== false) {
            $det = $this->detect_product_type($product);
            $out = str_replace('{product_category}', $det, $out);
        }

        // Primary category placeholders
        if (strpos($out, '{primary_category') !== false) {
            $ctx = $this->build_category_context($product);
            $primName = isset($ctx['primary']['name']) ? (string)$ctx['primary']['name'] : '';
            $primSlug = isset($ctx['primary']['slug']) ? (string)$ctx['primary']['slug'] : '';
            $out = str_replace('{primary_category}', $primName, $out);
            $out = str_replace('{primary_category_slug}', $primSlug, $out);
        }

        // Meta pattern {meta:key}
        if (preg_match_all('/\{meta:([^}]+)\}/i', $out, $m, PREG_SET_ORDER)) {
            foreach ($m as $match) {
                $meta_key = sanitize_key($match[1]);
                $val      = get_post_meta($product->get_id(), $meta_key, true);
                $out      = str_replace($match[0], is_scalar($val) ? (string)$val : '', $out);
            }
        }

        // Attribute pattern {attr:slug}
        if (preg_match_all('/\{attr:([^}]+)\}/i', $out, $m2, PREG_SET_ORDER)) {
            foreach ($m2 as $match) {
                $slug = wc_sanitize_taxonomy_name($match[1]);
                $val  = $this->get_product_attribute($product, $slug);
                $out  = str_replace($match[0], is_scalar($val) ? (string)$val : '', $out);
            }
        }

        return trim($out);
    }

    private function get_product_image_url(WC_Product $product) : string {
        $img_id = $product->get_image_id();
        if ($img_id) {
            $src = wp_get_attachment_image_src($img_id, 'full');
            if ($src && !empty($src[0])) return (string)$src[0];
        }
        return '';
    }

    private function get_product_attribute(WC_Product $product, string $attr_slug) : string {
        // Woo attributes are stored as taxonomy like pa_caliber
        $taxonomy = strtolower($attr_slug);
        if (strpos($taxonomy, 'pa_') !== 0) {
            $taxonomy = 'pa_' . $taxonomy;
        }
        $terms = wc_get_product_terms($product->get_id(), $taxonomy, ['fields' => 'names']);
        if (!empty($terms) && !is_wp_error($terms)) {
            return implode(', ', $terms);
        }
        return '';
    }

    public function apply_recommended_mapping() {
        if (!current_user_can('manage_woocommerce')) { 
            wp_die(esc_html__('Insufficient permissions', 'armory-atlas')); 
        }
        check_admin_referer(self::NONCE . '_apply_recommended');

        $opts = get_option(self::OPT_KEY, []);
        $map  = isset($opts['category_type_map']) && is_array($opts['category_type_map']) ? $opts['category_type_map'] : [];

        $terms = get_terms([
            'taxonomy'   => 'product_cat',
            'hide_empty' => false,
            'parent'     => 0,
            'orderby'    => 'name',
            'order'      => 'ASC',
        ]);
        if (!is_wp_error($terms) && !empty($terms)) {
            foreach ($terms as $t) {
                $name = is_string($t->name) ? $t->name : '';
                $slug = is_string($t->slug) ? $t->slug : '';
                $type = $this->classify_category_type($name, $slug);
                if ($type && (!isset($map[$t->term_id]) || empty($map[$t->term_id]))) {
                    $map[$t->term_id] = $type;
                }
            }
        }
        $opts['category_type_map'] = $map;
        update_option(self::OPT_KEY, $opts);

        wp_safe_redirect(admin_url('admin.php?page=armory-atlas&aa_recommended=1'));
        exit;
    }

    private function classify_category_type(string $name, string $slug) : string {
        $hay = strtolower($name.' '.$slug);
        $ammo = [ 'ammo', 'ammunition', 'round', 'rounds', 'cartridge', 'cartridges', 'shotshell', 'shotshells', 'shells' ];
        foreach ($ammo as $k) { if (strpos($hay, $k) !== false) return 'ammo'; }
        $fire = [ 'firearm', 'firearms', 'gun', 'guns', 'pistol', 'pistols', 'rifle', 'rifles', 'shotgun', 'shotguns', 'handgun', 'handguns', 'revolver', 'revolvers', 'carbine', 'carbines' ];
        foreach ($fire as $k) { if (strpos($hay, $k) !== false) return 'firearm'; }
        $acc = [ 'accessory', 'accessories', 'optic', 'optics', 'scope', 'scopes', 'sight', 'sights', 'red dot', 'magazine', 'magazines', 'mag', 'holster', 'holsters', 'sling', 'slings', 'mount', 'mounts', 'rail', 'rails', 'grip', 'grips', 'stock', 'stocks', 'chassis', 'case', 'cases', 'bag', 'bags', 'bipod', 'tripod', 'clean', 'cleaning', 'tool', 'tools', 'flashlight', 'light', 'laser', 'rangefinder', 'safe', 'safes', 'suppressor', 'silencer', 'muzzle', 'brake', 'compensator' ];
        foreach ($acc as $k) { if (strpos($hay, $k) !== false) return 'accessory'; }
        return '';
    }

    private function get_mapping_select_options(string $field_key = '') : array {
        $builtins = [
            '' => __('— Select —', 'armory-atlas'),
            'name' => __('Name/Title', 'armory-atlas'),
            'sku' => __('SKU', 'armory-atlas'),
            'price' => __('Price', 'armory-atlas'),
            'short_description' => __('Short Description', 'armory-atlas'),
            'description' => __('Full Description', 'armory-atlas'),
            'image' => __('Main Image URL', 'armory-atlas'),
            'permalink' => __('Product URL', 'armory-atlas'),
        ];
        if ($field_key === 'type') {
            $builtins = array_merge(['product_category' => __('Product category (auto: firearm/ammo/accessory)', 'armory-atlas')], $builtins);
        }
        if ($field_key === 'category') {
            $builtins = array_merge([
                'primary_category' => __('Primary Category (name)', 'armory-atlas')
            ], $builtins);
        }
        if ($field_key === 'category_slug') {
            $builtins = array_merge([
                'primary_category_slug' => __('Primary Category (slug)', 'armory-atlas')
            ], $builtins);
        }

        $attrs = [];
        $taxes = function_exists('wc_get_attribute_taxonomies') ? wc_get_attribute_taxonomies() : [];
        if (!empty($taxes)) {
            foreach ($taxes as $tax) {
                if (empty($tax->attribute_name)) continue;
                $slug = 'pa_'.sanitize_title($tax->attribute_name);
                $label = !empty($tax->attribute_label) ? $tax->attribute_label : $tax->attribute_name;
                // translators: %1$s is the attribute label, %2$s is the attribute slug
                $attrs['attr:'.$slug] = sprintf(__('Attribute: %1$s (%2$s)', 'armory-atlas'), $label, $slug);
            }
        }

        return [
            __('Built-in', 'armory-atlas') => $builtins,
            __('Attributes', 'armory-atlas') => (!empty($attrs) ? $attrs : ['' => __('No attributes found', 'armory-atlas')]),
            __('Other', 'armory-atlas') => [
                'meta' => __('Meta (enter key)', 'armory-atlas'),
                'custom' => __('Custom (literal or placeholder)', 'armory-atlas'),
            ],
        ];
    }

    private function parse_current_mapping_value(string $current) : array {
        $current = trim($current);
        if ($current === '') return ['select' => '', 'meta' => '', 'custom' => ''];
        // {meta:key}
        if (preg_match('/^\{meta:([^}]+)\}$/i', $current, $m)) {
            return ['select' => 'meta', 'meta' => sanitize_key($m[1]), 'custom' => ''];
        }
        // {product_category}
        if (preg_match('/^\{product_category\}$/i', $current)) {
            return ['select' => 'product_category', 'meta' => '', 'custom' => ''];
        }
        // {primary_category}
        if (preg_match('/^\{primary_category\}$/i', $current)) {
            return ['select' => 'primary_category', 'meta' => '', 'custom' => ''];
        }
        // {primary_category_slug}
        if (preg_match('/^\{primary_category_slug\}$/i', $current)) {
            return ['select' => 'primary_category_slug', 'meta' => '', 'custom' => ''];
        }
        // {attr:slug}
        if (preg_match('/^\{attr:([^}]+)\}$/i', $current, $m)) {
            $slug = wc_sanitize_taxonomy_name($m[1]);
            return ['select' => 'attr:'.$slug, 'meta' => '', 'custom' => ''];
        }
        // {builtin}
        if (preg_match('/^\{([a-z_]+)\}$/i', $current, $m)) {
            return ['select' => strtolower($m[1]), 'meta' => '', 'custom' => ''];
        }
        // Fallback: treat as custom
        return ['select' => 'custom', 'meta' => '', 'custom' => $current];
    }

    private function base64url(string $s) : string {
        return rtrim(strtr(base64_encode($s), '+/', '-_'), '=');
    }

    /**
     * Get appropriate meta key placeholder for each field
     */
    private function get_meta_placeholder(string $key) : string {
        $placeholders = [
            'type' => 'meta key (e.g. product_type)',
            'make' => 'meta key (e.g. manufacturer)',
            'model' => 'meta key (e.g. model)',
            'caliber' => 'meta key (e.g. caliber)',
            'upc' => 'meta key (e.g. upc)',
            'sku' => 'meta key (e.g. sku)',
            'ammo_type' => 'meta key (e.g. product_type)',
            'bullet_weight_grains' => 'meta key (e.g. bullet_weight)',
            'units_per_box' => 'meta key (e.g. units_per_box)',
            'cost_per_round' => 'meta key (e.g. cost_per_round)',
            'case_material' => 'meta key (e.g. case_material)',
            'mpn' => 'meta key (e.g. manufacturer_part_number)',
            'length' => 'meta key (e.g. cartridge_length)',
            'category' => 'meta key (e.g. primary_category)',
            'category_slug' => 'meta key (e.g. primary_category_slug)',
            'title' => 'meta key (e.g. product_title)',
            'description' => 'meta key (e.g. product_description)',
            'price' => 'meta key (e.g. product_price)',
            'image_url' => 'meta key (e.g. product_image)',
            'product_url' => 'meta key (e.g. product_url)',
        ];
        return isset($placeholders[$key]) ? $placeholders[$key] : 'meta key (e.g. custom_field)';
    }

    /**
     * Get appropriate custom placeholder for each field
     */
    private function get_custom_placeholder(string $key) : string {
        $placeholders = [
            'type' => 'custom literal or {product_category}',
            'make' => 'custom literal or {attr:pa_brand}',
            'model' => 'custom literal or {name}',
            'caliber' => 'custom literal or {attr:pa_caliber}',
            'upc' => 'custom literal or {meta:upc}',
            'sku' => 'custom literal or {sku}',
            'ammo_type' => 'custom literal or {attr:pa_product_type}',
            'bullet_weight_grains' => 'custom literal or {attr:pa_bullet_weight}',
            'units_per_box' => 'custom literal or {attr:pa_units_per_box}',
            'cost_per_round' => 'custom literal or {attr:pa_cost_per_round}',
            'case_material' => 'custom literal or {attr:pa_case_material}',
            'mpn' => 'custom literal or {attr:pa_manufacturer_part_number}',
            'length' => 'custom literal or {meta:cartridge_length}',
            'category' => 'custom literal or {primary_category}',
            'category_slug' => 'custom literal or {primary_category_slug}',
            'title' => 'custom literal or {name}',
            'description' => 'custom literal or {description}',
            'price' => 'custom literal or {price}',
            'image_url' => 'custom literal or {image}',
            'product_url' => 'custom literal or {permalink}',
        ];
        return isset($placeholders[$key]) ? $placeholders[$key] : 'custom literal or placeholder';
    }

    private function get_template_path(string $file) : string {
        $theme_child = trailingslashit(get_stylesheet_directory()) . 'armory-atlas/' . ltrim($file, '/');
        if (is_readable($theme_child)) return $theme_child;
        $theme_parent = trailingslashit(get_template_directory()) . 'armory-atlas/' . ltrim($file, '/');
        if (is_readable($theme_parent)) return $theme_parent;
        $plugin_path = plugin_dir_path(__FILE__) . 'templates/' . ltrim($file, '/');
        return $plugin_path;
    }
}

Armory_Atlas_Woo::init();