<?php
/**
 * TOC Generator Class
 *
 * @package AutoTOCSEO
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Auto_TOC_SEO_TOC_Generator
 */
class Auto_TOC_SEO_TOC_Generator {

	/**
	 * Instance of this class.
	 *
	 * @var Auto_TOC_SEO_TOC_Generator
	 */
	private static $instance = null;

	/**
	 * Get instance of the class.
	 *
	 * @return Auto_TOC_SEO_TOC_Generator
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_filter( 'the_content', array( $this, 'auto_insert_toc' ), 100 );
	}

	/**
	 * Automatically insert TOC into content.
	 *
	 * @param string $content Post content.
	 * @return string Modified content.
	 */
	public function auto_insert_toc( $content ) {
		if ( ! is_singular() || ! is_main_query() ) {
			return $content;
		}

		$settings = get_option( 'auto_toc_seo_settings', array() );
		$auto_insert = isset( $settings['auto_insert'] ) ? $settings['auto_insert'] : true;

		// Check if TOC block already exists in content.
		if ( has_block( 'aria-auto-table-of-contents/toc' ) || strpos( $content, '<!-- wp:aria-auto-table-of-contents/toc' ) !== false ) {
			return $content;
		}

		// If auto-insert is disabled and no TOC block exists, return original content.
		if ( ! $auto_insert ) {
			return $content;
		}

		$headings = $this->extract_headings( $content );

		if ( empty( $headings ) ) {
			return $content;
		}

		// Get minimum headings setting.
		$min_headings = isset( $settings['min_headings'] ) ? intval( $settings['min_headings'] ) : 3;

		if ( count( $headings ) < $min_headings ) {
			return $content;
		}

		$toc_html = $this->generate_toc_html( $headings, $settings );
		$content_with_ids = $this->add_ids_to_headings( $content, $headings );

		// Insert TOC after first paragraph.
		$position = isset( $settings['position'] ) ? $settings['position'] : 'after_first_paragraph';

		return $this->insert_toc( $content_with_ids, $toc_html, $position );
	}

	/**
	 * Extract headings from content.
	 *
	 * @param string $content Post content.
	 * @return array Array of headings.
	 */
	public function extract_headings( $content ) {
		$headings = array();

		// Remove script and style tags to avoid parsing their content.
		$content = preg_replace( '/<(script|style)[^>]*>.*?<\/\1>/si', '', $content );

		// Match h2 and h3 tags.
		preg_match_all( '/<h([2-3])([^>]*)>(.*?)<\/h\1>/i', $content, $matches, PREG_SET_ORDER );

		foreach ( $matches as $index => $match ) {
			$level = intval( $match[1] );
			$attributes = $match[2];
			$text = wp_strip_all_tags( $match[3] );

			// Skip empty headings.
			if ( empty( trim( $text ) ) ) {
				continue;
			}

			// Check if ID already exists.
			$existing_id = '';
			if ( preg_match( '/id=["\']([^"\']+)["\']/i', $attributes, $id_match ) ) {
				$existing_id = $id_match[1];
			}

			$headings[] = array(
				'level'       => $level,
				'text'        => $text,
				'id'          => $existing_id ? $existing_id : $this->generate_id( $text, $index ),
				'original'    => $match[0],
				'has_id'      => ! empty( $existing_id ),
			);
		}

		return $headings;
	}

	/**
	 * Generate ID from heading text.
	 *
	 * @param string $text Heading text.
	 * @param int    $index Heading index.
	 * @return string Generated ID.
	 */
	private function generate_id( $text, $index ) {
		// Remove HTML entities.
		$text = html_entity_decode( $text, ENT_QUOTES | ENT_HTML5, 'UTF-8' );

		// Sanitize the text.
		$id = sanitize_title( $text );

		// If sanitization results in empty string, use fallback.
		if ( empty( $id ) ) {
			$id = 'toc-heading-' . ( $index + 1 );
		}

		return $id;
	}

	/**
	 * Add IDs to headings in content.
	 *
	 * @param string $content Post content.
	 * @param array  $headings Array of headings.
	 * @return string Modified content.
	 */
	private function add_ids_to_headings( $content, $headings ) {
		foreach ( $headings as $heading ) {
			// Skip if heading already has an ID.
			if ( $heading['has_id'] ) {
				continue;
			}

			// Add ID to heading tag.
			$new_heading = preg_replace(
				'/^<h([2-3])([^>]*)>/i',
				'<h$1$2 id="' . esc_attr( $heading['id'] ) . '">',
				$heading['original']
			);

			$content = str_replace( $heading['original'], $new_heading, $content );
		}

		return $content;
	}

	/**
	 * Generate TOC HTML.
	 *
	 * @param array $headings Array of headings.
	 * @param array $settings Plugin settings.
	 * @return string TOC HTML.
	 */
	public function generate_toc_html( $headings, $settings = array() ) {
		$title = isset( $settings['title'] ) && ! empty( $settings['title'] )
			? $settings['title']
			: __( 'Table of Contents', 'aria-auto-table-of-contents' );

		$show_numbers = isset( $settings['show_numbers'] ) ? $settings['show_numbers'] : true;
		$collapsible = isset( $settings['collapsible'] ) ? $settings['collapsible'] : true;

		$html = '<div class="auto-toc-seo-container">';
		$html .= '<div class="auto-toc-seo-header">';
		$html .= '<h2 class="auto-toc-seo-title">' . esc_html( $title ) . '</h2>';

		if ( $collapsible ) {
			$html .= '<button class="auto-toc-seo-toggle" aria-label="' . esc_attr__( 'Toggle Table of Contents', 'aria-auto-table-of-contents' ) . '">';
			$html .= '<span class="auto-toc-seo-toggle-icon"></span>';
			$html .= '</button>';
		}

		$html .= '</div>';

		$html .= '<nav class="auto-toc-seo-nav" aria-label="' . esc_attr__( 'Table of Contents', 'aria-auto-table-of-contents' ) . '">';
		$html .= '<ol class="auto-toc-seo-list' . ( $show_numbers ? '' : ' no-numbers' ) . '">';

		$current_level = 2;
		$counter = array( 2 => 0, 3 => 0 );

		foreach ( $headings as $heading ) {
			$level = $heading['level'];

			// Close nested list if going back to h2 from h3.
			if ( $level === 2 && $current_level === 3 ) {
				$html .= '</ol></li>';
				$counter[3] = 0;
			}

			// Open nested list if going from h2 to h3.
			if ( $level === 3 && $current_level === 2 ) {
				$html .= '<ol class="auto-toc-seo-sublist">';
			}

			// Close previous list item if on same level.
			if ( $level === $current_level && $counter[ $level ] > 0 ) {
				$html .= '</li>';
			}

			$counter[ $level ]++;

			$html .= '<li class="auto-toc-seo-item auto-toc-seo-level-' . esc_attr( $level ) . '">';
			$html .= '<a href="#' . esc_attr( $heading['id'] ) . '" class="auto-toc-seo-link">';
			$html .= esc_html( $heading['text'] );
			$html .= '</a>';

			$current_level = $level;
		}

		// Close any open nested lists.
		if ( $current_level === 3 ) {
			$html .= '</li></ol></li>';
		} else {
			$html .= '</li>';
		}

		$html .= '</ol>';
		$html .= '</nav>';
		$html .= '</div>';

		return $html;
	}

	/**
	 * Insert TOC into content.
	 *
	 * @param string $content Post content.
	 * @param string $toc_html TOC HTML.
	 * @param string $position Position to insert TOC.
	 * @return string Modified content.
	 */
	private function insert_toc( $content, $toc_html, $position ) {
		if ( 'before_content' === $position ) {
			return $toc_html . $content;
		}

		if ( 'after_first_paragraph' === $position ) {
			// Find the first closing </p> tag.
			$pos = strpos( $content, '</p>' );
			if ( false !== $pos ) {
				return substr_replace( $content, '</p>' . $toc_html, $pos, 4 );
			}
		}

		// Fallback to before content.
		return $toc_html . $content;
	}
}

