<?php
/**
 * Settings Class
 *
 * @package AutoTOCSEO
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Auto_TOC_SEO_Settings
 */
class Auto_TOC_SEO_Settings {

	/**
	 * Instance of this class.
	 *
	 * @var Auto_TOC_SEO_Settings
	 */
	private static $instance = null;

	/**
	 * Settings page slug.
	 *
	 * @var string
	 */
	private $page_slug = 'auto-toc-seo-settings';

	/**
	 * Get instance of the class.
	 *
	 * @return Auto_TOC_SEO_Settings
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
	}

	/**
	 * Add settings page to admin menu.
	 */
	public function add_settings_page() {
		add_options_page(
			__( 'Auto TOC SEO Settings', 'aria-auto-table-of-contents' ),
			__( 'Auto TOC SEO', 'aria-auto-table-of-contents' ),
			'manage_options',
			$this->page_slug,
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Register plugin settings.
	 */
	public function register_settings() {
		register_setting(
			'auto_toc_seo_settings_group',
			'auto_toc_seo_settings',
			array(
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
				'default'           => $this->get_default_settings(),
			)
		);

		// General Settings Section.
		add_settings_section(
			'auto_toc_seo_general_section',
			__( 'General Settings', 'aria-auto-table-of-contents' ),
			array( $this, 'render_general_section' ),
			$this->page_slug
		);

		// Auto Insert.
		add_settings_field(
			'auto_insert',
			__( 'Auto Insert TOC', 'aria-auto-table-of-contents' ),
			array( $this, 'render_checkbox_field' ),
			$this->page_slug,
			'auto_toc_seo_general_section',
			array(
				'label_for'   => 'auto_insert',
				'description' => __( 'Automatically insert TOC into posts/pages', 'aria-auto-table-of-contents' ),
			)
		);

		// Position.
		add_settings_field(
			'position',
			__( 'TOC Position', 'aria-auto-table-of-contents' ),
			array( $this, 'render_select_field' ),
			$this->page_slug,
			'auto_toc_seo_general_section',
			array(
				'label_for'   => 'position',
				'options'     => array(
					'before_content'         => __( 'Before Content', 'aria-auto-table-of-contents' ),
					'after_first_paragraph'  => __( 'After First Paragraph', 'aria-auto-table-of-contents' ),
				),
				'description' => __( 'Where to insert the TOC', 'aria-auto-table-of-contents' ),
			)
		);

		// Minimum Headings.
		add_settings_field(
			'min_headings',
			__( 'Minimum Headings', 'aria-auto-table-of-contents' ),
			array( $this, 'render_number_field' ),
			$this->page_slug,
			'auto_toc_seo_general_section',
			array(
				'label_for'   => 'min_headings',
				'min'         => 1,
				'max'         => 10,
				'description' => __( 'Minimum number of headings required to show TOC', 'aria-auto-table-of-contents' ),
			)
		);

		// TOC Title.
		add_settings_field(
			'title',
			__( 'TOC Title', 'aria-auto-table-of-contents' ),
			array( $this, 'render_text_field' ),
			$this->page_slug,
			'auto_toc_seo_general_section',
			array(
				'label_for'   => 'title',
				'description' => __( 'Title to display above the table of contents', 'aria-auto-table-of-contents' ),
			)
		);

		// Show Numbers.
		add_settings_field(
			'show_numbers',
			__( 'Show Numbers', 'aria-auto-table-of-contents' ),
			array( $this, 'render_checkbox_field' ),
			$this->page_slug,
			'auto_toc_seo_general_section',
			array(
				'label_for'   => 'show_numbers',
				'description' => __( 'Display numbered list for TOC items', 'aria-auto-table-of-contents' ),
			)
		);

		// Collapsible.
		add_settings_field(
			'collapsible',
			__( 'Collapsible TOC', 'aria-auto-table-of-contents' ),
			array( $this, 'render_checkbox_field' ),
			$this->page_slug,
			'auto_toc_seo_general_section',
			array(
				'label_for'   => 'collapsible',
				'description' => __( 'Allow users to collapse/expand TOC', 'aria-auto-table-of-contents' ),
			)
		);

		// Schema Settings Section.
		add_settings_section(
			'auto_toc_seo_schema_section',
			__( 'Schema Settings', 'aria-auto-table-of-contents' ),
			array( $this, 'render_schema_section' ),
			$this->page_slug
		);

		// Enable TOC Schema.
		add_settings_field(
			'enable_toc_schema',
			__( 'Enable TOC Schema', 'aria-auto-table-of-contents' ),
			array( $this, 'render_checkbox_field' ),
			$this->page_slug,
			'auto_toc_seo_schema_section',
			array(
				'label_for'   => 'enable_toc_schema',
				'description' => __( 'Add Schema.org markup for table of contents', 'aria-auto-table-of-contents' ),
			)
		);

		// Enable FAQ Schema.
		add_settings_field(
			'enable_faq_schema',
			__( 'Enable FAQ Schema', 'aria-auto-table-of-contents' ),
			array( $this, 'render_checkbox_field' ),
			$this->page_slug,
			'auto_toc_seo_schema_section',
			array(
				'label_for'   => 'enable_faq_schema',
				'description' => __( 'Add Schema.org FAQ markup (automatically detected)', 'aria-auto-table-of-contents' ),
			)
		);
	}

	/**
	 * Get default settings.
	 *
	 * @return array Default settings.
	 */
	private function get_default_settings() {
		return array(
			'auto_insert'        => true,
			'position'           => 'after_first_paragraph',
			'min_headings'       => 3,
			'title'              => __( 'Table of Contents', 'aria-auto-table-of-contents' ),
			'show_numbers'       => true,
			'collapsible'        => true,
			'enable_toc_schema'  => true,
			'enable_faq_schema'  => false,
		);
	}

	/**
	 * Sanitize settings.
	 *
	 * @param array $input Raw input.
	 * @return array Sanitized settings.
	 */
	public function sanitize_settings( $input ) {
		$sanitized = array();
		$defaults = $this->get_default_settings();

		$sanitized['auto_insert'] = ! empty( $input['auto_insert'] );
		$sanitized['position'] = in_array( $input['position'], array( 'before_content', 'after_first_paragraph' ), true )
			? $input['position']
			: $defaults['position'];
		$sanitized['min_headings'] = absint( $input['min_headings'] );
		$sanitized['title'] = sanitize_text_field( $input['title'] );
		$sanitized['show_numbers'] = ! empty( $input['show_numbers'] );
		$sanitized['collapsible'] = ! empty( $input['collapsible'] );
		$sanitized['enable_toc_schema'] = ! empty( $input['enable_toc_schema'] );
		$sanitized['enable_faq_schema'] = ! empty( $input['enable_faq_schema'] );

		return $sanitized;
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_admin_assets( $hook ) {
		if ( 'settings_page_' . $this->page_slug !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'auto-toc-seo-admin',
			AUTO_TOC_SEO_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			AUTO_TOC_SEO_VERSION
		);
	}

	/**
	 * Render settings page.
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if settings were saved.
		if ( isset( $_GET['settings-updated'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			add_settings_error(
				'auto_toc_seo_messages',
				'auto_toc_seo_message',
				__( 'Settings saved successfully.', 'aria-auto-table-of-contents' ),
				'updated'
			);
		}

		settings_errors( 'auto_toc_seo_messages' );
		?>
		<div class="wrap auto-toc-seo-settings">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<div class="auto-toc-seo-intro">
				<p><?php esc_html_e( 'Configure the Aria Auto Table of Contents (SEO Friendly) plugin settings below.', 'aria-auto-table-of-contents' ); ?></p>
			</div>

			<form action="options.php" method="post">
				<?php
				settings_fields( 'auto_toc_seo_settings_group' );
				do_settings_sections( $this->page_slug );
				submit_button( __( 'Save Settings', 'aria-auto-table-of-contents' ) );
				?>
			</form>

			<div class="auto-toc-seo-footer">
				<p>
					<?php
					printf(
						/* translators: %s: Plugin developer name */
						esc_html__( 'Developed by %s', 'aria-auto-table-of-contents' ),
						'<strong>Alireza Aminzadeh</strong>'
					);
					?>
					| <a href="https://ariait.net" target="_blank">ariait.net</a>
				</p>
			</div>
		</div>
		<?php
	}

	/**
	 * Render general section description.
	 */
	public function render_general_section() {
		echo '<p>' . esc_html__( 'Configure how the table of contents is displayed on your site.', 'aria-auto-table-of-contents' ) . '</p>';
	}

	/**
	 * Render schema section description.
	 */
	public function render_schema_section() {
		echo '<p>' . esc_html__( 'Enable or disable Schema.org structured data for better SEO.', 'aria-auto-table-of-contents' ) . '</p>';
	}

	/**
	 * Render checkbox field.
	 *
	 * @param array $args Field arguments.
	 */
	public function render_checkbox_field( $args ) {
		$settings = get_option( 'auto_toc_seo_settings', $this->get_default_settings() );
		$value = isset( $settings[ $args['label_for'] ] ) ? $settings[ $args['label_for'] ] : false;
		?>
		<label>
			<input
				type="checkbox"
				id="<?php echo esc_attr( $args['label_for'] ); ?>"
				name="auto_toc_seo_settings[<?php echo esc_attr( $args['label_for'] ); ?>]"
				value="1"
				<?php checked( $value, true ); ?>
			/>
			<?php echo esc_html( $args['description'] ); ?>
		</label>
		<?php
	}

	/**
	 * Render select field.
	 *
	 * @param array $args Field arguments.
	 */
	public function render_select_field( $args ) {
		$settings = get_option( 'auto_toc_seo_settings', $this->get_default_settings() );
		$value = isset( $settings[ $args['label_for'] ] ) ? $settings[ $args['label_for'] ] : '';
		?>
		<select
			id="<?php echo esc_attr( $args['label_for'] ); ?>"
			name="auto_toc_seo_settings[<?php echo esc_attr( $args['label_for'] ); ?>]"
		>
			<?php foreach ( $args['options'] as $option_value => $option_label ) : ?>
				<option value="<?php echo esc_attr( $option_value ); ?>" <?php selected( $value, $option_value ); ?>>
					<?php echo esc_html( $option_label ); ?>
				</option>
			<?php endforeach; ?>
		</select>
		<p class="description"><?php echo esc_html( $args['description'] ); ?></p>
		<?php
	}

	/**
	 * Render number field.
	 *
	 * @param array $args Field arguments.
	 */
	public function render_number_field( $args ) {
		$settings = get_option( 'auto_toc_seo_settings', $this->get_default_settings() );
		$value = isset( $settings[ $args['label_for'] ] ) ? $settings[ $args['label_for'] ] : '';
		?>
		<input
			type="number"
			id="<?php echo esc_attr( $args['label_for'] ); ?>"
			name="auto_toc_seo_settings[<?php echo esc_attr( $args['label_for'] ); ?>]"
			value="<?php echo esc_attr( $value ); ?>"
			min="<?php echo esc_attr( $args['min'] ); ?>"
			max="<?php echo esc_attr( $args['max'] ); ?>"
			class="small-text"
		/>
		<p class="description"><?php echo esc_html( $args['description'] ); ?></p>
		<?php
	}

	/**
	 * Render text field.
	 *
	 * @param array $args Field arguments.
	 */
	public function render_text_field( $args ) {
		$settings = get_option( 'auto_toc_seo_settings', $this->get_default_settings() );
		$value = isset( $settings[ $args['label_for'] ] ) ? $settings[ $args['label_for'] ] : '';
		?>
		<input
			type="text"
			id="<?php echo esc_attr( $args['label_for'] ); ?>"
			name="auto_toc_seo_settings[<?php echo esc_attr( $args['label_for'] ); ?>]"
			value="<?php echo esc_attr( $value ); ?>"
			class="regular-text"
		/>
		<p class="description"><?php echo esc_html( $args['description'] ); ?></p>
		<?php
	}
}

