<?php
/**
 * Schema Generator Class
 *
 * @package AutoTOCSEO
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Auto_TOC_SEO_Schema_Generator
 */
class Auto_TOC_SEO_Schema_Generator {

	/**
	 * Instance of this class.
	 *
	 * @var Auto_TOC_SEO_Schema_Generator
	 */
	private static $instance = null;

	/**
	 * Get instance of the class.
	 *
	 * @return Auto_TOC_SEO_Schema_Generator
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'wp_footer', array( $this, 'add_schema_markup' ) );
	}

	/**
	 * Add schema markup to footer.
	 */
	public function add_schema_markup() {
		if ( ! is_singular() ) {
			return;
		}

		$settings = get_option( 'auto_toc_seo_settings', array() );
		$enable_toc_schema = isset( $settings['enable_toc_schema'] ) ? $settings['enable_toc_schema'] : true;
		$enable_faq_schema = isset( $settings['enable_faq_schema'] ) ? $settings['enable_faq_schema'] : false;

		$schemas = array();

		// Add TOC schema.
		if ( $enable_toc_schema ) {
			$toc_schema = $this->generate_toc_schema();
			if ( $toc_schema ) {
				$schemas[] = $toc_schema;
			}
		}

		// Add FAQ schema.
		if ( $enable_faq_schema ) {
			$faq_schema = $this->generate_faq_schema();
			if ( $faq_schema ) {
				$schemas[] = $faq_schema;
			}
		}

		if ( empty( $schemas ) ) {
			return;
		}

		// Output schema markup.
		$schema_json = wp_json_encode( count( $schemas ) === 1 ? $schemas[0] : $schemas );
		echo '<script type="application/ld+json">';
		echo esc_html( $schema_json );
		echo '</script>';
	}

	/**
	 * Generate TOC schema markup.
	 *
	 * @return array|null Schema array or null if no headings.
	 */
	private function generate_toc_schema() {
		global $post;

		if ( ! $post ) {
			return null;
		}

		// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
		$content = apply_filters( 'the_content', $post->post_content );
		$toc_generator = Auto_TOC_SEO_TOC_Generator::get_instance();
		$headings = $toc_generator->extract_headings( $content );

		if ( empty( $headings ) ) {
			return null;
		}

		$settings = get_option( 'auto_toc_seo_settings', array() );
		$min_headings = isset( $settings['min_headings'] ) ? intval( $settings['min_headings'] ) : 3;

		if ( count( $headings ) < $min_headings ) {
			return null;
		}

		$permalink = get_permalink( $post );

		$item_list_elements = array();

		foreach ( $headings as $index => $heading ) {
			$item_list_elements[] = array(
				'@type'    => 'ListItem',
				'position' => $index + 1,
				'name'     => $heading['text'],
				'url'      => $permalink . '#' . $heading['id'],
			);
		}

		return array(
			'@context'        => 'https://schema.org',
			'@type'           => 'ItemList',
			'name'            => get_the_title( $post ),
			'description'     => __( 'Table of Contents', 'aria-auto-table-of-contents' ),
			'itemListElement' => $item_list_elements,
		);
	}

	/**
	 * Generate FAQ schema markup.
	 *
	 * @return array|null Schema array or null if no FAQs.
	 */
	private function generate_faq_schema() {
		global $post;

		if ( ! $post ) {
			return null;
		}

		// Extract FAQ items from content.
		$faqs = $this->extract_faq_items( $post->post_content );

		if ( empty( $faqs ) ) {
			return null;
		}

		$main_entities = array();

		foreach ( $faqs as $faq ) {
			$main_entities[] = array(
				'@type'          => 'Question',
				'name'           => $faq['question'],
				'acceptedAnswer' => array(
					'@type' => 'Answer',
					'text'  => $faq['answer'],
				),
			);
		}

		return array(
			'@context'   => 'https://schema.org',
			'@type'      => 'FAQPage',
			'mainEntity' => $main_entities,
		);
	}

	/**
	 * Extract FAQ items from content.
	 *
	 * @param string $content Post content.
	 * @return array Array of FAQ items.
	 */
	private function extract_faq_items( $content ) {
		$faqs = array();

		// Check for FAQ blocks in Gutenberg.
		if ( has_block( 'aria-auto-table-of-contents/faq' ) ) {
			$blocks = parse_blocks( $content );
			$faqs = $this->extract_faq_from_blocks( $blocks );
		}

		// If no Gutenberg blocks, try to extract from HTML structure.
		if ( empty( $faqs ) ) {
			$faqs = $this->extract_faq_from_html( $content );
		}

		return $faqs;
	}

	/**
	 * Extract FAQ from Gutenberg blocks.
	 *
	 * @param array $blocks Array of blocks.
	 * @return array Array of FAQ items.
	 */
	private function extract_faq_from_blocks( $blocks ) {
		$faqs = array();

		foreach ( $blocks as $block ) {
			if ( 'aria-auto-table-of-contents/faq' === $block['blockName'] && ! empty( $block['attrs']['items'] ) ) {
				foreach ( $block['attrs']['items'] as $item ) {
					if ( ! empty( $item['question'] ) && ! empty( $item['answer'] ) ) {
						$faqs[] = array(
							'question' => wp_strip_all_tags( $item['question'] ),
							'answer'   => wp_strip_all_tags( $item['answer'] ),
						);
					}
				}
			}

			// Recursively check inner blocks.
			if ( ! empty( $block['innerBlocks'] ) ) {
				$inner_faqs = $this->extract_faq_from_blocks( $block['innerBlocks'] );
				$faqs = array_merge( $faqs, $inner_faqs );
			}
		}

		return $faqs;
	}

	/**
	 * Extract FAQ from HTML structure.
	 *
	 * @param string $content Post content.
	 * @return array Array of FAQ items.
	 */
	private function extract_faq_from_html( $content ) {
		$faqs = array();

		// Look for common FAQ patterns:
		// 1. Question in heading followed by answer in paragraph/div.
		// 2. DL/DT/DD structure.

		// Pattern 1: Heading + Paragraph.
		$pattern = '/<h([2-6])[^>]*>(?:.*?(?:سوال|سؤال|question|Q:|پرسش).*?)(.*?)<\/h\1>\s*<p[^>]*>(.*?)<\/p>/si';
		if ( preg_match_all( $pattern, $content, $matches, PREG_SET_ORDER ) ) {
			foreach ( $matches as $match ) {
				$question = wp_strip_all_tags( $match[2] );
				$answer = wp_strip_all_tags( $match[3] );

				if ( ! empty( $question ) && ! empty( $answer ) ) {
					$faqs[] = array(
						'question' => $question,
						'answer'   => $answer,
					);
				}
			}
		}

		// Pattern 2: DL/DT/DD structure.
		if ( preg_match_all( '/<dl[^>]*>(.*?)<\/dl>/si', $content, $dl_matches ) ) {
			foreach ( $dl_matches[1] as $dl_content ) {
				preg_match_all( '/<dt[^>]*>(.*?)<\/dt>\s*<dd[^>]*>(.*?)<\/dd>/si', $dl_content, $dd_matches, PREG_SET_ORDER );

				foreach ( $dd_matches as $dd_match ) {
					$question = wp_strip_all_tags( $dd_match[1] );
					$answer = wp_strip_all_tags( $dd_match[2] );

					if ( ! empty( $question ) && ! empty( $answer ) ) {
						$faqs[] = array(
							'question' => $question,
							'answer'   => $answer,
						);
					}
				}
			}
		}

		return $faqs;
	}
}

