<?php
/**
 * Gutenberg Block Class
 *
 * @package AutoTOCSEO
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Auto_TOC_SEO_Gutenberg_Block
 */
class Auto_TOC_SEO_Gutenberg_Block {

	/**
	 * Instance of this class.
	 *
	 * @var Auto_TOC_SEO_Gutenberg_Block
	 */
	private static $instance = null;

	/**
	 * Get instance of the class.
	 *
	 * @return Auto_TOC_SEO_Gutenberg_Block
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		add_action( 'init', array( $this, 'register_blocks' ) );
	}

	/**
	 * Register Gutenberg blocks.
	 */
	public function register_blocks() {
		// Register TOC block.
		register_block_type(
			'aria-auto-table-of-contents/toc',
			array(
				'api_version'     => 2,
				'editor_script'   => 'auto-toc-seo-toc-block',
				'editor_style'    => 'auto-toc-seo-toc-block-editor',
				'style'           => 'auto-toc-seo-style',
				'render_callback' => array( $this, 'render_toc_block' ),
				'attributes'      => array(
					'title'       => array(
						'type'    => 'string',
						'default' => __( 'Table of Contents', 'aria-auto-table-of-contents' ),
					),
					'showNumbers' => array(
						'type'    => 'boolean',
						'default' => true,
					),
					'collapsible' => array(
						'type'    => 'boolean',
						'default' => true,
					),
				),
			)
		);

		// Register FAQ block.
		register_block_type(
			'aria-auto-table-of-contents/faq',
			array(
				'api_version'     => 2,
				'editor_script'   => 'auto-toc-seo-faq-block',
				'editor_style'    => 'auto-toc-seo-faq-block-editor',
				'style'           => 'auto-toc-seo-style',
				'render_callback' => array( $this, 'render_faq_block' ),
				'attributes'      => array(
					'items' => array(
						'type'    => 'array',
						'default' => array(),
						'items'   => array(
							'type'       => 'object',
							'properties' => array(
								'question' => array(
									'type' => 'string',
								),
								'answer'   => array(
									'type' => 'string',
								),
							),
						),
					),
				),
			)
		);

		// Enqueue block assets.
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );
	}

	/**
	 * Enqueue block editor assets.
	 */
	public function enqueue_block_editor_assets() {
		// TOC Block.
		wp_enqueue_script(
			'auto-toc-seo-toc-block',
			AUTO_TOC_SEO_PLUGIN_URL . 'assets/js/blocks/toc-block.js',
			array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n', 'wp-server-side-render' ),
			AUTO_TOC_SEO_VERSION,
			true
		);

		wp_enqueue_style(
			'auto-toc-seo-toc-block-editor',
			AUTO_TOC_SEO_PLUGIN_URL . 'assets/css/blocks/toc-block-editor.css',
			array( 'wp-edit-blocks' ),
			AUTO_TOC_SEO_VERSION
		);

		// FAQ Block.
		wp_enqueue_script(
			'auto-toc-seo-faq-block',
			AUTO_TOC_SEO_PLUGIN_URL . 'assets/js/blocks/faq-block.js',
			array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n' ),
			AUTO_TOC_SEO_VERSION,
			true
		);

		wp_enqueue_style(
			'auto-toc-seo-faq-block-editor',
			AUTO_TOC_SEO_PLUGIN_URL . 'assets/css/blocks/faq-block-editor.css',
			array( 'wp-edit-blocks' ),
			AUTO_TOC_SEO_VERSION
		);
	}

	/**
	 * Render TOC block.
	 *
	 * @param array  $attributes Block attributes.
	 * @param string $content Block content.
	 * @return string Block HTML.
	 */
	public function render_toc_block( $attributes, $content ) {
		global $post;

		if ( ! $post ) {
			return '';
		}

		$toc_generator = Auto_TOC_SEO_TOC_Generator::get_instance();
		// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
		$post_content = apply_filters( 'the_content', $post->post_content );
		$headings = $toc_generator->extract_headings( $post_content );

		if ( empty( $headings ) ) {
			return '<div class="auto-toc-seo-container auto-toc-seo-empty">' .
				'<p>' . esc_html__( 'No headings found in the content.', 'aria-auto-table-of-contents' ) . '</p>' .
				'</div>';
		}

		$settings = array(
			'title'        => isset( $attributes['title'] ) ? $attributes['title'] : __( 'Table of Contents', 'aria-auto-table-of-contents' ),
			'show_numbers' => isset( $attributes['showNumbers'] ) ? $attributes['showNumbers'] : true,
			'collapsible'  => isset( $attributes['collapsible'] ) ? $attributes['collapsible'] : true,
		);

		return $toc_generator->generate_toc_html( $headings, $settings );
	}

	/**
	 * Render FAQ block.
	 *
	 * @param array  $attributes Block attributes.
	 * @param string $content Block content.
	 * @return string Block HTML.
	 */
	public function render_faq_block( $attributes, $content ) {
		$items = isset( $attributes['items'] ) ? $attributes['items'] : array();

		if ( empty( $items ) ) {
			return '<div class="auto-toc-seo-faq auto-toc-seo-empty">' .
				'<p>' . esc_html__( 'No FAQ items added yet.', 'aria-auto-table-of-contents' ) . '</p>' .
				'</div>';
		}

		$html = '<div class="auto-toc-seo-faq">';

		foreach ( $items as $index => $item ) {
			$question = isset( $item['question'] ) ? $item['question'] : '';
			$answer = isset( $item['answer'] ) ? $item['answer'] : '';

			if ( empty( $question ) || empty( $answer ) ) {
				continue;
			}

			$html .= '<div class="auto-toc-seo-faq-item" itemscope itemprop="mainEntity" itemtype="https://schema.org/Question">';
			$html .= '<h3 class="auto-toc-seo-faq-question" itemprop="name">' . wp_kses_post( $question ) . '</h3>';
			$html .= '<div class="auto-toc-seo-faq-answer" itemscope itemprop="acceptedAnswer" itemtype="https://schema.org/Answer">';
			$html .= '<div itemprop="text">' . wp_kses_post( $answer ) . '</div>';
			$html .= '</div>';
			$html .= '</div>';
		}

		$html .= '</div>';

		return $html;
	}
}

