/**
 * Frontend JavaScript
 */
(function () {
	'use strict';

	// Wait for DOM to be ready
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', init);
	} else {
		init();
	}

	function init() {
		initCollapsibleTOC();
		initSmoothScroll();
		initActiveHeading();
	}

	/**
	 * Initialize collapsible TOC functionality
	 */
	function initCollapsibleTOC() {
		const toggleButtons = document.querySelectorAll('.auto-toc-seo-toggle');

		toggleButtons.forEach(function (button) {
			button.addEventListener('click', function (e) {
				e.preventDefault();
				const container = button.closest('.auto-toc-seo-container');
				const nav = container.querySelector('.auto-toc-seo-nav');

				if (nav.style.display === 'none') {
					nav.style.display = 'block';
					container.classList.remove('collapsed');
					button.setAttribute('aria-expanded', 'true');
				} else {
					nav.style.display = 'none';
					container.classList.add('collapsed');
					button.setAttribute('aria-expanded', 'false');
				}
			});
		});
	}

	/**
	 * Initialize smooth scrolling for TOC links
	 */
	function initSmoothScroll() {
		const tocLinks = document.querySelectorAll('.auto-toc-seo-link');

		tocLinks.forEach(function (link) {
			link.addEventListener('click', function (e) {
				e.preventDefault();
				const targetId = this.getAttribute('href').substring(1);
				const targetElement = document.getElementById(targetId);

				if (targetElement) {
					const offset = 80; // Offset for fixed headers
					const targetPosition = targetElement.getBoundingClientRect().top + window.pageYOffset - offset;

					window.scrollTo({
						top: targetPosition,
						behavior: 'smooth',
					});

					// Update URL without jumping
					if (history.pushState) {
						history.pushState(null, null, '#' + targetId);
					}

					// Focus on the heading for accessibility
					targetElement.focus({ preventScroll: true });
				}
			});
		});
	}

	/**
	 * Highlight active TOC item based on scroll position
	 */
	function initActiveHeading() {
		const tocLinks = document.querySelectorAll('.auto-toc-seo-link');
		if (tocLinks.length === 0) return;

		const headings = [];
		tocLinks.forEach(function (link) {
			const targetId = link.getAttribute('href').substring(1);
			const targetElement = document.getElementById(targetId);
			if (targetElement) {
				headings.push({
					link: link,
					element: targetElement,
				});
			}
		});

		if (headings.length === 0) return;

		function updateActiveHeading() {
			const scrollPosition = window.pageYOffset + 100; // Offset for better UX

			let currentHeading = null;

			headings.forEach(function (heading) {
				const headingTop = heading.element.offsetTop;
				if (scrollPosition >= headingTop) {
					currentHeading = heading;
				}
			});

			// Remove active class from all links
			tocLinks.forEach(function (link) {
				link.classList.remove('active');
			});

			// Add active class to current link
			if (currentHeading) {
				currentHeading.link.classList.add('active');
			}
		}

		// Throttle scroll event
		let ticking = false;
		window.addEventListener('scroll', function () {
			if (!ticking) {
				window.requestAnimationFrame(function () {
					updateActiveHeading();
					ticking = false;
				});
				ticking = true;
			}
		});

		// Initial check
		updateActiveHeading();
	}
})();

