/**
 * FAQ Block for Gutenberg Editor
 */
(function (wp) {
	const { registerBlockType } = wp.blocks;
	const { InspectorControls, RichText } = wp.blockEditor;
	const { PanelBody, Button, IconButton } = wp.components;
	const { __ } = wp.i18n;
	const { Fragment } = wp.element;

	registerBlockType('aria-auto-table-of-contents/faq', {
		title: __('FAQ (SEO)', 'aria-auto-table-of-contents'),
		description: __('Add FAQ items with Schema.org markup', 'aria-auto-table-of-contents'),
		category: 'widgets',
		icon: 'editor-help',
		keywords: [__('faq', 'aria-auto-table-of-contents'), __('questions', 'aria-auto-table-of-contents'), __('schema', 'aria-auto-table-of-contents')],
		supports: {
			align: true,
			html: false,
		},
		attributes: {
			items: {
				type: 'array',
				default: [],
			},
		},

		edit: function (props) {
			const { attributes, setAttributes, className } = props;
			const { items } = attributes;

			const addItem = function () {
				const newItems = [...items, { question: '', answer: '' }];
				setAttributes({ items: newItems });
			};

			const updateItem = function (index, field, value) {
				const newItems = [...items];
				newItems[index][field] = value;
				setAttributes({ items: newItems });
			};

			const removeItem = function (index) {
				const newItems = items.filter(function (item, i) {
					return i !== index;
				});
				setAttributes({ items: newItems });
			};

			const moveItem = function (index, direction) {
				if (
					(direction === 'up' && index === 0) ||
					(direction === 'down' && index === items.length - 1)
				) {
					return;
				}

				const newItems = [...items];
				const targetIndex = direction === 'up' ? index - 1 : index + 1;
				const temp = newItems[index];
				newItems[index] = newItems[targetIndex];
				newItems[targetIndex] = temp;
				setAttributes({ items: newItems });
			};

			return wp.element.createElement(
				'div',
				{ className: className },
				[
					wp.element.createElement(
						'div',
						{ className: 'auto-toc-seo-faq-editor', key: 'editor' },
						[
							wp.element.createElement(
								'h3',
								{ className: 'auto-toc-seo-faq-title', key: 'title' },
								__('FAQ Items', 'aria-auto-table-of-contents')
							),
							items.map(function (item, index) {
								return wp.element.createElement(
									'div',
									{
										className: 'auto-toc-seo-faq-item-editor',
										key: 'item-' + index,
									},
									[
										wp.element.createElement(
											'div',
											{ className: 'auto-toc-seo-faq-item-controls', key: 'controls' },
											[
												wp.element.createElement(Button, {
													key: 'up',
													icon: 'arrow-up-alt2',
													label: __('Move Up', 'aria-auto-table-of-contents'),
													disabled: index === 0,
													onClick: function () {
														moveItem(index, 'up');
													},
												}),
												wp.element.createElement(Button, {
													key: 'down',
													icon: 'arrow-down-alt2',
													label: __('Move Down', 'aria-auto-table-of-contents'),
													disabled: index === items.length - 1,
													onClick: function () {
														moveItem(index, 'down');
													},
												}),
												wp.element.createElement(Button, {
													key: 'remove',
													icon: 'trash',
													label: __('Remove', 'aria-auto-table-of-contents'),
													isDestructive: true,
													onClick: function () {
														removeItem(index);
													},
												}),
											]
										),
										wp.element.createElement(RichText, {
											key: 'question',
											tagName: 'h4',
											className: 'auto-toc-seo-faq-question',
											placeholder: __('Enter question...', 'aria-auto-table-of-contents'),
											value: item.question,
											onChange: function (value) {
												updateItem(index, 'question', value);
											},
										}),
										wp.element.createElement(RichText, {
											key: 'answer',
											tagName: 'div',
											className: 'auto-toc-seo-faq-answer',
											placeholder: __('Enter answer...', 'aria-auto-table-of-contents'),
											value: item.answer,
											onChange: function (value) {
												updateItem(index, 'answer', value);
											},
										}),
									]
								);
							}),
							wp.element.createElement(
								Button,
								{
									key: 'add',
									isPrimary: true,
									className: 'auto-toc-seo-faq-add-button',
									onClick: addItem,
								},
								__('Add FAQ Item', 'aria-auto-table-of-contents')
							),
						]
					),
				]
			);
		},

		save: function () {
			return null; // Server-side render
		},
	});
})(window.wp);

