<?php
/**
 * Uninstall Script for Arewa Recently Viewed Content
 *
 * This file is executed when the plugin is uninstalled (deleted) from WordPress.
 * It provides options for data preservation and complete removal.
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly or not in uninstall context.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

/**
 * Handle plugin uninstallation.
 *
 * This function provides two modes:
 * 1. Preserve data (default) - Keeps all settings and user data for future reinstallation.
 * 2. Complete removal - Removes all plugin data when explicitly requested.
 */
function arwrev_handle_uninstall() {
	// Check if we should preserve data (default behavior).
	$preserve_data = get_option( 'arwrev_preserve_data_on_uninstall', true );

	if ( $preserve_data ) {
		// Preserve mode: Only clean up temporary data, keep settings and user history.
		arwrev_cleanup_temporary_data();

		// Set a flag to indicate the plugin was properly uninstalled but data preserved.
		update_option(
			'arwrev_data_preserved',
			array(
				'uninstalled_at' => time(),
				'version'        => get_option( 'arwrev_version', '1.0.0' ),
				'preserved'      => true,
			)
		);

	} else {
		// Complete removal mode: Remove all plugin data.
		arwrev_complete_data_removal();
	}
}

/**
 * Clean up only temporary data while preserving settings and user history.
 */
function arwrev_cleanup_temporary_data() {
	// Clear scheduled events.
	wp_clear_scheduled_hook( 'arwrev_cleanup' );

	// Remove temporary/cache options (if any exist in future versions).
	delete_option( 'arwrev_cache_version' );
	delete_option( 'arwrev_temp_data' );

	// Clear any transients.
	delete_transient( 'arwrev_layout_cache' );
	delete_transient( 'arwrev_settings_cache' );
}

/**
 * Completely remove all plugin data.
 */
function arwrev_complete_data_removal() {
	global $wpdb;

	// Remove all plugin options.
	delete_option( 'arwrev_settings' );
	delete_option( 'arwrev_version' );
	delete_option( 'arwrev_preserve_data_on_uninstall' );
	delete_option( 'arwrev_data_preserved' );
	delete_option( 'arwrev_options_migrated' );

	// Remove legacy options (from old Arewa Watch History plugin).
	delete_option( 'arewa_wh_settings' );

	// Clear scheduled events.
	wp_clear_scheduled_hook( 'arwrev_cleanup' );

	// Remove all user meta data.
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Required for complete uninstall.
	$wpdb->delete(
		$wpdb->usermeta,
		array( 'meta_key' => 'arewa_recently_viewed' ), // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Uninstall; one-time bulk delete.
		array( '%s' )
	);

	// Remove legacy user meta (from old plugin and old meta key).
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Required for complete uninstall.
	$wpdb->delete(
		$wpdb->usermeta,
		array( 'meta_key' => 'arwrev_history' ), // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Uninstall; one-time bulk delete.
		array( '%s' )
	);

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Required for complete uninstall.
	$wpdb->delete(
		$wpdb->usermeta,
		array( 'meta_key' => 'arewa_wh_history' ), // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Uninstall; one-time bulk delete.
		array( '%s' )
	);

	// Clear any transients.
	delete_transient( 'arwrev_layout_cache' );
	delete_transient( 'arwrev_settings_cache' );

	// Remove custom image sizes (WordPress will handle cleanup).
	// Note: Custom image sizes are automatically removed when plugin is deactivated.
}

/**
 * Add admin notice about data preservation.
 */
function arwrev_add_uninstall_notice() {
	if ( is_admin() && current_user_can( 'manage_options' ) ) {
		add_action(
			'admin_notices',
			function() {
				$preserve_data = get_option( 'arwrev_preserve_data_on_uninstall', true );
				if ( $preserve_data ) {
					echo '<div class="notice notice-info is-dismissible">';
					echo '<p><strong>' . esc_html__( 'AR Viewed Content:', 'arewa-recently-viewed-content' ) . '</strong> ';
					echo esc_html__( 'Your settings and watch history data will be preserved. To completely remove all data, go to the plugin settings and disable data preservation before uninstalling.', 'arewa-recently-viewed-content' );
					echo '</p></div>';
				}
			}
		);
	}
}

// Execute uninstall process.
arwrev_handle_uninstall();
