<?php
/**
 * Video Layout Class
 *
 * Handles video/card-style layout rendering
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure base class is loaded.
if ( ! class_exists( 'ARWREV_Base_Layout' ) ) {
	require_once ARWREV_PLUGIN_DIR . 'includes/layouts/class-arwrev-base-layout.php';
}

/**
 * Video Layout Class.
 *
 * Renders items in a card-style layout with overlay text.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Video_Layout extends ARWREV_Base_Layout {

	/**
	 * Layout type identifier
	 *
	 * @var string
	 */
	protected $layout_type = 'video';

	/**
	 * Supported display modes
	 *
	 * @var array
	 */
	protected $supported_modes = array( 'grid' );

	/**
	 * Render the video layout
	 *
	 * @param array  $items History items.
	 * @param array  $settings Plugin settings.
	 * @param array  $atts Shortcode attributes.
	 * @param string $display_mode Display mode.
	 * @return string
	 */
	public function render( $items, $settings, $atts, $display_mode ) {
		if ( empty( $items ) ) {
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'Your watch history is empty.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		$columns_desktop = isset( $settings['columns_desktop'] ) ? (int) $settings['columns_desktop'] : 6;
		$columns_tablet  = isset( $settings['columns_tablet'] ) ? (int) $settings['columns_tablet'] : 4;
		$columns_mobile  = isset( $settings['columns_mobile'] ) ? (int) $settings['columns_mobile'] : 2;
		$image_width     = isset( $settings['image_width'] ) ? (int) $settings['image_width'] : 155;
		$image_height    = isset( $settings['image_height'] ) ? (int) $settings['image_height'] : 250;
		$title_max_chars = isset( $settings['title_max_chars'] ) ? (int) $settings['title_max_chars'] : 0;

		$show_image = isset( $atts['show_image'] ) ? 'yes' === strtolower( $atts['show_image'] ) : true;
		$show_date  = isset( $atts['show_date'] ) ? 'yes' === strtolower( $atts['show_date'] ) : true;

		ob_start();
		?>
		<?php // phpcs:disable WordPress.WhiteSpace.PrecisionAlignment.Found -- HTML template alignment. ?>
		<div class="<?php echo esc_attr( $this->get_container_classes( $display_mode, $settings ) ); ?>"
			data-columns-desktop="<?php echo esc_attr( $columns_desktop ); ?>"
			data-columns-tablet="<?php echo esc_attr( $columns_tablet ); ?>"
			data-columns-mobile="<?php echo esc_attr( $columns_mobile ); ?>">

			<div class="arwrev-grid"
				style="--columns-desktop: <?php echo esc_attr( $columns_desktop ); ?>; --columns-tablet: <?php echo esc_attr( $columns_tablet ); ?>; --columns-mobile: <?php echo esc_attr( $columns_mobile ); ?>;">

			<?php foreach ( $items as $item ) : ?>
				<?php
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Method handles escaping internally
				echo $this->render_video_item( $item, $settings, $show_image, $show_date, $image_width, $image_height, $title_max_chars );
				?>
			<?php endforeach; ?>

			</div>
		</div>
		<?php // phpcs:enable WordPress.WhiteSpace.PrecisionAlignment.Found ?>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render individual video item
	 *
	 * @param array $item History item.
	 * @param array $settings Plugin settings.
	 * @param bool  $show_image Show image flag.
	 * @param bool  $show_date Show date flag.
	 * @param int   $image_width Image width.
	 * @param int   $image_height Image height.
	 * @param int   $title_max_chars Title max characters.
	 * @return string
	 */
	private function render_video_item( $item, $settings, $show_image, $show_date, $image_width, $image_height, $title_max_chars ) {
		$post_id   = isset( $item['post_id'] ) ? (int) $item['post_id'] : 0;
		$post_type = isset( $item['post_type'] ) ? $item['post_type'] : '';
		$timestamp = isset( $item['timestamp'] ) ? (int) $item['timestamp'] : 0;

		$post = get_post( $post_id );
		if ( ! $post ) {
			return '';
		}

		$post_type_label = $this->get_post_type_label( $post_type, $settings );
		$date_str        = $show_date ? $this->get_formatted_date( $timestamp ) : '';
		$title           = $this->get_truncated_title( get_the_title( $post_id ), $title_max_chars );

		ob_start();
		?>
		<?php // phpcs:disable WordPress.WhiteSpace.PrecisionAlignment.Found -- HTML template alignment. ?>
		<div class="<?php echo esc_attr( $this->get_item_classes( $item, 'grid' ) ); ?>"
			data-post-id="<?php echo esc_attr( $post_id ); ?>"
			style="--image-width: <?php echo esc_attr( $image_width ); ?>px; --image-height: <?php echo esc_attr( $image_height ); ?>px;">

			<?php if ( $show_image ) : ?>
				<div class="arwrev-item-image">
					<a href="<?php echo esc_url( get_permalink( $post_id ) ); ?>">
						<?php
						if ( has_post_thumbnail( $post_id ) ) {
							$thumbnail_id = get_post_thumbnail_id( $post_id );
							$image_src    = wp_get_attachment_image_src( $thumbnail_id, 'arwrev_thumb' );

							if ( ! $image_src || empty( $image_src[0] ) ) {
								$image_src = wp_get_attachment_image_src( $thumbnail_id, 'large' );
							}
							if ( ! $image_src || empty( $image_src[0] ) ) {
								$image_src = wp_get_attachment_image_src( $thumbnail_id, 'full' );
							}

							if ( $image_src && ! empty( $image_src[0] ) ) {
								printf(
									'<img src="%s" alt="%s" width="%d" height="%d" loading="lazy" />',
									esc_url( $image_src[0] ),
									esc_attr( get_the_title( $post_id ) ),
									esc_attr( $image_width ),
									esc_attr( $image_height )
								);
							} else {
								echo get_the_post_thumbnail( $post_id, 'large', array( 'alt' => esc_attr( get_the_title( $post_id ) ) ) );
							}
						} else {
							echo '<div class="arwrev-placeholder-image"></div>';
						}
						?>

						<!-- Play Button -->
						<div class="arwrev-play-button">
							<svg width="60" height="60" viewBox="0 0 60 60" fill="none" xmlns="http://www.w3.org/2000/svg">
								<circle cx="30" cy="30" r="30" fill="rgba(255, 255, 255, 0.9)"/>
								<path d="M25 20L40 30L25 40V20Z" fill="#333"/>
							</svg>
						</div>
						
						<!-- Text Overlay -->
						<div class="arwrev-overlay">
							<h3 class="arwrev-item-title"><?php echo esc_html( $title ); ?></h3>
							<div class="arwrev-item-meta">
								<span class="arwrev-item-type"><?php echo esc_html( $post_type_label ); ?></span>
								<?php if ( $date_str ) : ?>
									<span class="arwrev-item-date"><?php echo esc_html( $date_str ); ?></span>
								<?php endif; ?>
							</div>
						</div>
					</a>
				</div>
			<?php else : ?>
				<div class="arwrev-item-content">
					<h3 class="arwrev-item-title">
						<a href="<?php echo esc_url( get_permalink( $post_id ) ); ?>">
							<?php echo esc_html( $title ); ?>
						</a>
					</h3>
					<div class="arwrev-item-meta">
						<span class="arwrev-item-type"><?php echo esc_html( $post_type_label ); ?></span>
						<?php if ( $date_str ) : ?>
							<span class="arwrev-item-date"><?php echo esc_html( $date_str ); ?></span>
						<?php endif; ?>
					</div>
				</div>
			<?php endif; ?>

			<!-- Remove Button -->
			<button type="button" class="arwrev-remove-item"
				data-post-id="<?php echo esc_attr( $post_id ); ?>"
				data-nonce="<?php echo esc_attr( wp_create_nonce( 'arwrev_remove_item_' . $post_id ) ); ?>"
				aria-label="<?php esc_attr_e( 'Remove this item', 'arewa-recently-viewed-content' ); ?>">
				<?php esc_html_e( 'Remove', 'arewa-recently-viewed-content' ); ?>
			</button>
		</div>
		<?php // phpcs:enable WordPress.WhiteSpace.PrecisionAlignment.Found ?>
		<?php
		return ob_get_clean();
	}
}
