<?php
/**
 * Article Layout Class
 *
 * Handles article/blog-style layout rendering
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure base class is loaded.
if ( ! class_exists( 'ARWREV_Base_Layout' ) ) {
	require_once ARWREV_PLUGIN_DIR . 'includes/layouts/class-arwrev-base-layout.php';
}

/**
 * Article Layout Class.
 *
 * Renders items in a clean blog-style list layout.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Article_Layout extends ARWREV_Base_Layout {

	/**
	 * Layout type identifier
	 *
	 * @var string
	 */
	protected $layout_type = 'article';

	/**
	 * Supported display modes
	 *
	 * @var array
	 */
	protected $supported_modes = array( 'list' );

	/**
	 * Render the article layout
	 *
	 * @param array  $items History items.
	 * @param array  $settings Plugin settings.
	 * @param array  $atts Shortcode attributes.
	 * @param string $display_mode Display mode.
	 * @return string
	 */
	public function render( $items, $settings, $atts, $display_mode ) {
		if ( empty( $items ) ) {
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'Your reading history is empty.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		// Preserve history order: use WP_Query post__in + orderby post__in, then re-map to history order.
		$ordered = $this->get_posts_in_history_order( $items );
		if ( empty( $ordered ) ) {
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'Your reading history is empty.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		$excerpt_length = isset( $settings['article_excerpt_length'] ) ? (int) $settings['article_excerpt_length'] : 150;
		$show_image     = isset( $atts['show_image'] ) ? 'yes' === strtolower( $atts['show_image'] ) : true;
		$show_date      = isset( $atts['show_date'] ) ? 'yes' === strtolower( $atts['show_date'] ) : true;

		ob_start();
		?>
		<div class="<?php echo esc_attr( $this->get_container_classes( $display_mode, $settings ) ); ?>">
			<div class="arwrev-magazine-grid">
				<?php foreach ( $ordered as $entry ) : ?>
					<?php
					// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Method handles escaping internally
					echo $this->render_article_item( $entry['item'], $settings, $show_image, $show_date, $excerpt_length, $entry['post'] );
					?>
				<?php endforeach; ?>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Render individual article item
	 *
	 * @param array  $item History item.
	 * @param array  $settings Plugin settings.
	 * @param bool   $show_image Show image flag.
	 * @param bool   $show_date Show date flag.
	 * @param int    $excerpt_length Excerpt length.
	 * @param object $post           Optional. WP_Post from query (avoids extra get_post).
	 * @return string
	 */
	private function render_article_item( $item, $settings, $show_image, $show_date, $excerpt_length, $post = null ) {
		$post_id   = isset( $item['post_id'] ) ? (int) $item['post_id'] : 0;
		$post_type = isset( $item['post_type'] ) ? $item['post_type'] : '';
		$timestamp = isset( $item['timestamp'] ) ? (int) $item['timestamp'] : 0;

		if ( ! $post ) {
			$post = get_post( $post_id );
		}
		if ( ! $post ) {
			return '';
		}

		$post_type_label = $this->get_post_type_label( $post_type, $settings );
		$date_str        = $show_date ? $this->get_formatted_date( $timestamp ) : '';
		$title           = get_the_title( $post_id );
		$excerpt         = $this->get_post_excerpt( $post_id, $excerpt_length );
		$permalink       = get_permalink( $post_id );
		$categories      = get_the_category( $post_id );
		$category_name   = ! empty( $categories ) ? $categories[0]->name : '';

		ob_start();
		?>
		<article class="arwrev-magazine-card" data-post-id="<?php echo esc_attr( $post_id ); ?>">
			<div class="arwrev-magazine-card-inner">
				<a href="<?php echo esc_url( $permalink ); ?>" class="arwrev-magazine-link">
					<div class="arwrev-magazine-image">
						<?php if ( $show_image && has_post_thumbnail( $post_id ) ) : ?>
							<?php
							echo get_the_post_thumbnail(
								$post_id,
								'large',
								array(
									'alt'     => esc_attr( $title ),
									'loading' => 'lazy',
									'class'   => 'arwrev-magazine-thumbnail',
								)
							);
							?>
						<?php else : ?>
							<div class="arwrev-magazine-placeholder">
								<svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
									<rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
									<circle cx="8.5" cy="8.5" r="1.5"/>
									<polyline points="21,15 16,10 5,21"/>
								</svg>
							</div>
						<?php endif; ?>

						<!-- Gradient Overlay -->
						<div class="arwrev-magazine-overlay"></div>

						<!-- Content Overlay -->
						<div class="arwrev-magazine-content">
							<!-- Top Meta -->
							<div class="arwrev-magazine-meta">
								<?php if ( $category_name ) : ?>
									<span class="arwrev-magazine-category"><?php echo esc_html( $category_name ); ?></span>
								<?php endif; ?>
								<?php if ( $show_date && $date_str ) : ?>
									<time class="arwrev-magazine-date" datetime="<?php echo esc_attr( gmdate( 'c', $timestamp ) ); ?>">
										<?php echo esc_html( $date_str ); ?>
									</time>
								<?php endif; ?>
							</div>

							<!-- Title -->
							<h3 class="arwrev-magazine-title">
								<?php echo esc_html( $title ); ?>
							</h3>

							<!-- Excerpt (always visible) -->
							<?php if ( $excerpt ) : ?>
								<div class="arwrev-magazine-excerpt">
									<p><?php echo esc_html( $excerpt ); ?></p>
								</div>
							<?php endif; ?>

							<!-- Continue Reading Button -->
							<div class="arwrev-magazine-footer">
								<span class="arwrev-continue-reading-btn">
									<span class="arwrev-btn-text"><?php esc_html_e( 'Continue Reading', 'arewa-recently-viewed-content' ); ?></span>
									<svg class="arwrev-btn-arrow" width="16" height="16" viewBox="0 0 16 16" fill="none">
										<path d="M8.5 3L13.5 8L8.5 13M13 8H2.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
									</svg>
								</span>
							</div>
						</div>
					</div>
				</a>

				<!-- Remove Button -->
				<button type="button" class="arwrev-remove-magazine-btn"
					data-post-id="<?php echo esc_attr( $post_id ); ?>"
					data-nonce="<?php echo esc_attr( wp_create_nonce( 'arwrev_remove_item_' . $post_id ) ); ?>"
					aria-label="<?php esc_attr_e( 'Remove this article', 'arewa-recently-viewed-content' ); ?>"
					title="<?php esc_attr_e( 'Remove from history', 'arewa-recently-viewed-content' ); ?>">
					<svg width="14" height="14" viewBox="0 0 14 14" fill="none">
						<path d="M10.5 3.5L3.5 10.5M3.5 3.5L10.5 10.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
					</svg>
				</button>
			</div>
		</article>
		<?php
		return ob_get_clean();
	}
}
