<?php
/**
 * Arewa Recently Viewed Content Tracker
 *
 * Handles tracking and storing Arewa Recently Viewed Content for logged-in users
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure WordPress functions are available.
if ( ! function_exists( 'add_action' ) ) {
	return;
}

/**
 * Arewa Recently Viewed Content Tracker Class.
 *
 * Handles tracking and storing Arewa Recently Viewed Content for logged-in users.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Tracker {

	/**
	 * Instance of this class
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Get instance of this class
	 *
	 * @return ARWREV_Tracker
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'template_redirect', array( $this, 'track_visit' ) );
		add_action( 'arwrev_cleanup', array( $this, 'cleanup_old_entries' ) );
		add_action( 'wp_login', array( $this, 'merge_guest_history_on_login' ), 10, 2 );
	}

	/**
	 * Track visit to a post
	 */
	public function track_visit() {
		// Don't track in admin or during AJAX requests.
		if ( is_admin() || wp_doing_ajax() ) {
			return;
		}

		// Check if tracking is enabled.
		$settings = get_option( 'arwrev_settings', array() );
		if ( empty( $settings['enabled'] ) || '1' !== $settings['enabled'] ) {
			return;
		}

		// Only track singular posts.
		if ( ! is_singular() ) {
			return;
		}

		$post_id   = get_the_ID();
		$post_type = get_post_type( $post_id );

		// Check if this post type should be tracked.
		$tracked_post_types = isset( $settings['post_types'] ) ? $settings['post_types'] : array();
		if ( ! in_array( $post_type, $tracked_post_types, true ) ) {
			return;
		}

		// Track for both logged-in and guest users.
		if ( is_user_logged_in() ) {
			$this->add_to_history( $post_id, $post_type );
		} else {
			$this->add_to_guest_history( $post_id, $post_type );
		}
	}

	/**
	 * Add item to user's watch history.
	 *
	 * @param int    $post_id   Post ID.
	 * @param string $post_type Post type.
	 */
	public function add_to_history( $post_id, $post_type ) {
		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return;
		}

		// Get current history.
		$history = $this->get_user_history( $user_id );

		// Check if post already exists in history.
		$existing_key = false;
		foreach ( $history as $key => $item ) {
			if ( (int) $item['post_id'] === (int) $post_id ) {
				$existing_key = $key;
				break;
			}
		}

		$timestamp = time();

		// Move revisited item to beginning: update timestamp and reorder so it appears first.
		if ( false !== $existing_key ) {
			$item              = $history[ $existing_key ];
			$item['timestamp'] = $timestamp;
			unset( $history[ $existing_key ] );
			$history = array_values( $history );
			array_unshift( $history, $item );
		} else {
			array_unshift(
				$history,
				array(
					'post_id'   => (int) $post_id,
					'post_type' => sanitize_text_field( $post_type ),
					'timestamp' => $timestamp,
				)
			);
		}

		// Ensure order is by timestamp (newest first); no duplicates at this point.
		usort(
			$history,
			function( $a, $b ) {
				return $b['timestamp'] - $a['timestamp'];
			}
		);

		// Apply max items limit.
		$settings  = get_option( 'arwrev_settings', array() );
		$max_items = isset( $settings['max_items'] ) ? (int) $settings['max_items'] : 100;
		if ( $max_items > 0 && count( $history ) > $max_items ) {
			$history = array_slice( $history, 0, $max_items );
		}

		// Save history.
		update_user_meta( $user_id, 'arewa_recently_viewed', $history );

		// Schedule cleanup if enabled.
		$settings = get_option( 'arwrev_settings', array() );
		if ( ! empty( $settings['auto_cleanup_enabled'] ) && '1' === $settings['auto_cleanup_enabled'] ) {
			if ( ! wp_next_scheduled( 'arwrev_cleanup' ) ) {
				wp_schedule_event( time(), 'daily', 'arwrev_cleanup' );
			}
		}
	}

	/**
	 * Get user's watch history.
	 *
	 * @param int $user_id User ID.
	 * @return array
	 */
	public function get_user_history( $user_id = null ) {
		if ( null === $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( ! $user_id ) {
			return array();
		}

		// Try to get from new meta key first.
		$history = get_user_meta( $user_id, 'arewa_recently_viewed', true );

		// If empty or invalid, try to migrate from old keys.
		if ( empty( $history ) || ! is_array( $history ) ) {
			$history = $this->migrate_user_history_on_demand( $user_id );
		}

		if ( ! is_array( $history ) ) {
			return array();
		}

		// Filter out invalid entries.
		$history = array_filter(
			$history,
			function( $item ) {
				return isset( $item['post_id'] ) && get_post( $item['post_id'] );
			}
		);

		// Re-index array.
		$history = array_values( $history );

		// Always sort by timestamp (newest first) so display order is most-recent-first.
		usort(
			$history,
			function( $a, $b ) {
				$ts_a = isset( $a['timestamp'] ) ? (int) $a['timestamp'] : 0;
				$ts_b = isset( $b['timestamp'] ) ? (int) $b['timestamp'] : 0;
				return $ts_b - $ts_a;
			}
		);

		return $history;
	}

	/**
	 * Migrate user history on demand if new meta key is empty.
	 *
	 * @param int $user_id User ID.
	 * @return array Migrated history or empty array.
	 */
	private function migrate_user_history_on_demand( $user_id ) {
		// Check old arwrev_history key.
		$arwrev_history    = get_user_meta( $user_id, 'arwrev_history', true );
		$old_arewa_history = get_user_meta( $user_id, 'arewa_wh_history', true );

		$final_history = array();

		// Merge histories if they exist.
		if ( ! empty( $arwrev_history ) && is_array( $arwrev_history ) ) {
			$final_history = $arwrev_history;
		}

		if ( ! empty( $old_arewa_history ) && is_array( $old_arewa_history ) ) {
			if ( empty( $final_history ) ) {
				$final_history = $old_arewa_history;
			} else {
				// Merge the two histories.
				$final_history = $this->merge_histories( $final_history, $old_arewa_history );
			}
		}

		// If we found data, save it to the new key.
		if ( ! empty( $final_history ) ) {
			update_user_meta( $user_id, 'arewa_recently_viewed', $final_history );

			// Log migration for debugging.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when explicitly enabled
				error_log( "ARWREV: On-demand migration for user {$user_id}, found " . count( $final_history ) . ' items.' );
			}
		}

		return $final_history;
	}

	/**
	 * Remove item from user's history.
	 *
	 * @param int $post_id Post ID.
	 * @param int $user_id User ID.
	 * @return bool
	 */
	public function remove_from_history( $post_id, $user_id = null ) {
		if ( null === $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( ! $user_id ) {
			return false;
		}

		$history = $this->get_user_history( $user_id );

		$history = array_filter(
			$history,
			function( $item ) use ( $post_id ) {
				return (int) $item['post_id'] !== (int) $post_id;
			}
		);

		// Re-index array.
		$history = array_values( $history );

		update_user_meta( $user_id, 'arewa_recently_viewed', $history );

		return true;
	}

	/**
	 * Clear all history for a user.
	 *
	 * @param int $user_id User ID.
	 * @return bool
	 */
	public function clear_history( $user_id = null ) {
		if ( null === $user_id ) {
			$user_id = get_current_user_id();
		}

		if ( ! $user_id ) {
			return false;
		}

		return delete_user_meta( $user_id, 'arewa_recently_viewed' );
	}

	/**
	 * Add item to guest user's history via cookie.
	 *
	 * @param int    $post_id   Post ID.
	 * @param string $post_type Post type.
	 */
	public function add_to_guest_history( $post_id, $post_type ) {
		// Get current guest history from cookie.
		$history = $this->get_guest_history();

		// Check if post already exists in history.
		$existing_key = false;
		foreach ( $history as $key => $item ) {
			if ( (int) $item['post_id'] === (int) $post_id ) {
				$existing_key = $key;
				break;
			}
		}

		$timestamp = time();

		// Move revisited item to beginning: update timestamp and reorder so it appears first.
		if ( false !== $existing_key ) {
			$item              = $history[ $existing_key ];
			$item['timestamp'] = $timestamp;
			unset( $history[ $existing_key ] );
			$history = array_values( $history );
			array_unshift( $history, $item );
		} else {
			array_unshift(
				$history,
				array(
					'post_id'   => (int) $post_id,
					'post_type' => sanitize_text_field( $post_type ),
					'timestamp' => $timestamp,
				)
			);
		}

		// Ensure order is by timestamp (newest first); no duplicates.
		usort(
			$history,
			function( $a, $b ) {
				return $b['timestamp'] - $a['timestamp'];
			}
		);

		// Apply max items limit for guests.
		$settings        = get_option( 'arwrev_settings', array() );
		$max_guest_items = isset( $settings['max_guest_items'] ) ? (int) $settings['max_guest_items'] : 50;
		if ( $max_guest_items > 0 && count( $history ) > $max_guest_items ) {
			$history = array_slice( $history, 0, $max_guest_items );
		}

		// Save to cookie.
		$this->set_guest_history_cookie( $history );
	}

	/**
	 * Get guest user's history from cookie.
	 *
	 * @return array
	 */
	public function get_guest_history() {
		if ( ! isset( $_COOKIE['arewa_recently_viewed'] ) ) {
			return array();
		}

		$cookie_data = sanitize_text_field( wp_unslash( $_COOKIE['arewa_recently_viewed'] ) );
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode -- Cookie data encoding for guest history.
		$history = json_decode( base64_decode( $cookie_data ), true );

		if ( ! is_array( $history ) ) {
			return array();
		}

		// Get tracked post types from settings.
		$settings           = get_option( 'arwrev_settings', array() );
		$tracked_post_types = isset( $settings['post_types'] ) ? $settings['post_types'] : array();

		// Filter out invalid entries and non-tracked post types.
		$history = array_filter(
			$history,
			function( $item ) use ( $tracked_post_types ) {
				// Check if post exists.
				if ( ! isset( $item['post_id'] ) || ! get_post( $item['post_id'] ) ) {
					return false;
				}

				// If no post types configured, allow all.
				if ( empty( $tracked_post_types ) ) {
					return true;
				}

				// Check if post type is tracked.
				$post_type = isset( $item['post_type'] ) ? $item['post_type'] : '';
				return in_array( $post_type, $tracked_post_types, true );
			}
		);

		// Re-index array.
		$history = array_values( $history );

		// Sort by timestamp (newest first) - same as user history.
		usort(
			$history,
			function( $a, $b ) {
				$timestamp_a = isset( $a['timestamp'] ) ? (int) $a['timestamp'] : 0;
				$timestamp_b = isset( $b['timestamp'] ) ? (int) $b['timestamp'] : 0;
				return $timestamp_b - $timestamp_a; // Newest first.
			}
		);

		return $history;
	}

	/**
	 * Set guest history cookie.
	 *
	 * @param array $history History data.
	 */
	private function set_guest_history_cookie( $history ) {
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Cookie data encoding for guest history.
		$cookie_data = base64_encode( wp_json_encode( $history ) );
		$expiration  = time() + ( 30 * DAY_IN_SECONDS ); // 30 days.

		setcookie( 'arewa_recently_viewed', $cookie_data, $expiration, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true );
	}

	/**
	 * Get history for current user (logged-in or guest).
	 *
	 * @return array
	 */
	public function get_current_user_history() {
		if ( is_user_logged_in() ) {
			return $this->get_user_history();
		} else {
			return $this->get_guest_history();
		}
	}

	/**
	 * Merge guest history with user history on login.
	 *
	 * @param string  $user_login Username.
	 * @param WP_User $user       User object.
	 */
	public function merge_guest_history_on_login( $user_login, $user ) {
		// Get guest history from cookie.
		$guest_history = $this->get_guest_history();

		if ( empty( $guest_history ) ) {
			return;
		}

		// Get existing user history.
		$user_history = $this->get_user_history( $user->ID );

		// Merge histories.
		$merged_history = $this->merge_histories( $user_history, $guest_history );

		// Apply max items limit for logged-in users.
		$settings  = get_option( 'arwrev_settings', array() );
		$max_items = isset( $settings['max_items'] ) ? (int) $settings['max_items'] : 100;
		if ( $max_items > 0 && count( $merged_history ) > $max_items ) {
			$merged_history = array_slice( $merged_history, 0, $max_items );
		}

		// Save merged history to user meta.
		update_user_meta( $user->ID, 'arewa_recently_viewed', $merged_history );

		// Clear guest cookie after successful merge.
		$this->clear_guest_history_cookie();
	}

	/**
	 * Merge two history arrays, removing duplicates and sorting by timestamp.
	 *
	 * @param array $user_history   User's existing history.
	 * @param array $guest_history  Guest history from cookie.
	 * @return array Merged and sorted history.
	 */
	private function merge_histories( $user_history, $guest_history ) {
		$merged     = array();
		$seen_posts = array();

		// Combine both histories.
		$all_items = array_merge( $user_history, $guest_history );

		// Remove duplicates, keeping the most recent timestamp.
		foreach ( $all_items as $item ) {
			$post_id = (int) $item['post_id'];

			if ( ! isset( $seen_posts[ $post_id ] ) || $item['timestamp'] > $seen_posts[ $post_id ]['timestamp'] ) {
				$seen_posts[ $post_id ] = $item;
			}
		}

		// Convert back to indexed array.
		$merged = array_values( $seen_posts );

		// Sort by timestamp (newest first).
		usort(
			$merged,
			function( $a, $b ) {
				return $b['timestamp'] - $a['timestamp'];
			}
		);

		return $merged;
	}

	/**
	 * Clear guest history cookie.
	 */
	private function clear_guest_history_cookie() {
		setcookie( 'arewa_recently_viewed', '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true );
	}

	/**
	 * Clean up old entries when history exceeds max_items per user.
	 */
	public function cleanup_old_entries() {
		$settings  = get_option( 'arwrev_settings', array() );
		$max_items = isset( $settings['max_items'] ) ? (int) $settings['max_items'] : 100;

		if ( $max_items <= 0 ) {
			return;
		}

		$users = get_users( array( 'fields' => 'ID' ) );

		foreach ( $users as $user_id ) {
			$history = $this->get_user_history( $user_id );

			if ( count( $history ) > $max_items ) {
				$history = array_slice( $history, 0, $max_items );
				update_user_meta( $user_id, 'arewa_recently_viewed', $history );
			}
		}
	}
}
