<?php
/**
 * Watch History Shortcode (New Version with Layout Manager)
 *
 * Handles [watch_history] shortcode rendering with layout system
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure WordPress functions are available.
if ( ! function_exists( 'add_shortcode' ) ) {
	return;
}

/**
 * Watch History Shortcode Class (New Version).
 *
 * Handles [watch_history] shortcode rendering with layout system.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Shortcode_New {

	/**
	 * Instance of this class
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Get instance of this class
	 *
	 * @return ARWREV_Shortcode_New
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Delay shortcode registration until after textdomain is loaded.
		add_action( 'init', array( $this, 'init_shortcode_hooks' ) );
		// Ensure assets are loaded when shortcode is used.
		add_action( 'wp_enqueue_scripts', array( $this, 'maybe_enqueue_assets' ) );
	}

	/**
	 * Initialize shortcode hooks after textdomain is loaded
	 */
	public function init_shortcode_hooks() {
		add_shortcode( 'watch_history', array( $this, 'render_shortcode' ) );
	}

	/**
	 * Enqueue assets if shortcode might be used.
	 */
	public function maybe_enqueue_assets() {
		global $post;

		// Check if shortcode exists in post content or widgets.
		if ( is_a( $post, 'WP_Post' ) && has_shortcode( $post->post_content, 'watch_history' ) ) {
			// Force load assets.
			do_action( 'arwrev_enqueue_assets' );
		}

		// Also check if we're on a page that might have the shortcode (via filter).
		if ( apply_filters( 'arwrev_should_load_assets', false ) ) {
			do_action( 'arwrev_enqueue_assets' );
		}
	}

	/**
	 * Render shortcode.
	 *
	 * @param array $atts Shortcode attributes.
	 * @return string
	 */
	public function render_shortcode( $atts ) {
		// Ensure assets are loaded when shortcode is rendered.
		do_action( 'arwrev_enqueue_assets' );

		// Get settings.
		$settings = get_option( 'arwrev_settings', array() );

		// Check if guest users should be required to login.
		$require_login_for_guests = isset( $settings['require_login_for_guests'] ) ? $settings['require_login_for_guests'] : '0';

		// If not logged in and login is required for guests, show login message.
		if ( ! is_user_logged_in() && '1' === $require_login_for_guests ) {
			return $this->render_login_required();
		}

		// Parse attributes.
		$atts = $this->parse_shortcode_attributes( $atts, $settings );

		// Process limit attribute before merging settings.
		if ( ! empty( $atts['limit'] ) && is_numeric( $atts['limit'] ) ) {
			$atts['items_per_page'] = absint( $atts['limit'] );
		}

		// Override settings with shortcode attributes if provided.
		$settings = $this->merge_settings_with_attributes( $settings, $atts );

		// Get history for current user (logged-in or guest).
		$tracker = ARWREV_Tracker::get_instance();
		$history = $tracker->get_current_user_history();

		// Filter history based on shortcode attributes.
		$history = $this->filter_history( $history, $atts );

		// Check if history is empty.
		if ( empty( $history ) ) {
			if ( ! is_user_logged_in() ) {
				// Check if login is required for guests.
				if ( '1' === $require_login_for_guests ) {
					return $this->render_login_required();
				}
				// Show simple empty message without login button.
				return '<div class="arwrev-container arwrev-guest-empty"><p class="arwrev-message">' . esc_html__( 'Your recently viewed content will appear here as you browse.', 'arewa-recently-viewed-content' ) . '</p></div>';
			}
			return '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'Your watch history is empty.', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		// Handle pagination.
		$pagination_data = $this->handle_pagination( $history, $atts );

		ob_start();

		// Only show clear all button for logged-in users and if enabled in shortcode.
		$show_clear_button = 'yes' === strtolower( $atts['show_clear_all_history'] );
		if ( is_user_logged_in() && $show_clear_button ) {
			?>
			<div class="arwrev-actions">
				<button type="button" class="arwrev-clear-all button" data-nonce="<?php echo esc_attr( wp_create_nonce( 'arwrev_clear_all' ) ); ?>">
					<?php esc_html_e( 'Clear All History', 'arewa-recently-viewed-content' ); ?>
				</button>
			</div>
			<?php
		}

		// Use Layout Manager to render the content. Items are in history order (most recent first).
		if ( class_exists( 'ARWREV_Layout_Manager' ) ) {
			$layout_manager = ARWREV_Layout_Manager::get_instance();
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Layout manager escapes all dynamic parts internally.
			echo $layout_manager->render_layout( $pagination_data['items'], $settings, $atts );
		} else {
			// Fallback if Layout Manager is not available yet.
			echo '<div class="arwrev-container"><p class="arwrev-message">' . esc_html__( 'Layout system loading...', 'arewa-recently-viewed-content' ) . '</p></div>';
		}

		// Render pagination if needed (but never for carousel layout).
		if ( $pagination_data['show_pagination'] && 'carousel' !== $atts['layout_type'] ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- render_pagination escapes URLs and uses wp_kses_post for links.
			echo $this->render_pagination( $pagination_data );
		}

		return ob_get_clean();
	}

	/**
	 * Render login required message
	 *
	 * @return string
	 */
	private function render_login_required() {
		$settings          = get_option( 'arwrev_settings', array() );
		$login_url         = isset( $settings['login_url'] ) && ! empty( $settings['login_url'] ) ? $settings['login_url'] : wp_login_url();
		$login_button_text = isset( $settings['login_button_text'] ) && ! empty( $settings['login_button_text'] ) ? $settings['login_button_text'] : __( 'Login My Account', 'arewa-recently-viewed-content' );

		$output  = '<div class="arwrev-container arwrev-login-required">';
		$output .= '<p class="arwrev-message">' . esc_html__( 'Please log in to view your watch history.', 'arewa-recently-viewed-content' ) . '</p>';
		$output .= '<div class="arwrev-login-button-wrapper">';
		$output .= '<a href="' . esc_url( $login_url ) . '" class="arwrev-login-button button">' . esc_html( $login_button_text ) . '</a>';
		$output .= '</div>';
		$output .= '</div>';

		return $output;
	}

	/**
	 * Render guest empty message
	 *
	 * @return string
	 */
	private function render_guest_empty_message() {
		$settings          = get_option( 'arwrev_settings', array() );
		$login_url         = isset( $settings['login_url'] ) && ! empty( $settings['login_url'] ) ? $settings['login_url'] : wp_login_url();
		$login_button_text = isset( $settings['login_button_text'] ) && ! empty( $settings['login_button_text'] ) ? $settings['login_button_text'] : __( 'Login My Account', 'arewa-recently-viewed-content' );

		$output  = '<div class="arwrev-container arwrev-guest-empty">';
		$output .= '<p class="arwrev-message">' . esc_html__( 'Your recently viewed content will appear here as you browse.', 'arewa-recently-viewed-content' ) . '</p>';
		$output .= '<p class="arwrev-submessage">' . esc_html__( 'Log in to sync your history across devices and keep it permanently.', 'arewa-recently-viewed-content' ) . '</p>';
		$output .= '<div class="arwrev-login-button-wrapper">';
		$output .= '<a href="' . esc_url( $login_url ) . '" class="arwrev-login-button button">' . esc_html( $login_button_text ) . '</a>';
		$output .= '</div>';
		$output .= '</div>';

		return $output;
	}

	/**
	 * Parse shortcode attributes
	 *
	 * @param array $atts Raw attributes.
	 * @param array $settings Plugin settings.
	 * @return array
	 */
	private function parse_shortcode_attributes( $atts, $settings ) {
		$items_per_page_default = isset( $settings['items_per_page'] ) ? (int) $settings['items_per_page'] : 12;

		// Parse attributes first to check layout_type.
		$parsed_atts = shortcode_atts(
			array(
				'items_per_page'         => $items_per_page_default,
				'show_image'             => 'yes',
				'show_date'              => 'yes',
				'pagination'             => 'yes',
				'layout_type'            => '', // Allow override via shortcode.
				'layout'                 => '', // Alternative to layout_type for backward compatibility.
				'display_mode'           => '', // Allow override via shortcode.
				'post_type'              => '', // Filter by post type(s).
				'category'               => '', // Filter by category.
				'taxonomy'               => '', // Filter by custom taxonomy.
				'limit'                  => '', // Limit number of items (alternative to items_per_page).
				'show_clear_all_history' => 'yes', // Show/hide Clear All History button.
			),
			$atts,
			'watch_history'
		);

		// Handle layout attribute (backward compatibility).
		if ( ! empty( $parsed_atts['layout'] ) && empty( $parsed_atts['layout_type'] ) ) {
			$parsed_atts['layout_type'] = $parsed_atts['layout'];
		}

		// Special handling for carousel layout.
		if ( 'carousel' === $parsed_atts['layout_type'] ) {
			// Set higher default limit for carousel if not specified.
			if ( empty( $parsed_atts['limit'] ) && $parsed_atts['items_per_page'] === $items_per_page_default ) {
				$parsed_atts['items_per_page'] = 20; // Higher default for carousel.
			}
			// Disable pagination by default for carousel.
			if ( ! isset( $atts['pagination'] ) ) {
				$parsed_atts['pagination'] = 'no';
			}
		}

		return $parsed_atts;
	}

	/**
	 * Merge settings with shortcode attributes
	 *
	 * @param array $settings Plugin settings.
	 * @param array $atts Shortcode attributes.
	 * @return array
	 */
	private function merge_settings_with_attributes( $settings, $atts ) {
		// Handle layout attribute (backward compatibility).
		if ( ! empty( $atts['layout'] ) && empty( $atts['layout_type'] ) ) {
			$atts['layout_type'] = $atts['layout'];
		}

		// Override settings with shortcode attributes if provided.
		if ( ! empty( $atts['layout_type'] ) ) {
			$settings['layout_type'] = $atts['layout_type'];
		}
		if ( ! empty( $atts['display_mode'] ) ) {
			$settings['display_mode'] = $atts['display_mode'];
		}

		// Auto-infer post_type from layout if not explicitly provided.
		if ( empty( $atts['post_type'] ) && ! empty( $settings['layout_type'] ) ) {
			$atts['post_type'] = $this->infer_post_type_from_layout( $settings['layout_type'] );
		}

		// Set defaults if not set.
		if ( ! isset( $settings['layout_type'] ) ) {
			$settings['layout_type'] = 'video';
		}
		if ( ! isset( $settings['display_mode'] ) ) {
			$settings['display_mode'] = 'grid';
		}

		return $settings;
	}

	/**
	 * Infer post type from layout type
	 *
	 * @param string $layout_type Layout type.
	 * @return string
	 */
	private function infer_post_type_from_layout( $layout_type ) {
		$layout_to_post_type_map = array(
			'video'    => 'movies',
			'article'  => 'post',
			'shop'     => 'product',
			'carousel' => '', // No specific post type for carousel.
		);

		return isset( $layout_to_post_type_map[ $layout_type ] ) ? $layout_to_post_type_map[ $layout_type ] : '';
	}

	/**
	 * Filter history based on shortcode attributes
	 *
	 * @param array $history Full history array.
	 * @param array $atts Shortcode attributes.
	 * @return array
	 */
	private function filter_history( $history, $atts ) {
		if ( empty( $history ) ) {
			return $history;
		}

		$filtered_history = $history;

		// Filter by post type(s).
		if ( ! empty( $atts['post_type'] ) ) {
			$post_types       = $this->parse_comma_separated_values( $atts['post_type'] );
			$filtered_history = array_filter(
				$filtered_history,
				function( $item ) use ( $post_types ) {
					$post = get_post( $item['post_id'] );
					return $post && in_array( $post->post_type, $post_types, true );
				}
			);
		}

		// Filter by category.
		if ( ! empty( $atts['category'] ) ) {
			$categories       = $this->parse_comma_separated_values( $atts['category'] );
			$filtered_history = array_filter(
				$filtered_history,
				function( $item ) use ( $categories ) {
					return $this->post_has_terms( $item['post_id'], 'category', $categories );
				}
			);
		}

		// Filter by custom taxonomy.
		if ( ! empty( $atts['taxonomy'] ) ) {
			// Parse taxonomy:term format (e.g., "genre:action,comedy" or "genre:action").
			$taxonomy_filters = $this->parse_taxonomy_filter( $atts['taxonomy'] );

			foreach ( $taxonomy_filters as $taxonomy => $terms ) {
				$filtered_history = array_filter(
					$filtered_history,
					function( $item ) use ( $taxonomy, $terms ) {
						return $this->post_has_terms( $item['post_id'], $taxonomy, $terms );
					}
				);
			}
		}

		// Re-index array to maintain proper array structure.
		$filtered_history = array_values( $filtered_history );

		// Mandatory: sort by visit time (newest first). Never use orderby date for recently viewed.
		// This order is passed to all layouts; they must preserve it and must not re-sort.
		usort(
			$filtered_history,
			function( $a, $b ) {
				$timestamp_a = isset( $a['timestamp'] ) ? (int) $a['timestamp'] : 0;
				$timestamp_b = isset( $b['timestamp'] ) ? (int) $b['timestamp'] : 0;
				return $timestamp_b - $timestamp_a; // Newest first.
			}
		);

		return $filtered_history;
	}

	/**
	 * Parse comma-separated values into array
	 *
	 * @param string $value Comma-separated values.
	 * @return array
	 */
	private function parse_comma_separated_values( $value ) {
		if ( empty( $value ) ) {
			return array();
		}

		$values = explode( ',', $value );
		return array_map( 'trim', $values );
	}

	/**
	 * Parse taxonomy filter format
	 *
	 * @param string $taxonomy_filter Taxonomy filter string.
	 * @return array
	 */
	private function parse_taxonomy_filter( $taxonomy_filter ) {
		$filters = array();

		// Handle multiple taxonomy filters separated by semicolon.
		$taxonomy_parts = explode( ';', $taxonomy_filter );

		foreach ( $taxonomy_parts as $part ) {
			$part = trim( $part );
			if ( strpos( $part, ':' ) !== false ) {
				list( $taxonomy, $terms_string ) = explode( ':', $part, 2 );
				$taxonomy                        = trim( $taxonomy );
				$terms                           = $this->parse_comma_separated_values( $terms_string );

				if ( ! empty( $taxonomy ) && ! empty( $terms ) ) {
					$filters[ $taxonomy ] = $terms;
				}
			}
		}

		return $filters;
	}

	/**
	 * Check if post has specific terms in taxonomy
	 *
	 * @param int    $post_id Post ID.
	 * @param string $taxonomy Taxonomy name.
	 * @param array  $terms Array of term names or slugs.
	 * @return bool
	 */
	private function post_has_terms( $post_id, $taxonomy, $terms ) {
		if ( empty( $terms ) ) {
			return true;
		}

		$post_terms = wp_get_post_terms( $post_id, $taxonomy, array( 'fields' => 'all' ) );

		if ( is_wp_error( $post_terms ) || empty( $post_terms ) ) {
			return false;
		}

		foreach ( $post_terms as $post_term ) {
			// Check both name and slug.
			if ( in_array( $post_term->name, $terms, true ) || in_array( $post_term->slug, $terms, true ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Handle pagination logic
	 *
	 * @param array $history Full history array.
	 * @param array $atts Shortcode attributes.
	 * @return array
	 */
	private function handle_pagination( $history, $atts ) {
		$items_per_page = absint( $atts['items_per_page'] );
		$pagination     = 'yes' === strtolower( $atts['pagination'] );

		// Ensure minimum items per page.
		if ( $items_per_page < 1 ) {
			$items_per_page = 12; // Default fallback.
		}

		// Get current page.
		global $wp_query;

		$paged = 1;
		if ( get_query_var( 'paged' ) ) {
			$paged = absint( get_query_var( 'paged' ) );
		} elseif ( isset( $wp_query->query_vars['paged'] ) && $wp_query->query_vars['paged'] > 0 ) {
			$paged = absint( $wp_query->query_vars['paged'] );
		} elseif ( get_query_var( 'wh_page' ) ) {
			$paged = absint( get_query_var( 'wh_page' ) );
		} else {
			// Try to extract page number from URL (format: /page/2/).
			$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput
			if ( preg_match( '#/page/(\d+)/?#', $request_uri, $matches ) ) {
				$paged = absint( $matches[1] );
			}
		}

		$current_page = max( $paged, 1 );
		$total_items  = count( $history );

		// Always apply items_per_page limit, regardless of pagination setting.
		if ( $pagination ) {
			// With pagination: calculate pages and slice accordingly.
			$total_pages = ceil( $total_items / $items_per_page );
			$offset      = ( $current_page - 1 ) * $items_per_page;
			$items       = array_slice( $history, $offset, $items_per_page );
		} else {
			// Without pagination: just limit to items_per_page from the beginning.
			$total_pages = 1;
			$items       = array_slice( $history, 0, $items_per_page );
		}

		return array(
			'items'           => $items,
			'current_page'    => $current_page,
			'total_pages'     => $total_pages,
			'total_items'     => $total_items,
			'show_pagination' => $pagination && $total_pages > 1,
		);
	}

	/**
	 * Render pagination
	 *
	 * @param array $pagination_data Pagination data.
	 * @return string
	 */
	private function render_pagination( $pagination_data ) {
		// Get the base URL for pagination.
		$page_permalink = get_permalink();

		if ( $page_permalink ) {
			$page_permalink = preg_replace( '#/page/\d+/?#', '/', $page_permalink );
		} else {
			$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput
			$request_uri = strtok( $request_uri, '?' );
			$request_uri = preg_replace( '#/page/\d+/?#', '', $request_uri );
			$request_uri = rtrim( $request_uri, '/' );

			if ( ! empty( $request_uri ) ) {
				$page_permalink = home_url( $request_uri );
			} else {
				$page_permalink = home_url();
			}
		}

		$page_permalink  = preg_replace( '#/page/\d+/?#', '/', $page_permalink );
		$page_permalink  = rtrim( $page_permalink, '/' );
		$pagination_base = trailingslashit( $page_permalink );
		$pagination_base = preg_replace( '#(/page/\d+)+/#', '/', $pagination_base );
		$pagination_base = trailingslashit( $pagination_base );

		$pagination_links = paginate_links(
			array(
				'base'      => $pagination_base . 'page/%#%/',
				'format'    => '',
				'current'   => $pagination_data['current_page'],
				'total'     => $pagination_data['total_pages'],
				'prev_text' => __( '&laquo; Previous', 'arewa-recently-viewed-content' ),
				'next_text' => __( 'Next &raquo;', 'arewa-recently-viewed-content' ),
				'add_args'  => false,
			)
		);

		return '<div class="arwrev-pagination">' . wp_kses_post( $pagination_links ) . '</div>';
	}
}
