<?php
/**
 * Data Management Class
 *
 * Handles data preservation, backup, and restoration for the plugin
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Data Manager Class.
 *
 * Handles data preservation, backup, and restoration functionality.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Data_Manager {

	/**
	 * Instance of this class
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Get instance of this class
	 *
	 * @return ARWREV_Data_Manager
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		// Hook into WordPress actions.
		add_action( 'admin_init', array( $this, 'check_restoration_status' ) );
	}

	/**
	 * Check if data needs to be restored and handle it.
	 */
	public function check_restoration_status() {
		$preserved_data = get_option( 'arwrev_data_preserved', false );

		if ( $preserved_data && is_array( $preserved_data ) && isset( $preserved_data['preserved'] ) && ! isset( $preserved_data['restored'] ) ) {
			// Data was preserved but not yet restored, trigger restoration.
			$this->restore_preserved_data();
		}
	}

	/**
	 * Create a backup of all plugin data
	 *
	 * @return array Backup data array
	 */
	public function create_backup() {
		global $wpdb;

		$backup_data = array(
			'timestamp'      => time(),
			'version'        => get_option( 'arwrev_version', ARWREV_VERSION ),
			'settings'       => get_option( 'arwrev_settings', array() ),
			'user_histories' => array(),
			'options'        => array(),
		);

		// Backup all plugin-related options.
		$plugin_options = array(
			'arwrev_preserve_data_on_uninstall',
			'arwrev_version',
			'arewa_wh_settings', // Legacy option.
		);

		foreach ( $plugin_options as $option ) {
			$value = get_option( $option, null );
			if ( null !== $value ) {
				$backup_data['options'][ $option ] = $value;
			}
		}

		// Backup user history data.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for data backup functionality.
		$user_meta_results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT user_id, meta_value FROM {$wpdb->usermeta} WHERE meta_key IN (%s, %s, %s)",
				'arewa_recently_viewed',
				'arwrev_history',
				'arewa_wh_history'
			)
		);

		foreach ( $user_meta_results as $meta ) {
			$history_data = maybe_unserialize( $meta->meta_value );
			if ( is_array( $history_data ) && ! empty( $history_data ) ) {
				$backup_data['user_histories'][ $meta->user_id ] = $history_data;
			}
		}

		return $backup_data;
	}

	/**
	 * Restore data from backup
	 *
	 * @param array $backup_data Backup data to restore.
	 * @return bool Success status
	 */
	public function restore_from_backup( $backup_data ) {
		if ( ! is_array( $backup_data ) || empty( $backup_data ) ) {
			return false;
		}

		$restored = true;

		// Restore settings.
		if ( isset( $backup_data['settings'] ) && is_array( $backup_data['settings'] ) ) {
			update_option( 'arwrev_settings', $backup_data['settings'] );
		}

		// Restore other options.
		if ( isset( $backup_data['options'] ) && is_array( $backup_data['options'] ) ) {
			foreach ( $backup_data['options'] as $option_name => $option_value ) {
				update_option( $option_name, $option_value );
			}
		}

		// Restore user histories.
		if ( isset( $backup_data['user_histories'] ) && is_array( $backup_data['user_histories'] ) ) {
			foreach ( $backup_data['user_histories'] as $user_id => $history_data ) {
				if ( is_array( $history_data ) && ! empty( $history_data ) ) {
					update_user_meta( $user_id, 'arewa_recently_viewed', $history_data );
				}
			}
		}

		// Update version if available.
		if ( isset( $backup_data['version'] ) ) {
			update_option( 'arwrev_version', $backup_data['version'] );
		}

		return $restored;
	}

	/**
	 * Restore preserved data after plugin reactivation
	 */
	public function restore_preserved_data() {
		$preserved_data = get_option( 'arwrev_data_preserved', false );

		if ( ! $preserved_data || ! is_array( $preserved_data ) ) {
			return false;
		}

		// Verify that settings still exist (they should in preserve mode).
		$settings = get_option( 'arwrev_settings', false );
		if ( false === $settings ) {
			// This shouldn't happen in preserve mode, but handle gracefully.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when explicitly enabled
				error_log( 'ARWREV: Settings not found during restoration, this may indicate a problem with data preservation.' );
			}
			return false;
		}

		// Mark as restored.
		$preserved_data['restored']        = true;
		$preserved_data['restored_at']     = time();
		$preserved_data['current_version'] = ARWREV_VERSION;

		update_option( 'arwrev_data_preserved', $preserved_data );

		// Show restoration notice.
		add_option( 'arwrev_show_restoration_notice', true );

		// Log successful restoration.
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when explicitly enabled
			error_log( 'ARWREV: Data successfully restored after plugin reactivation.' );
		}

		return true;
	}

	/**
	 * Get data statistics for admin display
	 *
	 * @return array Statistics array
	 */
	public function get_data_statistics() {
		global $wpdb;

		$stats = array(
			'settings_exist'           => false,
			'total_users_with_history' => 0,
			'total_history_items'      => 0,
			'oldest_entry'             => null,
			'newest_entry'             => null,
			'plugin_version'           => get_option( 'arwrev_version', 'Unknown' ),
		);

		// Check if settings exist.
		$settings                = get_option( 'arwrev_settings', false );
		$stats['settings_exist'] = ( false !== $settings && is_array( $settings ) );

		// Count users with history.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for statistics functionality.
		$user_count                        = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(DISTINCT user_id) FROM {$wpdb->usermeta} WHERE meta_key = %s",
				'arewa_recently_viewed'
			)
		);
		$stats['total_users_with_history'] = (int) $user_count;

		// Get history statistics.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Required for statistics functionality.
		$history_meta = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT meta_value FROM {$wpdb->usermeta} WHERE meta_key = %s",
				'arewa_recently_viewed'
			)
		);

		$total_items = 0;
		$timestamps  = array();

		foreach ( $history_meta as $meta ) {
			$history = maybe_unserialize( $meta->meta_value );
			if ( is_array( $history ) ) {
				$total_items += count( $history );

				foreach ( $history as $item ) {
					if ( isset( $item['timestamp'] ) ) {
						$timestamps[] = $item['timestamp'];
					}
				}
			}
		}

		$stats['total_history_items'] = $total_items;

		if ( ! empty( $timestamps ) ) {
			$stats['oldest_entry'] = min( $timestamps );
			$stats['newest_entry'] = max( $timestamps );
		}

		return $stats;
	}

	/**
	 * Export plugin data for manual backup
	 *
	 * @return string JSON encoded backup data
	 */
	public function export_data() {
		$backup_data = $this->create_backup();
		return wp_json_encode( $backup_data, JSON_PRETTY_PRINT );
	}

	/**
	 * Import plugin data from manual backup
	 *
	 * @param string $json_data JSON encoded backup data.
	 * @return bool Success status
	 */
	public function import_data( $json_data ) {
		$backup_data = json_decode( $json_data, true );

		if ( json_last_error() !== JSON_ERROR_NONE ) {
			return false;
		}

		return $this->restore_from_backup( $backup_data );
	}
}
