<?php
/**
 * AJAX Handlers
 *
 * Handles AJAX requests for watch history operations
 *
 * @package Arewa_Recently_Viewed_Content
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure WordPress functions are available.
if ( ! function_exists( 'add_action' ) ) {
	return;
}

/**
 * AJAX Handlers Class.
 *
 * Handles AJAX requests for watch history operations.
 *
 * @package Arewa_Recently_Viewed_Content
 */
class ARWREV_Ajax {

	/**
	 * Instance of this class
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Get instance of this class
	 *
	 * @return ARWREV_Ajax
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'wp_ajax_arwrev_remove_item', array( $this, 'remove_item' ) );
		add_action( 'wp_ajax_arwrev_clear_all', array( $this, 'clear_all' ) );
		add_action( 'wp_ajax_arwrev_export_data', array( $this, 'export_data' ) );
		add_action( 'wp_ajax_arwrev_import_data', array( $this, 'import_data' ) );

		// Guest user AJAX handlers (no login required).
		add_action( 'wp_ajax_nopriv_arwrev_track_guest_view', array( $this, 'track_guest_view' ) );
		add_action( 'wp_ajax_arwrev_track_guest_view', array( $this, 'track_guest_view' ) );

		// Guest user remove item (no login required).
		add_action( 'wp_ajax_nopriv_arwrev_guest_remove_item', array( $this, 'guest_remove_item' ) );

		// Guest user clear all (no login required).
		add_action( 'wp_ajax_nopriv_arwrev_guest_clear_all', array( $this, 'guest_clear_all' ) );
	}

	/**
	 * Remove single item from history.
	 */
	public function remove_item() {
		// Check nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'arwrev_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'arewa-recently-viewed-content' ) ) );
		}

		// Check user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'arewa-recently-viewed-content' ) ) );
		}

		// Get post ID.
		if ( ! isset( $_POST['post_id'] ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid request.', 'arewa-recently-viewed-content' ) ) );
		}

		$post_id = absint( wp_unslash( $_POST['post_id'] ) );

		// Verify nonce for specific item.
		if ( ! isset( $_POST['item_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['item_nonce'] ) ), 'arwrev_remove_item_' . $post_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'arewa-recently-viewed-content' ) ) );
		}

		// Remove item.
		$tracker = ARWREV_Tracker::get_instance();
		$removed = $tracker->remove_from_history( $post_id );

		if ( $removed ) {
			wp_send_json_success(
				array(
					'message' => __( 'Item removed successfully.', 'arewa-recently-viewed-content' ),
				)
			);
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to remove item.', 'arewa-recently-viewed-content' ) ) );
		}
	}

	/**
	 * Clear all history.
	 */
	public function clear_all() {
		// Check nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'arwrev_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'arewa-recently-viewed-content' ) ) );
		}

		// Check user is logged in.
		if ( ! is_user_logged_in() ) {
			wp_send_json_error( array( 'message' => __( 'You must be logged in.', 'arewa-recently-viewed-content' ) ) );
		}

		// Verify clear all nonce.
		if ( ! isset( $_POST['clear_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['clear_nonce'] ) ), 'arwrev_clear_all' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'arewa-recently-viewed-content' ) ) );
		}

		// Clear history.
		$tracker = ARWREV_Tracker::get_instance();
		$cleared = $tracker->clear_history();

		if ( $cleared ) {
			wp_send_json_success(
				array(
					'message' => __( 'All history cleared successfully.', 'arewa-recently-viewed-content' ),
				)
			);
		} else {
			wp_send_json_error( array( 'message' => __( 'Failed to clear history.', 'arewa-recently-viewed-content' ) ) );
		}
	}

	/**
	 * Export plugin data.
	 */
	public function export_data() {
		// Check nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'arwrev_export_nonce' ) ) {
			wp_send_json_error( __( 'Security check failed.', 'arewa-recently-viewed-content' ) );
		}

		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'arewa-recently-viewed-content' ) );
		}

		// Get data manager and export data.
		$data_manager = ARWREV_Data_Manager::get_instance();
		$export_data  = $data_manager->export_data();

		if ( $export_data ) {
			wp_send_json_success( $export_data );
		} else {
			wp_send_json_error( __( 'Failed to export data.', 'arewa-recently-viewed-content' ) );
		}
	}

	/**
	 * Import plugin data.
	 */
	public function import_data() {
		// Check nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'arwrev_import_nonce' ) ) {
			wp_send_json_error( __( 'Security check failed.', 'arewa-recently-viewed-content' ) );
		}

		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'arewa-recently-viewed-content' ) );
		}

		// Get backup data.
		if ( ! isset( $_POST['backup_data'] ) ) {
			wp_send_json_error( __( 'No backup data provided.', 'arewa-recently-viewed-content' ) );
		}

		$backup_data = sanitize_textarea_field( wp_unslash( $_POST['backup_data'] ) );

		// Import data.
		$data_manager = ARWREV_Data_Manager::get_instance();
		$imported     = $data_manager->import_data( $backup_data );

		if ( $imported ) {
			wp_send_json_success( __( 'Data imported successfully.', 'arewa-recently-viewed-content' ) );
		} else {
			wp_send_json_error( __( 'Failed to import data. Please check the file format.', 'arewa-recently-viewed-content' ) );
		}
	}

	/**
	 * Track guest user view (AJAX fallback for JavaScript tracking).
	 */
	public function track_guest_view() {
		// Check nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'arwrev_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'arewa-recently-viewed-content' ) ) );
		}

		// Get post data.
		if ( ! isset( $_POST['post_id'] ) || ! isset( $_POST['post_type'] ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid request.', 'arewa-recently-viewed-content' ) ) );
		}

		$post_id   = absint( wp_unslash( $_POST['post_id'] ) );
		$post_type = sanitize_text_field( wp_unslash( $_POST['post_type'] ) );

		// Verify post exists.
		if ( ! get_post( $post_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Post not found.', 'arewa-recently-viewed-content' ) ) );
		}

		// Track the view.
		$tracker = ARWREV_Tracker::get_instance();

		if ( is_user_logged_in() ) {
			$tracker->add_to_history( $post_id, $post_type );
		} else {
			$tracker->add_to_guest_history( $post_id, $post_type );
		}

		wp_send_json_success(
			array(
				'message' => __( 'View tracked successfully.', 'arewa-recently-viewed-content' ),
			)
		);
	}

	/**
	 * Remove item from guest history.
	 * CRITICAL: Must update the server-side cookie so it persists after page reload.
	 */
	public function guest_remove_item() {
		// Check nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'arwrev_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'arewa-recently-viewed-content' ) ) );
		}

		// Get post ID.
		if ( ! isset( $_POST['post_id'] ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid request.', 'arewa-recently-viewed-content' ) ) );
		}

		$post_id = absint( wp_unslash( $_POST['post_id'] ) );

		// Get current guest history from cookie.
		$tracker = ARWREV_Tracker::get_instance();
		$history = $tracker->get_guest_history();

		// Remove the item from history.
		$history = array_filter(
			$history,
			function( $item ) use ( $post_id ) {
				return (int) $item['post_id'] !== (int) $post_id;
			}
		);

		// Re-index array.
		$history = array_values( $history );

		// Update the cookie on the server.
		if ( empty( $history ) ) {
			// If empty, expire the cookie.
			setcookie( 'arewa_recently_viewed', '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true );
		} else {
			// Save updated history to cookie.
			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Cookie data encoding for guest history.
			$cookie_data = base64_encode( wp_json_encode( $history ) );
			$expiration  = time() + ( 30 * DAY_IN_SECONDS );
			setcookie( 'arewa_recently_viewed', $cookie_data, $expiration, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true );
		}

		wp_send_json_success(
			array(
				'message'         => __( 'Item removed successfully.', 'arewa-recently-viewed-content' ),
				'remaining_items' => count( $history ),
			)
		);
	}

	/**
	 * Clear all guest history.
	 * CRITICAL: Must expire the server-side cookie so it persists after page reload.
	 */
	public function guest_clear_all() {
		// Check nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'arwrev_nonce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'arewa-recently-viewed-content' ) ) );
		}

		// Expire the cookie on the server.
		setcookie( 'arewa_recently_viewed', '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true );

		wp_send_json_success(
			array(
				'message' => __( 'All history cleared successfully.', 'arewa-recently-viewed-content' ),
			)
		);
	}
}
