/**
 * Arewa Recently Viewed Content - Frontend JavaScript
 * Enhanced with hybrid tracking for logged-in and guest users
 *
 * @package Arewa_Recently_Viewed_Content
 */

// IMMEDIATE TEST - This should appear in console.
console.log( 'Arewa Recently Viewed Content: JavaScript file loaded successfully' );

(function($) {
	'use strict';

	// Guest user tracking functionality..
	var GuestTracker = {
		cookieName: 'arewa_recently_viewed',
		maxItems: 50,
		trackedPostTypes: [], // Will be populated from arwrev.tracked_post_types.

		init: function() {
			// Get tracked post types from arwrev object.
			if (typeof arwrev !== 'undefined' && arwrev.tracked_post_types) {
				this.trackedPostTypes = arwrev.tracked_post_types;
				console.log( 'Guest Tracker: Tracked post types:', this.trackedPostTypes );
			}

			// Only track for guest users.
			if (typeof arwrev !== 'undefined' && arwrev.is_user_logged_in) {
				return;
			}

			// Track current page if it's a singular post.
			this.trackCurrentPage();
		},

		trackCurrentPage: function() {
			// Check if we're on a singular post page.
			if ( ! document.body.classList.contains( 'single' ) &&
				! document.body.classList.contains( 'page' )) {
				return;
			}

			// Get post data from body classes or meta tags.
			var postId   = this.getCurrentPostId();
			var postType = this.getCurrentPostType();

			if (postId && postType) {
				// Check if this post type should be tracked.
				if (this.trackedPostTypes.length > 0 && this.trackedPostTypes.indexOf( postType ) === -1) {
					console.log( 'Guest Tracker: Post type "' + postType + '" is not tracked. Skipping.' );
					return;
				}

				this.addToHistory( postId, postType );
			}
		},

		getCurrentPostId: function() {
			// Try to get post ID from arwrev object first (most reliable).
			if (typeof arwrev !== 'undefined' && arwrev.current_post_id) {
				return parseInt( arwrev.current_post_id );
			}

			// Try to get post ID from body class.
			var bodyClasses = document.body.className;
			var match       = bodyClasses.match( /postid-(\d+)/ );
			if (match) {
				return parseInt( match[1] );
			}

			// Try to get from meta tag.
			var metaTag = document.querySelector( 'meta[name="post-id"]' );
			if (metaTag) {
				return parseInt( metaTag.getAttribute( 'content' ) );
			}

			// Try to get from data attribute on body.
			var postIdAttr = document.body.getAttribute( 'data-post-id' );
			if (postIdAttr) {
				return parseInt( postIdAttr );
			}

			return null;
		},

		getCurrentPostType: function() {
			// Try to get post type from arwrev object first (most reliable).
			if (typeof arwrev !== 'undefined' && arwrev.current_post_type) {
				return arwrev.current_post_type;
			}

			// Try to get post type from body class.
			var bodyClasses = document.body.className;
			var match       = bodyClasses.match( /single-(\w+)/ );
			if (match) {
				return match[1];
			}

			// Check for page.
			if (document.body.classList.contains( 'page' )) {
				return 'page';
			}

			// Try to get from meta tag.
			var metaTag = document.querySelector( 'meta[name="post-type"]' );
			if (metaTag) {
				return metaTag.getAttribute( 'content' );
			}

			// Try to get from data attribute on body.
			var postTypeAttr = document.body.getAttribute( 'data-post-type' );
			if (postTypeAttr) {
				return postTypeAttr;
			}

			return 'post'; // Default fallback.
		},

		addToHistory: function(postId, postType) {
			var history   = this.getHistory();
			var timestamp = Math.floor( Date.now() / 1000 );

			// Remove existing entry if it exists.
			history = history.filter(
				function(item) {
					return item.post_id !== postId;
				}
			);

			// Add new entry at the beginning.
			history.unshift(
				{
					post_id: postId,
					post_type: postType,
					timestamp: timestamp
				}
			);

			// Limit to max items.
			if (history.length > this.maxItems) {
				history = history.slice( 0, this.maxItems );
			}

			// Save to both localStorage and cookie.
			this.saveHistory( history );

			console.log( 'Guest tracking: Added post', postId, 'to history' );
		},

		getHistory: function() {
			var history = [];

			// Try localStorage first (primary storage).
			if (typeof Storage !== 'undefined') {
				try {
					var stored = localStorage.getItem( this.cookieName );
					if (stored) {
						history = JSON.parse( stored );
						console.log( 'Guest Tracker: Loaded', history.length, 'items from localStorage' );
					}
				} catch (e) {
					console.warn( 'Guest Tracker: Failed to read from localStorage:', e );
				}
			}

			// Fallback to cookie if localStorage failed or is empty.
			if (history.length === 0) {
				history = this.getHistoryFromCookie();
				if (history.length > 0) {
					console.log( 'Guest Tracker: Loaded', history.length, 'items from cookie (fallback)' );
					// Sync to localStorage if we got data from cookie.
					if (typeof Storage !== 'undefined') {
						try {
							localStorage.setItem( this.cookieName, JSON.stringify( history ) );
							console.log( 'Guest Tracker: Synced cookie data to localStorage' );
						} catch (e) {
							console.warn( 'Guest Tracker: Failed to sync to localStorage:', e );
						}
					}
				}
			}

			return Array.isArray( history ) ? history : [];
		},

		getHistoryFromCookie: function() {
			var cookieValue = this.getCookie( this.cookieName );
			if ( ! cookieValue) {
				return [];
			}

			try {
				return JSON.parse( atob( cookieValue ) );
			} catch (e) {
				console.warn( 'Failed to parse cookie data:', e );
				return [];
			}
		},

		saveHistory: function(history, skipIndicator) {
			console.log( 'Guest Tracker: Saving history with', history.length, 'items' );

			// Save to localStorage (primary storage).
			if (typeof Storage !== 'undefined') {
				try {
					if (history.length === 0) {
						// If empty, remove the key entirely.
						localStorage.removeItem( this.cookieName );
						console.log( 'Guest Tracker: Removed empty history from localStorage' );
					} else {
						localStorage.setItem( this.cookieName, JSON.stringify( history ) );
						console.log( 'Guest Tracker: Saved to localStorage successfully' );
					}
				} catch (e) {
					console.error( 'Guest Tracker: Failed to save to localStorage:', e );
				}
			}

			// Save to cookie as fallback (important for WebView compatibility).
			var cookiePath   = (typeof arwrev !== 'undefined' && arwrev.cookie_path) ? arwrev.cookie_path : '/';
			var cookieDomain = (typeof arwrev !== 'undefined' && arwrev.cookie_domain) ? arwrev.cookie_domain : '';

			if (history.length === 0) {
				// If empty, expire the cookie immediately.
				var expireCookieString = this.cookieName + '=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=' + cookiePath;
				if (cookieDomain) {
					expireCookieString += '; domain=' + cookieDomain;
				}
				document.cookie = expireCookieString;
				console.log( 'Guest Tracker: Expired empty cookie' );
			} else {
				// Save non-empty history to cookie.
				var cookieData = btoa( JSON.stringify( history ) );
				var expiration = new Date();
				expiration.setTime( expiration.getTime() + (30 * 24 * 60 * 60 * 1000) ); // 30 days.

				// Build cookie string with proper attributes for WebView compatibility.
				var cookieString = this.cookieName + '=' + cookieData +
					';expires=' + expiration.toUTCString() +
					';path=' + cookiePath +
					';SameSite=Lax';

				if (cookieDomain) {
					cookieString += ';domain=' + cookieDomain;
				}

				// Add Secure flag only for HTTPS.
				if (location.protocol === 'https:') {
					cookieString += ';Secure';
				}

				document.cookie = cookieString;
				console.log( 'Guest Tracker: Saved to cookie successfully' );
			}

			// Show brief indicator for guest users (optional) - skip when removing items.
			if ( ! skipIndicator) {
				this.showTrackingIndicator();
			}
		},

		showTrackingIndicator: function() {
			// Only show for guest users and not too frequently.
			if (typeof arwrev !== 'undefined' && arwrev.is_user_logged_in) {
				return;
			}

			// Check if we've shown this recently.
			var lastShown = localStorage.getItem( 'arwrev_indicator_shown' );
			var now       = Date.now();
			if (lastShown && (now - parseInt( lastShown )) < 300000) { // 5 minutes.
				return;
			}

			// Create and show indicator.
			var indicator         = document.createElement( 'div' );
			indicator.className   = 'arwrev-guest-indicator';
			indicator.textContent = 'Content tracked';
			document.body.appendChild( indicator );

			// Show with animation.
			setTimeout(
				function() {
					indicator.classList.add( 'show' );
				},
				100
			);

			// Hide after 3 seconds.
			setTimeout(
				function() {
					indicator.classList.remove( 'show' );
					setTimeout(
						function() {
							if (indicator.parentNode) {
								indicator.parentNode.removeChild( indicator );
							}
						},
						300
					);
				},
				3000
			);

			// Remember we showed this.
			try {
				localStorage.setItem( 'arwrev_indicator_shown', now.toString() );
			} catch (e) {
				// Ignore localStorage errors.
			}
		},

		getCookie: function(name) {
			var nameEQ = name + '=';
			var ca     = document.cookie.split( ';' );
			var caLen  = ca.length;
			for ( var i = 0; i < caLen; i++ ) {
				var c = ca[i];
				while (c.charAt( 0 ) === ' ') {
					c = c.substring( 1, c.length );
				}
				if (c.indexOf( nameEQ ) === 0) {
					return c.substring( nameEQ.length, c.length );
				}
			}
			return null;
		},

		/**
		 * Remove item from guest history.
		 */
		removeFromHistory: function(postId) {
			console.log( 'Guest Tracker: Attempting to remove post ID:', postId, 'Type:', typeof postId );

			var history = this.getHistory();
			console.log( 'Guest Tracker: Current history before removal:', history );

			// Ensure postId is an integer for comparison.
			postId = parseInt( postId, 10 );

			// Filter out the item - compare as integers.
			var filteredHistory = history.filter(
				function(item) {
					var itemPostId = parseInt( item.post_id, 10 );
					var keep       = itemPostId !== postId;
					if ( ! keep) {
						console.log( 'Guest Tracker: Removing item with post_id:', item.post_id );
					}
					return keep;
				}
			);

			console.log( 'Guest Tracker: History after filtering:', filteredHistory );
			console.log( 'Guest Tracker: Items removed:', history.length - filteredHistory.length );

			// Save updated history (skip indicator when removing).
			this.saveHistory( filteredHistory, true );

			// Verify it was saved.
			var verifyHistory = this.getHistory();
			console.log( 'Guest Tracker: Verified history after save:', verifyHistory );

			return true;
		},

		/**
		 * Clear all guest history.
		 */
		clearHistory: function() {
			console.log( 'Guest Tracker: Clearing all history' );

			// Clear localStorage.
			if (typeof Storage !== 'undefined') {
				try {
					localStorage.removeItem( this.cookieName );
					console.log( 'Guest Tracker: Cleared localStorage' );
				} catch (e) {
					console.error( 'Guest Tracker: Failed to clear localStorage:', e );
				}
			}

			// Clear cookie by setting expiration to past date.
			var cookiePath  = (typeof arwrev !== 'undefined' && arwrev.cookie_path) ? arwrev.cookie_path : '/';
			document.cookie = this.cookieName + '=; expires=Thu, 01 Jan 1970 00:00:00 UTC; path=' + cookiePath + ';';
			console.log( 'Guest Tracker: Cleared cookie' );

			// Verify it was cleared.
			var verifyHistory = this.getHistory();
			console.log( 'Guest Tracker: Verified history after clear:', verifyHistory );

			return true;
		}
	};

	// Modal functions.
	function showModal(options) {
		var title              = options.title || 'Confirm';
		var message            = options.message || '';
		var confirmText        = options.confirmText || 'OK';
		var cancelText         = options.cancelText || 'Cancel';
		var onConfirm          = options.onConfirm || function() {};
		var onCancel           = options.onCancel || function() {};
		var showCancel         = options.showCancel !== false;
		var confirmButtonClass = options.confirmButtonClass || 'arwrev-modal-button-primary';

		// Remove existing modal if any and clean up event handlers.
		$( '.arwrev-modal-overlay' ).remove();
		$( document ).off( 'keydown.arwrevModal' );

		var modalHtml = '<div class="arwrev-modal-overlay">' +
			'<div class="arwrev-modal">' +
			'<div class="arwrev-modal-header">' +
			'<h3 class="arwrev-modal-title">' + title + '</h3>' +
			(showCancel ? '<button type="button" class="arwrev-modal-close" aria-label="Close">&times;</button>' : '') +
			'</div>' +
			'<div class="arwrev-modal-body">' + message + '</div>' +
			'<div class="arwrev-modal-footer">' +
			(showCancel ? '<button type="button" class="arwrev-modal-button arwrev-modal-button-secondary arwrev-modal-cancel">' + cancelText + '</button>' : '') +
			'<button type="button" class="arwrev-modal-button ' + confirmButtonClass + ' arwrev-modal-confirm">' + confirmText + '</button>' +
			'</div>' +
			'</div>' +
			'</div>';

		var $modal = $( modalHtml );
		$( 'body' ).append( $modal );

		console.log( 'WP Video Watch History: Modal created and appended to body', $modal.length );

		// Use requestAnimationFrame to ensure DOM is ready.
		requestAnimationFrame(
			function() {
				// Force display to ensure it's visible.
				$modal.css(
					{
						'display': 'flex',
						'opacity': '1',
						'visibility': 'visible'
					}
				);
				console.log( 'WP Video Watch History: Modal CSS applied', $modal.is( ':visible' ) );
			}
		);

		// Use a flag to prevent immediate closing.
		var modalClosing = false;

		// Close on overlay click.
		$modal.on(
			'click.modal',
			function(e) {
				if ( modalClosing ) {
					return false;
				}
				if ( $( e.target ).hasClass( 'arwrev-modal-overlay' ) ) {
					modalClosing = true;
					e.preventDefault();
					e.stopPropagation();
					e.stopImmediatePropagation();
					closeModal();
					if ( typeof onCancel === 'function' ) {
						onCancel();
					}
					return false;
				}
			}
		);

		// Close button.
		$modal.find( '.arwrev-modal-close, .arwrev-modal-cancel' ).on(
			'click.modal',
			function(e) {
				if ( modalClosing ) {
					return false;
				}
				modalClosing = true;
				e.preventDefault();
				e.stopPropagation();
				e.stopImmediatePropagation();
				closeModal();
				if ( typeof onCancel === 'function' ) {
					onCancel();
				}
				return false;
			}
		);

		// Confirm button.
		$modal.find( '.arwrev-modal-confirm' ).on(
			'click.modal',
			function(e) {
				if ( modalClosing ) {
					return false;
				}
				modalClosing = true;
				e.preventDefault();
				e.stopPropagation();
				e.stopImmediatePropagation();
				closeModal();
				if ( typeof onConfirm === 'function' ) {
					onConfirm();
				}
				return false;
			}
		);

		// Prevent clicks inside modal from closing it.
		$modal.find( '.arwrev-modal' ).on(
			'click.modal',
			function(e) {
				e.stopPropagation();
				e.stopImmediatePropagation();
				return false;
			}
		);

		// ESC key to close.
		$( document ).on(
			'keydown.arwrevModal',
			function(e) {
				if ( modalClosing ) {
					return false;
				}
				if ( e.key === 'Escape' || e.keyCode === 27 ) {
					modalClosing = true;
					e.preventDefault();
					e.stopPropagation();
					closeModal();
					if ( typeof onCancel === 'function' ) {
						onCancel();
					}
					return false;
				}
			}
		);

		// Return the modal element for external use.
		return $modal;
	}

	function closeModal() {
		var $modal = $( '.arwrev-modal-overlay' );
		if ( $modal.length === 0 ) {
			return;
		}
		$modal.css( 'animation', 'arwrevFadeOut 0.2s ease-out forwards' );
		$modal.find( '.arwrev-modal' ).css( 'animation', 'arwrevModalSlideOut 0.2s ease-out forwards' );
		$modal.off( 'click.modal' );
		setTimeout(
			function() {
				$modal.remove();
				$( document ).off( 'keydown.arwrevModal' );
			},
			200
		);
	}

	// Wait for both jQuery and DOM to be ready.
	function initArwrev() {
		// Check if arwrev is available.
		if ( typeof arwrev === 'undefined' ) {
			console.warn( 'Arewa Recently Viewed Content: arwrev object not yet available, retrying...' );
			setTimeout( initArwrev, 100 );
			return;
		}

		console.log( 'Arewa Recently Viewed Content: Initializing...', arwrev );

		// Initialize guest tracking.
		GuestTracker.init();
			// Remove single item.
			$( document ).on(
				'click.arwrev',
				'.arwrev-remove-item, .arwrev-remove-magazine-btn, .arwrev-shop-remove-btn, .arwrev-carousel-remove',
				function(e) {
					e.preventDefault();
					e.stopPropagation();
					e.stopImmediatePropagation();

					var $button = $( this );

					// Prevent multiple clicks.
					if ( $button.prop( 'disabled' ) || $button.hasClass( 'processing' ) ) {
						return false;
					}

					// Mark as processing.
					$button.addClass( 'processing' );

					var $item     = $button.closest( '.arwrev-item, .arwrev-magazine-card, .arwrev-shop-card, .arwrev-item-carousel' );
					var postId    = $button.data( 'post-id' );
					var itemNonce = $button.data( 'nonce' );

					if ( ! postId) {
						console.error( 'WP Video Watch History: post_id is missing' );
						$button.removeClass( 'processing' );
						return false;
					}

					if ( typeof arwrev === 'undefined' ) {
						console.error( 'Arewa Recently Viewed Content: arwrev object is not defined. Make sure the script is loaded correctly.' );
						$button.removeClass( 'processing' );
						showModal(
							{
								title: 'Error',
								message: 'Error: Script not loaded properly. Please refresh the page and try again.',
								confirmText: 'OK',
								showCancel: false,
								confirmButtonClass: 'arwrev-modal-button-primary'
							}
						);
						return false;
					}

					// Show confirmation modal immediately.
					console.log( 'WP Video Watch History: Showing remove modal for post ID:', postId );
					var modalInstance = showModal(
						{
							title: 'Remove Item',
							message: 'Are you sure you want to remove this item from your watch history?',
							confirmText: 'Remove',
							cancelText: 'Cancel',
							confirmButtonClass: 'arwrev-modal-button-danger',
							onConfirm: function() {
								// Disable button.
								$button.removeClass( 'processing' );
								$button.prop( 'disabled', true ).html( '<svg width="12" height="12" viewBox="0 0 12 12" fill="none"><circle cx="6" cy="6" r="5" stroke="currentColor" stroke-width="2" opacity="0.3"/><path d="M6 1v5l3 3" stroke="currentColor" stroke-width="2" stroke-linecap="round"/></svg>' );

								// Check if user is logged in.
								var isLoggedIn = arwrev.is_user_logged_in;

								if ( ! isLoggedIn) {
									// Guest user - use AJAX to update server-side cookie.
									console.log( 'Guest user: Sending AJAX request to remove item from server' );

									$.ajax(
										{
											url: arwrev.ajaxurl,
											type: 'POST',
											data: {
												action: 'arwrev_guest_remove_item',
												post_id: postId,
												nonce: arwrev.nonce
											},
											success: function(response) {
												console.log( 'Guest user: Server response:', response );

												if (response && response.success) {
													// Also update client-side storage.
													GuestTracker.removeFromHistory( postId );

													// Fade out and remove item.
													$item.fadeOut(
														300,
														function() {
															$( this ).remove();

															// Check if container is empty.
															var $container     = $( '.arwrev-container' );
															var $grid          = $container.find( '.arwrev-grid, .arwrev-magazine-grid, .arwrev-shop-grid, .arwrev-carousel-track' );
															var remainingItems = $grid.find( '.arwrev-item, .arwrev-magazine-card, .arwrev-shop-card, .arwrev-item-carousel' ).length;

															if ( remainingItems === 0) {
																// No items left, show empty message.
																$grid.after( '<p class="arwrev-message">Your watch history is empty.</p>' );
																$container.find( '.arwrev-actions, .arwrev-pagination' ).remove();
															}
														}
													);
												} else {
													var errorMessage = (response && response.data && response.data.message) ? response.data.message : 'Failed to remove item. Please try again.';
													showModal(
														{
															title: 'Error',
															message: errorMessage,
															confirmText: 'OK',
															showCancel: false,
															confirmButtonClass: 'arwrev-modal-button-primary'
														}
													);
													$button.prop( 'disabled', false ).html( '<svg width="14" height="14" viewBox="0 0 14 14" fill="none"><path d="M10.5 3.5L3.5 10.5M3.5 3.5L10.5 10.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>' );
												}
											},
											error: function(xhr, status, error) {
												console.error( 'Guest user AJAX Error:', status, error, xhr.responseText );
												showModal(
													{
														title: 'Error',
														message: 'An error occurred. Please try again or refresh the page.',
														confirmText: 'OK',
														showCancel: false,
														confirmButtonClass: 'arwrev-modal-button-primary'
													}
												);
												$button.prop( 'disabled', false ).html( '<svg width="14" height="14" viewBox="0 0 14 14" fill="none"><path d="M10.5 3.5L3.5 10.5M3.5 3.5L10.5 10.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>' );
											}
										}
									);

									return;
								}

								// Logged-in user - AJAX request.
								$.ajax(
									{
										url: arwrev.ajaxurl,
										type: 'POST',
										data: {
											action: 'arwrev_remove_item',
											post_id: postId,
											nonce: arwrev.nonce,
											item_nonce: itemNonce
										},
										success: function(response) {
											if (response && response.success) {
												// Fade out and remove item.
												$item.fadeOut(
													300,
													function() {
														$( this ).remove();

														// Check if container is empty.
														var $container     = $( '.arwrev-container' );
														var $grid          = $container.find( '.arwrev-grid, .arwrev-magazine-grid, .arwrev-shop-grid, .arwrev-carousel-track' );
														var remainingItems = $grid.find( '.arwrev-item, .arwrev-magazine-card, .arwrev-shop-card, .arwrev-item-carousel' ).length;

														if ( remainingItems === 0) {
															// No items left, show empty message.
															$grid.after( '<p class="arwrev-message">Your watch history is empty.</p>' );
															$container.find( '.arwrev-actions, .arwrev-pagination' ).remove();
														} else {
															// Items remain, reload the page to update pagination and grid properly.
															setTimeout(
																function() {
																	window.location.reload();
																},
																100
															);
														}
													}
												);
											} else {
												var errorMessage = (response && response.data && response.data.message) ? response.data.message : 'Failed to remove item. Please try again.';
												showModal(
													{
														title: 'Error',
														message: errorMessage,
														confirmText: 'OK',
														showCancel: false,
														confirmButtonClass: 'arwrev-modal-button-primary'
													}
												);
												$button.prop( 'disabled', false ).html( '<svg width="14" height="14" viewBox="0 0 14 14" fill="none"><path d="M10.5 3.5L3.5 10.5M3.5 3.5L10.5 10.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>' );
											}
										},
										error: function(xhr, status, error) {
											console.error( 'WP Video Watch History AJAX Error:', status, error, xhr.responseText );
											showModal(
												{
													title: 'Error',
													message: 'An error occurred. Please try again or refresh the page.',
													confirmText: 'OK',
													showCancel: false,
													confirmButtonClass: 'arwrev-modal-button-primary'
												}
											);
											$button.prop( 'disabled', false ).html( '<svg width="14" height="14" viewBox="0 0 14 14" fill="none"><path d="M10.5 3.5L3.5 10.5M3.5 3.5L10.5 10.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>' );
										}
									}
								);
							},
							onCancel: function() {
								// User cancelled, remove processing class.
								$button.removeClass( 'processing' );
							}
						}
					);

					return false;
				}
			);

			// Clear all history.
			$( document ).on(
				'click.arwrev',
				'.arwrev-clear-all',
				function(e) {
					e.preventDefault();
					e.stopPropagation();
					e.stopImmediatePropagation();

					var $button = $( this );

					// Prevent multiple clicks.
					if ( $button.prop( 'disabled' ) || $button.hasClass( 'processing' ) ) {
						return false;
					}

					// Mark as processing.
					$button.addClass( 'processing' );

					var clearNonce = $button.data( 'nonce' );

					if ( typeof arwrev === 'undefined' ) {
						console.error( 'Arewa Recently Viewed Content: arwrev object is not defined. Make sure the script is loaded correctly.' );
						$button.removeClass( 'processing' );
						showModal(
							{
								title: 'Error',
								message: 'Error: Script not loaded properly. Please refresh the page and try again.',
								confirmText: 'OK',
								showCancel: false,
								confirmButtonClass: 'arwrev-modal-button-primary'
							}
						);
						return false;
					}

					// Show confirmation modal immediately.
					console.log( 'WP Video Watch History: Showing clear all modal' );
					var modalInstance = showModal(
						{
							title: 'Clear All History',
							message: 'Are you sure you want to clear all watch history? This action cannot be undone.',
							confirmText: 'Clear All',
							cancelText: 'Cancel',
							confirmButtonClass: 'arwrev-modal-button-danger',
							onConfirm: function() {
								// Disable button.
								$button.removeClass( 'processing' );
								$button.prop( 'disabled', true ).text( 'Clearing...' );

								// Check if user is logged in to use correct endpoint.
								var isLoggedIn = arwrev.is_user_logged_in;
								var ajaxAction = isLoggedIn ? 'arwrev_clear_all' : 'arwrev_guest_clear_all';

								console.log( 'Clear All: User type:', isLoggedIn ? 'Logged in' : 'Guest', 'Action:', ajaxAction );

								// AJAX request.
								$.ajax(
									{
										url: arwrev.ajaxurl,
										type: 'POST',
										data: {
											action: ajaxAction,
											nonce: arwrev.nonce,
											clear_nonce: clearNonce
										},
										success: function(response) {
											if (response && response.success) {
												// For guest users, also clear client-side storage.
												if ( ! isLoggedIn) {
													GuestTracker.clearHistory();
												}

												// Show message and reload page.
												$( '.arwrev-container' ).html(
													'<p class="arwrev-message">All watch history has been cleared.</p>'
												);
											} else {
												var errorMessage = (response && response.data && response.data.message) ? response.data.message : 'Failed to clear history. Please try again.';
												showModal(
													{
														title: 'Error',
														message: errorMessage,
														confirmText: 'OK',
														showCancel: false,
														confirmButtonClass: 'arwrev-modal-button-primary'
													}
												);
												$button.prop( 'disabled', false ).text( 'Clear All History' );
											}
										},
										error: function(xhr, status, error) {
											console.error( 'WP Video Watch History AJAX Error:', status, error, xhr.responseText );
											showModal(
												{
													title: 'Error',
													message: 'An error occurred. Please try again or refresh the page.',
													confirmText: 'OK',
													showCancel: false,
													confirmButtonClass: 'arwrev-modal-button-primary'
												}
											);
											$button.prop( 'disabled', false ).text( 'Clear All History' );
										}
									}
								);
							},
							onCancel: function() {
								// User cancelled, remove processing class.
								$button.removeClass( 'processing' );
							}
						}
					);

					return false;
				}
			);
	}

	// Initialize when DOM is ready.
	if ( document.readyState === 'loading' ) {
		document.addEventListener( 'DOMContentLoaded', initArwrev );
	} else {
		// DOM is already ready.
		initArwrev();
	}

	// Fallback: also try with jQuery ready if available.
	if ( typeof jQuery !== 'undefined' ) {
		jQuery( document ).ready( initArwrev );
	}

})( jQuery );

/**
 * Carousel with auto-play - clean implementation.
 */
(function($) {
	'use strict';

	console.log( 'arwrev Carousel: Loading with Auto-Play...' );

	// Global carousel state.
	var carousels = {};

	function initializeCarousel($container) {
		var containerId = $container.attr( 'id' ) || 'carousel-' + Math.random().toString( 36 ).substr( 2, 9 );
		$container.attr( 'id', containerId );

		// Skip if already initialized.
		if (carousels[containerId]) {
			console.log( 'arwrev Carousel: Already initialized', containerId );
			return;
		}

		console.log( 'arwrev Carousel: Initializing with auto-play', containerId );

		var $wrapper = $container.find( '.arwrev-carousel-wrapper' );
		var $track   = $container.find( '.arwrev-carousel-track' );
		// Use items in DOM order only (history order from server). Do not re-sort; only handle UI/animation.
		var $items   = $track.find( '.arwrev-item-carousel' );
		var $prevBtn = $container.find( '.arwrev-carousel-prev' );
		var $nextBtn = $container.find( '.arwrev-carousel-next' );

		if ($items.length === 0) {
			console.log( 'arwrev Carousel: No items found' );
			return;
		}

		// Carousel state with auto-play.
		var state = {
			currentIndex: 0,
			itemWidth: 0,
			visibleItems: 3, // Default to 3 items visible on desktop.
			maxIndex: 0,
			isAnimating: false,
			autoPlayInterval: null,
			autoPlayDelay: 4000, // 4 seconds between slides.
			isAutoPlaying: false,
			isPaused: false
		};

		// Calculate dimensions.
		function updateDimensions() {
			var windowWidth  = $( window ).width();
			var wrapperWidth = $wrapper.width(); // Get actual wrapper width.
			var gap          = 10; // Default gap.

			// Mobile: no gap to prevent showing next slide.
			if (windowWidth <= 767) {
				gap = 0;
			}

			// Responsive visible items.
			if (windowWidth <= 767) {
				state.visibleItems = 1; // Mobile: 1 item.
			} else if (windowWidth <= 1023) {
				state.visibleItems = 2; // Tablet: 2 items.
			} else {
				state.visibleItems = 3; // Desktop: 3 items.
			}

			// Calculate item width based on wrapper width and visible items.
			var totalGaps = (state.visibleItems - 1) * gap;
			var itemWidth = Math.floor( (wrapperWidth - totalGaps) / state.visibleItems ); // FLOOR to prevent partial pixels.

			// Total width to scroll per item = item width + gap.
			// Round to prevent sub-pixel rendering issues.
			state.itemWidth = Math.round( itemWidth + gap );

			state.maxIndex = Math.max( 0, $items.length - state.visibleItems );

			console.log(
				'arwrev Carousel: Dimensions updated',
				{
					wrapperWidth: wrapperWidth,
					itemWidth: itemWidth,
					gap: gap,
					scrollWidth: state.itemWidth,
					visibleItems: state.visibleItems,
					maxIndex: state.maxIndex,
					totalItems: $items.length,
					windowWidth: windowWidth
				}
			);
		}

		// Move carousel to specific index.
		function moveTo(index, animate = true, isAutoPlay = false) {
			if (state.isAnimating) {
				return;
			}

			state.currentIndex = Math.max( 0, Math.min( index, state.maxIndex ) );

			// Round translateX to prevent sub-pixel rendering.
			var translateX = Math.round( -(state.currentIndex * state.itemWidth) );

			console.log( 'arwrev Carousel: Moving to index', state.currentIndex, 'translateX:', translateX, 'isAutoPlay:', isAutoPlay );

			if (animate) {
				state.isAnimating = true;

				// Different transition timing for auto-play vs manual.
				var transitionDuration = isAutoPlay ? '1.2s' : '0.6s';
				var transitionEasing   = isAutoPlay ?
					'cubic-bezier(0.23, 1, 0.32, 1)' :
					'cubic-bezier(0.25, 0.46, 0.45, 0.94)';

				$track.css(
					{
						'transition': 'transform ' + transitionDuration + ' ' + transitionEasing,
						'transform': 'translateX(' + translateX + 'px)'
					}
				);

				// Add auto-play class for different styling.
				if (isAutoPlay) {
					$track.addClass( 'arwrev-auto-playing' );
				} else {
					$track.removeClass( 'arwrev-auto-playing' );
				}

				var animationDuration = isAutoPlay ? 1200 : 600;
				setTimeout(
					function() {
						state.isAnimating = false;
						$track.removeClass( 'arwrev-auto-playing' );
					},
					animationDuration
				);
			} else {
				$track.css(
					{
						'transition': 'none',
						'transform': 'translateX(' + translateX + 'px)'
					}
				);
			}

			updateControls();
		}

		// Update button states.
		function updateControls() {
			$prevBtn.prop( 'disabled', state.currentIndex <= 0 );
			$nextBtn.prop( 'disabled', state.currentIndex >= state.maxIndex );

			// Update button opacity for visual feedback.
			$prevBtn.css( 'opacity', state.currentIndex <= 0 ? 0.3 : 1 );
			$nextBtn.css( 'opacity', state.currentIndex >= state.maxIndex ? 0.3 : 1 );
		}

		// Auto-play functionality.
		function startAutoPlay() {
			if (state.isAutoPlaying || state.isPaused || $items.length <= state.visibleItems) {
				return;
			}

			console.log( 'arwrev Carousel: Starting auto-play for', containerId );
			state.isAutoPlaying = true;

			state.autoPlayInterval = setInterval(
				function() {
					if (state.isPaused || state.isAnimating) {
						return;
					}

					// Move to next slide, or loop back to beginning.
					if (state.currentIndex >= state.maxIndex) {
						// Loop back to start with smooth transition.
						moveTo( 0, true, true );
						console.log( 'arwrev Carousel: Auto-play looped to start' );
					} else {
						// Move to next slide.
						moveTo( state.currentIndex + 1, true, true );
						console.log( 'arwrev Carousel: Auto-play moved to index', state.currentIndex + 1 );
					}
				},
				state.autoPlayDelay
			);
		}

		function stopAutoPlay() {
			if (state.autoPlayInterval) {
				console.log( 'arwrev Carousel: Stopping auto-play for', containerId );
				clearInterval( state.autoPlayInterval );
				state.autoPlayInterval = null;
				state.isAutoPlaying    = false;
			}
		}

		function pauseAutoPlay() {
			console.log( 'arwrev Carousel: Pausing auto-play for', containerId );
			state.isPaused = true;
		}

		function resumeAutoPlay() {
			console.log( 'arwrev Carousel: Resuming auto-play for', containerId );
			state.isPaused = false;
			if ( ! state.isAutoPlaying) {
				startAutoPlay();
			}
		}

		// Previous button handler.
		function handlePrev(e) {
			e.preventDefault();
			e.stopPropagation();

			console.log( 'arwrev Carousel: Previous clicked, current index:', state.currentIndex );

			// Pause auto-play when user interacts.
			pauseAutoPlay();

			if (state.currentIndex > 0) {
				moveTo( state.currentIndex - 1 );
			}

			// Resume auto-play after a delay.
			setTimeout( resumeAutoPlay, 5000 ); // 5 seconds delay.
		}

		// Next button handler.
		function handleNext(e) {
			e.preventDefault();
			e.stopPropagation();

			console.log( 'arwrev Carousel: Next clicked, current index:', state.currentIndex );

			// Pause auto-play when user interacts.
			pauseAutoPlay();

			if (state.currentIndex < state.maxIndex) {
				moveTo( state.currentIndex + 1 );
			}

			// Resume auto-play after a delay.
			setTimeout( resumeAutoPlay, 5000 ); // 5 seconds delay.
		}

		// Attach event handlers.
		$prevBtn.off( 'click.arwrev-carousel' ).on( 'click.arwrev-carousel', handlePrev );
		$nextBtn.off( 'click.arwrev-carousel' ).on( 'click.arwrev-carousel', handleNext );

		// Pause auto-play on hover.
		$container.on(
			'mouseenter.arwrev-carousel',
			function() {
				console.log( 'arwrev Carousel: Mouse enter - pausing auto-play' );
				pauseAutoPlay();
			}
		);

		// Resume auto-play when mouse leaves (with delay).
		$container.on(
			'mouseleave.arwrev-carousel',
			function() {
				console.log( 'arwrev Carousel: Mouse leave - will resume auto-play' );
				setTimeout( resumeAutoPlay, 2000 ); // 2 second delay before resuming.
			}
		);

		// Pause on focus (accessibility).
		$container.on(
			'focusin.arwrev-carousel',
			function() {
				pauseAutoPlay();
			}
		);

		// Resume on blur (accessibility).
		$container.on(
			'focusout.arwrev-carousel',
			function() {
				setTimeout( resumeAutoPlay, 3000 );
			}
		);

		// Window resize handler.
		var resizeTimeout;
		$( window ).off( 'resize.arwrev-carousel-' + containerId ).on(
			'resize.arwrev-carousel-' + containerId,
			function() {
				clearTimeout( resizeTimeout );
				resizeTimeout = setTimeout(
					function() {
						updateDimensions();
						// Adjust current index if needed.
						if (state.currentIndex > state.maxIndex) {
							state.currentIndex = state.maxIndex;
						}
						moveTo( state.currentIndex, false );
					},
					250
				);
			}
		);

		// Initialize.
		updateDimensions();
		updateControls();

		// Store carousel state.
		carousels[containerId] = {
			state: state,
			moveTo: moveTo,
			updateDimensions: updateDimensions,
			startAutoPlay: startAutoPlay,
			stopAutoPlay: stopAutoPlay,
			pauseAutoPlay: pauseAutoPlay,
			resumeAutoPlay: resumeAutoPlay
		};

		// Mark as initialized.
		$container.addClass( 'arwrev-carousel-initialized' );

		// Start auto-play after initialization (with delay).
		setTimeout(
			function() {
				startAutoPlay();
				console.log( 'arwrev Carousel: Auto-play started for', containerId );
			},
			2000
		); // 2 second delay before starting auto-play.

		console.log( 'arwrev Carousel: Initialized successfully with auto-play', containerId );
	}

	// Initialize all carousels.
	function initAllCarousels() {
		$( '.arwrev-carousel-container' ).each(
			function() {
				initializeCarousel( $( this ) );
			}
		);
	}

	// DOM ready initialization.
	$( document ).ready(
		function() {
			console.log( 'arwrev Carousel: DOM ready' );

			// Small delay to ensure all elements are rendered.
			setTimeout(
				function() {
					initAllCarousels();
				},
				100
			);
		}
	);

	// Window load fallback.
	$( window ).on(
		'load',
		function() {
			console.log( 'arwrev Carousel: Window loaded' );
			initAllCarousels();
		}
	);

	// Re-initialize after AJAX updates.
	$( document ).on(
		'arwrev:content-updated',
		function() {
			console.log( 'arwrev Carousel: Content updated, reinitializing' );
			setTimeout( initAllCarousels, 100 );
		}
	);

	// Intersection Observer for lazy initialization.
	if ('IntersectionObserver' in window) {
		var observer = new IntersectionObserver(
			function(entries) {
				entries.forEach(
					function(entry) {
						if (entry.isIntersecting) {
							var $container = $( entry.target );
							if ( ! $container.hasClass( 'arwrev-carousel-initialized' )) {
								setTimeout(
									function() {
										initializeCarousel( $container );
									},
									50
								);
							}
						}
					}
				);
			},
			{
				threshold: 0.1,
				rootMargin: '50px'
			}
		);

		// Observe carousel containers.
		setTimeout(
			function() {
				$( '.arwrev-carousel-container' ).each(
					function() {
						observer.observe( this );
					}
				);
			},
			100
		);
	}

})( jQuery );

// Debug function to check carousel status.
function debugCarousel() {
	console.log( '=== arwrev CAROUSEL DEBUG ===' );
	console.log( 'Containers found:', $( '.arwrev-carousel-container' ).length );
	console.log( 'Tracks found:', $( '.arwrev-carousel-track' ).length );
	console.log( 'Items found:', $( '.arwrev-item-carousel' ).length );
	console.log( 'Prev buttons found:', $( '.arwrev-carousel-prev' ).length );
	console.log( 'Next buttons found:', $( '.arwrev-carousel-next' ).length );
	console.log( 'Pagination elements found:', $( '.arwrev-pagination' ).length );

	$( '.arwrev-carousel-container' ).each(
		function(index) {
			var $container = $( this );
			console.log(
				'Container ' + index + ':',
				{
					id: $container.attr( 'id' ),
					initialized: $container.hasClass( 'arwrev-carousel-initialized' ),
					items: $container.find( '.arwrev-item-carousel' ).length,
					prevBtn: $container.find( '.arwrev-carousel-prev' ).length,
					nextBtn: $container.find( '.arwrev-carousel-next' ).length
				}
			);
		}
	);

	console.log( '=== END DEBUG ===' );
}

// Run debug after page load.
$( window ).on(
	'load',
	function() {
		setTimeout( debugCarousel, 3000 );
	}
);

// Manual debug function (can be called from browser console).
window.arwrevDebugCarousel = debugCarousel;
