<?php
/*
Plugin Name: Aptranet CDN Integration
Description: Integrates Aptranet CDN to accelerate your WordPress site.
Version: 1.8
Author: Aptranet
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/* --------------------------
   ADMIN SETTINGS & UI
-------------------------- */

function aptranet_cdn_add_settings_page() {
    add_options_page(
        'Aptranet CDN Integration',
        'Aptranet CDN',
        'manage_options',
        'aptranet-cdn-integration',
        'aptranet_cdn_render_settings_page'
    );
}
add_action( 'admin_menu', 'aptranet_cdn_add_settings_page' );

/**
 * Render the Settings Page
 */
function aptranet_cdn_render_settings_page() {
    // Retrieve saved options
    $saved_id   = get_option( 'aptranet_cdn_distribution' );
    $saved_name = get_option( 'aptranet_cdn_distribution_name' );
    ?>
    <div class="wrap aptranet-cdn-settings-container">
        <!-- Aptranet Logo -->
        <img 
            src="<?php echo esc_url(plugin_dir_url( __FILE__ ) . 'images/logo.svg'); ?>" 
            alt="Aptranet Logo" 
            class="aptranet-logo" 
        />

        <h1>Aptranet CDN Integration Settings</h1>

        <form method="post" action="options.php" class="aptranet-cdn-form">
            <?php
                settings_fields( 'aptranet_cdn_options_group' );
                do_settings_sections( 'aptranet-cdn-integration' );
            ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><label for="aptranet_cdn_access_key">Access Key</label></th>
                    <td>
                        <input 
                            type="text" 
                            id="aptranet_cdn_access_key" 
                            name="aptranet_cdn_access_key"
                            value="<?php echo esc_attr( get_option( 'aptranet_cdn_access_key' ) ); ?>" 
                            class="regular-text" 
                        />
                        <p class="description">
                            Your API Key only requires the <strong>CDN.ListDistributions</strong> permission.
                        </p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="aptranet_cdn_secret">API Secret</label></th>
                    <td>
                        <input 
                            type="text" 
                            id="aptranet_cdn_secret" 
                            name="aptranet_cdn_secret"
                            value="<?php echo esc_attr( get_option( 'aptranet_cdn_secret' ) ); ?>" 
                            class="regular-text" 
                        />
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="aptranet_cdn_enable">Enable CDN</label></th>
                    <td>
                        <input 
                            type="checkbox" 
                            id="aptranet_cdn_enable" 
                            name="aptranet_cdn_enable" 
                            value="1" 
                            <?php checked( 1, get_option( 'aptranet_cdn_enable' ), true ); ?> 
                        />
                        <p class="description">
                            Check to enable rewriting of front-end assets to serve from the CDN.
                        </p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><label for="aptranet_cdn_distribution">Select Distribution</label></th>
                    <td>
                        <div class="aptranet-cdn-distribution-row">
                            <select 
                                id="aptranet_cdn_distribution" 
                                name="aptranet_cdn_distribution"
                            >
                                <option value="">-- Select Distribution --</option>
                                <?php 
                                if ( $saved_id && $saved_name ) {
                                    echo '<option value="' . esc_attr( $saved_id ) . '" selected>' . esc_html( $saved_name ) . '</option>';
                                }
                                ?>
                            </select>
                            <!-- Inline Fetch Distributions button -->
                            <button id="fetch-distributions" class="button button-secondary">Fetch Distributions</button>
                            
                            <!-- Manage Distribution button (dynamically updated via JS) -->
                            <a 
                                id="aptranet-manage-distribution" 
                                href="#" 
                                target="_blank" 
                                class="button button-secondary" 
                                style="display:none;"
                            >
                                Manage Distribution
                            </a>
                        </div>

                        <!-- Hidden field to store the distribution name -->
                        <input 
                            type="hidden" 
                            id="aptranet_cdn_distribution_name" 
                            name="aptranet_cdn_distribution_name"
                            value="<?php echo esc_attr( get_option( 'aptranet_cdn_distribution_name', '' ) ); ?>"
                        />
                        <p class="description">
                            Fetch available distributions, select one, then click "Save Changes."
                        </p>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row">Current CDN Base URL</th>
                    <td>
                        <input 
                            type="text" 
                            id="aptranet_cdn_base_url" 
                            name="aptranet_cdn_base_url" 
                            readonly
                            value="<?php echo esc_attr( get_option( 'aptranet_cdn_base_url', '' ) ); ?>" 
                            class="regular-text" 
                        />
                        <p class="description">
                            Automatically set when you select a distribution above.
                        </p>
                    </td>
                </tr>
            </table>
            <?php submit_button( 'Save Changes', 'primary', 'submit', false, array( 'id' => 'aptranet-cdn-save-button' ) ); ?>
        </form>

        <!-- Message box for success/error. Initially hidden. -->
        <div id="aptranet-cdn-message"></div>

        <!-- Hidden overlay for the loading spinner -->
        <div id="aptranet-cdn-save-spinner-overlay">
            <div class="aptranet-cdn-spinner"></div>
            <p>Saving, please wait...</p>
        </div>
    </div>
    <?php
}

function aptranet_cdn_register_settings() {
    register_setting( 'aptranet_cdn_options_group', 'aptranet_cdn_access_key', 'sanitize_text_field' );
    register_setting( 'aptranet_cdn_options_group', 'aptranet_cdn_secret', 'sanitize_text_field' );
    register_setting( 'aptranet_cdn_options_group', 'aptranet_cdn_enable', 'absint' );
    register_setting( 'aptranet_cdn_options_group', 'aptranet_cdn_distribution', 'sanitize_text_field' );
    register_setting( 'aptranet_cdn_options_group', 'aptranet_cdn_distribution_name', 'sanitize_text_field' );
    register_setting( 'aptranet_cdn_options_group', 'aptranet_cdn_base_url', 'esc_url_raw' );

    add_settings_section( 'aptranet_cdn_section', 'Aptranet CDN Settings', null, 'aptranet-cdn-integration' );
}
add_action( 'admin_init', 'aptranet_cdn_register_settings' );

/* --------------------------
   ENQUEUE ADMIN ASSETS
--------------------------- */
function aptranet_cdn_enqueue_admin_assets( $hook ) {
    if ( 'settings_page_aptranet-cdn-integration' !== $hook ) {
        return;
    }
    // Enqueue custom styles with a version number
    wp_enqueue_style( 'aptranet-cdn-styles', plugin_dir_url( __FILE__ ) . 'css/styles.css', array(), '1.8' );
    
    // Enqueue the main plugin script with a version number
    wp_enqueue_script( 'aptranet-cdn-script', plugin_dir_url( __FILE__ ) . 'js/aptranet-cdn.js', array('jquery'), '1.8', true );
    
    // Localize the script to pass in AJAX info
    wp_localize_script( 'aptranet-cdn-script', 'aptranetCdn', array(
        'ajax_url' => admin_url( 'admin-ajax.php' ),
        'nonce'    => wp_create_nonce( 'aptranet_cdn_nonce' )
    ) );
}
add_action( 'admin_enqueue_scripts', 'aptranet_cdn_enqueue_admin_assets' );

/* --------------------------
   AJAX HANDLER: Fetch Distributions
--------------------------- */
function aptranet_cdn_fetch_distributions() {
    check_ajax_referer( 'aptranet_cdn_nonce', 'nonce' );

    $access_key = get_option( 'aptranet_cdn_access_key' );
    $secret     = get_option( 'aptranet_cdn_secret' );

    if ( empty( $access_key ) || empty( $secret ) ) {
        wp_send_json_error( 'API credentials are missing.' );
    }

    // Endpoint as per API reference
    $endpoint = 'https://api.aptranet.com/cdn/distributions';

    $args = array(
        'headers' => array(
            'Authorization'   => $access_key,
            'Aptranet-Secret' => $secret,
            'Accept'          => 'application/json',
        ),
        'timeout' => 10,
    );

    $response = wp_remote_get( $endpoint, $args );

    if ( is_wp_error( $response ) ) {
        wp_send_json_error( 'API request failed: ' . $response->get_error_message() );
    }

    $code = wp_remote_retrieve_response_code( $response );
    if ( $code !== 200 ) {
        wp_send_json_error( 'API request returned HTTP code ' . $code );
    }

    $body = wp_remote_retrieve_body( $response );
    $data = json_decode( $body, true );

    if ( empty( $data ) || ! is_array( $data ) ) {
        wp_send_json_error( 'No distributions found in API response.' );
    }

    wp_send_json_success( $data );
}
add_action( 'wp_ajax_aptranet_cdn_fetch_distributions', 'aptranet_cdn_fetch_distributions' );

/* --------------------------
   FRONT-END FUNCTIONALITY
   URL REWRITING
--------------------------- */

/**
 * Rewrites scripts, styles, and attachments, excluding wp-admin to avoid 415 errors.
 */
function aptranet_cdn_rewrite_urls( $url ) {
    // Don't rewrite anything in the admin area
    if ( is_admin() ) {
        return $url;
    }

    if ( get_option( 'aptranet_cdn_enable' ) ) {
        $base_url = get_option( 'aptranet_cdn_base_url' );
        if ( $base_url ) {
            $base_url = rtrim( $base_url, '/' ) . '/';
            $home_url = rtrim( home_url(), '/' ) . '/';
            $url = str_replace( $home_url, $base_url, $url );
        }
    }
    return $url;
}
add_filter( 'wp_get_attachment_url', 'aptranet_cdn_rewrite_urls' );
add_filter( 'script_loader_src', 'aptranet_cdn_rewrite_urls' );
add_filter( 'style_loader_src', 'aptranet_cdn_rewrite_urls' );

/* --------------------------
   IMAGES IN POST CONTENT
--------------------------- */
function aptranet_cdn_rewrite_content_images( $content ) {
    // Don't rewrite anything in the admin area
    if ( is_admin() ) {
        return $content;
    }

    if ( get_option( 'aptranet_cdn_enable' ) ) {
        $base_url = get_option( 'aptranet_cdn_base_url' );
        if ( $base_url ) {
            $base_url = rtrim( $base_url, '/' ) . '/';
            $home_url = rtrim( home_url(), '/' ) . '/';
            $content = str_replace( $home_url, $base_url, $content );
        }
    }
    return $content;
}
add_filter( 'the_content', 'aptranet_cdn_rewrite_content_images' );

/* --------------------------
   RESPONSIVE IMAGE SRCSET
--------------------------- */
function aptranet_cdn_rewrite_srcset( $sources ) {
    // Don't rewrite anything in the admin area
    if ( is_admin() ) {
        return $sources;
    }

    if ( get_option( 'aptranet_cdn_enable' ) ) {
        $base_url = get_option( 'aptranet_cdn_base_url' );
        if ( $base_url ) {
            $base_url = rtrim( $base_url, '/' ) . '/';
            $home_url = rtrim( home_url(), '/' ) . '/';

            foreach ( $sources as $width => $source ) {
                if ( isset( $source['url'] ) ) {
                    $sources[$width]['url'] = str_replace( $home_url, $base_url, $source['url'] );
                }
            }
        }
    }
    return $sources;
}
add_filter( 'wp_calculate_image_srcset', 'aptranet_cdn_rewrite_srcset' );
