<?php
/**
 * Maintenance mode functionality for Anti-Cache Emergency Kit
 * 
 * @package AntiCache
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Hook into template_redirect to show maintenance page
add_action( 'template_redirect', 'ayudawp_anticache_maintenance_mode_check', 1 );

/**
 * Check if maintenance mode should be displayed
 */
function ayudawp_anticache_maintenance_mode_check() {
    // Skip if maintenance mode is not enabled
    if ( ! get_option( 'ayudawp_anticache_maintenance_mode', false ) ) {
        return;
    }

    // Skip for logged-in users with admin capabilities
    if ( is_user_logged_in() && current_user_can( 'manage_options' ) ) {
        return;
    }

    // Skip for admin pages, login, and wp-cron
    if ( is_admin() || is_user_logged_in() || 
         isset( $GLOBALS['pagenow'] ) && $GLOBALS['pagenow'] === 'wp-login.php' ||
         ( defined( 'DOING_CRON' ) && DOING_CRON ) ||
         ( defined( 'WP_CLI' ) && WP_CLI ) ) {
        return;
    }

    // Skip for AJAX requests
    if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
        return;
    }

    // Skip for REST API requests
    if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
        return;
    }

    // Skip for XML-RPC requests
    if ( defined( 'XMLRPC_REQUEST' ) && XMLRPC_REQUEST ) {
        return;
    }

    // Show maintenance page
    ayudawp_anticache_show_maintenance_page();
}

/**
 * Display the maintenance page
 */
function ayudawp_anticache_show_maintenance_page() {
    // Set proper HTTP status and headers BEFORE any output
    status_header( 503 );
    nocache_headers();
    header( 'Retry-After: 1800' ); // 30 minutes
    
    // Get site information
    $site_name = get_bloginfo( 'name' );
    $site_description = get_bloginfo( 'description' );
    $site_url = home_url();
    
    // Allow customization via filter
    $maintenance_data = apply_filters( 'ayudawp_anticache_maintenance_data', array(
        // translators: %s is the site name
        'title' => sprintf( __( '%s - Under Maintenance', 'anticache' ), $site_name ),
        'heading' => __( 'Site Under Maintenance', 'anticache' ),
        'message' => __( 'We are currently performing scheduled maintenance. Please check back soon.', 'anticache' ),
        'retry_after' => __( 'Estimated time: 30 minutes', 'anticache' ),
        'contact_info' => '',
        'background_color' => '#f8f9fa',
        'text_color' => '#333333',
        'accent_color' => '#0073aa'
    ) );

    // Enqueue styles and scripts
    add_action( 'wp_enqueue_scripts', function() use ( $maintenance_data ) {
        ayudawp_anticache_maintenance_styles( $maintenance_data );
        ayudawp_anticache_maintenance_script();
    });

    // Output the maintenance page
    ?>
    <!DOCTYPE html>
    <html <?php language_attributes(); ?>>
    <head>
        <meta charset="<?php bloginfo( 'charset' ); ?>">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <meta name="robots" content="noindex, nofollow">
        <title><?php echo esc_html( $maintenance_data['title'] ); ?></title>
        
        <?php wp_head(); ?>
    </head>
    <body>
        <div class="maintenance-container">
            <div class="maintenance-icon"></div>
            
            <h1><?php echo esc_html( $maintenance_data['heading'] ); ?></h1>
            
            <p><?php echo esc_html( $maintenance_data['message'] ); ?></p>
            
            <div class="progress-bar">
                <div class="progress-fill"></div>
            </div>
            
            <?php if ( ! empty( $maintenance_data['retry_after'] ) ) : ?>
                <div class="retry-time">
                    <?php echo esc_html( $maintenance_data['retry_after'] ); ?>
                </div>
            <?php endif; ?>
            
            <?php if ( ! empty( $maintenance_data['contact_info'] ) ) : ?>
                <div class="contact-info">
                    <?php echo wp_kses_post( $maintenance_data['contact_info'] ); ?>
                </div>
            <?php endif; ?>
            
            <a href="<?php echo esc_url( $site_url ); ?>" class="site-link">
                <?php esc_html_e( 'Try Again', 'anticache' ); ?>
            </a>
            
            <!-- Admin notice (only visible in source) -->
            <?php if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) : ?>
                <div class="admin-notice">
                    <strong><?php esc_html_e( 'Admin Notice:', 'anticache' ); ?></strong>
                    <?php esc_html_e( 'This maintenance mode is activated by Anti-Cache Emergency Kit. Log in as administrator to bypass this screen.', 'anticache' ); ?>
                </div>
            <?php endif; ?>
        </div>
        
        <?php wp_footer(); ?>
    </body>
    </html>
    <?php
    
    // Stop execution to prevent WordPress from continuing
    exit;
}

/**
 * Output maintenance page styles
 */
function ayudawp_anticache_maintenance_styles( $maintenance_data ) {
    // Register a dummy style to attach inline CSS
    wp_register_style( 'ayudawp-maintenance-style', false, array(), '3.0.2' );
    wp_enqueue_style( 'ayudawp-maintenance-style' );
    
    $css = '
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
            background-color: ' . esc_attr( $maintenance_data['background_color'] ) . ';
            color: ' . esc_attr( $maintenance_data['text_color'] ) . ';
            line-height: 1.6;
            display: flex;
            min-height: 100vh;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .maintenance-container {
            max-width: 600px;
            width: 100%;
            text-align: center;
            background: #ffffff;
            padding: 60px 40px;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .maintenance-icon {
            width: 80px;
            height: 80px;
            margin: 0 auto 30px;
            background-color: ' . esc_attr( $maintenance_data['accent_color'] ) . ';
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
        }
        
        .maintenance-icon::before {
            content: "🔧";
            font-size: 40px;
            animation: rotate 2s linear infinite;
        }
        
        @keyframes rotate {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        
        .maintenance-container h1 {
            font-size: 32px;
            margin-bottom: 20px;
            color: ' . esc_attr( $maintenance_data['accent_color'] ) . ';
            font-weight: 600;
        }
        
        .maintenance-container p {
            font-size: 18px;
            margin-bottom: 20px;
            color: #666;
        }
        
        .retry-time {
            background-color: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 6px;
            padding: 15px;
            margin: 30px 0;
            font-weight: 500;
            color: ' . esc_attr( $maintenance_data['accent_color'] ) . ';
        }
        
        .contact-info {
            margin-top: 40px;
            padding-top: 30px;
            border-top: 1px solid #e9ecef;
            font-size: 16px;
            color: #888;
        }
        
        .site-link {
            display: inline-block;
            margin-top: 20px;
            color: ' . esc_attr( $maintenance_data['accent_color'] ) . ';
            text-decoration: none;
            font-weight: 500;
            border: 2px solid ' . esc_attr( $maintenance_data['accent_color'] ) . ';
            padding: 12px 24px;
            border-radius: 6px;
            transition: all 0.3s ease;
        }
        
        .site-link:hover {
            background-color: ' . esc_attr( $maintenance_data['accent_color'] ) . ';
            color: white;
        }
        
        .progress-bar {
            width: 100%;
            height: 4px;
            background-color: #e9ecef;
            border-radius: 2px;
            margin: 30px 0;
            overflow: hidden;
        }
        
        .progress-fill {
            height: 100%;
            background-color: ' . esc_attr( $maintenance_data['accent_color'] ) . ';
            border-radius: 2px;
            animation: progress 3s ease-in-out infinite;
        }
        
        @keyframes progress {
            0% { width: 0%; }
            50% { width: 70%; }
            100% { width: 0%; }
        }
        
        .admin-notice {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            color: #856404;
            padding: 15px;
            border-radius: 6px;
            margin-top: 30px;
            font-size: 14px;
        }
        
        @media (max-width: 600px) {
            .maintenance-container {
                padding: 40px 20px;
            }
            
            .maintenance-container h1 {
                font-size: 24px;
            }
            
            .maintenance-container p {
                font-size: 16px;
            }
        }
    ';
    
    wp_add_inline_style( 'ayudawp-maintenance-style', $css );
}

/**
 * Output maintenance page script
 */
function ayudawp_anticache_maintenance_script() {
    // Register a dummy script to attach inline JavaScript
    wp_register_script( 'ayudawp-maintenance-script', false, array(), '3.0.2', true );
    wp_enqueue_script( 'ayudawp-maintenance-script' );
    
    $js = '
        // Auto-refresh every 5 minutes
        setTimeout(function() {
            location.reload();
        }, 300000); // 5 minutes
    ';
    
    wp_add_inline_script( 'ayudawp-maintenance-script', $js );
}

/**
 * Add maintenance mode toggle to admin bar for quick access
 */
add_action( 'admin_bar_menu', 'ayudawp_anticache_admin_bar_maintenance', 100 );

/**
 * Add maintenance mode toggle to admin bar
 */
function ayudawp_anticache_admin_bar_maintenance( $wp_admin_bar ) {
    // Only show for administrators
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $maintenance_mode = get_option( 'ayudawp_anticache_maintenance_mode', false );
    
    $wp_admin_bar->add_node( array(
        'id'    => 'anticache-maintenance',
        'title' => $maintenance_mode 
            ? '🔧 ' . __( 'Maintenance ON', 'anticache' )
            : '⚙️ ' . __( 'Maintenance OFF', 'anticache' ),
        'href'  => wp_nonce_url( 
            add_query_arg( 'anticache_toggle_maintenance', '1' ), 
            'anticache_toggle_maintenance' 
        ),
        'meta'  => array(
            'class' => $maintenance_mode ? 'anticache-maintenance-on' : 'anticache-maintenance-off'
        )
    ) );
}

/**
 * Handle maintenance mode toggle from admin bar
 */
add_action( 'init', 'ayudawp_anticache_handle_admin_bar_toggle' );

/**
 * Handle admin bar maintenance toggle
 */
function ayudawp_anticache_handle_admin_bar_toggle() {
    if ( ! isset( $_GET['anticache_toggle_maintenance'] ) || 
         ! current_user_can( 'manage_options' ) ) {
        return;
    }

    // Verify nonce
    if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'anticache_toggle_maintenance' ) ) {
        wp_die( esc_html__( 'Security check failed', 'anticache' ) );
    }

    // Toggle maintenance mode
    $current_mode = get_option( 'ayudawp_anticache_maintenance_mode', false );
    update_option( 'ayudawp_anticache_maintenance_mode', ! $current_mode );

    // Redirect back to same page
    $redirect_url = remove_query_arg( array( 'anticache_toggle_maintenance', '_wpnonce' ) );
    wp_safe_redirect( $redirect_url );
    exit;
}

/**
 * Add admin bar styles for maintenance mode indicator
 */
add_action( 'wp_head', 'ayudawp_anticache_admin_bar_styles' );
add_action( 'admin_head', 'ayudawp_anticache_admin_bar_styles' );

/**
 * Add styles for admin bar maintenance indicator
 */
function ayudawp_anticache_admin_bar_styles() {
    if ( ! is_admin_bar_showing() || ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    $css = '
        #wp-admin-bar-anticache-maintenance .anticache-maintenance-on > a {
            background-color: #dc3232 !important;
            color: white !important;
        }
        
        #wp-admin-bar-anticache-maintenance .anticache-maintenance-off > a {
            background-color: #00a32a !important;
            color: white !important;
        }
        
        #wp-admin-bar-anticache-maintenance a:hover {
            opacity: 0.8;
        }
    ';
    
    wp_add_inline_style( 'admin-bar', $css );
}