<?php
/**
 * Admin panel functionality for Anti-Cache Emergency Kit
 * 
 * @package AntiCache
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Add admin menu
add_action( 'admin_menu', 'ayudawp_anticache_admin_menu' );

/**
 * Add admin menu item under Tools
 */
function ayudawp_anticache_admin_menu() {
    add_management_page(
        __( 'Anti-Cache Emergency Kit', 'anticache' ),
        __( 'Anti-Cache Kit', 'anticache' ),
        'manage_options',
        'anticache-settings',
        'ayudawp_anticache_admin_page'
    );
}

/**
 * Admin page content
 */
function ayudawp_anticache_admin_page() {
    // Handle form submissions
    if ( isset( $_POST['ayudawp_anticache_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['ayudawp_anticache_nonce'] ) ), 'ayudawp_anticache_admin' ) ) {
        if ( isset( $_POST['reactivate_plugins'] ) ) {
            ayudawp_anticache_reactivate_plugins();
        }
        
        if ( isset( $_POST['toggle_maintenance'] ) ) {
            ayudawp_anticache_toggle_maintenance_mode();
        }

        if ( isset( $_POST['clear_caches'] ) ) {
            ayudawp_anticache_manual_cache_clear();
        }
    }

    $deactivated_plugins = get_option( 'ayudawp_anticache_deactivated_plugins', array() );
    $maintenance_mode = get_option( 'ayudawp_anticache_maintenance_mode', false );
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
        
        <div class="notice notice-warning">
            <p><strong><?php esc_html_e( 'Warning:', 'anticache' ); ?></strong> 
               <?php esc_html_e( 'This plugin is designed for development and troubleshooting only. Do not leave it active on production sites!', 'anticache' ); ?>
            </p>
        </div>

        <div class="ayudawp-anticache-admin">
            
            <!-- Plugin Status Card -->
            <div class="card">
                <h2><?php esc_html_e( 'Current Status', 'anticache' ); ?></h2>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Debug Mode:', 'anticache' ); ?></th>
                        <td>
                            <span class="status-indicator active">
                                <?php esc_html_e( 'ENABLED', 'anticache' ); ?>
                            </span>
                            <p class="description"><?php esc_html_e( 'WP_DEBUG, WP_DEBUG_LOG, and WP_DEBUG_DISPLAY are active', 'anticache' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Cache Prevention:', 'anticache' ); ?></th>
                        <td>
                            <span class="status-indicator active">
                                <?php esc_html_e( 'ACTIVE FOR ADMINS', 'anticache' ); ?>
                            </span>
                            <p class="description"><?php esc_html_e( 'All cache disabled for administrator users only', 'anticache' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e( 'Maintenance Mode:', 'anticache' ); ?></th>
                        <td>
                            <span class="status-indicator <?php echo $maintenance_mode ? 'active' : 'inactive'; ?>">
                                <?php echo $maintenance_mode ? esc_html__( 'ENABLED', 'anticache' ) : esc_html__( 'DISABLED', 'anticache' ); ?>
                            </span>
                            <p class="description">
                                <?php esc_html_e( 'Simple maintenance screen for non-logged users during development', 'anticache' ); ?>
                            </p>
                        </td>
                    </tr>
                </table>
            </div>

            <!-- Deactivated Plugins Card -->
            <?php if ( ! empty( $deactivated_plugins ) ) : ?>
            <div class="card">
                <h2><?php esc_html_e( 'Deactivated Cache Plugins', 'anticache' ); ?></h2>
                <p><?php esc_html_e( 'These cache plugins were automatically deactivated when Anti-Cache Kit was activated:', 'anticache' ); ?></p>
                
                <ul class="deactivated-plugins-list">
                    <?php foreach ( $deactivated_plugins as $plugin ) : ?>
                        <li><span class="dashicons dashicons-dismiss"></span> <?php echo esc_html( $plugin ); ?></li>
                    <?php endforeach; ?>
                </ul>

                <form method="post" style="margin-top: 15px;">
                    <?php wp_nonce_field( 'ayudawp_anticache_admin', 'ayudawp_anticache_nonce' ); ?>
                    <input type="submit" name="reactivate_plugins" class="button button-secondary" 
                           value="<?php esc_attr_e( 'Reactivate All Cache Plugins', 'anticache' ); ?>" 
                           onclick="return confirm('<?php echo esc_js( esc_attr__( 'This will reactivate all cache plugins. Are you sure?', 'anticache' ) ); ?>');">
                    <p class="description">
                        <?php esc_html_e( 'Warning: This will reactivate all cache plugins, potentially hiding your changes again.', 'anticache' ); ?>
                    </p>
                </form>
            </div>
            <?php endif; ?>

            <!-- Deactivation Card -->
            <div class="card deactivation-warning">
                <h2><?php esc_html_e( '⚠️ Important: Deactivation', 'anticache' ); ?></h2>
                <p><strong><?php esc_html_e( 'When you deactivate this plugin:', 'anticache' ); ?></strong></p>
                <ul>
                    <li><?php esc_html_e( 'Debug mode will be disabled in wp-config.php', 'anticache' ); ?></li>
                    <li><?php esc_html_e( 'All anti-cache measures will stop', 'anticache' ); ?></li>
                    <li><?php esc_html_e( 'Maintenance mode will be disabled', 'anticache' ); ?></li>
                    <li><?php esc_html_e( 'Cache plugins will remain deactivated (reactivate them manually if needed)', 'anticache' ); ?></li>
                </ul>
                <p class="highlight"><?php esc_html_e( 'Remember to reactivate your cache plugins after finishing development!', 'anticache' ); ?></p>
            </div>

            <!-- Actions Card -->
            <div class="card">
                <h2><?php esc_html_e( 'Quick Actions', 'anticache' ); ?></h2>
                
                <form method="post" style="margin-bottom: 20px;">
                    <?php wp_nonce_field( 'ayudawp_anticache_admin', 'ayudawp_anticache_nonce' ); ?>
                    
                    <p>
                        <input type="submit" name="toggle_maintenance" class="button button-primary" 
                               value="<?php echo $maintenance_mode ? esc_attr__( 'Disable Maintenance Mode', 'anticache' ) : esc_attr__( 'Enable Maintenance Mode', 'anticache' ); ?>">
                        <span class="description">
                            <?php esc_html_e( 'Show a simple maintenance screen to non-logged users while you develop', 'anticache' ); ?>
                        </span>
                    </p>

                    <p>
                        <input type="submit" name="clear_caches" class="button button-secondary" 
                               value="<?php esc_attr_e( 'Manual Cache Clear', 'anticache' ); ?>">
                        <span class="description">
                            <?php esc_html_e( 'Manually clear all possible caches again', 'anticache' ); ?>
                        </span>
                    </p>
                </form>
            </div>

            <!-- Information Card -->
            <div class="card full-width">
                <h2><?php esc_html_e( 'How It Works', 'anticache' ); ?></h2>
                <div class="ayudawp-info-grid">
                    <div class="info-item">
                        <h4><?php esc_html_e( 'For Administrators Only', 'anticache' ); ?></h4>
                        <p><?php esc_html_e( 'All anti-cache measures only affect logged-in administrators. Regular visitors see the normal cached site.', 'anticache' ); ?></p>
                    </div>
                    
                    <div class="info-item">
                        <h4><?php esc_html_e( 'Cache Plugins', 'anticache' ); ?></h4>
                        <p><?php esc_html_e( 'Automatically detects, flushes, and deactivates major cache plugins including WP Rocket, W3 Total Cache, LiteSpeed, and more.', 'anticache' ); ?></p>
                    </div>
                    
                    <div class="info-item">
                        <h4><?php esc_html_e( 'Debug Mode', 'anticache' ); ?></h4>
                        <p><?php esc_html_e( 'Enables WP_DEBUG, WP_DEBUG_LOG, and WP_DEBUG_DISPLAY in wp-config.php to show PHP errors and warnings.', 'anticache' ); ?></p>
                    </div>
                    
                    <div class="info-item">
                        <h4><?php esc_html_e( 'Browser Cache', 'anticache' ); ?></h4>
                        <p><?php esc_html_e( 'Sends no-cache headers and adds random versions to CSS/JS files to prevent browser caching for admins.', 'anticache' ); ?></p>
                    </div>
                    
                    <div class="info-item">
                        <h4><?php esc_html_e( 'Object Cache', 'anticache' ); ?></h4>
                        <p><?php esc_html_e( 'Clears Redis, Memcached, and other object caches. Also clears OPcache if available.', 'anticache' ); ?></p>
                    </div>
                    
                    <div class="info-item">
                        <h4><?php esc_html_e( 'Maintenance Mode', 'anticache' ); ?></h4>
                        <p><?php esc_html_e( 'Optional simple maintenance screen for non-logged users to hide development errors and changes.', 'anticache' ); ?></p>
                    </div>
                </div>
            </div>

        </div>
    </div>
    <?php
}

/**
 * Reactivate deactivated cache plugins
 */
function ayudawp_anticache_reactivate_plugins() {
    $deactivated_plugins = get_option( 'ayudawp_anticache_deactivated_plugins', array() );
    $cache_plugins = ayudawp_anticache_get_cache_plugins();
    $reactivated = array();
    $errors = array();

    if ( ! function_exists( 'activate_plugin' ) ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }

    foreach ( $cache_plugins as $plugin_file => $plugin_data ) {
        if ( in_array( $plugin_data['name'], $deactivated_plugins ) ) {
            // Reactivate cache plugins - This is safe as it only reactivates plugins 
            // that were previously active and were deactivated by this plugin
            // User explicitly requested reactivation via admin interface
            $result = activate_plugin( $plugin_file );
            if ( is_wp_error( $result ) ) {
                $errors[] = sprintf( 
                    // translators: %1$s is the plugin name, %2$s is the error message
                    esc_html__( 'Could not reactivate %1$s: %2$s', 'anticache' ), 
                    $plugin_data['name'], 
                    $result->get_error_message() 
                );
            } else {
                $reactivated[] = $plugin_data['name'];
            }
        }
    }

    // Show results
    if ( $reactivated ) {
        add_action( 'admin_notices', function() use ( $reactivated ) {
            echo '<div class="notice notice-success is-dismissible"><p>';
            echo sprintf(
                // translators: %s is the list of reactivated plugins 
                esc_html__( 'Successfully reactivated: %s', 'anticache' ), 
                esc_html( implode( ', ', $reactivated ) )
            );
            echo '</p></div>';
        });
        
        // Clear the deactivated plugins list
        delete_option( 'ayudawp_anticache_deactivated_plugins' );
    }

    if ( $errors ) {
        add_action( 'admin_notices', function() use ( $errors ) {
            echo '<div class="notice notice-error is-dismissible"><p>';
            echo wp_kses_post( implode( '<br>', $errors ) );
            echo '</p></div>';
        });
    }
}

/**
 * Toggle maintenance mode
 */
function ayudawp_anticache_toggle_maintenance_mode() {
    $current_mode = get_option( 'ayudawp_anticache_maintenance_mode', false );
    $new_mode = ! $current_mode;
    
    update_option( 'ayudawp_anticache_maintenance_mode', $new_mode );
    
    $message = $new_mode 
        ? esc_html__( 'Maintenance mode enabled. Non-logged users will see a maintenance screen.', 'anticache' )
        : esc_html__( 'Maintenance mode disabled. Site is now visible to all users.', 'anticache' );
    
    add_action( 'admin_notices', function() use ( $message ) {
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html( $message ) . '</p></div>';
    });
}

/**
 * Manual cache clearing
 */
function ayudawp_anticache_manual_cache_clear() {
    $cleared = array();
    
    // Clear object cache
    if ( function_exists( 'wp_cache_flush' ) ) {
        wp_cache_flush();
        $cleared[] = esc_html__( 'Object Cache', 'anticache' );
    }
    
    // Clear OPcache
    if ( function_exists( 'opcache_reset' ) ) {
        opcache_reset();
        $cleared[] = esc_html__( 'OPcache', 'anticache' );
    }
    
    // Clear hosting caches
    ayudawp_anticache_clear_hosting_cache();
    $cleared[] = esc_html__( 'Hosting Cache', 'anticache' );
    
    $message = sprintf(
        // translators: %s is the list of cleared caches 
        esc_html__( 'Manual cache clear completed: %s', 'anticache' ), 
        implode( ', ', $cleared ) 
    );
    
    add_action( 'admin_notices', function() use ( $message ) {
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html( $message ) . '</p></div>';
    });
}

/**
 * Enqueue admin styles
 */
function ayudawp_anticache_enqueue_admin_styles( $hook ) {
    // Only load on our admin page
    if ( 'tools_page_anticache-settings' !== $hook ) {
        return;
    }
    
    $css = '
        .ayudawp-anticache-admin {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(450px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .ayudawp-anticache-admin .card {
            background: #fff;
            border: 1px solid #c3c4c7;
            border-left: 4px solid #0073aa;
            box-shadow: 0 1px 1px rgba(0, 0, 0, 0.04);
            padding: 20px;
            margin: 0;
            border-radius: 4px;
            height: fit-content;
        }

        .ayudawp-anticache-admin .card h2 {
            margin-top: 0;
            color: #0073aa;
        }

        .ayudawp-anticache-admin .card.full-width {
            grid-column: 1 / -1;
        }

        .status-indicator {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 3px;
            font-weight: bold;
            font-size: 12px;
        }

        .status-indicator.active {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .status-indicator.inactive {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .deactivated-plugins-list {
            background: #f9f9f9;
            border: 1px solid #ddd;
            padding: 15px;
            margin: 10px 0;
            border-radius: 4px;
        }

        .deactivated-plugins-list li {
            margin: 5px 0;
            color: #d35400;
        }

        .deactivated-plugins-list .dashicons {
            margin-right: 5px;
        }

        .ayudawp-info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 15px;
            margin-top: 15px;
        }

        .info-item {
            background: #f9f9f9;
            padding: 15px;
            border-radius: 4px;
            border-left: 3px solid #0073aa;
        }

        .info-item h4 {
            margin-top: 0;
            color: #0073aa;
        }

        .deactivation-warning {
            border-left-color: #dc3232 !important;
        }

        .deactivation-warning h2 {
            color: #dc3232 !important;
        }

        .deactivation-warning .highlight {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            padding: 10px;
            border-radius: 4px;
            font-weight: bold;
            color: #856404;
        }

        .deactivation-warning ul {
            background: #f8f9fa;
            padding: 15px 30px;
            border-radius: 4px;
            border: 1px solid #dee2e6;
        }

        @media (max-width: 1200px) {
            .ayudawp-anticache-admin {
                grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            }
        }

        @media (max-width: 900px) {
            .ayudawp-anticache-admin {
                grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
                gap: 15px;
            }
        }

        @media (max-width: 768px) {
            .ayudawp-anticache-admin {
                grid-template-columns: 1fr;
                gap: 15px;
            }
            
            .ayudawp-anticache-admin .card {
                padding: 15px;
            }

            .ayudawp-info-grid {
                grid-template-columns: 1fr;
                gap: 10px;
            }
        }

        @media (max-width: 600px) {
            .info-item {
                padding: 12px;
            }
        }
    ';
    
    wp_add_inline_style( 'wp-admin', $css );
}

// Hook to enqueue admin styles
add_action( 'admin_enqueue_scripts', 'ayudawp_anticache_enqueue_admin_styles' );