<?php
/*
Plugin Name: Anti-Cache Emergency Kit
Plugin URI: https://servicios.ayudawp.com
Description: Instantly flushes and deactivates major WordPress cache plugins, enables debug mode, disables all browser/proxy cache for administrators, and forces random versioning in assets. Includes optional maintenance mode for non-logged users during development. Designed for troubleshooting after updates or design changes.
Version: 3.0.5
Author: Fernando Tellado
Author URI: https://ayudawp.com
License: GPL2
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Requires at least: 5.0
Requires PHP: 7.4
Tested up to: 6.9
Text Domain: anticache
Domain Path: /languages
*/

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Plugin constants
define( 'AYUDAWP_ANTICACHE_VERSION', '3.0.5' );
define( 'AYUDAWP_ANTICACHE_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'AYUDAWP_ANTICACHE_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

// Note: load_plugin_textdomain() is not needed for WordPress.org hosted plugins
// WordPress automatically loads translations when needed since version 4.6

// Activation and deactivation hooks
register_activation_hook( __FILE__, 'ayudawp_anticache_activate' );
register_deactivation_hook( __FILE__, 'ayudawp_anticache_deactivate' );

/**
 * Plugin activation - Main function to flush caches and setup debug mode
 */
function ayudawp_anticache_activate() {
    // Ensure required WordPress functions are loaded
    if ( ! function_exists( 'get_plugins' ) ) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }

    $results = array(
        'flushed' => array(),
        'deactivated' => array(),
        'errors' => array(),
        'actions' => array(),
        'cache_found' => false,
        'debug_enabled' => false,
        'object_cache_cleared' => false,
        'backup_created' => false
    );

    // Create backup directory if it doesn't exist
    $backup_dir = AYUDAWP_ANTICACHE_PLUGIN_DIR . 'backup';
    if ( ! is_dir( $backup_dir ) ) {
        wp_mkdir_p( $backup_dir );
    }

    // Comprehensive list of cache plugins to handle
    $cache_plugins = ayudawp_anticache_get_cache_plugins();

    // Process each cache plugin
    $all_plugins = get_plugins();
    foreach ( $cache_plugins as $plugin_file => $plugin_data ) {
        if ( is_plugin_active( $plugin_file ) ) {
            $results['cache_found'] = true;
            
            // Try to flush cache
            $flush_result = ayudawp_anticache_flush_plugin_cache( $plugin_file, $plugin_data['name'] );
            if ( $flush_result ) {
                $results['flushed'][] = $plugin_data['name'];
            }

            // Deactivate plugin
            if ( array_key_exists( $plugin_file, $all_plugins ) ) {
                deactivate_plugins( $plugin_file, false, false );
                if ( ! is_plugin_active( $plugin_file ) ) {
                    $results['deactivated'][] = $plugin_data['name'];
                } else {
                    $results['errors'][] = sprintf(
                        // translators: %s is the plugin name 
                        esc_html__( 'Could not deactivate %s, please deactivate it manually.', 'anticache' ), 
                        $plugin_data['name'] 
                    );
                }
            }
        }
    }

    // Clear object cache if available
    if ( ayudawp_anticache_clear_object_cache() ) {
        $results['object_cache_cleared'] = true;
    }

    // Enable debug mode with backup
    $debug_result = ayudawp_anticache_enable_debug_mode();
    if ( $debug_result['success'] ) {
        $results['debug_enabled'] = true;
        if ( $debug_result['backup_created'] ) {
            $results['backup_created'] = true;
        }
    } else {
        $results['errors'][] = esc_html__( 'Could not enable debug mode in wp-config.php', 'anticache' );
        if ( ! empty( $debug_result['error'] ) ) {
            $results['errors'][] = $debug_result['error'];
        }
    }

    // Clear hosting-specific caches
    ayudawp_anticache_clear_hosting_cache();

    // Generate and save admin notice
    ayudawp_anticache_generate_admin_notice( $results );

    // Save deactivated plugins list for potential reactivation
    update_option( 'ayudawp_anticache_deactivated_plugins', $results['deactivated'] );
}

/**
 * Get comprehensive list of cache plugins to handle
 */
function ayudawp_anticache_get_cache_plugins() {
    return array(
        'sg-cachepress/sg-cachepress.php' => array(
            'name' => 'SiteGround Optimizer',
            'flush_function' => 'sg_cachepress'
        ),
        'wp-rocket/wp-rocket.php' => array(
            'name' => 'WP Rocket',
            'flush_function' => 'wp_rocket'
        ),
        'w3-total-cache/w3-total-cache.php' => array(
            'name' => 'W3 Total Cache',
            'flush_function' => 'w3_total_cache'
        ),
        'wp-super-cache/wp-cache.php' => array(
            'name' => 'WP Super Cache',
            'flush_function' => 'wp_super_cache'
        ),
        'wp-fastest-cache/wpFastestCache.php' => array(
            'name' => 'WP Fastest Cache',
            'flush_function' => 'wp_fastest_cache'
        ),
        'litespeed-cache/litespeed-cache.php' => array(
            'name' => 'LiteSpeed Cache',
            'flush_function' => 'litespeed_cache'
        ),
        'surge/surge.php' => array(
            'name' => 'Surge',
            'flush_function' => 'surge'
        ),
        'cache-enabler/cache-enabler.php' => array(
            'name' => 'Cache Enabler',
            'flush_function' => 'cache_enabler'
        ),
        'wp-optimize/wp-optimize.php' => array(
            'name' => 'WP Optimize',
            'flush_function' => 'wp_optimize'
        ),
        'hummingbird-performance/wp-hummingbird.php' => array(
            'name' => 'Hummingbird',
            'flush_function' => 'hummingbird'
        ),
        'breeze/breeze.php' => array(
            'name' => 'Breeze',
            'flush_function' => 'breeze'
        ),
        'autoptimize/autoptimize.php' => array(
            'name' => 'Autoptimize',
            'flush_function' => 'autoptimize'
        ),
        'redis-cache/redis-cache.php' => array(
            'name' => 'Redis Object Cache',
            'flush_function' => 'redis_cache'
        )
    );
}

/**
 * Flush cache for specific plugin
 */
function ayudawp_anticache_flush_plugin_cache( $plugin_file, $plugin_name ) {
    try {
        switch ( $plugin_file ) {
            case 'sg-cachepress/sg-cachepress.php':
                if ( class_exists( 'SG_CachePress_Supercacher' ) ) {
                    SG_CachePress_Supercacher::purge_cache();
                    return true;
                }
                break;

            case 'wp-rocket/wp-rocket.php':
                if ( function_exists( 'rocket_clean_domain' ) ) {
                    rocket_clean_domain();
                    return true;
                }
                break;

            case 'w3-total-cache/w3-total-cache.php':
                if ( function_exists( 'w3tc_flush_all' ) ) {
                    w3tc_flush_all();
                    return true;
                }
                break;

            case 'wp-super-cache/wp-cache.php':
                if ( function_exists( 'wp_cache_clean_cache' ) ) {
                    global $file_prefix;
                    wp_cache_clean_cache( $file_prefix, true );
                    return true;
                }
                break;

            case 'wp-fastest-cache/wpFastestCache.php':
                if ( class_exists( 'WpFastestCache' ) ) {
                    $wfc = new WpFastestCache();
                    $wfc->deleteCache();
                    return true;
                }
                break;

            case 'litespeed-cache/litespeed-cache.php':
                if ( class_exists( 'LiteSpeed_Cache_API' ) ) {
                    LiteSpeed_Cache_API::purge_all();
                    return true;
                }
                break;

            case 'surge/surge.php':
                if ( function_exists( 'surge_purge_cache' ) ) {
                    surge_purge_cache();
                    return true;
                }
                break;

            case 'cache-enabler/cache-enabler.php':
                if ( class_exists( 'Cache_Enabler' ) && method_exists( 'Cache_Enabler', 'clear_site_cache' ) ) {
                    Cache_Enabler::clear_site_cache();
                    return true;
                } elseif ( function_exists( 'ce_cache_enabler_clear_site_cache' ) ) {
                    ce_cache_enabler_clear_site_cache();
                    return true;
                }
                break;

            case 'wp-optimize/wp-optimize.php':
                if ( class_exists( 'WP_Optimize_Cache_Commands' ) ) {
                    $wp_optimize_cache = new WP_Optimize_Cache_Commands();
                    $wp_optimize_cache->purge_page_cache();
                    return true;
                }
                break;

            case 'hummingbird-performance/wp-hummingbird.php':
                if ( class_exists( 'Hummingbird\WP_Hummingbird' ) ) {
                    do_action( 'wphb_clear_page_cache' );
                    return true;
                }
                break;

            case 'breeze/breeze.php':
                if ( class_exists( 'Breeze_PurgeCache' ) ) {
                    Breeze_PurgeCache::breeze_cache_flush();
                    return true;
                }
                break;

            case 'autoptimize/autoptimize.php':
                if ( class_exists( 'autoptimizeCache' ) ) {
                    autoptimizeCache::clearall();
                    return true;
                }
                break;

            case 'redis-cache/redis-cache.php':
                if ( function_exists( 'wp_cache_flush' ) ) {
                    wp_cache_flush();
                    return true;
                }
                break;
        }
    } catch ( Exception $e ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            // Use WordPress logging instead of error_log
            if ( function_exists( 'wp_debug_log' ) ) {
                wp_debug_log( 'Anti-Cache Kit: Error flushing ' . $plugin_name . ' - ' . $e->getMessage() );
            }
        }
    }

    return false;
}

/**
 * Clear object cache (Redis, Memcached)
 */
function ayudawp_anticache_clear_object_cache() {
    $cleared = false;

    // WordPress object cache
    if ( function_exists( 'wp_cache_flush' ) ) {
        try {
            wp_cache_flush();
            $cleared = true;
        } catch ( Exception $e ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                // Use WordPress logging instead of error_log
                if ( function_exists( 'wp_debug_log' ) ) {
                    wp_debug_log( 'Anti-Cache Kit: Error clearing object cache - ' . $e->getMessage() );
                }
            }
        }
    }

    // Additional object cache clearing
    if ( function_exists( 'opcache_reset' ) ) {
        try {
            opcache_reset();
            $cleared = true;
        } catch ( Exception $e ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                // Use WordPress logging instead of error_log
                if ( function_exists( 'wp_debug_log' ) ) {
                    wp_debug_log( 'Anti-Cache Kit: Error clearing opcache - ' . $e->getMessage() );
                }
            }
        }
    }

    return $cleared;
}

/**
 * Clear hosting-specific caches
 */
function ayudawp_anticache_clear_hosting_cache() {
    // SiteGround hosting cache
    if ( class_exists( 'SG_CachePress_Supercacher' ) ) {
        try {
            SG_CachePress_Supercacher::purge_cache();
        } catch ( Exception $e ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                // Use WordPress logging instead of error_log
                if ( function_exists( 'wp_debug_log' ) ) {
                    wp_debug_log( 'Anti-Cache Kit: Error clearing SiteGround cache - ' . $e->getMessage() );
                }
            }
        }
    }

    // Cloudflare (if CF plugin is available)
    // Check new namespace first (Cloudflare for WordPress v4.14.0+), then fallback to legacy namespace
    $cloudflare_available = class_exists( 'Cloudflare\APO\WordPress\Hooks' ) || class_exists( 'CF\WordPress\Hooks' );
    
    if ( $cloudflare_available ) {
        try {
            do_action( 'cloudflare_purge_cache' );
        } catch ( Exception $e ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                // Use WordPress logging instead of error_log
                if ( function_exists( 'wp_debug_log' ) ) {
                    wp_debug_log( 'Anti-Cache Kit: Error clearing Cloudflare cache - ' . $e->getMessage() );
                }
            }
        }
    }
}

/**
 * Enable debug mode in wp-config.php with backup
 */
function ayudawp_anticache_enable_debug_mode() {
    $result = array(
        'success' => false,
        'backup_created' => false,
        'error' => ''
    );

    // Use get_home_path() for proper wp-config.php location
    if ( ! function_exists( 'get_home_path' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    $wp_config = get_home_path() . 'wp-config.php';
    
    // Fallback to ABSPATH if get_home_path() doesn't work
    if ( ! file_exists( $wp_config ) ) {
        $wp_config = ABSPATH . 'wp-config.php';
    }
    
    // Initialize WP_Filesystem
    if ( ! function_exists( 'request_filesystem_credentials' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    global $wp_filesystem;
    if ( empty( $wp_filesystem ) ) {
        WP_Filesystem();
    }

    if ( ! $wp_filesystem || ! $wp_filesystem->exists( $wp_config ) || ! $wp_filesystem->is_writable( $wp_config ) ) {
        $result['error'] = esc_html__( 'wp-config.php file not found or not writable', 'anticache' );
        return $result;
    }

    $content = $wp_filesystem->get_contents( $wp_config );
    
    // Step 1: Create backup of original wp-config.php
    $backup_file = AYUDAWP_ANTICACHE_PLUGIN_DIR . 'backup/wp-config-backup.php';
    if ( $wp_filesystem->put_contents( $backup_file, $content, FS_CHMOD_FILE ) ) {
        $result['backup_created'] = true;
    }
    
    // Step 2: Remove existing debug definitions (any value, true or false)
    $content = preg_replace( "/^\s*define\s*\(\s*['\"]WP_DEBUG([A-Z_]*)?['\"]\s*,.*\);.*$/mi", '', $content );
    $content = preg_replace( "/^\s*define\s*\(\s*['\"]WP_DEBUG_LOG['\"]\s*,.*\);.*$/mi", '', $content );
    $content = preg_replace( "/^\s*define\s*\(\s*['\"]WP_DEBUG_DISPLAY['\"]\s*,.*\);.*$/mi", '', $content );
    
    // Step 3: Remove existing cache definitions (any value, true or false)
    $content = preg_replace( "/^\s*define\s*\(\s*['\"]WP_CACHE['\"]\s*,.*\);.*$/mi", '', $content );
    
    // Remove previous plugin comments
    $content = preg_replace( "/^\s*\/\*\s*Debug mode enabled by.*Anti-Cache.*Kit\s*\*\/\s*$/mi", '', $content );
    $content = preg_replace( "/^\s*\/\*\s*Cache disabled by.*Anti-Cache.*Kit\s*\*\/\s*$/mi", '', $content );

    // Step 4: Add new debug definitions
    $defines = array(
        "/* Debug mode enabled by Anti-Cache Emergency Kit */",
        "define('WP_DEBUG', true);",
        "define('WP_DEBUG_LOG', true);",
        "define('WP_DEBUG_DISPLAY', true);",
        "/* Cache disabled by Anti-Cache Emergency Kit */",
        "define('WP_CACHE', false);"
    );
    
    $content = preg_replace( 
        '/(\/\*\s*That[^\n]+stop editing![^\n]*\*\/)/i', 
        implode( "\n", $defines ) . "\n$1", 
        $content 
    );

    if ( $wp_filesystem->put_contents( $wp_config, $content, FS_CHMOD_FILE ) ) {
        $result['success'] = true;
    } else {
        $result['error'] = esc_html__( 'Could not write to wp-config.php', 'anticache' );
    }

    return $result;
}

/**
 * Generate admin notice with results
 */
function ayudawp_anticache_generate_admin_notice( $results ) {
    $notice = '<h2 style="margin-top:0">' . esc_html__( '⚠️ Anti-Cache Emergency Kit ACTIVATED', 'anticache' ) . '</h2>';
    
    // Actions performed
    $actions = array(
        esc_html__( 'All possible actions to avoid cache and show errors have been performed for administrator users only:', 'anticache' ),
        sprintf(
            // translators: %s is the list of cache plugins that were flushed and deactivated 
            esc_html__( '— Cache plugins detected, flushed and deactivated: %s', 'anticache' ), 
            $results['flushed'] ? implode( ', ', $results['flushed'] ) : esc_html__( 'None', 'anticache' )
        )
    );

    if ( $results['backup_created'] ) {
        $actions[] = esc_html__( '— Created backup of wp-config.php', 'anticache' );
    }

    if ( $results['debug_enabled'] ) {
        $actions[] = esc_html__( '—  Enabled debug mode (WP_DEBUG, WP_DEBUG_LOG, WP_DEBUG_DISPLAY) in wp-config.php', 'anticache' );
        $actions[] = esc_html__( '— Disabled WordPress cache (WP_CACHE = false) in wp-config.php', 'anticache' );
    }

    if ( $results['object_cache_cleared'] ) {
        $actions[] = esc_html__( '— Object cache (Redis/Memcached) cleared', 'anticache' );
    }

    $actions[] = esc_html__( '— All possible anti-cache HTTP headers sent for admin users', 'anticache' );
    $actions[] = esc_html__( '— Forced random version in all scripts and styles for admins', 'anticache' );
    $actions[] = esc_html__( '— HTML comment added in <head> for admin users as cache/debug signal', 'anticache' );
    $actions[] = esc_html__( '— All changes are only visible to logged-in administrators', 'anticache' );

    $notice .= '<ul style="margin: 8px 0 12px 18px;">';
    foreach ( $actions as $action ) {
        $notice .= '<li style="color:#0073aa">' . esc_html( $action ) . '</li>';
    }
    $notice .= '</ul>';

    // Deactivated plugins
    if ( $results['deactivated'] ) {
        $notice .= '<ul style="margin: 8px 0 12px 18px;">';
        foreach ( $results['deactivated'] as $plugin ) {
            $notice .= '<li style="color:#d35400">' . sprintf(
                // translators: %s is the name of the deactivated plugin 
                esc_html__( 'Plugin deactivated: <strong>%s</strong>', 'anticache' ), 
                esc_html( $plugin ) 
            ) . '</li>';
        }
        $notice .= '</ul>';
    }

    // Warnings and errors
    if ( ! $results['cache_found'] ) {
        $notice .= '<div style="color:orange; font-weight:bold;">' . 
                   esc_html__( 'No cache plugin was detected or active.', 'anticache' ) . 
                   '</div>';
    }

    if ( $results['errors'] ) {
        foreach ( $results['errors'] as $error ) {
            $notice .= '<div style="color:red"><strong>' . esc_html( $error ) . '</strong></div>';
        }
    }

    // Final warnings
    $notice .= '<div style="color:#c00;font-weight:bold;margin-top:12px;font-size:1.1em;">' . 
               esc_html__( 'Do NOT leave this plugin active! Deactivate it after troubleshooting.', 'anticache' ) . 
               '</div>';
    $notice .= '<div style="color:#888;margin-top:8px;">' . 
               esc_html__( 'All changes will be reverted when you deactivate the plugin.', 'anticache' ) . 
               '</div>';
    
    // Add settings link - Simple approach with direct link
    $settings_url = admin_url( 'tools.php?page=anticache-settings' );
    $notice .= '<div style="margin-top:15px;padding:10px;background:#f0f6fc;border:1px solid #c3e6cb;border-radius:4px;">' .
               '<strong>' . esc_html__( 'Need more control?', 'anticache' ) . '</strong> ' .
               '<a href="' . esc_url( $settings_url ) . '" style="color:#0073aa;font-weight:bold;text-decoration:none;">' .
               esc_html__( 'Go to Anti-Cache Kit Settings', 'anticache' ) . '</a>' .
               '</div>';

    update_option( 'ayudawp_anticache_admin_notice', $notice );
}

/**
 * Plugin deactivation - Revert all changes
 */
function ayudawp_anticache_deactivate() {
    // Restore wp-config.php from backup
    ayudawp_anticache_restore_wp_config();

    // Clean up options
    delete_option( 'ayudawp_anticache_admin_notice' );
    delete_option( 'ayudawp_anticache_deactivated_plugins' );
    delete_option( 'ayudawp_anticache_maintenance_mode' );
    delete_option( 'ayudawp_anticache_backup_file' );
}

/**
 * Restore wp-config.php from backup
 */
function ayudawp_anticache_restore_wp_config() {
    $backup_file = AYUDAWP_ANTICACHE_PLUGIN_DIR . 'backup/wp-config-backup.php';
    
    // Check if backup exists
    if ( ! file_exists( $backup_file ) ) {
        // Fallback to manual cleanup if backup doesn't exist
        ayudawp_anticache_disable_debug_mode();
        return false;
    }

    // Use get_home_path() for proper wp-config.php location
    if ( ! function_exists( 'get_home_path' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    $wp_config = get_home_path() . 'wp-config.php';
    
    // Fallback to ABSPATH if get_home_path() doesn't work
    if ( ! file_exists( $wp_config ) ) {
        $wp_config = ABSPATH . 'wp-config.php';
    }

    if ( ! function_exists( 'request_filesystem_credentials' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    global $wp_filesystem;
    if ( empty( $wp_filesystem ) ) {
        WP_Filesystem();
    }

    if ( ! $wp_filesystem || ! $wp_filesystem->exists( $backup_file ) || ! $wp_filesystem->is_writable( $wp_config ) ) {
        // Fallback to manual cleanup if filesystem issues
        ayudawp_anticache_disable_debug_mode();
        return false;
    }

    // Get backup content
    $backup_content = $wp_filesystem->get_contents( $backup_file );
    
    // Restore the backup
    if ( $wp_filesystem->put_contents( $wp_config, $backup_content, FS_CHMOD_FILE ) ) {
        // Delete backup file after successful restoration
        $wp_filesystem->delete( $backup_file );
        return true;
    }

    return false;
}

/**
 * Disable debug mode in wp-config.php (fallback method)
 */
function ayudawp_anticache_disable_debug_mode() {
    // Use get_home_path() for proper wp-config.php location
    if ( ! function_exists( 'get_home_path' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    $wp_config = get_home_path() . 'wp-config.php';
    
    // Fallback to ABSPATH if get_home_path() doesn't work
    if ( ! file_exists( $wp_config ) ) {
        $wp_config = ABSPATH . 'wp-config.php';
    }

    if ( ! function_exists( 'request_filesystem_credentials' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    global $wp_filesystem;
    if ( empty( $wp_filesystem ) ) {
        WP_Filesystem();
    }

    if ( ! $wp_filesystem || ! $wp_filesystem->exists( $wp_config ) || ! $wp_filesystem->is_writable( $wp_config ) ) {
        return false;
    }

    $content = $wp_filesystem->get_contents( $wp_config );
    
    // Remove debug mode additions
    $content = preg_replace( "/^\s*\/\*\s*Debug mode enabled by.*Anti-Cache.*Kit\s*\*\/\s*$/mi", '', $content );
    $content = preg_replace( "/^\s*\/\*\s*Cache disabled by.*Anti-Cache.*Kit\s*\*\/\s*$/mi", '', $content );
    $content = preg_replace( "/^\s*define\s*\(\s*['\"]WP_DEBUG([A-Z_]*)?['\"]\s*,.*\);.*$/mi", '', $content );
    $content = preg_replace( "/^\s*define\s*\(\s*['\"]WP_DEBUG_LOG['\"]\s*,.*\);.*$/mi", '', $content );
    $content = preg_replace( "/^\s*define\s*\(\s*['\"]WP_DEBUG_DISPLAY['\"]\s*,.*\);.*$/mi", '', $content );
    $content = preg_replace( "/^\s*define\s*\(\s*['\"]WP_CACHE['\"]\s*,.*\);.*$/mi", '', $content );

    return $wp_filesystem->put_contents( $wp_config, $content, FS_CHMOD_FILE );
}

// Load admin panel
if ( is_admin() ) {
    require_once AYUDAWP_ANTICACHE_PLUGIN_DIR . 'includes/admin.php';
}

// Add settings link in plugins list
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'ayudawp_anticache_plugin_action_links' );

/**
 * Add settings link to plugin actions in plugins list
 */
function ayudawp_anticache_plugin_action_links( $links ) {
    $settings_link = sprintf(
        '<a href="%s" style="color:#d35400;font-weight:bold;">%s</a>',
        admin_url( 'tools.php?page=anticache-settings' ),
        esc_html__( 'Settings', 'anticache' )
    );
    
    // Add settings link at the beginning
    array_unshift( $links, $settings_link );
    
    return $links;
}

// Load maintenance mode functionality
require_once AYUDAWP_ANTICACHE_PLUGIN_DIR . 'includes/maintenance.php';

// Display admin notice
add_action( 'admin_notices', 'ayudawp_anticache_show_admin_notice' );

/**
 * Show admin notice in dashboard
 */
function ayudawp_anticache_show_admin_notice() {
    if ( $notice = get_option( 'ayudawp_anticache_admin_notice' ) ) {
        echo '<div class="notice notice-warning is-dismissible" style="border-left: 5px solid #0073aa;background:#fff;padding:18px 22px;font-size:16px;">' . 
             wp_kses( $notice, array(
                 'h2' => array( 'style' => array() ),
                 'ul' => array( 'style' => array() ),
                 'li' => array( 'style' => array() ),
                 'div' => array( 'style' => array() ),
                 'strong' => array(),
                 'br' => array(),
                 'a' => array( 
                     'href' => array(), 
                     'style' => array(),
                     'target' => array(),
                     'rel' => array()
                 )
             ) ) . 
             '</div>';
    }
}

// Anti-cache functionality for administrators
add_action( 'send_headers', 'ayudawp_anticache_headers_for_admins', 99 );
add_filter( 'script_loader_src', 'ayudawp_anticache_randomize_assets', 99 );
add_filter( 'style_loader_src', 'ayudawp_anticache_randomize_assets', 99 );
add_action( 'wp_head', 'ayudawp_anticache_html_comment_admin', 1 );
add_action( 'admin_head', 'ayudawp_anticache_html_comment_admin', 1 );

/**
 * Send anti-cache headers for admin users
 */
function ayudawp_anticache_headers_for_admins() {
    if ( ! is_user_logged_in() || ! current_user_can( 'manage_options' ) ) {
        return;
    }

    @header( 'Expires: Wed, 11 Jan 1984 05:00:00 GMT' );
    @header( 'Cache-Control: no-store, no-cache, must-revalidate, max-age=0, s-maxage=0' );
    @header( 'Cache-Control: post-check=0, pre-check=0', false );
    @header( 'Pragma: no-cache' );
    @header( 'X-Cache-By: Anti-Cache Emergency Kit' );
}

/**
 * Add random version to scripts and styles for admin users
 */
function ayudawp_anticache_randomize_assets( $src ) {
    if ( ! is_user_logged_in() || ! current_user_can( 'manage_options' ) ) {
        return $src;
    }

    $random = 'nocache=' . wp_rand( 100000, 999999 );
    $src .= ( strpos( $src, '?' ) === false ? '?' : '&' ) . $random;
    
    return $src;
}

/**
 * Add HTML comment in head for admin users
 */
function ayudawp_anticache_html_comment_admin() {
    if ( ! is_user_logged_in() || ! current_user_can( 'manage_options' ) ) {
        return;
    }

    echo "\n<!-- AyudaWP Anti-Cache Kit: All possible cache disabled for this session (admin user) -->\n";
}