<?php
/**
 * Anti-Bloat Plugin Uninstall
 *
 * Fired when the plugin is uninstalled.
 *
 * @package AntiBloat
 * @version 1.1.0
 * @since 1.0.0
 */

// If uninstall not called from WordPress, then exit.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
    exit;
}

// Check if user has permission to delete plugins
if ( ! current_user_can( 'delete_plugins' ) ) {
    exit;
}

/**
 * Clean up all plugin data
 */
function anti_bloat_cleanup_data() {
    // Remove plugin options
    delete_option( 'anti_bloat_options' );
    
    // Remove legacy options (if any)
    delete_option( 'anti-bloat-settings' );
    
    // Remove user meta for welcome panel (if set by plugin)
    $users = get_users( array( 'fields' => 'ID' ) );
    foreach ( $users as $user_id ) {
        // Only remove if it was set to 0 (disabled by our plugin)
        $welcome_panel = get_user_meta( $user_id, 'show_welcome_panel', true );
        if ( $welcome_panel === '0' || $welcome_panel === 0 ) {
            delete_user_meta( $user_id, 'show_welcome_panel' );
        }
    }
    
    // Clean up any transients (if plugin used any)
    delete_transient( 'anti_bloat_cache' );
    delete_transient( 'anti_bloat_settings_cache' );
    
    // Remove any custom database tables (none for this plugin, but good practice)
    // global $wpdb;
    // $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}anti_bloat_data");
    
    // Clear any cached data
    wp_cache_flush();
}

/**
 * Multisite cleanup
 */
function anti_bloat_multisite_cleanup() {
    global $wpdb;
    
    if ( is_multisite() ) {
        // Get all blog IDs
        $blog_ids = $wpdb->get_col( "SELECT blog_id FROM $wpdb->blogs" );
        
        foreach ( $blog_ids as $blog_id ) {
            switch_to_blog( $blog_id );
            anti_bloat_cleanup_data();
            restore_current_blog();
        }
        
        // Clean up network-wide options if any
        delete_site_option( 'anti_bloat_network_options' );
    } else {
        anti_bloat_cleanup_data();
    }
}

// Perform the cleanup
anti_bloat_multisite_cleanup();

// Log the uninstall (optional - for debugging)
if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
    error_log( 'Anti-Bloat Plugin: Uninstall completed at ' . current_time( 'mysql' ) );
}

