<?php
/**
 * Disable Admin Bloats Plugin Main Class
 * 
 * @package DisableAdminBloats
 * @version 1.1.0
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DisableAdminBloatsPlugin {
    
    /**
     * Plugin version
     */
    private $version = '1.1.0';
    
    /**
     * Default options
     */
    private $default_options = array(
        'disable_all_bloat' => 0,
        'disable_admin_notices' => 0,
        'disable_woocommerce_notices' => 0,
        'disable_plugin_notices' => 0,
        'disable_update_notices' => 0,
        'disable_dashboard_widgets' => 0,
        'disable_wp_news' => 0,
        'disable_welcome_panel' => 0,
        'disable_admin_bar_wp_logo' => 0,
        'disable_howdy' => 0,
        'disable_screen_options' => 0,
        'disable_help_tabs' => 0,
        'remove_wp_version' => 0,
        'disable_file_editing' => 0,
        'performance_mode' => 0,
        'developer_mode' => 0
    );
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
        $this->apply_bloat_removal();
    }
    
    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_bar_menu', array( $this, 'add_admin_bar_menu' ), 999 );
        add_action( 'admin_head', array( $this, 'admin_head_actions' ) );
        add_action( 'wp_head', array( $this, 'wp_head_actions' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
        add_filter( 'plugin_action_links_' . DISABLE_ADMIN_BLOATS_BASENAME, array( $this, 'add_settings_link' ) );
        
        // AJAX handlers for import/export
        add_action( 'wp_ajax_disable_admin_bloats_import_settings', array( $this, 'ajax_import_settings' ) );
        add_action( 'wp_ajax_disable_admin_bloats_validate_import', array( $this, 'ajax_validate_import' ) );
        
        // Security enhancements
        add_action( 'init', array( $this, 'security_enhancements' ) );
    }
    
    /**
     * Apply bloat removal based on settings
     */
    private function apply_bloat_removal() {
        $options = $this->get_options();
        
        // Legacy compatibility - if old setting exists, migrate it
        if ( isset( $options['disable-all-bloat'] ) ) {
            $options['disable_all_bloat'] = $options['disable-all-bloat'];
            unset( $options['disable-all-bloat'] );
            update_option( 'disable_admin_bloats_options', $options );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_admin_notices'] ) {
            $this->disable_admin_notices();
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_woocommerce_notices'] ) {
            $this->disable_woocommerce_notices();
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_plugin_notices'] ) {
            $this->disable_plugin_notices();
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_update_notices'] ) {
            $this->disable_update_notices();
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_dashboard_widgets'] ) {
            add_action( 'wp_dashboard_setup', array( $this, 'remove_dashboard_widgets' ) );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_wp_news'] ) {
            add_action( 'wp_dashboard_setup', array( $this, 'remove_wp_news' ) );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_welcome_panel'] ) {
            add_action( 'admin_init', array( $this, 'disable_welcome_panel' ) );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_admin_bar_wp_logo'] ) {
            add_action( 'admin_bar_menu', array( $this, 'remove_wp_logo' ), 999 );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_howdy'] ) {
            add_action( 'admin_bar_menu', array( $this, 'replace_howdy' ), 999 );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_screen_options'] ) {
            add_filter( 'screen_options_show_screen', '__return_false' );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_help_tabs'] ) {
            add_action( 'admin_head', array( $this, 'remove_help_tabs' ) );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_file_editing'] ) {
            if ( ! defined( 'DISALLOW_FILE_EDIT' ) ) {
                define( 'DISALLOW_FILE_EDIT', true );
            }
        }
        
        if ( $options['disable_all_bloat'] || $options['performance_mode'] ) {
            $this->enable_performance_mode();
        }
    }
    
    /**
     * Get plugin options
     */
    private function get_options() {
        return wp_parse_args( get_option( 'disable_admin_bloats_options', array() ), $this->default_options );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        register_setting( 'disable-admin-bloats-settings', 'disable_admin_bloats_options', array( $this, 'sanitize_options' ) );
        
        // General settings section
        add_settings_section(
            'disable-admin-bloats-general-section',
            __( 'General Settings', 'disable-admin-bloats' ),
            array( $this, 'general_section_callback' ),
            'disable-admin-bloats'
        );
        
        // Notices settings section
        add_settings_section(
            'disable-admin-bloats-notices-section',
            __( 'Notice Controls', 'disable-admin-bloats' ),
            array( $this, 'notices_section_callback' ),
            'disable-admin-bloats'
        );
        
        // Interface settings section
        add_settings_section(
            'disable-admin-bloats-interface-section',
            __( 'Interface Cleanup', 'disable-admin-bloats' ),
            array( $this, 'interface_section_callback' ),
            'disable-admin-bloats'
        );
        
        // Performance settings section
        add_settings_section(
            'disable-admin-bloats-performance-section',
            __( 'Performance Settings', 'disable-admin-bloats' ),
            array( $this, 'performance_section_callback' ),
            'disable-admin-bloats'
        );
        
        // Add all settings fields
        $this->add_settings_fields();
    }
    
    /**
     * Add settings fields
     */
    private function add_settings_fields() {
        // Master toggle
        add_settings_field(
            'disable_all_bloat',
            __( 'Master Toggle', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-general-section',
            array( 
                'key' => 'disable_all_bloat',
                'label' => __( 'Disable all admin bloats (overrides individual settings)', 'disable-admin-bloats' ),
                'description' => __( 'This will enable all bloat removal features at once.', 'disable-admin-bloats' )
            )
        );
        
        // Notice controls
        add_settings_field(
            'disable_admin_notices',
            __( 'Admin Notices', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-notices-section',
            array( 
                'key' => 'disable_admin_notices',
                'label' => __( 'Hide all admin notices and messages', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_woocommerce_notices',
            __( 'WooCommerce Notices', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-notices-section',
            array( 
                'key' => 'disable_woocommerce_notices',
                'label' => __( 'Hide WooCommerce admin notices', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_plugin_notices',
            __( 'Plugin Notices', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-notices-section',
            array( 
                'key' => 'disable_plugin_notices',
                'label' => __( 'Hide plugin update and activation notices', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_update_notices',
            __( 'Update Notices', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-notices-section',
            array( 
                'key' => 'disable_update_notices',
                'label' => __( 'Hide WordPress update notifications', 'disable-admin-bloats' )
            )
        );
        
        // Interface controls
        add_settings_field(
            'disable_dashboard_widgets',
            __( 'Dashboard Widgets', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-interface-section',
            array( 
                'key' => 'disable_dashboard_widgets',
                'label' => __( 'Remove default dashboard widgets', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_wp_news',
            __( 'WordPress News', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-interface-section',
            array( 
                'key' => 'disable_wp_news',
                'label' => __( 'Remove WordPress news dashboard widget', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_welcome_panel',
            __( 'Welcome Panel', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-interface-section',
            array( 
                'key' => 'disable_welcome_panel',
                'label' => __( 'Hide WordPress welcome panel', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_admin_bar_wp_logo',
            __( 'WordPress Logo', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-interface-section',
            array( 
                'key' => 'disable_admin_bar_wp_logo',
                'label' => __( 'Remove WordPress logo from admin bar', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_howdy',
            __( 'Howdy Text', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-interface-section',
            array( 
                'key' => 'disable_howdy',
                'label' => __( 'Replace "Howdy" with "Welcome"', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_screen_options',
            __( 'Screen Options', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-interface-section',
            array( 
                'key' => 'disable_screen_options',
                'label' => __( 'Hide screen options tab', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_help_tabs',
            __( 'Help Tabs', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-interface-section',
            array( 
                'key' => 'disable_help_tabs',
                'label' => __( 'Remove help tabs from admin pages', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'remove_wp_version',
            __( 'WordPress Version', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-interface-section',
            array( 
                'key' => 'remove_wp_version',
                'label' => __( 'Remove WordPress version from source', 'disable-admin-bloats' )
            )
        );
        
        add_settings_field(
            'disable_file_editing',
            __( 'File Editing', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-interface-section',
            array( 
                'key' => 'disable_file_editing',
                'label' => __( 'Disable file editing in admin', 'disable-admin-bloats' )
            )
        );
        
        // Performance settings
        add_settings_field(
            'performance_mode',
            __( 'Performance Mode', 'disable-admin-bloats' ),
            array( $this, 'render_checkbox' ),
            'disable-admin-bloats',
            'disable-admin-bloats-performance-section',
            array( 
                'key' => 'performance_mode',
                'label' => __( 'Enable performance optimizations', 'disable-admin-bloats' ),
                'description' => __( 'Removes unnecessary features to improve admin performance.', 'disable-admin-bloats' )
            )
        );
    }

    /**
     * Render checkbox field
     */
    public function render_checkbox( $args ) {
        $options = $this->get_options();
        $key = $args['key'];
        $label = $args['label'];
        $description = isset( $args['description'] ) ? $args['description'] : '';
        $checked = isset( $options[ $key ] ) && $options[ $key ] == 1 ? 'checked' : '';
        
        echo '<tr>';
        echo '<th scope="row">' . esc_html( $this->format_field_title( $key ) ) . '</th>';
        echo '<td>';
        echo '<label>';
        echo '<input type="checkbox" name="disable_admin_bloats_options[' . esc_attr( $key ) . ']" value="1" ' . $checked . '> ';
        echo esc_html( $label );
        echo '</label>';
        
        if ( $description ) {
            echo '<p class="description">' . esc_html( $description ) . '</p>';
        }
        echo '</td>';
        echo '</tr>';
    }
    
    /**
     * Format field title from key
     */
    private function format_field_title( $key ) {
        // Convert key to readable title
        $title = str_replace( array( 'disable_', 'remove_', '_' ), array( '', '', ' ' ), $key );
        return ucwords( $title );
    }
    
    /**
     * Section callbacks
     */
    public function general_section_callback() {
        echo '<p>' . esc_html__( 'General plugin settings and master toggle controls.', 'disable-admin-bloats' ) . '</p>';
    }
    
    public function notices_section_callback() {
        echo '<p>' . esc_html__( 'Control which admin notices and messages to hide from your dashboard.', 'disable-admin-bloats' ) . '</p>';
    }
    
    public function interface_section_callback() {
        echo '<p>' . esc_html__( 'Clean up the WordPress admin interface by removing unnecessary elements and bloat.', 'disable-admin-bloats' ) . '</p>';
    }
    
    public function performance_section_callback() {
        echo '<p>' . esc_html__( 'Performance optimizations for a faster and cleaner admin experience.', 'disable-admin-bloats' ) . '</p>';
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __( 'Disable Admin Bloats Settings', 'disable-admin-bloats' ),
            __( 'Disable Admin Bloats', 'disable-admin-bloats' ),
            'manage_options',
            'disable-admin-bloats-settings',
            array( $this, 'settings_page' ),
            'dashicons-shield-alt',
            99
        );
    }
    
    /**
     * Settings page
     */
    public function settings_page() {
        include( DISABLE_ADMIN_BLOATS_PLUGIN_DIR . 'includes/class-disable-admin-bloats-settings.php' );
    }
    
    /**
     * Sanitize options
     */
    public function sanitize_options( $input ) {
        if ( ! is_array( $input ) ) {
            return array();
        }
        
        $sanitized = array();
        foreach ( $this->default_options as $key => $default ) {
            $sanitized[ $key ] = isset( $input[ $key ] ) ? absint( $input[ $key ] ) : 0;
        }
        
        return $sanitized;
    }
    
    /**
     * Add settings link to plugin actions
     */
    public function add_settings_link( $links ) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url( 'admin.php?page=disable-admin-bloats-settings' ),
            __( 'Settings', 'disable-admin-bloats' )
        );
        array_unshift( $links, $settings_link );
        return $links;
    }
    
    /**
     * Admin bar menu
     */
    public function add_admin_bar_menu( $wp_admin_bar ) {
        $options = $this->get_options();
        $is_active = $options['disable_all_bloat'] || 
                    $options['disable_admin_notices'] || 
                    $options['disable_woocommerce_notices'];
        
        if ( $is_active ) {
            $wp_admin_bar->add_menu( array(
                'id' => 'disable-admin-bloats-status',
                'title' => __( 'Disable Admin Bloats Active', 'disable-admin-bloats' ),
                'parent' => 'top-secondary',
                'meta' => array( 'class' => 'disable-admin-bloats-active' ),
                'href' => admin_url( 'admin.php?page=disable-admin-bloats-settings' ),
            ) );
        }
    }
    
    /**
     * Admin head actions
     */
    public function admin_head_actions() {
        $this->output_admin_css();
        $this->disable_admin_notices();
    }
    
    /**
     * WP head actions
     */
    public function wp_head_actions() {
        $options = $this->get_options();
        
        if ( $options['developer_mode'] ) {
            echo '<!-- Disable Admin Bloats Plugin v' . esc_attr( $this->version ) . ' Active -->' . "\n";
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts( $hook ) {
        if ( 'toplevel_page_disable-admin-bloats-settings' === $hook ) {
            wp_enqueue_script( 'jquery' );
            wp_add_inline_script( 'jquery', $this->get_admin_js() );
            
            // Localize script for AJAX
            wp_localize_script( 'jquery', 'disableAdminBloatsAjax', array(
                'ajaxurl' => admin_url( 'admin-ajax.php' ),
                'nonce' => wp_create_nonce( 'disable_admin_bloats_import' ),
                'strings' => array(
                    'importSuccess' => __( 'Settings imported successfully!', 'disable-admin-bloats' ),
                    'importError' => __( 'Error importing settings.', 'disable-admin-bloats' ),
                    'invalidFile' => __( 'Please select a valid JSON file.', 'disable-admin-bloats' ),
                    'invalidFormat' => __( 'Invalid Disable Admin Bloats settings file.', 'disable-admin-bloats' ),
                    'noSettings' => __( 'No settings found in the file.', 'disable-admin-bloats' ),
                    'confirmImport' => __( 'This will replace your current settings. Continue?', 'disable-admin-bloats' ),
                    'exportDate' => __( 'Export Date:', 'disable-admin-bloats' ),
                    'version' => __( 'Version:', 'disable-admin-bloats' ),
                    'settings' => __( 'settings', 'disable-admin-bloats' ),
                    'dontForgetSave' => __( 'Don\'t forget to save your changes.', 'disable-admin-bloats' ),
                    'readError' => __( 'Error reading the settings file. Please check if it\'s a valid JSON file.', 'disable-admin-bloats' )
                )
            ) );
        }
    }
    
    /**
     * Get admin JavaScript
     */
    private function get_admin_js() {
        return "
        jQuery(document).ready(function($) {
            // Toggle all options when 'disable all bloat' is checked
            $('#disable_admin_bloats_options_disable_all_bloat').change(function() {
                var isChecked = $(this).is(':checked');
                if (isChecked) {
                    if (confirm('" . esc_js( __( 'This will enable all bloat removal features. Continue?', 'disable-admin-bloats' ) ) . "')) {
                        $('input[type=\"checkbox\"]').not(this).prop('checked', false).prop('disabled', true);
                    } else {
                        $(this).prop('checked', false);
                    }
                } else {
                    $('input[type=\"checkbox\"]').prop('disabled', false);
                }
            });
            
            // Initial state
            if ($('#disable_admin_bloats_options_disable_all_bloat').is(':checked')) {
                $('input[type=\"checkbox\"]').not('#disable_admin_bloats_options_disable_all_bloat').prop('disabled', true);
            }
        });
        ";
    }

    /**
     * Disable admin notices
     */
    private function disable_admin_notices() {
        $options = $this->get_options();
        
        if ( $options['disable_all_bloat'] || $options['disable_admin_notices'] ) {
            remove_all_actions( 'admin_notices' );
            remove_all_actions( 'user_admin_notices' );
            remove_all_actions( 'network_admin_notices' );
            remove_all_actions( 'all_admin_notices' );
        }
    }
    
    /**
     * Disable WooCommerce notices
     */
    private function disable_woocommerce_notices() {
        if ( class_exists( 'WooCommerce' ) ) {
            add_action( 'wp_print_styles', function() {
                wp_add_inline_style( 'wp-admin', '
                    .woocommerce-message,
                    .woocommerce-error,
                    .woocommerce-info,
                    .wc-admin-notice,
                    .woocommerce-admin-notice {
                        display: none !important;
                    }
                ' );
            } );
        }
    }
    
    /**
     * Disable plugin notices
     */
    private function disable_plugin_notices() {
        add_action( 'admin_head', function() {
            echo '<style>
                .plugin-update-tr,
                .plugin-install-php,
                .update-message,
                .updating-message,
                .updated-message {
                    display: none !important;
                }
            </style>';
        } );
    }
    
    /**
     * Disable update notices
     */
    private function disable_update_notices() {
        add_action( 'admin_head', function() {
            echo '<style>
                .update-nag,
                .update-core-php,
                .update-php,
                .notice-warning.update-php {
                    display: none !important;
                }
            </style>';
        } );
        
        // Remove update notifications for non-admins
        if ( ! current_user_can( 'update_core' ) ) {
            add_action( 'init', function() {
                remove_action( 'init', 'wp_version_check' );
            }, 2 );
        }
    }
    
    /**
     * Remove dashboard widgets
     */
    public function remove_dashboard_widgets() {
        global $wp_meta_boxes;
        
        $dashboard_widgets = array(
            'dashboard_incoming_links',
            'dashboard_plugins',
            'dashboard_primary',
            'dashboard_secondary',
            'dashboard_quick_press',
            'dashboard_recent_drafts',
            'dashboard_recent_comments',
            'dashboard_right_now',
            'dashboard_activity',
            'dashboard_site_health',
            'network_dashboard_right_now',
            'wp_mail_smtp_reports_widget',
            'woocommerce_dashboard_overview',
        );
        
        foreach ( $dashboard_widgets as $widget ) {
            remove_meta_box( $widget, 'dashboard', 'normal' );
            remove_meta_box( $widget, 'dashboard', 'side' );
            remove_meta_box( $widget, 'dashboard', 'core' );
        }
    }
    
    /**
     * Remove WordPress news widget
     */
    public function remove_wp_news() {
        remove_meta_box( 'dashboard_primary', 'dashboard', 'side' );
        remove_meta_box( 'dashboard_secondary', 'dashboard', 'side' );
    }
    
    /**
     * Disable welcome panel
     */
    public function disable_welcome_panel() {
        update_user_meta( get_current_user_id(), 'show_welcome_panel', 0 );
    }
    
    /**
     * Remove WordPress logo from admin bar
     */
    public function remove_wp_logo( $wp_admin_bar ) {
        $wp_admin_bar->remove_node( 'wp-logo' );
    }
    
    /**
     * Replace "Howdy" with "Welcome"
     */
    public function replace_howdy( $wp_admin_bar ) {
        $my_account = $wp_admin_bar->get_node( 'my-account' );
        if ( $my_account ) {
            $new_title = str_replace( 'Howdy,', __( 'Welcome,', 'disable-admin-bloats' ), $my_account->title );
            $wp_admin_bar->add_node( array(
                'id' => 'my-account',
                'title' => $new_title,
            ) );
        }
    }
    
    /**
     * Remove help tabs
     */
    public function remove_help_tabs() {
        $screen = get_current_screen();
        if ( $screen ) {
            $screen->remove_help_tabs();
        }
    }
    
    /**
     * Enable performance mode
     */
    private function enable_performance_mode() {
        // Disable WordPress heartbeat on non-post pages
        add_action( 'init', function() {
            global $pagenow;
            if ( $pagenow !== 'post.php' && $pagenow !== 'post-new.php' ) {
                wp_deregister_script( 'heartbeat' );
            }
        } );
        
        // Reduce heartbeat frequency
        add_filter( 'heartbeat_settings', function( $settings ) {
            $settings['interval'] = 60; // 60 seconds
            return $settings;
        } );
        
        // Disable theme/plugin editor
        if ( ! defined( 'DISALLOW_FILE_EDIT' ) ) {
            define( 'DISALLOW_FILE_EDIT', true );
        }
        
        // Remove unnecessary dashboard widgets
        add_action( 'wp_dashboard_setup', array( $this, 'remove_dashboard_widgets' ) );
    }
    
    /**
     * Security enhancements
     */
    public function security_enhancements() {
        $options = $this->get_options();
        
        if ( $options['remove_wp_version'] || $options['disable_all_bloat'] ) {
            // Remove version from RSS feeds
            add_filter( 'the_generator', '__return_empty_string' );
            
            // Remove version from scripts and styles
            add_filter( 'style_loader_src', array( $this, 'remove_version_from_src' ), 9999 );
            add_filter( 'script_loader_src', array( $this, 'remove_version_from_src' ), 9999 );
        }
    }
    
    /**
     * Remove version from source URLs
     */
    public function remove_version_from_src( $src ) {
        if ( strpos( $src, 'ver=' ) ) {
            $src = remove_query_arg( 'ver', $src );
        }
        return $src;
    }
    
    /**
     * Output admin CSS
     */
    private function output_admin_css() {
        $options = $this->get_options();
        $css = '<style>';
        
        // Basic notice hiding CSS
        if ( $options['disable_all_bloat'] || $options['disable_admin_notices'] ) {
            $css .= '
                .notice, .notice-success, .notice-error, .notice-warning, .notice-info,
                .updated, .error, .update-nag, .update-message,
                .woocommerce-message, .woocommerce-error, .woocommerce-info {
                    display: none !important;
                }
            ';
        }
        
        // Admin bar indicator
        $css .= '
            #wp-admin-bar-disable-admin-bloats-status {
                background: #dc3232 !important;
            }
            #wp-admin-bar-disable-admin-bloats-status .ab-item {
                color: white !important;
                font-weight: bold !important;
            }
        ';
        
        $css .= '</style>';
        echo $css;
    }
    
    /**
     * AJAX handler for importing settings
     */
    public function ajax_import_settings() {
        // Check permissions and nonce
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have sufficient permissions.', 'disable-admin-bloats' ) );
        }
        
        check_ajax_referer( 'disable_admin_bloats_import', 'nonce' );
        
        if ( ! isset( $_POST['settings'] ) ) {
            wp_send_json_error( __( 'No settings data provided.', 'disable-admin-bloats' ) );
        }
        
        $import_data = json_decode( stripslashes( $_POST['settings'] ), true );
        
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            wp_send_json_error( __( 'Invalid JSON data.', 'disable-admin-bloats' ) );
        }
        
        // Validate import data
        if ( ! isset( $import_data['plugin'] ) || $import_data['plugin'] !== 'disable-admin-bloats' ) {
            wp_send_json_error( __( 'Invalid plugin settings file.', 'disable-admin-bloats' ) );
        }
        
        if ( ! isset( $import_data['settings'] ) || ! is_array( $import_data['settings'] ) ) {
            wp_send_json_error( __( 'No valid settings found in file.', 'disable-admin-bloats' ) );
        }
        
        // Import settings
        $current_options = $this->get_options();
        $imported_count = 0;
        
        foreach ( $import_data['settings'] as $key => $value ) {
            if ( array_key_exists( $key, $this->default_options ) ) {
                $current_options[ $key ] = absint( $value );
                $imported_count++;
            }
        }
        
        // Save options
        update_option( 'disable_admin_bloats_options', $current_options );
        
        wp_send_json_success( array(
            'message' => sprintf( __( 'Successfully imported %d settings.', 'disable-admin-bloats' ), $imported_count ),
            'imported_count' => $imported_count
        ) );
    }
    
    /**
     * AJAX handler for validating import data
     */
    public function ajax_validate_import() {
        // Check permissions and nonce
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have sufficient permissions.', 'disable-admin-bloats' ) );
        }
        
        check_ajax_referer( 'disable_admin_bloats_import', 'nonce' );
        
        if ( ! isset( $_POST['settings'] ) ) {
            wp_send_json_error( __( 'No settings data provided.', 'disable-admin-bloats' ) );
        }
        
        $import_data = json_decode( stripslashes( $_POST['settings'] ), true );
        
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            wp_send_json_error( __( 'Invalid JSON data.', 'disable-admin-bloats' ) );
        }
        
        $validation_result = array(
            'valid' => true,
            'plugin_match' => isset( $import_data['plugin'] ) && $import_data['plugin'] === 'disable-admin-bloats',
            'has_settings' => isset( $import_data['settings'] ) && is_array( $import_data['settings'] ),
            'export_date' => isset( $import_data['export_date'] ) ? $import_data['export_date'] : null,
            'version' => isset( $import_data['version'] ) ? $import_data['version'] : null,
            'valid_settings_count' => 0
        );
        
        if ( $validation_result['has_settings'] ) {
            foreach ( $import_data['settings'] as $key => $value ) {
                if ( array_key_exists( $key, $this->default_options ) ) {
                    $validation_result['valid_settings_count']++;
                }
            }
        }
        
        $validation_result['valid'] = $validation_result['plugin_match'] && 
                                    $validation_result['has_settings'] && 
                                    $validation_result['valid_settings_count'] > 0;
        
        wp_send_json_success( $validation_result );
    }
}