<?php
/**
 * Anti-Bloat Plugin Main Class
 * 
 * @package AntiBloat
 * @version 1.1.0
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class AntiBloatPlugin {
    
    /**
     * Plugin version
     */
    private $version = '1.1.0';
    
    /**
     * Default options
     */
    private $default_options = array(
        'disable_all_bloat' => 0,
        'disable_admin_notices' => 0,
        'disable_woocommerce_notices' => 0,
        'disable_plugin_notices' => 0,
        'disable_update_notices' => 0,
        'disable_dashboard_widgets' => 0,
        'disable_wp_news' => 0,
        'disable_welcome_panel' => 0,
        'disable_admin_bar_wp_logo' => 0,
        'disable_howdy' => 0,
        'disable_screen_options' => 0,
        'disable_help_tabs' => 0,
        'remove_wp_version' => 0,
        'disable_file_editing' => 0,
        'performance_mode' => 0,
        'developer_mode' => 0
    );
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
        $this->apply_bloat_removal();
    }
    
    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_bar_menu', array( $this, 'add_admin_bar_menu' ), 999 );
        add_action( 'admin_head', array( $this, 'admin_head_actions' ) );
        add_action( 'wp_head', array( $this, 'wp_head_actions' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
        add_filter( 'plugin_action_links_' . ANTI_BLOAT_BASENAME, array( $this, 'add_settings_link' ) );
        
        // AJAX handlers for import/export
        add_action( 'wp_ajax_anti_bloat_import_settings', array( $this, 'ajax_import_settings' ) );
        add_action( 'wp_ajax_anti_bloat_validate_import', array( $this, 'ajax_validate_import' ) );
        
        // Security enhancements
        add_action( 'init', array( $this, 'security_enhancements' ) );
    }
    
    /**
     * Apply bloat removal based on settings
     */
    private function apply_bloat_removal() {
        $options = $this->get_options();
        
        // Legacy compatibility - if old setting exists, migrate it
        if ( isset( $options['disable-all-bloat'] ) ) {
            $options['disable_all_bloat'] = $options['disable-all-bloat'];
            unset( $options['disable-all-bloat'] );
            update_option( 'anti_bloat_options', $options );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_admin_notices'] ) {
            $this->disable_admin_notices();
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_woocommerce_notices'] ) {
            $this->disable_woocommerce_notices();
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_plugin_notices'] ) {
            $this->disable_plugin_notices();
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_update_notices'] ) {
            $this->disable_update_notices();
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_dashboard_widgets'] ) {
            add_action( 'wp_dashboard_setup', array( $this, 'remove_dashboard_widgets' ) );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_welcome_panel'] ) {
            add_action( 'admin_init', array( $this, 'disable_welcome_panel' ) );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_admin_bar_wp_logo'] ) {
            add_action( 'admin_bar_menu', array( $this, 'remove_wp_logo' ), 999 );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_howdy'] ) {
            add_filter( 'admin_bar_menu', array( $this, 'replace_howdy' ), 25 );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_screen_options'] ) {
            add_filter( 'screen_options_show_screen', '__return_false' );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_help_tabs'] ) {
            add_action( 'admin_head', array( $this, 'remove_help_tabs' ) );
        }
        
        if ( $options['disable_all_bloat'] || $options['remove_wp_version'] ) {
            remove_action( 'wp_head', 'wp_generator' );
            add_filter( 'the_generator', '__return_empty_string' );
        }
        
        if ( $options['disable_all_bloat'] || $options['disable_file_editing'] ) {
            if ( ! defined( 'DISALLOW_FILE_EDIT' ) ) {
                define( 'DISALLOW_FILE_EDIT', true );
            }
        }
        
        if ( $options['performance_mode'] ) {
            $this->enable_performance_mode();
        }
    }

    /**
     * Get plugin options with defaults
     */
    private function get_options() {
        $options = get_option( 'anti_bloat_options', array() );
        return wp_parse_args( $options, $this->default_options );
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting( 
            'anti-bloat-settings', 
            'anti_bloat_options', 
            array( $this, 'sanitize_options' ) 
        );
        
        // General Settings Section
        add_settings_section(
            'anti-bloat-general-section',
            __( 'General Settings', 'anti-bloat' ),
            array( $this, 'general_section_callback' ),
            'anti-bloat'
        );
        
        // Notice Control Section  
        add_settings_section(
            'anti-bloat-notices-section',
            __( 'Notice Controls', 'anti-bloat' ),
            array( $this, 'notices_section_callback' ),
            'anti-bloat'
        );
        
        // Interface Cleanup Section
        add_settings_section(
            'anti-bloat-interface-section',
            __( 'Interface Cleanup', 'anti-bloat' ),
            array( $this, 'interface_section_callback' ),
            'anti-bloat'
        );
        
        // Performance Section
        add_settings_section(
            'anti-bloat-performance-section',
            __( 'Performance Settings', 'anti-bloat' ),
            array( $this, 'performance_section_callback' ),
            'anti-bloat'
        );
        
        // Add settings fields
        $this->add_settings_fields();
    }
    
    /**
     * Add all settings fields with modern descriptions
     */
    private function add_settings_fields() {
        // General settings
        add_settings_field(
            'disable_all_bloat',
            'Legacy Mode',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-general-section',
            array( 
                'key' => 'disable_all_bloat',
                'label' => 'Enable all anti-bloat features at once',
                'description' => 'Activates all cleanup features. Individual settings below will be overridden.'
            )
        );
        
        add_settings_field(
            'developer_mode',
            'Developer Mode',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-general-section',
            array( 
                'key' => 'developer_mode',
                'label' => 'Enable developer-friendly features',
                'description' => 'Shows additional debugging information and developer tools.'
            )
        );
        
        // Notice controls
        add_settings_field(
            'disable_admin_notices',
            'Admin Notices',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-notices-section',
            array( 
                'key' => 'disable_admin_notices',
                'label' => 'Hide admin notices and messages',
                'description' => 'Removes general WordPress admin notifications and alerts.'
            )
        );
        
        add_settings_field(
            'disable_woocommerce_notices',
            'WooCommerce Notices',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-notices-section',
            array( 
                'key' => 'disable_woocommerce_notices',
                'label' => 'Hide WooCommerce admin notices',
                'description' => 'Hides WooCommerce-specific notifications and setup prompts.'
            )
        );
        
        add_settings_field(
            'disable_plugin_notices',
            'Plugin Notices',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-notices-section',
            array( 
                'key' => 'disable_plugin_notices',
                'label' => 'Hide plugin activation/update notices',
                'description' => 'Removes plugin activation confirmations and update notifications.'
            )
        );
        
        add_settings_field(
            'disable_update_notices',
            'Update Notices',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-notices-section',
            array( 
                'key' => 'disable_update_notices',
                'label' => 'Hide WordPress and plugin update notices',
                'description' => 'Suppresses core WordPress and plugin update notifications.'
            )
        );
        
        // Interface cleanup
        add_settings_field(
            'disable_dashboard_widgets',
            'Dashboard Widgets',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-interface-section',
            array( 
                'key' => 'disable_dashboard_widgets',
                'label' => 'Remove default dashboard widgets',
                'description' => 'Cleans up the dashboard by removing unnecessary widget boxes.'
            )
        );
        
        add_settings_field(
            'disable_welcome_panel',
            'Welcome Panel',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-interface-section',
            array( 
                'key' => 'disable_welcome_panel',
                'label' => 'Hide the WordPress welcome panel',
                'description' => 'Removes the "Welcome to WordPress" panel from the dashboard.'
            )
        );
        
        add_settings_field(
            'disable_admin_bar_wp_logo',
            'WordPress Logo',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-interface-section',
            array( 
                'key' => 'disable_admin_bar_wp_logo',
                'label' => 'Remove WordPress logo from admin bar',
                'description' => 'Hides the WordPress logo and menu from the top admin bar.'
            )
        );
        
        add_settings_field(
            'disable_howdy',
            'Howdy Text',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-interface-section',
            array( 
                'key' => 'disable_howdy',
                'label' => 'Replace "Howdy" with "Welcome"',
                'description' => 'Changes the greeting text in the admin bar to be more professional.'
            )
        );
        
        add_settings_field(
            'disable_screen_options',
            'Screen Options',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-interface-section',
            array( 
                'key' => 'disable_screen_options',
                'label' => 'Hide screen options tab',
                'description' => 'Removes the screen options dropdown from admin pages.'
            )
        );
        
        add_settings_field(
            'disable_help_tabs',
            'Help Tabs',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-interface-section',
            array( 
                'key' => 'disable_help_tabs',
                'label' => 'Remove help tabs from admin pages',
                'description' => 'Hides the contextual help tabs throughout the admin area.'
            )
        );
        
        add_settings_field(
            'remove_wp_version',
            'WordPress Version',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-interface-section',
            array( 
                'key' => 'remove_wp_version',
                'label' => 'Remove WordPress version from source',
                'description' => 'Hides version information for improved security.'
            )
        );
        
        add_settings_field(
            'disable_file_editing',
            'File Editing',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-interface-section',
            array( 
                'key' => 'disable_file_editing',
                'label' => 'Disable file editing in admin',
                'description' => 'Prevents theme and plugin file editing from the admin panel for security.'
            )
        );
        
        // Performance settings
        add_settings_field(
            'performance_mode',
            'Performance Mode',
            array( $this, 'render_checkbox' ),
            'anti-bloat',
            'anti-bloat-performance-section',
            array( 
                'key' => 'performance_mode',
                'label' => 'Enable performance optimizations',
                'description' => 'Applies various optimizations to improve admin interface speed.'
            )
        );
    }

    /**
     * Render modern toggle switch field
     */
    public function render_checkbox( $args ) {
        $options = $this->get_options();
        $key = $args['key'];
        $label = $args['label'];
        $description = isset( $args['description'] ) ? $args['description'] : '';
        $checked = isset( $options[ $key ] ) && $options[ $key ] == 1 ? 'checked' : '';
        
        // Create modern toggle switch HTML
        echo '<div class="ab-setting-item">';
        echo '<div class="ab-setting-header">';
        echo '<div class="ab-setting-info">';
        echo '<h4 class="ab-setting-title">' . esc_html( $this->format_setting_title( $key ) ) . '</h4>';
        if ( $description ) {
            echo '<p class="ab-setting-description">' . esc_html( $description ) . '</p>';
        }
        echo '</div>';
        echo '<div class="ab-toggle-container">';
        echo '<label class="ab-toggle">';
        echo '<input type="checkbox" name="anti_bloat_options[' . esc_attr( $key ) . ']" value="1" ' . $checked . ' class="ab-toggle-input">';
        echo '<span class="ab-toggle-slider"></span>';
        echo '</label>';
        echo '</div>';
        echo '</div>';
        echo '</div>';
    }
    
    /**
     * Format setting title from key (improved version)
     */
    private function format_setting_title( $key ) {
        $titles = array(
            'disable_all_bloat' => 'Legacy Mode',
            'developer_mode' => 'Developer Mode',
            'disable_admin_notices' => 'Admin Notices',
            'disable_woocommerce_notices' => 'WooCommerce Notices',
            'disable_plugin_notices' => 'Plugin Notices',
            'disable_update_notices' => 'Update Notices',
            'disable_dashboard_widgets' => 'Dashboard Widgets',
            'disable_welcome_panel' => 'Welcome Panel',
            'disable_admin_bar_wp_logo' => 'WordPress Logo',
            'disable_howdy' => 'Howdy Text',
            'disable_screen_options' => 'Screen Options',
            'disable_help_tabs' => 'Help Tabs',
            'remove_wp_version' => 'WordPress Version',
            'disable_file_editing' => 'File Editing',
            'performance_mode' => 'Performance Mode'
        );
        
        return isset( $titles[ $key ] ) ? $titles[ $key ] : ucwords( str_replace( '_', ' ', $key ) );
    }
    
    /**
     * Section callbacks
     */
    public function general_section_callback() {
        echo '<p>' . esc_html__( 'General plugin settings and legacy options.', 'anti-bloat' ) . '</p>';
    }
    
    public function notices_section_callback() {
        echo '<p>' . esc_html__( 'Control which admin notices and messages to hide.', 'anti-bloat' ) . '</p>';
    }
    
    public function interface_section_callback() {
        echo '<p>' . esc_html__( 'Clean up the WordPress admin interface by removing unnecessary elements.', 'anti-bloat' ) . '</p>';
    }
    
    public function performance_section_callback() {
        echo '<p>' . esc_html__( 'Performance optimizations for a faster admin experience.', 'anti-bloat' ) . '</p>';
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __( 'Anti-Bloat Settings', 'anti-bloat' ),
            __( 'Anti-Bloat', 'anti-bloat' ),
            'manage_options',
            'anti-bloat-settings',
            array( $this, 'settings_page' ),
            'dashicons-shield',
            99
        );
    }
    
    /**
     * Settings page
     */
    public function settings_page() {
        include( ANTI_BLOAT_PLUGIN_DIR . 'includes/class-anti-bloat-settings.php' );
    }
    
    /**
     * Sanitize options
     */
    public function sanitize_options( $input ) {
        if ( ! is_array( $input ) ) {
            return array();
        }
        
        $sanitized = array();
        foreach ( $this->default_options as $key => $default ) {
            $sanitized[ $key ] = isset( $input[ $key ] ) ? absint( $input[ $key ] ) : 0;
        }
        
        return $sanitized;
    }
    
    /**
     * Add settings link to plugin actions
     */
    public function add_settings_link( $links ) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url( 'admin.php?page=anti-bloat-settings' ),
            __( 'Settings', 'anti-bloat' )
        );
        array_unshift( $links, $settings_link );
        return $links;
    }
    
    /**
     * Admin bar menu
     */
    public function add_admin_bar_menu( $wp_admin_bar ) {
        $options = $this->get_options();
        $is_active = $options['disable_all_bloat'] || 
                    $options['disable_admin_notices'] || 
                    $options['disable_woocommerce_notices'];
        
        if ( $is_active ) {
            $wp_admin_bar->add_menu( array(
                'id' => 'anti-bloat-status',
                'title' => __( 'Anti-Bloat Active', 'anti-bloat' ),
                'parent' => 'top-secondary',
                'meta' => array( 'class' => 'anti-bloat-active' ),
                'href' => admin_url( 'admin.php?page=anti-bloat-settings' ),
            ) );
        }
    }
    
    /**
     * Admin head actions
     */
    public function admin_head_actions() {
        $this->output_admin_css();
        $this->disable_admin_notices();
    }
    
    /**
     * WP head actions
     */
    public function wp_head_actions() {
        $options = $this->get_options();
        
        if ( $options['developer_mode'] ) {
            echo '<!-- Anti-Bloat Plugin v' . esc_attr( $this->version ) . ' Active -->' . "\n";
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts( $hook ) {
        if ( 'toplevel_page_anti-bloat-settings' === $hook ) {
            wp_enqueue_script( 'jquery' );
            wp_add_inline_script( 'jquery', $this->get_admin_js() );
            
            // Localize script for AJAX
            wp_localize_script( 'jquery', 'antiBloatAjax', array(
                'ajaxurl' => admin_url( 'admin-ajax.php' ),
                'nonce' => wp_create_nonce( 'anti_bloat_import' ),
                'strings' => array(
                    'importSuccess' => __( 'Settings imported successfully!', 'anti-bloat' ),
                    'importError' => __( 'Error importing settings.', 'anti-bloat' ),
                    'invalidFile' => __( 'Please select a valid JSON file.', 'anti-bloat' ),
                    'invalidFormat' => __( 'Invalid Anti-Bloat settings file.', 'anti-bloat' ),
                    'noSettings' => __( 'No settings found in the file.', 'anti-bloat' ),
                    'confirmImport' => __( 'This will replace your current settings. Continue?', 'anti-bloat' ),
                    'exportDate' => __( 'Export Date:', 'anti-bloat' ),
                    'version' => __( 'Version:', 'anti-bloat' ),
                    'settings' => __( 'settings', 'anti-bloat' ),
                    'dontForgetSave' => __( 'Don\'t forget to save your changes.', 'anti-bloat' ),
                    'readError' => __( 'Error reading the settings file. Please check if it\'s a valid JSON file.', 'anti-bloat' )
                )
            ) );
        }
    }
    
    /**
     * Get admin JavaScript
     */
    private function get_admin_js() {
        return "
        jQuery(document).ready(function($) {
            // Toggle all options when 'disable all bloat' is checked
            $('#anti_bloat_options_disable_all_bloat').change(function() {
                var isChecked = $(this).is(':checked');
                if (isChecked) {
                    if (confirm('" . esc_js( __( 'This will enable all anti-bloat features. Continue?', 'anti-bloat' ) ) . "')) {
                        $('input[type=\"checkbox\"]').not(this).prop('checked', false).prop('disabled', true);
                    } else {
                        $(this).prop('checked', false);
                    }
                } else {
                    $('input[type=\"checkbox\"]').prop('disabled', false);
                }
            });
            
            // Initial state
            if ($('#anti_bloat_options_disable_all_bloat').is(':checked')) {
                $('input[type=\"checkbox\"]').not('#anti_bloat_options_disable_all_bloat').prop('disabled', true);
            }
        });
        ";
    }

    /**
     * Disable admin notices
     */
    private function disable_admin_notices() {
        $options = $this->get_options();
        
        if ( $options['disable_all_bloat'] || $options['disable_admin_notices'] ) {
            remove_all_actions( 'admin_notices' );
            remove_all_actions( 'user_admin_notices' );
            remove_all_actions( 'network_admin_notices' );
            remove_all_actions( 'all_admin_notices' );
        }
    }
    
    /**
     * Disable WooCommerce notices
     */
    private function disable_woocommerce_notices() {
        if ( class_exists( 'WooCommerce' ) ) {
            add_action( 'wp_print_styles', function() {
                wp_add_inline_style( 'wp-admin', '
                    .woocommerce-message,
                    .woocommerce-error,
                    .woocommerce-info,
                    .wc-admin-notice,
                    .woocommerce-admin-notice {
                        display: none !important;
                    }
                ' );
            } );
        }
    }
    
    /**
     * Disable plugin notices
     */
    private function disable_plugin_notices() {
        add_action( 'admin_head', function() {
            echo '<style>
                .plugin-update-tr,
                .plugin-install-php,
                .update-message,
                .updating-message,
                .updated-message {
                    display: none !important;
                }
            </style>';
        } );
    }
    
    /**
     * Disable update notices
     */
    private function disable_update_notices() {
        add_action( 'admin_head', function() {
            echo '<style>
                .update-nag,
                .update-core-php,
                .update-php,
                .notice-warning.update-php {
                    display: none !important;
                }
            </style>';
        } );
        
        // Remove update notifications for non-admins
        if ( ! current_user_can( 'update_core' ) ) {
            add_action( 'init', function() {
                remove_action( 'init', 'wp_version_check' );
            }, 2 );
        }
    }
    
    /**
     * Remove dashboard widgets
     */
    public function remove_dashboard_widgets() {
        global $wp_meta_boxes;
        
        $dashboard_widgets = array(
            'dashboard_incoming_links',
            'dashboard_plugins',
            'dashboard_primary',
            'dashboard_secondary',
            'dashboard_quick_press',
            'dashboard_recent_drafts',
            'dashboard_recent_comments',
            'dashboard_right_now',
            'dashboard_activity',
            'dashboard_site_health',
            'network_dashboard_right_now',
            'wp_mail_smtp_reports_widget',
            'woocommerce_dashboard_overview',
        );
        
        foreach ( $dashboard_widgets as $widget ) {
            remove_meta_box( $widget, 'dashboard', 'normal' );
            remove_meta_box( $widget, 'dashboard', 'side' );
            remove_meta_box( $widget, 'dashboard', 'core' );
        }
    }
    
    /**
     * Disable welcome panel
     */
    public function disable_welcome_panel() {
        update_user_meta( get_current_user_id(), 'show_welcome_panel', 0 );
    }
    
    /**
     * Remove WordPress logo from admin bar
     */
    public function remove_wp_logo( $wp_admin_bar ) {
        $wp_admin_bar->remove_node( 'wp-logo' );
    }
    
    /**
     * Replace "Howdy" with "Welcome"
     */
    public function replace_howdy( $wp_admin_bar ) {
        $my_account = $wp_admin_bar->get_node( 'my-account' );
        if ( $my_account ) {
            $new_title = str_replace( 'Howdy,', __( 'Welcome,', 'anti-bloat' ), $my_account->title );
            $wp_admin_bar->add_node( array(
                'id' => 'my-account',
                'title' => $new_title,
            ) );
        }
    }
    
    /**
     * Remove help tabs
     */
    public function remove_help_tabs() {
        $screen = get_current_screen();
        if ( $screen ) {
            $screen->remove_help_tabs();
        }
    }
    
    /**
     * Enable performance mode
     */
    private function enable_performance_mode() {
        // Disable WordPress heartbeat on non-post pages
        add_action( 'init', function() {
            global $pagenow;
            if ( $pagenow !== 'post.php' && $pagenow !== 'post-new.php' ) {
                wp_deregister_script( 'heartbeat' );
            }
        } );
        
        // Reduce heartbeat frequency
        add_filter( 'heartbeat_settings', function( $settings ) {
            $settings['interval'] = 60; // 60 seconds
            return $settings;
        } );
        
        // Disable theme/plugin editor
        if ( ! defined( 'DISALLOW_FILE_EDIT' ) ) {
            define( 'DISALLOW_FILE_EDIT', true );
        }
        
        // Remove unnecessary dashboard widgets
        add_action( 'wp_dashboard_setup', array( $this, 'remove_dashboard_widgets' ) );
    }
    
    /**
     * Security enhancements
     */
    public function security_enhancements() {
        $options = $this->get_options();
        
        if ( $options['remove_wp_version'] || $options['disable_all_bloat'] ) {
            // Remove version from RSS feeds
            add_filter( 'the_generator', '__return_empty_string' );
            
            // Remove version from scripts and styles
            add_filter( 'style_loader_src', array( $this, 'remove_version_from_src' ), 9999 );
            add_filter( 'script_loader_src', array( $this, 'remove_version_from_src' ), 9999 );
        }
    }
    
    /**
     * Remove version from source URLs
     */
    public function remove_version_from_src( $src ) {
        if ( strpos( $src, 'ver=' ) ) {
            $src = remove_query_arg( 'ver', $src );
        }
        return $src;
    }
    
    /**
     * Output admin CSS
     */
    private function output_admin_css() {
        $options = $this->get_options();
        $css = '<style>';
        
        // Basic notice hiding CSS
        if ( $options['disable_all_bloat'] || $options['disable_admin_notices'] ) {
            $css .= '
                .notice, .notice-success, .notice-error, .notice-warning, .notice-info,
                .updated, .error, .update-nag, .update-message,
                .woocommerce-message, .woocommerce-error, .woocommerce-info {
                    display: none !important;
                }
            ';
        }
        
        // Admin bar indicator
        $css .= '
            #wp-admin-bar-anti-bloat-status {
                background: #dc3232 !important;
            }
            #wp-admin-bar-anti-bloat-status .ab-item {
                color: white !important;
                font-weight: bold !important;
            }
        ';
        
        // Settings page improvements
        if ( isset( $_GET['page'] ) && $_GET['page'] === 'anti-bloat-settings' ) {
            $css .= '
                .anti-bloat-settings .form-table th {
                    width: 200px;
                }
                .anti-bloat-settings .description {
                    font-style: italic;
                    color: #666;
                    margin-top: 5px;
                }
                .anti-bloat-section {
                    background: #f9f9f9;
                    border: 1px solid #e5e5e5;
                    padding: 15px;
                    margin: 20px 0;
                    border-radius: 3px;
                }
                .anti-bloat-section h3 {
                    margin-top: 0;
                    color: #333;
                }
                .import-drop-zone {
                    border: 2px dashed #ccc;
                    border-radius: 5px;
                    padding: 20px;
                    text-align: center;
                    margin: 10px 0;
                    background: #fafafa;
                    transition: all 0.3s ease;
                }
                .import-drop-zone.dragover {
                    border-color: #0073aa;
                    background: #e8f4fd;
                }
                .import-drop-zone p {
                    margin: 5px 0;
                    color: #666;
                }
            ';
        }
        
        $css .= '</style>';
        echo $css;
    }
    
    /**
     * AJAX handler for importing settings
     */
    public function ajax_import_settings() {
        // Check permissions and nonce
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have sufficient permissions.', 'anti-bloat' ) );
        }
        
        check_ajax_referer( 'anti_bloat_import', 'nonce' );
        
        if ( ! isset( $_POST['settings'] ) ) {
            wp_send_json_error( __( 'No settings data provided.', 'anti-bloat' ) );
        }
        
        $import_data = json_decode( stripslashes( $_POST['settings'] ), true );
        
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            wp_send_json_error( __( 'Invalid JSON data.', 'anti-bloat' ) );
        }
        
        // Validate import data structure
        if ( ! isset( $import_data['plugin'] ) || $import_data['plugin'] !== 'anti-bloat' ) {
            wp_send_json_error( __( 'Invalid Anti-Bloat settings file.', 'anti-bloat' ) );
        }
        
        if ( ! isset( $import_data['settings'] ) || ! is_array( $import_data['settings'] ) ) {
            wp_send_json_error( __( 'No valid settings found in the file.', 'anti-bloat' ) );
        }
        
        // Sanitize and validate each setting
        $sanitized_settings = array();
        foreach ( $import_data['settings'] as $key => $value ) {
            if ( array_key_exists( $key, $this->default_options ) ) {
                $sanitized_settings[ $key ] = absint( $value );
            }
        }
        
        // Update options
        update_option( 'anti_bloat_options', $sanitized_settings );
        
        wp_send_json_success( array(
            'message' => __( 'Settings imported successfully!', 'anti-bloat' ),
            'imported_count' => count( $sanitized_settings )
        ) );
    }
    
    /**
     * AJAX handler for validating import data
     */
    public function ajax_validate_import() {
        // Check permissions and nonce
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have sufficient permissions.', 'anti-bloat' ) );
        }
        
        check_ajax_referer( 'anti_bloat_import', 'nonce' );
        
        if ( ! isset( $_POST['settings'] ) ) {
            wp_send_json_error( __( 'No settings data provided.', 'anti-bloat' ) );
        }
        
        $import_data = json_decode( stripslashes( $_POST['settings'] ), true );
        
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            wp_send_json_error( __( 'Invalid JSON data.', 'anti-bloat' ) );
        }
        
        $validation_result = array(
            'valid' => true,
            'plugin_match' => isset( $import_data['plugin'] ) && $import_data['plugin'] === 'anti-bloat',
            'has_settings' => isset( $import_data['settings'] ) && is_array( $import_data['settings'] ),
            'export_date' => isset( $import_data['export_date'] ) ? $import_data['export_date'] : null,
            'version' => isset( $import_data['version'] ) ? $import_data['version'] : null,
            'valid_settings_count' => 0
        );
        
        if ( $validation_result['has_settings'] ) {
            foreach ( $import_data['settings'] as $key => $value ) {
                if ( array_key_exists( $key, $this->default_options ) ) {
                    $validation_result['valid_settings_count']++;
                }
            }
        }
        
        $validation_result['valid'] = $validation_result['plugin_match'] && 
                                    $validation_result['has_settings'] && 
                                    $validation_result['valid_settings_count'] > 0;
        
        wp_send_json_success( $validation_result );
    }
}
