<?php
/**
 * Modern Anti-Bloat Settings Page
 * TypeScript-inspired interface with modern UI components
 * 
 * @package AntiBloat
 * @version 1.2.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Get current options
$options = get_option( 'anti_bloat_options', array() );
$is_active = isset( $options['disable_all_bloat'] ) && $options['disable_all_bloat'];
?>

<div class="wrap anti-bloat-modern">
    <div class="ab-header">
        <div class="ab-title">
            <span class="ab-icon">🛡️</span>
            <h1>Anti-Bloat</h1>
            <span class="ab-version">v<?php echo esc_html( ANTI_BLOAT_VERSION ); ?></span>
        </div>
        
        <div class="ab-status <?php echo $is_active ? 'active' : 'inactive'; ?>">
            <span class="ab-status-dot"></span>
            <span class="ab-status-text">
                <?php echo $is_active ? __( 'Active', 'anti-bloat' ) : __( 'Inactive', 'anti-bloat' ); ?>
            </span>
        </div>
    </div>

    <div class="ab-description">
        <p><?php _e( 'Modern WordPress admin interface cleanup for developers and power users.', 'anti-bloat' ); ?></p>
    </div>

    <div class="ab-container">
        <div class="ab-sidebar">
            <nav class="ab-nav">
                <a href="#general" class="ab-nav-item active" data-tab="general">
                    <span class="ab-nav-icon">⚙️</span>
                    <span>General</span>
                </a>
                <a href="#notices" class="ab-nav-item" data-tab="notices">
                    <span class="ab-nav-icon">🔕</span>
                    <span>Notices</span>
                </a>
                <a href="#interface" class="ab-nav-item" data-tab="interface">
                    <span class="ab-nav-icon">🎨</span>
                    <span>Interface</span>
                </a>
                <a href="#performance" class="ab-nav-item" data-tab="performance">
                    <span class="ab-nav-icon">⚡</span>
                    <span>Performance</span>
                </a>
                <a href="#system" class="ab-nav-item" data-tab="system">
                    <span class="ab-nav-icon">📊</span>
                    <span>System Info</span>
                </a>
            </nav>
        </div>

        <div class="ab-content">
            <form method="post" action="options.php" class="ab-form">
                <?php settings_fields( 'anti-bloat-settings' ); ?>
                
                <!-- General Tab -->
                <div id="general" class="ab-tab-content active">
                    <div class="ab-section">
                        <h2>General Configuration</h2>
                        <p class="ab-subtitle">Core plugin settings and legacy options</p>
                        
                        <div class="ab-settings-grid">
                            <?php do_settings_fields( 'anti-bloat', 'anti-bloat-general-section' ); ?>
                        </div>
                    </div>
                </div>
                
                <!-- Notices Tab -->
                <div id="notices" class="ab-tab-content">
                    <div class="ab-section">
                        <h2>Notice Management</h2>
                        <p class="ab-subtitle">Control admin notifications and messages</p>
                        
                        <div class="ab-settings-grid">
                            <?php do_settings_fields( 'anti-bloat', 'anti-bloat-notices-section' ); ?>
                        </div>
                    </div>
                </div>
                
                <!-- Interface Tab -->
                <div id="interface" class="ab-tab-content">
                    <div class="ab-section">
                        <h2>Interface Cleanup</h2>
                        <p class="ab-subtitle">Remove unnecessary UI elements</p>
                        
                        <div class="ab-settings-grid">
                            <?php do_settings_fields( 'anti-bloat', 'anti-bloat-interface-section' ); ?>
                        </div>
                    </div>
                </div>
                
                <!-- Performance Tab -->
                <div id="performance" class="ab-tab-content">
                    <div class="ab-section">
                        <h2>Performance Optimization</h2>
                        <p class="ab-subtitle">Speed up WordPress admin experience</p>
                        
                        <div class="ab-settings-grid">
                            <?php do_settings_fields( 'anti-bloat', 'anti-bloat-performance-section' ); ?>
                        </div>
                    </div>
                </div>
                
                <!-- System Info Tab -->
                <div id="system" class="ab-tab-content">
                    <div class="ab-section">
                        <h2>System Information</h2>
                        <p class="ab-subtitle">Environment details and plugin status</p>
                        
                        <div class="ab-info-grid">
                            <div class="ab-info-card">
                                <h3>Environment</h3>
                                <div class="ab-info-item">
                                    <span class="ab-info-label">Plugin Version</span>
                                    <span class="ab-info-value"><?php echo esc_html( ANTI_BLOAT_VERSION ); ?></span>
                                </div>
                                <div class="ab-info-item">
                                    <span class="ab-info-label">WordPress</span>
                                    <span class="ab-info-value"><?php echo esc_html( get_bloginfo( 'version' ) ); ?></span>
                                </div>
                                <div class="ab-info-item">
                                    <span class="ab-info-label">PHP Version</span>
                                    <span class="ab-info-value"><?php echo esc_html( PHP_VERSION ); ?></span>
                                </div>
                                <?php if ( class_exists( 'WooCommerce' ) ): ?>
                                <div class="ab-info-item">
                                    <span class="ab-info-label">WooCommerce</span>
                                    <span class="ab-info-value"><?php echo esc_html( WC()->version ); ?></span>
                                </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="ab-info-card">
                                <h3>Active Features</h3>
                                <div class="ab-feature-list">
                                    <?php 
                                    $active_features = array();
                                    if ( $options['disable_all_bloat'] ) {
                                        echo '<div class="ab-feature active">🔒 All Bloats Disabled</div>';
                                    } else {
                                        $features = array(
                                            'disable_admin_notices' => '🔕 Admin Notices',
                                            'disable_woocommerce_notices' => '🛒 WooCommerce Notices',
                                            'disable_dashboard_widgets' => '📊 Dashboard Widgets',
                                            'performance_mode' => '⚡ Performance Mode',
                                            'developer_mode' => '👨‍💻 Developer Mode'
                                        );
                                        
                                        $has_active = false;
                                        foreach ( $features as $key => $label ) {
                                            if ( isset( $options[$key] ) && $options[$key] ) {
                                                echo '<div class="ab-feature active">' . esc_html( $label ) . '</div>';
                                                $has_active = true;
                                            }
                                        }
                                        
                                        if ( !$has_active ) {
                                            echo '<div class="ab-feature inactive">No features currently active</div>';
                                        }
                                    }
                                    ?>
                                </div>
                            </div>
                        </div>
                        
                        <div class="ab-actions">
                            <button type="button" class="ab-btn ab-btn-secondary" id="reset-settings">
                                <span>🔄</span> Reset Settings
                            </button>
                            <button type="button" class="ab-btn ab-btn-secondary" id="export-settings">
                                <span>📤</span> Export
                            </button>
                            <button type="button" class="ab-btn ab-btn-secondary" id="import-settings">
                                <span>📥</span> Import
                            </button>
                            <button type="button" class="ab-btn ab-btn-secondary" id="contact-support">
                                <span>💬</span> Contact Support
                            </button>
                            <button type="button" class="ab-btn ab-btn-secondary" id="add-review">
                                <span>⭐</span> Add Review
                            </button>
                            <input type="file" id="import-file" accept=".json" style="display: none;">
                        </div>
                    </div>
                </div>
                
                <div class="ab-footer">
                    <button type="submit" class="ab-btn ab-btn-primary">
                        <span>💾</span> Save Configuration
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
/* Modern TypeScript-inspired Anti-Bloat Settings */
:root {
    --ab-primary: #007acc;
    --ab-primary-dark: #005a9e;
    --ab-secondary: #6c757d;
    --ab-success: #28a745;
    --ab-warning: #ffc107;
    --ab-danger: #dc3545;
    --ab-dark: #1e1e1e;
    --ab-light: #f8f9fa;
    --ab-border: #e1e5e9;
    --ab-text: #2c3e50;
    --ab-text-muted: #6c757d;
    --ab-bg: #ffffff;
    --ab-bg-secondary: #f8f9fa;
    --ab-shadow: 0 2px 4px rgba(0,0,0,0.1);
    --ab-shadow-lg: 0 4px 12px rgba(0,0,0,0.15);
    --ab-radius: 8px;
    --ab-radius-sm: 4px;
    --ab-transition: all 0.3s ease;
}

.anti-bloat-modern {
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    color: var(--ab-text);
    background: var(--ab-bg-secondary);
    margin: 0 -20px -10px;
    padding: 0;
}

/* Header */
.ab-header {
    background: linear-gradient(135deg, var(--ab-primary) 0%, var(--ab-primary-dark) 100%);
    color: white;
    padding: 2rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    box-shadow: var(--ab-shadow-lg);
}

.ab-title {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.ab-title .ab-icon {
    font-size: 2rem;
}

.ab-title h1 {
    margin: 0;
    font-size: 2rem;
    font-weight: 600;
    color: white;
}

.ab-version {
    background: rgba(255,255,255,0.2);
    padding: 0.25rem 0.75rem;
    border-radius: var(--ab-radius-sm);
    font-size: 0.875rem;
    font-weight: 500;
}

.ab-status {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: var(--ab-radius);
    background: rgba(255,255,255,0.1);
    font-weight: 500;
}

.ab-status-dot {
    width: 8px;
    height: 8px;
    border-radius: 50%;
    background: var(--ab-warning);
}

.ab-status.active .ab-status-dot {
    background: var(--ab-success);
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.5; }
}

.ab-description {
    background: var(--ab-bg);
    padding: 1.5rem 2rem;
    border-bottom: 1px solid var(--ab-border);
    margin: 0;
}

.ab-description p {
    margin: 0;
    font-size: 1.1rem;
    color: var(--ab-text-muted);
}

/* Container */
.ab-container {
    display: flex;
    min-height: 600px;
    background: var(--ab-bg);
    box-shadow: var(--ab-shadow);
}

/* Sidebar */
.ab-sidebar {
    width: 240px;
    background: var(--ab-bg);
    border-right: 1px solid var(--ab-border);
    padding: 1.5rem 0;
}

.ab-nav-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.875rem 1.5rem;
    color: var(--ab-text);
    text-decoration: none;
    border: none;
    background: none;
    width: 100%;
    text-align: left;
    transition: var(--ab-transition);
    font-size: 0.95rem;
    font-weight: 500;
}

.ab-nav-item:hover {
    background: var(--ab-bg-secondary);
    color: var(--ab-primary);
}

.ab-nav-item.active {
    background: var(--ab-primary);
    color: white;
    border-right: 3px solid var(--ab-primary-dark);
    max-width: 150px;
}

.ab-nav-icon {
    font-size: 1.1rem;
    width: 20px;
    text-align: center;
}

/* Content */
.ab-content {
    flex: 1;
    padding: 2rem;
    overflow-y: auto;
}

.ab-tab-content {
    display: none;
}

.ab-tab-content.active {
    display: block;
    animation: fadeIn 0.3s ease;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

.ab-section h2 {
    margin: 0 0 0.5rem 0;
    font-size: 1.5rem;
    font-weight: 600;
    color: var(--ab-text);
}

.ab-subtitle {
    color: var(--ab-text-muted);
    margin: 0 0 2rem 0;
    font-size: 1rem;
}

/* Settings Grid */
.ab-settings-grid {
    display: grid;
    gap: 1rem;
}

/* Info Grid */
.ab-info-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.ab-info-card {
    background: var(--ab-bg-secondary);
    padding: 1.5rem;
    border-radius: var(--ab-radius);
    border: 1px solid var(--ab-border);
}

.ab-info-card h3 {
    margin: 0 0 1rem 0;
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--ab-text);
}

.ab-info-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid var(--ab-border);
}

.ab-info-item:last-child {
    border-bottom: none;
}

.ab-info-label {
    font-weight: 500;
    color: var(--ab-text-muted);
}

.ab-info-value {
    font-weight: 600;
    color: var(--ab-text);
}

.ab-feature {
    padding: 0.5rem 0.75rem;
    margin: 0.25rem 0;
    border-radius: var(--ab-radius-sm);
    font-size: 0.9rem;
    font-weight: 500;
}

.ab-feature.active {
    background: rgba(40, 167, 69, 0.1);
    color: var(--ab-success);
    border: 1px solid rgba(40, 167, 69, 0.2);
}

.ab-feature.inactive {
    background: rgba(108, 117, 125, 0.1);
    color: var(--ab-text-muted);
    border: 1px solid rgba(108, 117, 125, 0.2);
}

/* Buttons */
.ab-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: var(--ab-radius);
    font-size: 0.95rem;
    font-weight: 500;
    text-decoration: none;
    cursor: pointer;
    transition: var(--ab-transition);
    text-align: center;
}

.ab-btn-primary {
    background: var(--ab-primary);
    color: white;
    box-shadow: var(--ab-shadow);
}

.ab-btn-primary:hover {
    background: var(--ab-primary-dark);
    transform: translateY(-1px);
    box-shadow: var(--ab-shadow-lg);
}

.ab-btn-secondary {
    background: var(--ab-bg-secondary);
    color: var(--ab-text);
    border: 1px solid var(--ab-border);
}

.ab-btn-secondary:hover {
    background: var(--ab-border);
    transform: translateY(-1px);
}

.ab-actions {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
    margin-top: 2rem;
    padding-top: 2rem;
    border-top: 1px solid var(--ab-border);
}

.ab-footer {
    margin-top: 2rem;
    padding-top: 2rem;
    border-top: 1px solid var(--ab-border);
    text-align: center;
}

/* Form styling will be handled by the updated render method */

/* Responsive */
@media (max-width: 768px) {
    .ab-container {
        flex-direction: column;
    }
    
    .ab-sidebar {
        width: 100%;
        border-right: none;
        border-bottom: 1px solid var(--ab-border);
    }
    
    .ab-nav {
        display: flex;
        overflow-x: auto;
        padding: 0;
    }
    
    .ab-nav-item {
        white-space: nowrap;
        min-width: 120px;
        justify-content: center;
    }
    
    .ab-header {
        flex-direction: column;
        gap: 1rem;
        text-align: center;
    }
    
    .ab-info-grid {
        grid-template-columns: 1fr;
    }
    
    .ab-actions {
        justify-content: center;
    }
}

/* Settings Item Styles */
.ab-setting-item {
    background: var(--ab-bg);
    border: 1px solid var(--ab-border);
    border-radius: var(--ab-radius);
    padding: 1.5rem;
    margin-bottom: 1rem;
    transition: var(--ab-transition);
}

.ab-setting-item:hover {
    border-color: var(--ab-primary);
    box-shadow: var(--ab-shadow);
}

.ab-setting-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 1rem;
}

.ab-setting-info {
    flex: 1;
}

.ab-setting-title {
    margin: 0 0 0.5rem 0;
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--ab-text);
    line-height: 1.3;
}

.ab-setting-description {
    margin: 0;
    color: var(--ab-text-muted);
    font-size: 0.9rem;
    line-height: 1.4;
}

/* Modern Toggle Switch */
.ab-toggle-container {
    flex-shrink: 0;
    margin-top: 0.25rem;
}

.ab-toggle {
    position: relative;
    display: inline-block;
    width: 52px;
    height: 28px;
    cursor: pointer;
}

.ab-toggle-input {
    opacity: 0;
    width: 0;
    height: 0;
    position: absolute;
}

.ab-toggle-slider {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: #cdd4da;
    border-radius: 28px;
    transition: var(--ab-transition);
    box-shadow: inset 0 2px 4px rgba(0,0,0,0.1);
}

.ab-toggle-slider:before {
    content: "";
    position: absolute;
    height: 22px;
    width: 22px;
    left: 3px;
    top: 3px;
    background: white;
    border-radius: 50%;
    transition: var(--ab-transition);
    box-shadow: 0 2px 4px rgba(0,0,0,0.2);
}

.ab-toggle-input:checked + .ab-toggle-slider {
    background: var(--ab-primary);
}

.ab-toggle-input:checked + .ab-toggle-slider:before {
    transform: translateX(24px);
}

.ab-toggle:hover .ab-toggle-slider {
    box-shadow: inset 0 2px 4px rgba(0,0,0,0.15), 0 0 0 3px rgba(0, 122, 204, 0.1);
}

.ab-toggle-input:focus + .ab-toggle-slider {
    box-shadow: inset 0 2px 4px rgba(0,0,0,0.1), 0 0 0 3px rgba(0, 122, 204, 0.3);
}

/* Additional TypeScript-inspired enhancements */

/* Code-like font for certain elements */
.ab-version, .ab-info-value {
    font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
    font-size: 0.85rem;
}

/* Improved form styling */
.ab-form {
    position: relative;
}

/* Loading state for form */
.ab-form.loading::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(255, 255, 255, 0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: var(--ab-radius);
    z-index: 1000;
}

/* Better scrollbar for content area */
.ab-content::-webkit-scrollbar {
    width: 8px;
}

.ab-content::-webkit-scrollbar-track {
    background: var(--ab-bg-secondary);
}

.ab-content::-webkit-scrollbar-thumb {
    background: var(--ab-border);
    border-radius: 4px;
}

.ab-content::-webkit-scrollbar-thumb:hover {
    background: var(--ab-text-muted);
}

/* Enhanced section styling */
.ab-section {
    background: transparent;
    border: none;
    padding: 0;
    border-radius: 0;
}

/* Empty state styling */
.ab-empty-state {
    text-align: center;
    padding: 3rem 1rem;
    color: var(--ab-text-muted);
}

.ab-empty-state h3 {
    margin: 0 0 1rem 0;
    color: var(--ab-text);
}

/* Keyboard navigation improvements */
.ab-nav-item:focus,
.ab-toggle:focus-within,
.ab-btn:focus {
    outline: 2px solid var(--ab-primary);
    outline-offset: 2px;
}

/* Animation for status changes */
.ab-status-dot {
    transition: var(--ab-transition);
}

/* Better mobile experience */
@media (max-width: 480px) {
    .ab-header {
        padding: 1rem;
    }
    
    .ab-title h1 {
        font-size: 1.5rem;
    }
    
    .ab-content {
        padding: 1rem;
    }
    
    .ab-setting-header {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .ab-toggle-container {
        margin-top: 1rem;
        align-self: flex-end;
    }
}

/* Print styles */
@media print {
    .ab-header,
    .ab-sidebar,
    .ab-actions,
    .ab-footer {
        display: none;
    }
    
    .ab-container {
        display: block;
    }
    
    .ab-content {
        padding: 0;
    }
    
    .ab-setting-item {
        break-inside: avoid;
        margin-bottom: 1rem;
    }
}

/* High contrast mode support */
@media (prefers-contrast: high) {
    :root {
        --ab-border: #000000;
        --ab-text: #000000;
        --ab-text-muted: #333333;
    }
    
    .ab-toggle-slider {
        border: 2px solid var(--ab-text);
    }
}

/* Reduce motion for accessibility */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* Hide old WordPress admin notices on this page */
.anti-bloat-modern .notice,
.anti-bloat-modern .updated,
.anti-bloat-modern .error {
    display: none !important;
}
</style>

<script>
jQuery(document).ready(function($) {
    // Modern tab functionality
    $('.ab-nav-item').click(function(e) {
        e.preventDefault();
        
        // Remove active class from all tabs and content
        $('.ab-nav-item').removeClass('active');
        $('.ab-tab-content').removeClass('active');
        
        // Add active class to clicked tab
        $(this).addClass('active');
        
        // Show corresponding content
        var target = $(this).data('tab');
        $('#' + target).addClass('active');
    });
    
    // Reset settings with modern confirmation
    $('#reset-settings').click(function() {
        if (confirm('⚠️ Reset all settings?\n\nThis action cannot be undone and will restore all options to their default values.')) {
            $('input[type="checkbox"]').prop('checked', false);
            
            // Show toast notification
            showNotification('Settings have been reset. Remember to save your changes.', 'warning');
        }
    });
    
    // Export settings with modern JSON format
    $('#export-settings').click(function() {
        var settings = {};
        $('input[type="checkbox"]').each(function() {
            var name = $(this).attr('name');
            if (name && name.indexOf('anti_bloat_options[') === 0) {
                var key = name.replace('anti_bloat_options[', '').replace(']', '');
                settings[key] = $(this).is(':checked') ? 1 : 0;
            }
        });
        
        var exportData = {
            plugin: 'anti-bloat',
            version: '<?php echo esc_js( ANTI_BLOAT_VERSION ); ?>',
            exportDate: new Date().toISOString(),
            timestamp: Date.now(),
            settings: settings,
            meta: {
                exportedBy: 'Anti-Bloat Settings Panel',
                wpVersion: '<?php echo esc_js( get_bloginfo( 'version' ) ); ?>',
                phpVersion: '<?php echo esc_js( PHP_VERSION ); ?>'
            }
        };
        
        var dataStr = "data:text/json;charset=utf-8," + encodeURIComponent(JSON.stringify(exportData, null, 2));
        var downloadAnchorNode = document.createElement('a');
        downloadAnchorNode.setAttribute("href", dataStr);
        downloadAnchorNode.setAttribute("download", "anti-bloat-config-" + new Date().toISOString().slice(0,10) + ".json");
        document.body.appendChild(downloadAnchorNode);
        downloadAnchorNode.click();
        downloadAnchorNode.remove();
        
        showNotification('Configuration exported successfully! 📁', 'success');
    });
    
    // Import settings
    $('#import-settings').click(function() {
        $('#import-file').click();
    });
    
    $('#import-file').change(function(e) {
        var file = e.target.files[0];
        if (!file) return;
        
        if (file.type !== 'application/json') {
            showNotification('Please select a valid JSON configuration file.', 'error');
            return;
        }
        
        var reader = new FileReader();
        reader.onload = function(e) {
            try {
                var importData = JSON.parse(e.target.result);
                
                // Validate the import data
                if (!importData.plugin || importData.plugin !== 'anti-bloat') {
                    showNotification('Invalid Anti-Bloat configuration file.', 'error');
                    return;
                }
                
                if (!importData.settings) {
                    showNotification('No settings found in the configuration file.', 'error');
                    return;
                }
                
                // Modern confirmation dialog
                var confirmMessage = '📥 Import Configuration\n\n';
                confirmMessage += 'This will replace your current settings.\n';
                if (importData.exportDate) {
                    confirmMessage += 'Export Date: ' + new Date(importData.exportDate).toLocaleDateString() + '\n';
                }
                if (importData.version) {
                    confirmMessage += 'Version: ' + importData.version + '\n';
                }
                confirmMessage += '\nContinue with import?';
                
                if (!confirm(confirmMessage)) {
                    return;
                }
                
                // Apply settings
                var importedCount = 0;
                $('input[type="checkbox"]').each(function() {
                    var name = $(this).attr('name');
                    if (name && name.indexOf('anti_bloat_options[') === 0) {
                        var key = name.replace('anti_bloat_options[', '').replace(']', '');
                        if (importData.settings.hasOwnProperty(key)) {
                            $(this).prop('checked', importData.settings[key] == 1);
                            importedCount++;
                        }
                    }
                });
                
                showNotification('Configuration imported successfully! (' + importedCount + ' settings)\n\nDon\'t forget to save your changes.', 'success');
                
                // Trigger the disable_all_bloat logic if it was imported
                if (importData.settings.disable_all_bloat == 1) {
                    $('#anti_bloat_options_disable_all_bloat').trigger('change');
                }
                
            } catch (error) {
                console.error('Import error:', error);
                showNotification('Error reading the configuration file. Please check if it\'s a valid JSON file.', 'error');
            }
        };
        
        reader.readAsText(file);
        
        // Reset the file input
        $(this).val('');
    });
    
    // Contact Support button
    $('#contact-support').click(function() {
        var supportUrl = 'https://wordpress.org/support/plugin/anti-bloat/';
        var subject = 'Anti-Bloat Plugin Support Request';
        var body = 'Plugin Version: <?php echo esc_js( ANTI_BLOAT_VERSION ); ?>\n';
        body += 'WordPress Version: <?php echo esc_js( get_bloginfo( 'version' ) ); ?>\n';
        body += 'PHP Version: <?php echo esc_js( PHP_VERSION ); ?>\n\n';
        body += 'Please describe your issue or question:\n';
        
        // Try to open WordPress support forum first, fallback to email
        if (confirm('📧 Contact Support\n\nThis will open the WordPress support forum for Anti-Bloat plugin.\n\nWould you like to continue?')) {
            window.open(supportUrl, '_blank');
        }
    });
    
    // Add Review button
    $('#add-review').click(function() {
        var reviewUrl = 'https://wordpress.org/plugins/anti-bloat/#reviews';
        
        if (confirm('⭐ Add Review\n\nWe\'d love to hear your feedback!\n\nThis will open the WordPress plugin repository where you can leave a review.\n\nWould you like to continue?')) {
            window.open(reviewUrl, '_blank');
        }
    });
    
    // Modern notification system
    function showNotification(message, type = 'info') {
        // Remove existing notifications
        $('.ab-notification').remove();
        
        var icons = {
            success: '✅',
            warning: '⚠️',
            error: '❌',
            info: 'ℹ️'
        };
        
        var notification = $('<div class="ab-notification ab-notification-' + type + '">')
            .html('<span class="ab-notification-icon">' + icons[type] + '</span>' + 
                  '<span class="ab-notification-message">' + message.replace(/\n/g, '<br>') + '</span>' +
                  '<button class="ab-notification-close">×</button>');
        
        $('body').append(notification);
        
        // Auto dismiss after 5 seconds
        setTimeout(function() {
            notification.fadeOut(300, function() {
                $(this).remove();
            });
        }, 5000);
        
        // Manual dismiss
        notification.find('.ab-notification-close').click(function() {
            notification.fadeOut(300, function() {
                $(this).remove();
            });
        });
        
        // Show with animation
        notification.hide().fadeIn(300);
    }
    
    // Form submission feedback
    $('.ab-form').on('submit', function() {
        var submitBtn = $('.ab-btn-primary');
        var originalText = submitBtn.html();
        
        submitBtn.html('<span>⏳</span> Saving...');
        submitBtn.prop('disabled', true);
        
        // Reset after a moment (form will redirect anyway)
        setTimeout(function() {
            submitBtn.html(originalText);
            submitBtn.prop('disabled', false);
        }, 2000);
    });
});
</script>

<style>
/* Notification styles */
.ab-notification {
    position: fixed;
    top: 32px;
    right: 20px;
    z-index: 100000;
    max-width: 400px;
    padding: 1rem 1.5rem;
    border-radius: var(--ab-radius);
    box-shadow: var(--ab-shadow-lg);
    display: flex;
    align-items: flex-start;
    gap: 0.75rem;
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    font-size: 0.9rem;
    line-height: 1.4;
}

.ab-notification-success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.ab-notification-warning {
    background: #fff3cd;
    color: #856404;
    border: 1px solid #ffeaa7;
}

.ab-notification-error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.ab-notification-info {
    background: #d1ecf1;
    color: #0c5460;
    border: 1px solid #bee5eb;
}

.ab-notification-icon {
    font-size: 1.1rem;
    margin-top: 0.1rem;
}

.ab-notification-message {
    flex: 1;
}

.ab-notification-close {
    background: none;
    border: none;
    font-size: 1.2rem;
    cursor: pointer;
    color: inherit;
    opacity: 0.7;
    padding: 0;
    margin-left: 0.5rem;
}

.ab-notification-close:hover {
    opacity: 1;
}
</style>