<?php
/**
 * Plugin Name: Announceo – Scrolling Notification Bar
 * Description: Display a responsive notification bar with static or scrolling text.
 * Version: 1.9.0
 * Author: Inu Etc
 * Author URI: https://inuetc.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: announceo-scrolling-notification-bar
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/* SANITIZE EDITOR CONTENT */
function annoscno_sanitize_editor_content( $content ) {
    return wp_kses(
        $content,
        array(
            'a' => array(
                'href' => array(),
                'title' => array(),
                'target' => array(),
                'rel' => array(),
            ),
            'strong' => array(),
            'b' => array(),
            'em' => array(),
            'i' => array(),
            'br' => array(),
            'span' => array(),
            'p' => array(),
        )
    );
}

/* FRONTEND STYLES */
function annoscno_enqueue_frontend_assets() {
    wp_enqueue_style(
        'announceo-bar',
        plugin_dir_url(__FILE__) . 'frontend.css',
        array(),
        '1.9.1'
    );

    $link  = esc_attr( get_option('annoscno_link_color', '#ffffff') );
    $hover = esc_attr( get_option('annoscno_link_hover_color', '#dddddd') );

    wp_add_inline_style(
        'announceo-bar',
        ".annoscno-bar a{color:$link}.annoscno-bar a:hover{color:$hover}"
    );
}
add_action('wp_enqueue_scripts','annoscno_enqueue_frontend_assets');

/* ADMIN SCRIPTS */
function annoscno_enqueue_admin_assets( $hook ) {
    if ( $hook !== 'settings_page_announceo-scrolling-notification-bar' ) return;

    wp_register_script( 'annoscno-admin', '', array(), '1.9.1', true );
    wp_enqueue_script( 'annoscno-admin' );

    wp_add_inline_script( 'annoscno-admin', "
        document.addEventListener('DOMContentLoaded', function(){
            var mode = document.getElementById('annoscno_mode');
            var speedRow = document.getElementById('annoscno_speed_row');
            if(!mode || !speedRow) return;
            function toggle(){ speedRow.style.display = mode.value === 'scroll' ? 'table-row' : 'none'; }
            mode.addEventListener('change', toggle);
            toggle();
        });
    ");
}
add_action('admin_enqueue_scripts','annoscno_enqueue_admin_assets');

/* SETTINGS */
function annoscno_register_settings() {
    register_setting('annoscno_settings','annoscno_enabled',['sanitize_callback'=>'absint']);
    register_setting('annoscno_settings','annoscno_text',['sanitize_callback'=>'annoscno_sanitize_editor_content']);
    register_setting('annoscno_settings','annoscno_mode',['sanitize_callback'=>'sanitize_text_field']);
    register_setting('annoscno_settings','annoscno_speed',['sanitize_callback'=>'absint']);
    register_setting('annoscno_settings','annoscno_font_size',['sanitize_callback'=>'absint']);
    register_setting('annoscno_settings','annoscno_font_family',['sanitize_callback'=>'sanitize_text_field']);
    register_setting('annoscno_settings','annoscno_bg_color',['sanitize_callback'=>'sanitize_hex_color']);
    register_setting('annoscno_settings','annoscno_text_color',['sanitize_callback'=>'sanitize_hex_color']);
    register_setting('annoscno_settings','annoscno_link_color',['sanitize_callback'=>'sanitize_hex_color']);
    register_setting('annoscno_settings','annoscno_link_hover_color',['sanitize_callback'=>'sanitize_hex_color']);
}
add_action('admin_init','annoscno_register_settings');

/* MENU */
function annoscno_register_menu() {
    add_options_page(
        'Scrolling Notification Bar',
        'Scrolling Notification Bar',
        'manage_options',
        'announceo-scrolling-notification-bar',
        'annoscno_settings_page'
    );
}
add_action('admin_menu','annoscno_register_menu');

/* SETTINGS PAGE */
function annoscno_settings_page() {
    $mode = get_option('annoscno_mode','scroll');
?>
<div class="wrap">
<h1>Scrolling Notification Bar</h1>
<form method="post" action="options.php">
<?php settings_fields('annoscno_settings'); ?>
<table class="form-table">

<tr>
<th>Enable</th>
<td><input type="checkbox" name="annoscno_enabled" value="1" <?php checked(1,get_option('annoscno_enabled',1)); ?>></td>
</tr>

<tr>
<th>Notification Text</th>
<td><?php
    wp_editor(
        get_option('annoscno_text'),
        'annoscno_text',
        array(
            'textarea_name' => 'annoscno_text',
            'teeny' => true,
            'media_buttons' => false
        )
    );
?></td>
</tr>

<tr>
<th>Mode</th>
<td>
<select name="annoscno_mode" id="annoscno_mode">
<option value="static" <?php selected($mode,'static'); ?>>Static</option>
<option value="scroll" <?php selected($mode,'scroll'); ?>>Scroll</option>
</select>
</td>
</tr>

<tr id="annoscno_speed_row">
<th>Scroll Speed (seconds)</th>
<td><input type="range" min="5" max="30" name="annoscno_speed" value="<?php echo esc_attr(get_option('annoscno_speed',15)); ?>"></td>
</tr>

<tr>
<th>Font Size (px)</th>
<td><input type="range" min="10" max="30" name="annoscno_font_size" value="<?php echo esc_attr(get_option('annoscno_font_size',14)); ?>"></td>
</tr>

<tr>
<th>Font Family</th>
<td>
<select name="annoscno_font_family">
<option value="inherit">Inherit</option>
<option value="Arial">Arial</option>
<option value="Georgia">Georgia</option>
<option value="Poppins">Poppins</option>
</select>
</td>
</tr>

<tr>
<th>Background Color</th>
<td><input type="color" name="annoscno_bg_color" value="<?php echo esc_attr(get_option('annoscno_bg_color','#8b0000')); ?>"></td>
</tr>

<tr>
<th>Text Color</th>
<td><input type="color" name="annoscno_text_color" value="<?php echo esc_attr(get_option('annoscno_text_color','#ffffff')); ?>"></td>
</tr>

<tr>
<th>Link Color</th>
<td><input type="color" name="annoscno_link_color" value="<?php echo esc_attr(get_option('annoscno_link_color','#ffffff')); ?>"></td>
</tr>

<tr>
<th>Link Hover Color</th>
<td><input type="color" name="annoscno_link_hover_color" value="<?php echo esc_attr(get_option('annoscno_link_hover_color','#dddddd')); ?>"></td>
</tr>

</table>
<?php submit_button(); ?>
</form>

<hr>
<p>
If you enjoy using Scrolling Notification Bar, please consider leaving a review to help it reach more people.
You can also support me on <a href="https://buymeacoffee.com/inuetc" target="_blank" rel="noopener">Buy Me a Coffee</a>
or visit my website to learn more about me, hire me, or collaborate.
Your support keeps me inspired to build useful tools for the WordPress community.
</p>
<p>
☕ <a href="https://buymeacoffee.com/inuetc" target="_blank" rel="noopener">Buy Me a Coffee</a><br>
🌐 <a href="https://inuetc.com/" target="_blank" rel="noopener">https://inuetc.com/</a>
</p>
</div>
<?php }

/* FRONTEND OUTPUT */
function annoscno_render_bar() {
    if ( ! get_option('annoscno_enabled',1) ) return;

    $text = get_option('annoscno_text');
    if ( empty($text) ) return;

    $mode = get_option('annoscno_mode','scroll');
?>
<div class="annoscno-bar" style="
background:<?php echo esc_attr(get_option('annoscno_bg_color','#8b0000')); ?>;
color:<?php echo esc_attr(get_option('annoscno_text_color','#ffffff')); ?>;
font-size:<?php echo esc_attr(get_option('annoscno_font_size',14)); ?>px;
font-family:<?php echo esc_attr(get_option('annoscno_font_family','inherit')); ?>;">
<?php if ( $mode === 'scroll' ): ?>
<div class="annoscno-marquee">
<span style="animation-duration:<?php echo esc_attr(get_option('annoscno_speed',15)); ?>s">
<?php echo wp_kses_post($text); ?>
</span>
</div>
<?php else: ?>
<div class="annoscno-static"><?php echo wp_kses_post($text); ?></div>
<?php endif; ?>
</div>
<?php }
add_action('wp_body_open','annoscno_render_bar');
