<?php
/**
 * AnchorKit TOC Widget
 *
 * Provides a WordPress widget to display the Table of Contents in sidebars and widget areas.
 *
 * @package AnchorKit_TOC
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * AnchorKit Table of Contents Widget.
 */
class AnchorKit_TOC_Widget extends WP_Widget {

	/**
	 * Constructor.
	 */
	public function __construct() {
		parent::__construct(
			'anchorkit_toc_widget',
			__( 'AnchorKit TOC', 'anchorkit-table-of-contents' ),
			array(
				'description' => __( 'Display a Table of Contents in your sidebar', 'anchorkit-table-of-contents' ),
				'classname'   => 'anchorkit-toc-widget',
			)
		);
	}

	/**
	 * Widget output on the frontend.
	 *
	 * @param array $args     Widget arguments.
	 * @param array $instance Widget instance settings.
	 * @return void
	 */
	public function widget( $args, $instance ) {
		// Only display on singular posts/pages.
		if ( ! is_singular() ) {
			return;
		}

		// Check if TOC is enabled globally.
		if ( ! anchorkit_get_option( 'anchorkit_toc_enabled', false ) ) {
			return;
		}

		// Check if current post type is enabled.
		$post_types = anchorkit_get_option( 'anchorkit_toc_post_types', array( 'post', 'page' ) );
		if ( ! in_array( get_post_type(), $post_types, true ) ) {
			return;
		}

		// Get widget settings.
		$title                      = ! empty( $instance['title'] ) ? $instance['title'] : '';
		$show_for_current_post_only = isset( $instance['show_for_current_post_only'] ) ? (bool) $instance['show_for_current_post_only'] : true;

		// Get the current post content.
		global $post;
		if ( ! $post ) {
			return;
		}

		$content = $post->post_content;

		// Apply content filters to ensure we get the same content as the main loop.
		$content = apply_filters( 'the_content', $content ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core hook.

		// Parse headings from content.
		if ( ! function_exists( 'anchorkit_toc_parse_headings' ) ) {
			return;
		}

		$headings = anchorkit_toc_parse_headings( $content );

		// Check minimum headings threshold.
		$min_headings = (int) anchorkit_get_option( 'anchorkit_toc_min_headings', 2 );
		if ( count( $headings ) < $min_headings ) {
			return;
		}

		// Output widget.
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Widget args are pre-escaped by WordPress core.
		echo $args['before_widget'];

		if ( ! empty( $title ) ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Widget args (before/after_title) are pre-escaped by WordPress core.
			echo $args['before_title'] . esc_html( $title ) . $args['after_title'];
		}

		// Generate and display TOC.
		if ( function_exists( 'anchorkit_toc_generate_html' ) ) {
			// Add a class to identify this is a widget TOC.
			add_filter(
				'anchorkit_toc_container_class',
				function ( $classes ) {
					$classes[] = 'anchorkit-toc-widget-container';
					return $classes;
				}
			);

			$widget_context = array(
				'source'  => 'widget',
				'post_id' => ( $post && isset( $post->ID ) ) ? (int) $post->ID : 0,
				'data'    => array(
					'widget_args'     => $args,
					'widget_instance' => $instance,
					'widget_id'       => $this->id,
				),
			);

			// We use wp_kses() for late escaping, allowing our TOC-specific tags and attributes.
			echo wp_kses( anchorkit_toc_generate_html( $headings, $content, $widget_context ), anchorkit_get_allowed_html_tags() );

			// Remove the filter after use.
			remove_all_filters( 'anchorkit_toc_container_class' );
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Widget args are pre-escaped by WordPress core.
		echo $args['after_widget'];
	}

	/**
	 * Widget settings form in admin.
	 *
	 * @param array $instance Widget instance settings.
	 * @return void
	 */
	public function form( $instance ) {
		$title                      = isset( $instance['title'] ) ? $instance['title'] : __( 'Table of Contents', 'anchorkit-table-of-contents' );
		$show_for_current_post_only = isset( $instance['show_for_current_post_only'] ) ? (bool) $instance['show_for_current_post_only'] : true;
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>">
				<?php esc_html_e( 'Title:', 'anchorkit-table-of-contents' ); ?>
			</label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text"
				value="<?php echo esc_attr( $title ); ?>">
			<small class="description" style="display: block; margin-top: 5px;">
				<?php esc_html_e( 'Leave empty to use the title from plugin settings.', 'anchorkit-table-of-contents' ); ?>
			</small>
		</p>

		<p>
			<input class="checkbox" type="checkbox" <?php checked( $show_for_current_post_only ); ?>
				id="<?php echo esc_attr( $this->get_field_id( 'show_for_current_post_only' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'show_for_current_post_only' ) ); ?>" value="1">
			<label for="<?php echo esc_attr( $this->get_field_id( 'show_for_current_post_only' ) ); ?>">
				<?php esc_html_e( 'Show only for current post/page', 'anchorkit-table-of-contents' ); ?>
			</label>
			<small class="description" style="display: block; margin-top: 5px;">
				<?php esc_html_e( 'When checked, the TOC will only display on singular post/page views.', 'anchorkit-table-of-contents' ); ?>
			</small>
		</p>

		<p class="description" style="padding: 10px; background: #f0f0f1; border-left: 3px solid #2271b1;">
			<strong><?php esc_html_e( 'Note:', 'anchorkit-table-of-contents' ); ?></strong>
			<?php esc_html_e( 'This widget uses the TOC settings configured in the AnchorKit settings page. To customize appearance, visit Settings > AnchorKit.', 'anchorkit-table-of-contents' ); ?>
		</p>
		<?php
	}

	/**
	 * Update widget settings.
	 *
	 * @param array $new_instance New widget instance settings.
	 * @param array $old_instance Old widget instance settings.
	 * @return array Updated instance settings.
	 */
	public function update( $new_instance, $old_instance ) {
		$instance                               = array();
		$instance['title']                      = ! empty( $new_instance['title'] ) ? sanitize_text_field( $new_instance['title'] ) : '';
		$instance['show_for_current_post_only'] = isset( $new_instance['show_for_current_post_only'] ) ? 1 : 0;
		return $instance;
	}
}

/**
 * Register the widget.
 *
 * @return void
 */
function anchorkit_register_widgets() {
	register_widget( 'AnchorKit_TOC_Widget' );
}
add_action( 'widgets_init', 'anchorkit_register_widgets' );
