<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// includes/settings.php

/**
 * Register all plugin settings with WordPress Settings API
 */
function anchorkit_register_settings() {
	// Get all TOC default settings
	$defaults = function_exists( 'anchorkit_get_toc_default_settings' )
		? anchorkit_get_toc_default_settings()
		: array();

	// Define all option names that should be registered
	$option_names = array(
		// Core TOC settings
		'anchorkit_toc_enabled',
		'anchorkit_toc_automatic_insertion',
		'anchorkit_toc_position',
		'anchorkit_toc_post_types',
		'anchorkit_toc_min_headings',
		'anchorkit_toc_include_headings',
		'anchorkit_toc_exclude_selectors',
		'anchorkit_toc_exclude_keywords',
		'anchorkit_toc_exclude_regex',
		'anchorkit_toc_min_heading_depth',
		'anchorkit_toc_max_heading_depth',

		// Appearance settings
		'anchorkit_toc_show_label',
		'anchorkit_toc_title_text',
		'anchorkit_toc_style_preset',
		'anchorkit_toc_theme',
		'anchorkit_toc_custom_styling',
		'anchorkit_toc_hierarchical_view',
		'anchorkit_toc_collapsible',
		'anchorkit_toc_initial_state',
		'anchorkit_toc_bullet_style',
		'anchorkit_toc_bullet_character',
		'anchorkit_toc_show_numerals',
		'anchorkit_toc_numbering_style',
		'anchorkit_toc_numbering_format',
		'anchorkit_toc_numbering_separator',

		// Behavior settings
		'anchorkit_toc_smooth_scroll',
		'anchorkit_toc_scroll_offset',
		'anchorkit_toc_scroll_easing',
		'anchorkit_toc_scroll_duration',
		'anchorkit_toc_hide_on_mobile',
		'anchorkit_toc_mobile_breakpoint',
		'anchorkit_toc_width',
		'anchorkit_toc_alignment',
		'anchorkit_toc_float',

		// Pro features
		'anchorkit_toc_sticky_enabled',
		'anchorkit_toc_sticky_position',
		'anchorkit_toc_sticky_offset',
		'anchorkit_toc_scroll_spy',
		'anchorkit_toc_back_to_top_link',
		'anchorkit_toc_back_to_top_text',
		'anchorkit_toc_back_to_top_font_size',
		'anchorkit_toc_view_more_enabled',
		'anchorkit_toc_initial_count',
		'anchorkit_toc_view_more_text',
		'anchorkit_toc_view_less_text',
		'anchorkit_toc_entrance_animation',
		'anchorkit_toc_animation_type',

		// Typography
		'anchorkit_toc_font_family',
		'anchorkit_toc_font_size',
		'anchorkit_toc_line_height',
		'anchorkit_toc_letter_spacing',
		'anchorkit_toc_text_transform',
		'anchorkit_toc_link_underline',
		'anchorkit_toc_advanced_typography_override',
		'anchorkit_toc_title_font_size',
		'anchorkit_toc_h2_font_size',
		'anchorkit_toc_h3_font_size',
		'anchorkit_toc_h4_font_size',
		'anchorkit_toc_h5_font_size',
		'anchorkit_toc_h6_font_size',

		// Layout
		'anchorkit_toc_padding',
		'anchorkit_toc_border_width',
		'anchorkit_toc_border_radius',

		// Colors - Light mode
		'anchorkit_toc_bg_color_light',
		'anchorkit_toc_text_color_light',
		'anchorkit_toc_link_color_light',
		'anchorkit_toc_link_hover_color_light',
		'anchorkit_toc_border_color_light',
		'anchorkit_toc_bullet_color_light',
		'anchorkit_toc_active_link_color_light',

		// Colors - Dark mode
		'anchorkit_toc_bg_color_dark',
		'anchorkit_toc_text_color_dark',
		'anchorkit_toc_link_color_dark',
		'anchorkit_toc_link_hover_color_dark',
		'anchorkit_toc_border_color_dark',
		'anchorkit_toc_bullet_color_dark',
		'anchorkit_toc_active_link_color_dark',

		// Shadow settings
		'anchorkit_toc_box_shadow_enabled',
		'anchorkit_toc_box_shadow_custom',
		'anchorkit_toc_box_shadow_light',
		'anchorkit_toc_box_shadow_dark',
		'anchorkit_toc_shadow_h_offset_light',
		'anchorkit_toc_shadow_v_offset_light',
		'anchorkit_toc_shadow_blur_light',
		'anchorkit_toc_shadow_spread_light',
		'anchorkit_toc_shadow_color_light',
		'anchorkit_toc_shadow_opacity_light',
		'anchorkit_toc_shadow_h_offset_dark',
		'anchorkit_toc_shadow_v_offset_dark',
		'anchorkit_toc_shadow_blur_dark',
		'anchorkit_toc_shadow_spread_dark',
		'anchorkit_toc_shadow_color_dark',
		'anchorkit_toc_shadow_opacity_dark',

		// Advanced settings
		'anchorkit_toc_anchor_format',
		'anchorkit_toc_anchor_prefix',
		'anchorkit_toc_custom_class',
		'anchorkit_toc_custom_labels',
		'anchorkit_toc_custom_css',
		'anchorkit_toc_aria_label',
		'anchorkit_toc_show_reading_time',
		'anchorkit_toc_show_word_count',
		'anchorkit_toc_reading_speed',
		'anchorkit_toc_time_format',

		// ACF Integration
		'anchorkit_toc_acf_enabled',
		'anchorkit_toc_acf_field_names',
		'anchorkit_toc_acf_merge_mode',

		// Schema & SEO
		'anchorkit_toc_schema_enabled',
		'anchorkit_toc_schema_type',
		'anchorkit_toc_schema_type_per_post_type',
		'anchorkit_toc_amp_enabled',

		// Preview
		'anchorkit_toc_live_preview_enabled',
	);

	// Register each setting
	foreach ( $option_names as $option_name ) {
		$args = array(
			'type'              => 'string', // Default to string
			'sanitize_callback' => 'anchorkit_sanitize_registered_setting',
			'default'           => null,
			'show_in_rest'      => false,
		);

		// Explicitly set type to array for array options
		if (
			in_array(
				$option_name,
				array(
					'anchorkit_toc_post_types',
					'anchorkit_toc_include_headings',
					'anchorkit_toc_schema_type_per_post_type',
				),
				true
			)
		) {
			$args['type'] = 'array';
		}

		register_setting(
			'anchorkit_settings', // Option group
			$option_name,
			$args
		);
	}
}
add_action( 'admin_init', 'anchorkit_register_settings' );

/**
 * Sanitization callback for registered settings
 *
 * @param mixed $value The setting value
 * @return mixed Sanitized value
 */
function anchorkit_sanitize_registered_setting( $value ) {
	// Get the option name from the current filter
	$option_name = str_replace( 'sanitize_option_', '', current_filter() );

	// Use our existing comprehensive sanitization function
	if ( function_exists( 'anchorkit_sanitize_option_input' ) ) {
		return anchorkit_sanitize_option_input( $option_name, $value );
	}

	// Fallback sanitization
	if ( is_array( $value ) ) {
		return array_map( 'sanitize_text_field', $value );
	}

	return sanitize_text_field( $value );
}

add_action( 'admin_menu', 'anchorkit_settings_menu' );
function anchorkit_settings_menu() {
	add_options_page(
		'AnchorKit Settings',
		'AnchorKit',
		'manage_options',
		'anchorkit-settings',
		'anchorkit_settings_page'
	);
}

function anchorkit_settings_page() {
	// Sanitize and validate the tab parameter
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab parameter is for display only, not state-changing
	$active_tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'quick-start';

	// Define tabs with their colors
	$tabs = array(
		'quick-start'            => array(
			'name'  => 'Quick Start',
			'title' => 'Set up your table of contents in seconds',
			'icon'  => 'quick-start-icon.svg',
			'color' => '#3f51b5', // Deep Blue
		),
		'appearance-behavior'    => array(
			'name'  => 'Appearance & Behavior',
			'title' => 'Style and customize your table of contents',
			'icon'  => 'appearance-and-settings-icon.svg',
			'color' => '#5c6bc0', // Indigo
		),
		'structure-seo'          => array(
			'name'  => 'Structure & SEO',
			'title' => 'Structure your table of contents and optimize for SEO',
			'icon'  => 'structure-and-seo-icon.svg',
			'color' => '#7b1fa2', // Deep Purple
		),
		'advanced-accessibility' => array(
			'name'  => 'Advanced & Accessibility',
			'title' => 'Control advanced settings and ensure maximum accessibility',
			'icon'  => 'advanced-and-accessibility-icon.svg',
			'color' => '#8e24aa', // Purple
		),
		'help'                   => array(
			'name'  => 'Help',
			'title' => 'Useful info and documentation',
			'icon'  => 'help-icon.svg',
			'color' => '#673ab7', // Blue Purple
		),
		'anchorkit-pro'          => array(
			'name'  => 'AnchorKit Pro',
			'title' => 'Unlock the full features by upgrading to AnchorKit Pro',
			'icon'  => 'pro-icon.svg',
			'color' => '#9c27b0', // Violet
		),
	);

	// Validate that the active tab exists in our allowed tabs
	if ( ! array_key_exists( $active_tab, $tabs ) ) {
		$active_tab = 'quick-start';
	}
	?>
	<div class="wrap anchorkit-admin" data-active-tab="<?php echo esc_attr( $active_tab ); ?>">
		<div class="anchorkit-header-section">
			<?php
			anchorkit_render_branding(); // Renders logo and branding
			?>

			<!-- Multi-Tab Navigation -->
			<h2 class="nav-tab-wrapper anchorkit-tabs">
				<?php foreach ( $tabs as $tab_key => $tab_data ) : ?>
					<a href="?page=anchorkit-settings&tab=<?php echo esc_attr( $tab_key ); ?>"
						class="nav-tab <?php echo esc_attr( $active_tab === $tab_key ? 'nav-tab-active' : '' ); ?>">
						<img src="<?php echo esc_url( plugins_url( 'assets/tab_icons/' . $tab_data['icon'], ANCHORKIT_PLUGIN_FILE ) ); ?>"
							alt="<?php echo esc_attr( $tab_data['name'] ); ?> Icon">
						<?php echo esc_html( $tab_data['name'] ); ?>
					</a>
				<?php endforeach; ?>
			</h2>

			<!-- Mobile Dropdown -->
			<div class="anchorkit-tabs-dropdown">
				<button class="anchorkit-tabs-dropdown-button" data-active-tab="<?php echo esc_attr( $active_tab ); ?>">
					<span class="dropdown-tab-text">
						<img src="<?php echo esc_url( plugins_url( 'assets/tab_icons/' . $tabs[ $active_tab ]['icon'], ANCHORKIT_PLUGIN_FILE ) ); ?>"
							alt="<?php echo esc_attr( $tabs[ $active_tab ]['name'] ); ?> Icon">
						<?php echo esc_html( $tabs[ $active_tab ]['name'] ); ?>
					</span>
					<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
						<polyline points="6,9 12,15 18,9"></polyline>
					</svg>
				</button>
				<ul class="anchorkit-tabs-dropdown-menu">
					<?php foreach ( $tabs as $tab_key => $tab_data ) : ?>
						<li>
							<a href="?page=anchorkit-settings&tab=<?php echo esc_attr( $tab_key ); ?>"
								class="<?php echo esc_attr( $active_tab === $tab_key ? 'active' : '' ); ?>">
								<img src="<?php echo esc_url( plugins_url( 'assets/tab_icons/' . $tab_data['icon'], ANCHORKIT_PLUGIN_FILE ) ); ?>"
									alt="<?php echo esc_attr( $tab_data['name'] ); ?> Icon">
								<?php echo esc_html( $tab_data['name'] ); ?>
							</a>
						</li>
					<?php endforeach; ?>
				</ul>
			</div>
		</div>

		<?php
		$toc_tabs         = array( 'quick-start', 'appearance-behavior', 'structure-seo', 'advanced-accessibility' );
		$show_toc_preview = in_array( $active_tab, $toc_tabs, true );
		$current_theme    = $show_toc_preview ? anchorkit_get_option( 'anchorkit_toc_theme', 'system' ) : 'system';
		?>

		<!-- Two-column layout: Settings + Live Preview -->
		<div class="anchorkit-layout-shell <?php echo esc_attr( $show_toc_preview ? 'has-preview' : '' ); ?>">
			<!-- Left: Settings Content -->
			<div class="anchorkit-layout-container">
				<div class="anchorkit-tab-content">
					<!-- Tab Title -->
					<h1><?php echo esc_html( $tabs[ $active_tab ]['title'] ); ?></h1>

					<?php do_action( 'anchorkit_settings_tabs', $active_tab ); ?>
				</div>
			</div>

			<!-- Right: Live Preview Sidebar (Only visible on TOC tabs) -->
			<?php if ( $show_toc_preview ) : ?>
				<aside class="anchorkit-preview-sidebar" id="anchorkit-preview-sidebar">
					<div class="anchorkit-preview-sidebar-inner">
						<div class="anchorkit-preview-header">
							<div class="anchorkit-preview-header-left">
								<h3><?php esc_html_e( 'Live Preview', 'anchorkit-table-of-contents' ); ?></h3>
								<!-- Always render the mode indicator, let JS control visibility -->
								<span
									class="anchorkit-preview-mode-indicator<?php echo ( $current_theme !== 'system' ) ? ' is-hidden' : ''; ?>"
									style="<?php echo ( $current_theme !== 'system' ) ? 'display: none;' : ''; ?>">
									<span class="mode-text"><?php esc_html_e( 'Light', 'anchorkit-table-of-contents' ); ?></span>
								</span>
							</div>
							<div class="anchorkit-preview-header-right">
								<!-- Always render the theme toggle, let JS control visibility -->
								<button
									class="anchorkit-theme-test-toggle<?php echo ( $current_theme !== 'system' ) ? ' is-hidden' : ''; ?>"
									aria-label="<?php esc_attr_e( 'Test dark mode preview', 'anchorkit-table-of-contents' ); ?>"
									data-test-mode="light" title="<?php esc_attr_e( 'Test Dark Mode', 'anchorkit-table-of-contents' ); ?>"
									style="<?php echo ( $current_theme !== 'system' ) ? 'display: none;' : ''; ?>">
									<svg class="theme-icon sun-icon" width="18" height="18" viewBox="0 0 24 24" fill="none"
										stroke="currentColor" stroke-width="2">
										<circle cx="12" cy="12" r="5"></circle>
										<line x1="12" y1="1" x2="12" y2="3"></line>
										<line x1="12" y1="21" x2="12" y2="23"></line>
										<line x1="4.22" y1="4.22" x2="5.64" y2="5.64"></line>
										<line x1="18.36" y1="18.36" x2="19.78" y2="19.78"></line>
										<line x1="1" y1="12" x2="3" y2="12"></line>
										<line x1="21" y1="12" x2="23" y2="12"></line>
										<line x1="4.22" y1="19.78" x2="5.64" y2="18.36"></line>
										<line x1="18.36" y1="5.64" x2="19.78" y2="4.22"></line>
									</svg>
									<svg class="theme-icon moon-icon" width="18" height="18" viewBox="0 0 24 24" fill="none"
										stroke="currentColor" stroke-width="2" style="display: none;">
										<path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"></path>
									</svg>
								</button>
								<button class="anchorkit-preview-toggle"
									aria-label="<?php esc_attr_e( 'Toggle preview', 'anchorkit-table-of-contents' ); ?>" aria-expanded="true">
									<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor"
										stroke-width="2">
										<polyline points="18 15 12 9 6 15"></polyline>
									</svg>
								</button>
							</div>
						</div>
						<div class="anchorkit-preview-content" id="anchorkit-sidebar-toc-preview">
							<!-- TOC Preview will be rendered here -->
							<div class="anchorkit-preview-loading">
								<p><?php esc_html_e( 'Adjust settings to see live preview', 'anchorkit-table-of-contents' ); ?></p>
							</div>
						</div>
					</div>
				</aside>
			<?php endif; ?>
		</div>
	</div>
	<?php
}
