<?php

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Get CSS variables array for TOC styling (single source of truth)
 * This function extracts the CSS variable logic from anchorkit_toc_inline_styles()
 * so it can be used both for frontend styles and AJAX preview responses
 *
 * @return array Associative array of CSS variable names => values
 */
if (!function_exists('anchorkit_collect_toc_tokens')) {
	function anchorkit_collect_toc_tokens()
	{
		// Check if custom styling is enabled - explicitly convert to boolean
		$custom_styling_raw = anchorkit_get_option('anchorkit_toc_custom_styling', false);
		$custom_styling = filter_var($custom_styling_raw, FILTER_VALIDATE_BOOLEAN);
		// PRO: Advanced Typography - Apply to ALL TOC containers (presets and custom)
		if (anchorkit_fs() && anchorkit_fs()->is__premium_only()) {
			if (anchorkit_fs()->can_use_premium_code()) {
				$advanced_typography_override = anchorkit_get_option('anchorkit_toc_advanced_typography_override', false);

				if ($advanced_typography_override) {
					$title_font_size = anchorkit_get_option('anchorkit_toc_title_font_size', '');
					$h2_font_size = anchorkit_get_option('anchorkit_toc_h2_font_size', '');
					$h3_font_size = anchorkit_get_option('anchorkit_toc_h3_font_size', '');
					$h4_font_size = anchorkit_get_option('anchorkit_toc_h4_font_size', '');
					$h5_font_size = anchorkit_get_option('anchorkit_toc_h5_font_size', '');
					$h6_font_size = anchorkit_get_option('anchorkit_toc_h6_font_size', '');

					if (!empty($title_font_size) && is_numeric($title_font_size)) {
						$tokens['light']['--anchorkit-toc-title-font-size'] = intval($title_font_size) . 'px';
						$tokens['dark']['--anchorkit-toc-title-font-size'] = intval($title_font_size) . 'px';
					}
					if (!empty($h2_font_size) && is_numeric($h2_font_size)) {
						$tokens['light']['--anchorkit-toc-h2-font-size'] = intval($h2_font_size) . 'px';
						$tokens['dark']['--anchorkit-toc-h2-font-size'] = intval($h2_font_size) . 'px';
					}
					if (!empty($h3_font_size) && is_numeric($h3_font_size)) {
						$tokens['light']['--anchorkit-toc-h3-font-size'] = intval($h3_font_size) . 'px';
						$tokens['dark']['--anchorkit-toc-h3-font-size'] = intval($h3_font_size) . 'px';
					}
					if (!empty($h4_font_size) && is_numeric($h4_font_size)) {
						$tokens['light']['--anchorkit-toc-h4-font-size'] = intval($h4_font_size) . 'px';
						$tokens['dark']['--anchorkit-toc-h4-font-size'] = intval($h4_font_size) . 'px';
					}
					if (!empty($h5_font_size) && is_numeric($h5_font_size)) {
						$tokens['light']['--anchorkit-toc-h5-font-size'] = intval($h5_font_size) . 'px';
						$tokens['dark']['--anchorkit-toc-h5-font-size'] = intval($h5_font_size) . 'px';
					}
					if (!empty($h6_font_size) && is_numeric($h6_font_size)) {
						$tokens['light']['--anchorkit-toc-h6-font-size'] = intval($h6_font_size) . 'px';
						$tokens['dark']['--anchorkit-toc-h6-font-size'] = intval($h6_font_size) . 'px';
					}
				}
			}
		}
		$theme = anchorkit_get_option('anchorkit_toc_theme', 'system');
		$style_preset = anchorkit_get_option('anchorkit_toc_style_preset', 'minimal');

		$shared = array(
			'--anchorkit-toc-padding' => (int) anchorkit_get_option('anchorkit_toc_padding', 15) . 'px',
			'--anchorkit-toc-border-width' => (int) anchorkit_get_option('anchorkit_toc_border_width', 1) . 'px',
			'--anchorkit-toc-border-radius' => (int) anchorkit_get_option('anchorkit_toc_border_radius', 4) . 'px',
			'--anchorkit-toc-font-size' => (int) anchorkit_get_option('anchorkit_toc_font_size', 14) . 'px',
			'--anchorkit-toc-font-family' => esc_attr(anchorkit_get_option('anchorkit_toc_font_family', 'inherit')),
			'--anchorkit-toc-bullet-gap' => '12px',
		);

		$tokens = array(
			'light' => array_merge(
				$shared,
				array(
					'--anchorkit-toc-bg' => '#ffffff',
					'--anchorkit-toc-text-color' => '#333333',
					'--anchorkit-toc-link-color' => '#0073AA',
					'--anchorkit-toc-link-hover-color' => '#005177',
					'--anchorkit-toc-active-link-color' => '#00A0D2',
					'--anchorkit-toc-border-color' => '#DDDDDD',
					'--anchorkit-toc-hover-bg' => 'rgba(0,0,0,0.04)',
					'--anchorkit-toc-focus-bg' => 'rgba(0,0,0,0.08)',
					'--anchorkit-toc-sublevel-bg' => 'transparent',
					'--anchorkit-toc-bullet-color' => '#0073AA',
				)
			),
			'dark' => array_merge(
				$shared,
				array(
					'--anchorkit-toc-bg' => '#1e1e1e',
					'--anchorkit-toc-text-color' => '#e5e5e5',
					'--anchorkit-toc-link-color' => '#59a9ff',
					'--anchorkit-toc-link-hover-color' => '#a3c8ff',
					'--anchorkit-toc-active-link-color' => '#ffffff',
					'--anchorkit-toc-border-color' => '#444444',
					'--anchorkit-toc-hover-bg' => 'rgba(255,255,255,0.08)',
					'--anchorkit-toc-focus-bg' => 'rgba(255,255,255,0.12)',
					'--anchorkit-toc-sublevel-bg' => 'transparent',
					'--anchorkit-toc-bullet-color' => '#59a9ff',
				)
			),
		);

		// Modern preset relies on gradient CSS; leave defaults intact
		if ($style_preset === 'modern' && !$custom_styling) {
			// Use minimal inline overrides; tokens remain but CSS will supersede.
		}

		$legacy_active_link_color = sanitize_hex_color(anchorkit_get_option('anchorkit_toc_active_link_color', '#00A0D2'));
		if (!$legacy_active_link_color) {
			$legacy_active_link_color = '#00A0D2';
		}
		$dark_active_fallback = ($legacy_active_link_color && $legacy_active_link_color !== '#00A0D2') ? $legacy_active_link_color : '#5ba3d0';

		$active_link_color_light = anchorkit_get_sanitized_color_option('anchorkit_toc_active_link_color_light', $legacy_active_link_color);
		$active_link_color_dark = anchorkit_get_sanitized_color_option('anchorkit_toc_active_link_color_dark', $dark_active_fallback);

		if ($custom_styling) {
			$light_overrides = array(
				'--anchorkit-toc-bg' => anchorkit_get_sanitized_color_option('anchorkit_toc_bg_color_light', '#FFFFFF'),
				'--anchorkit-toc-text-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_text_color_light', '#333333'),
				'--anchorkit-toc-link-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_link_color_light', '#0073AA'),
				'--anchorkit-toc-link-hover-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_link_hover_color_light', '#005177'),
				'--anchorkit-toc-border-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_border_color_light', '#DDDDDD'),
				'--anchorkit-toc-bullet-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_bullet_color_light', '#0073AA'),
				'--anchorkit-toc-hover-bg' => anchorkit_get_sanitized_color_option('anchorkit_toc_hover_bg_color_light', 'rgba(0,0,0,0.04)'),
				'--anchorkit-toc-focus-bg' => anchorkit_get_sanitized_color_option('anchorkit_toc_focus_bg_color_light', 'rgba(0,0,0,0.08)'),
			);

			$dark_overrides = array(
				'--anchorkit-toc-bg' => anchorkit_get_sanitized_color_option('anchorkit_toc_bg_color_dark', '#1e1e1e'),
				'--anchorkit-toc-text-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_text_color_dark', '#e0e0e0'),
				'--anchorkit-toc-link-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_link_color_dark', '#7ec4ee'),
				'--anchorkit-toc-link-hover-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_link_hover_color_dark', '#a8d8f0'),
				'--anchorkit-toc-border-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_border_color_dark', '#404040'),
				'--anchorkit-toc-bullet-color' => anchorkit_get_sanitized_color_option('anchorkit_toc_bullet_color_dark', '#7ec4ee'),
				'--anchorkit-toc-hover-bg' => anchorkit_get_sanitized_color_option('anchorkit_toc_hover_bg_color_dark', 'rgba(255,255,255,0.08)'),
				'--anchorkit-toc-focus-bg' => anchorkit_get_sanitized_color_option('anchorkit_toc_focus_bg_color_dark', 'rgba(255,255,255,0.12)'),
			);

			$tokens['light'] = array_merge($tokens['light'], $light_overrides);
			$tokens['dark'] = array_merge($tokens['dark'], $dark_overrides);
		} else {
			// When custom styling is disabled, still apply bullet colors from Bullets & Numbering section
			$tokens['light']['--anchorkit-toc-bullet-color'] = anchorkit_get_sanitized_color_option('anchorkit_toc_bullet_color_light', '#0073AA');
			$tokens['dark']['--anchorkit-toc-bullet-color'] = anchorkit_get_sanitized_color_option('anchorkit_toc_bullet_color_dark', '#7ec4ee');
		}

		$tokens['light']['--anchorkit-toc-active-link-color'] = $active_link_color_light;
		$tokens['dark']['--anchorkit-toc-active-link-color'] = $active_link_color_dark;

		// Basic typography settings - always apply (available to all users)
		$tokens['light']['--anchorkit-toc-line-height'] = floatval(anchorkit_get_option('anchorkit_toc_line_height', 1.6));
		$tokens['dark']['--anchorkit-toc-line-height'] = floatval(anchorkit_get_option('anchorkit_toc_line_height', 1.6));
		$tokens['light']['--anchorkit-toc-letter-spacing'] = intval(anchorkit_get_option('anchorkit_toc_letter_spacing', 0)) . 'px';
		$tokens['dark']['--anchorkit-toc-letter-spacing'] = intval(anchorkit_get_option('anchorkit_toc_letter_spacing', 0)) . 'px';
		$tokens['light']['--anchorkit-toc-text-transform'] = esc_attr(anchorkit_get_option('anchorkit_toc_text_transform', 'none'));
		$tokens['dark']['--anchorkit-toc-text-transform'] = esc_attr(anchorkit_get_option('anchorkit_toc_text_transform', 'none'));

		// Link underline setting
		$link_underline = anchorkit_get_option('anchorkit_toc_link_underline', 'none');
		$allowed_underlines = array('none', 'always', 'hover', 'hover_none');
		$link_underline = in_array($link_underline, $allowed_underlines, true) ? $link_underline : 'none';
		$link_underline_value = ($link_underline === 'always' || $link_underline === 'hover_none') ? 'underline' : 'none';
		$link_underline_hover = ($link_underline === 'hover') ? 'underline' : (($link_underline === 'always') ? 'underline' : 'none');
		$tokens['light']['--anchorkit-toc-link-decoration'] = $link_underline_value;
		$tokens['dark']['--anchorkit-toc-link-decoration'] = $link_underline_value;
		$tokens['light']['--anchorkit-toc-link-decoration-hover'] = $link_underline_hover;
		$tokens['dark']['--anchorkit-toc-link-decoration-hover'] = $link_underline_hover;



		if (anchorkit_get_option('anchorkit_toc_box_shadow_enabled', false)) {
			// Build shadow from components
			$h_offset_light = (int) anchorkit_get_option('anchorkit_toc_shadow_h_offset_light', 0);
			$v_offset_light = (int) anchorkit_get_option('anchorkit_toc_shadow_v_offset_light', 4);
			$blur_light = (int) anchorkit_get_option('anchorkit_toc_shadow_blur_light', 6);
			$spread_light = (int) anchorkit_get_option('anchorkit_toc_shadow_spread_light', 0);
			$color_light = anchorkit_get_option('anchorkit_toc_shadow_color_light', '#000000');
			$opacity_light = (float) anchorkit_get_option('anchorkit_toc_shadow_opacity_light', 0.1);

			$h_offset_dark = (int) anchorkit_get_option('anchorkit_toc_shadow_h_offset_dark', 0);
			$v_offset_dark = (int) anchorkit_get_option('anchorkit_toc_shadow_v_offset_dark', 2);
			$blur_dark = (int) anchorkit_get_option('anchorkit_toc_shadow_blur_dark', 8);
			$spread_dark = (int) anchorkit_get_option('anchorkit_toc_shadow_spread_dark', 0);
			$color_dark = anchorkit_get_option('anchorkit_toc_shadow_color_dark', '#000000');
			$opacity_dark = (float) anchorkit_get_option('anchorkit_toc_shadow_opacity_dark', 0.3);

			// Convert hex to RGB
			list($r_light, $g_light, $b_light) = sscanf($color_light, '#%02x%02x%02x');
			list($r_dark, $g_dark, $b_dark) = sscanf($color_dark, '#%02x%02x%02x');

			// Build shadow strings
			$shadow_light = "{$h_offset_light}px {$v_offset_light}px {$blur_light}px {$spread_light}px rgba({$r_light}, {$g_light}, {$b_light}, {$opacity_light})";
			$shadow_dark = "{$h_offset_dark}px {$v_offset_dark}px {$blur_dark}px {$spread_dark}px rgba({$r_dark}, {$g_dark}, {$b_dark}, {$opacity_dark})";

			$tokens['light']['--anchorkit-toc-shadow-light'] = $shadow_light;
			$tokens['dark']['--anchorkit-toc-shadow-dark'] = $shadow_dark;
		}

		// PRO: Back-to-top font size
		if (!isset($tokens['light']['--anchorkit-toc-back-to-top-font-size'])) {
			if (anchorkit_fs() && anchorkit_fs()->is__premium_only()) {
				if (anchorkit_fs()->can_use_premium_code() && anchorkit_get_option('anchorkit_toc_back_to_top_link', false)) {
					$font_size = (int) anchorkit_get_option('anchorkit_toc_back_to_top_font_size', 14) . 'px';
					$tokens['light']['--anchorkit-toc-back-to-top-font-size'] = $font_size;
					$tokens['dark']['--anchorkit-toc-back-to-top-font-size'] = $font_size;
				}
			}
		}

		foreach (array('light', 'dark') as $mode) {
			if (!isset($tokens[$mode]['--anchorkit-toc-back-to-top-border-color'])) {
				$tokens[$mode]['--anchorkit-toc-back-to-top-border-color'] = $tokens[$mode]['--anchorkit-toc-border-color'] ?? '#dddddd';
			}
			if (!isset($tokens[$mode]['--anchorkit-toc-back-to-top-color'])) {
				$tokens[$mode]['--anchorkit-toc-back-to-top-color'] = $tokens[$mode]['--anchorkit-toc-link-color'] ?? '#0073AA';
			}
			if (!isset($tokens[$mode]['--anchorkit-toc-back-to-top-hover-color'])) {
				$tokens[$mode]['--anchorkit-toc-back-to-top-hover-color'] = $tokens[$mode]['--anchorkit-toc-link-hover-color'] ?? '#005177';
			}
		}

		return $tokens;
	}
}

if (!function_exists('anchorkit_get_toc_css_variables')) {
	function anchorkit_get_toc_css_variables($mode = null)
	{
		$tokens = anchorkit_collect_toc_tokens();
		$theme = anchorkit_get_option('anchorkit_toc_theme', 'system');

		if ($mode === 'light' || $mode === 'dark') {
			return $tokens[$mode];
		}

		if ($theme === 'light') {
			return $tokens['light'];
		}

		if ($theme === 'dark') {
			return $tokens['dark'];
		}

		// System default: use light tokens; dark styling handled via CSS media queries.
		return $tokens['light'];
	}
}
if (!function_exists('anchorkit_get_toc_inline_css')) {
	/**
	 * Generate the TOC inline CSS content (without style tags)
	 *
	 * @return string CSS content
	 */
	function anchorkit_get_toc_inline_css()
	{
		// Check if custom styling is enabled - explicitly convert to boolean
		$custom_styling_raw = anchorkit_get_option('anchorkit_toc_custom_styling', false);
		$custom_styling = filter_var($custom_styling_raw, FILTER_VALIDATE_BOOLEAN);

		// Pro: Advanced Typography - Apply to ALL TOC containers (presets and custom)
		if (anchorkit_fs() && anchorkit_fs()->is__premium_only()) {
			if (anchorkit_fs()->can_use_premium_code()) {
				// Check if advanced typography override is enabled
				$advanced_typography_override = anchorkit_get_option('anchorkit_toc_advanced_typography_override', false);

				// Advanced Typography - Font Sizes (only if override is enabled)
				if ($advanced_typography_override) {
					$title_font_size = anchorkit_get_option('anchorkit_toc_title_font_size', '');
					$h2_font_size = anchorkit_get_option('anchorkit_toc_h2_font_size', '');
					$h3_font_size = anchorkit_get_option('anchorkit_toc_h3_font_size', '');
					$h4_font_size = anchorkit_get_option('anchorkit_toc_h4_font_size', '');
					$h5_font_size = anchorkit_get_option('anchorkit_toc_h5_font_size', '');
					$h6_font_size = anchorkit_get_option('anchorkit_toc_h6_font_size', '');

					// Title font size
					if (!empty($title_font_size) && is_numeric($title_font_size)) {
						$css .= '.anchorkit-toc-container .anchorkit-toc-title{';
						$css .= 'font-size:' . intval($title_font_size) . 'px !important;';
						$css .= '}';
					}

					// H2 level font size
					if (!empty($h2_font_size) && is_numeric($h2_font_size)) {
						$css .= '.anchorkit-toc-container .anchorkit-toc-heading-level-2 > .anchorkit-toc-link{';
						$css .= 'font-size:' . intval($h2_font_size) . 'px !important;';
						$css .= '}';
					}

					// H3 level font size
					if (!empty($h3_font_size) && is_numeric($h3_font_size)) {
						$css .= '.anchorkit-toc-container .anchorkit-toc-heading-level-3 > .anchorkit-toc-link{';
						$css .= 'font-size:' . intval($h3_font_size) . 'px !important;';
						$css .= '}';
					}

					// H4 level font size
					if (!empty($h4_font_size) && is_numeric($h4_font_size)) {
						$css .= '.anchorkit-toc-container .anchorkit-toc-heading-level-4 > .anchorkit-toc-link{';
						$css .= 'font-size:' . intval($h4_font_size) . 'px !important;';
						$css .= '}';
					}

					// H5 level font size
					if (!empty($h5_font_size) && is_numeric($h5_font_size)) {
						$css .= '.anchorkit-toc-container .anchorkit-toc-heading-level-5 > .anchorkit-toc-link{';
						$css .= 'font-size:' . intval($h5_font_size) . 'px !important;';
						$css .= '}';
					}

					// H6 level font size
					if (!empty($h6_font_size) && is_numeric($h6_font_size)) {
						$css .= '.anchorkit-toc-container .anchorkit-toc-heading-level-6 > .anchorkit-toc-link{';
						$css .= 'font-size:' . intval($h6_font_size) . 'px !important;';
						$css .= '}';
					}
				}
			}
		}

		$theme = anchorkit_get_option('anchorkit_toc_theme', 'system');
		$all_tokens = anchorkit_collect_toc_tokens();
		$light_tokens = isset($all_tokens['light']) ? $all_tokens['light'] : array();
		$dark_tokens = isset($all_tokens['dark']) ? $all_tokens['dark'] : array();
		$css_vars = anchorkit_get_toc_css_variables(); // Returns active tokens per selected theme

		// Width and float settings (max-width/min-width handled by CSS)
		$width = (int) anchorkit_get_option('anchorkit_toc_width', 80);
		$alignment = anchorkit_get_option('anchorkit_toc_alignment', 'center');
		$float = anchorkit_get_option('anchorkit_toc_float', 'none');

		$valid_alignments = array('left', 'center', 'right');
		if (!in_array($alignment, $valid_alignments, true)) {
			$alignment = 'center';
		}

		$css = '';

		// Safari fix: When theme is 'system', define BOTH light and dark with class-based selectors
		if ($theme === 'system') {
			// Light mode: Apply via class AND as default (fallback)
			$css .= '.anchorkit-toc-container.anchorkit-toc-theme-system,';
			$css .= 'html.anchorkit-prefers-light .anchorkit-toc-container.anchorkit-toc-theme-system{';
			foreach ($light_tokens as $k => $v) {
				$css .= $k . ':' . esc_attr($v) . ';';
			}
			// Apply direct properties for Safari
			$css .= 'background:' . esc_attr($light_tokens['--anchorkit-toc-bg']) . ';';
			$css .= 'color:' . esc_attr($light_tokens['--anchorkit-toc-text-color']) . ';';
			$css .= 'border-color:' . esc_attr($light_tokens['--anchorkit-toc-border-color']) . ';';
			$css .= '}';
		} else {
			// For explicit light/dark themes, set defaults normally
			$css .= '.anchorkit-toc-container{';
			foreach ($css_vars as $k => $v) {
				$css .= $k . ':' . esc_attr($v) . ';';
			}
			$css .= '}';
		}

		// Apply width and float settings ONLY to non-Elementor/non-Gutenberg TOCs
		// (max-width is handled globally via CSS at 600px)
		$css .= '.anchorkit-toc-container:not(.anchorkit-toc-elementor){';
		if ($width > 0 && $width <= 100) {
			$css .= 'width:' . $width . '% !important;';
		}
		if ($float !== 'none') {
			$css .= 'float:' . esc_attr($float) . ';';
			$css .= 'margin:' . ($float === 'left' ? '0 20px 20px 0' : '0 0 20px 20px') . ' !important;';
		} elseif ($alignment === 'left') {
			$css .= 'margin:20px auto 20px 0 !important;';
		} elseif ($alignment === 'right') {
			$css .= 'margin:20px 0 20px auto !important;';
		} else {
			$css .= 'margin:20px auto !important;';
		}
		$css .= '}';

		// Apply bullet colors to ALL TOC containers (not just custom styling)
		$css .= '.anchorkit-toc-container .anchorkit-toc-item::before{';
		$css .= 'color:var(--anchorkit-toc-bullet-color);';
		$css .= '}';

		// Apply bullet colors for list markers (for disc, circle, square bullets)
		$css .= '.anchorkit-toc-container .anchorkit-toc-item::marker{';
		$css .= 'color:var(--anchorkit-toc-bullet-color);';
		$css .= '}';

		if ($theme === 'system') {
			// Safari fix: Apply dark mode with absolute maximum specificity
			// Use body tag in selector for even higher specificity
			// IMPORTANT: Exclude instance TOCs (Elementor/Gutenberg) - they manage their own colors via scoped style blocks
			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]){';
			foreach ($dark_tokens as $k => $v) {
				$css .= $k . ':' . esc_attr($v) . ' !important;';
			}
			// Apply direct properties (bypassing CSS variables entirely for Safari)
			$css .= 'background:' . esc_attr($dark_tokens['--anchorkit-toc-bg']) . ' !important;';
			$css .= 'color:' . esc_attr($dark_tokens['--anchorkit-toc-text-color']) . ' !important;';
			$css .= 'border-color:' . esc_attr($dark_tokens['--anchorkit-toc-border-color']) . ' !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]) .anchorkit-toc-title{';
			$css .= 'color:' . esc_attr($dark_tokens['--anchorkit-toc-text-color']) . ' !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-link{';
			$css .= 'color:' . esc_attr($dark_tokens['--anchorkit-toc-link-color']) . ' !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-link:visited{';
			$css .= 'color:' . esc_attr($dark_tokens['--anchorkit-toc-link-color']) . ' !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-link:hover{';
			$css .= 'color:' . esc_attr($dark_tokens['--anchorkit-toc-link-hover-color']) . ' !important;';
			$css .= 'background:' . esc_attr($dark_tokens['--anchorkit-toc-hover-bg']) . ' !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-link.active,';
			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-link.anchorkit-toc-active{';
			$css .= 'color:' . esc_attr($dark_tokens['--anchorkit-toc-active-link-color']) . ' !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-toggle-button{';
			$css .= 'color:' . esc_attr($dark_tokens['--anchorkit-toc-link-color']) . ' !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-view-more-btn{';
			$css .= 'color:' . esc_attr($dark_tokens['--anchorkit-toc-link-color']) . ' !important;';
			$css .= 'border-color:' . esc_attr($dark_tokens['--anchorkit-toc-border-color']) . ' !important;';
			$css .= '}';

			// Special handling for Modern preset (has gradient backgrounds)
			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-preset-modern.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling){';
			$css .= 'background:-webkit-linear-gradient(135deg, #0d1117 0%, #161b22 50%, #1c2128 100%) !important;';
			$css .= 'background:linear-gradient(135deg, #0d1117 0%, #161b22 50%, #1c2128 100%) !important;';
			$css .= 'border:1px solid rgba(88, 166, 255, 0.2) !important;';
			$css .= 'color:#f0f6fc !important;';
			$css .= '}';

			// Special handling for Classic preset (border-top color should match link color)
			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-preset-classic.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling){';
			$css .= 'border-top-color:' . esc_attr($dark_tokens['--anchorkit-toc-link-color']) . ' !important;';
			$css .= '}';

			// Add custom back-to-top border override for system dark mode or general
			// We'll add this generally in the custom styling section later, but cleaning this up allows it to work.

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-preset-modern.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]) .anchorkit-toc-title{';
			$css .= 'color:#f0f6fc !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-preset-modern.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-link{';
			$css .= 'color:#58a6ff !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-preset-modern.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-link:hover{';
			$css .= 'color:#79c0ff !important;';
			$css .= 'background:rgba(88, 166, 255, 0.15) !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-preset-modern.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-toggle-button{';
			$css .= 'color:#58a6ff !important;';
			$css .= '}';

			$css .= 'html.anchorkit-prefers-dark body .anchorkit-toc-preset-modern.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]):not(.anchorkit-toc-custom-styling) .anchorkit-toc-view-more-btn{';
			$css .= 'color:#58a6ff !important;';
			$css .= 'border-color:rgba(88, 166, 255, 0.4) !important;';
			$css .= 'background:rgba(88, 166, 255, 0.08) !important;';
			$css .= '}';

			// Fallback: Keep media query for browsers without JS or if script doesn't run
			// IMPORTANT: Exclude instance TOCs (Elementor/Gutenberg) - they manage their own colors via scoped style blocks
			$css .= '@media (prefers-color-scheme: dark){';
			$css .= '.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]){';
			foreach ($dark_tokens as $k => $v) {
				$css .= $k . ':' . esc_attr($v) . ';';
			}
			$css .= '}';
			// Also target preset classes specifically to ensure custom colors override preset defaults
			$css .= '.anchorkit-toc-container.anchorkit-toc-theme-system.anchorkit-toc-preset-minimal:not([data-anchorkit-instance]),';
			$css .= '.anchorkit-toc-container.anchorkit-toc-theme-system.anchorkit-toc-preset-classic:not([data-anchorkit-instance]),';
			$css .= '.anchorkit-toc-container.anchorkit-toc-theme-system.anchorkit-toc-preset-modern:not([data-anchorkit-instance]){';
			$css .= '--anchorkit-toc-bullet-color:' . esc_attr($dark_tokens['--anchorkit-toc-bullet-color'] ?? '#7ec4ee') . ' !important;';
			$css .= '}';
			$css .= '}';
		} elseif ($theme === 'light') {
			// Explicitly set light mode when theme is 'light' (already set above, but be explicit)
			$css .= '.anchorkit-toc-container.anchorkit-toc-theme-light{';
			foreach ($light_tokens as $k => $v) {
				$css .= $k . ':' . esc_attr($v) . ';';
			}
			$css .= '}';
		} elseif ($theme === 'dark') {
			// When theme is explicitly set to dark, apply dark tokens directly
			$css .= '.anchorkit-toc-container.anchorkit-toc-theme-dark{';
			foreach ($dark_tokens as $k => $v) {
				$css .= $k . ':' . esc_attr($v) . ';';
			}
			$css .= '}';
		}

		// Custom styling class - override presets with !important
		// Always emit these rules so per-block Gutenberg overrides work even when global custom styling is disabled.
		$css .= '.anchorkit-toc-container.anchorkit-toc-custom-styling{';
		$css .= 'background:var(--anchorkit-toc-bg) !important;';
		$css .= 'color:var(--anchorkit-toc-text-color) !important;';
		$css .= 'padding:var(--anchorkit-toc-padding) !important;';
		$css .= 'border:var(--anchorkit-toc-border-width) solid var(--anchorkit-toc-border-color) !important;';
		$css .= 'border-radius:var(--anchorkit-toc-border-radius) !important;';
		$css .= 'font-size:var(--anchorkit-toc-font-size) !important;';
		$css .= 'font-family:var(--anchorkit-toc-font-family) !important;';
		$css .= '}';

		// Apply direct styles to custom styling elements
		$css .= '.anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-title{';
		$css .= 'color:var(--anchorkit-toc-text-color) !important;';
		$css .= '}';

		$css .= '.anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-link{';
		$css .= 'color:var(--anchorkit-toc-link-color) !important;';
		$css .= 'text-decoration:var(--anchorkit-toc-link-decoration, none) !important;';
		$css .= '}';

		$css .= '.anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-link:hover{';
		$css .= 'color:var(--anchorkit-toc-link-hover-color) !important;';
		$css .= 'text-decoration:var(--anchorkit-toc-link-decoration-hover, none) !important;';
		$css .= '}';

		$css .= '.anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-link.anchorkit-toc-active,';
		$css .= '.anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-link.active{';
		$css .= 'color:var(--anchorkit-toc-active-link-color) !important;';
		$css .= '}';

		// Back to top button - ensure it follows link styling (decoration AND color for custom styling)
		$css .= '.anchorkit-toc-container .anchorkit-toc-back-to-top a{';
		$css .= 'text-decoration:var(--anchorkit-toc-link-decoration, none) !important;';
		$css .= '}';

		$css .= '.anchorkit-toc-container .anchorkit-toc-back-to-top a:hover{';
		$css .= 'text-decoration:var(--anchorkit-toc-link-decoration-hover, none) !important;';
		$css .= '}';

		// Ensure custom colors apply to back to top button
		$css .= 'html body .anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-back-to-top{';
		$css .= 'border-top-color:var(--anchorkit-toc-link-color) !important;';
		$css .= '}';

		$css .= 'html body .anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-back-to-top a{';
		$css .= 'color:var(--anchorkit-toc-link-color) !important;';
		$css .= 'opacity: 1 !important;';
		$css .= '}';

		$css .= 'html body .anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-back-to-top a:hover{';
		$css .= 'color:var(--anchorkit-toc-link-hover-color) !important;';
		$css .= 'opacity: 1 !important;';
		$css .= '}';

		// View More button custom colors - High Specificity Override
		// Non-hover: Outlined style with link color for text and border
		$css .= 'html body .anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-view-more-btn{';
		$css .= 'color:var(--anchorkit-toc-link-color) !important;';
		$css .= 'border-color:var(--anchorkit-toc-link-color) !important;';
		$css .= 'background-color: transparent !important;';
		$css .= '}';

		// Hover: Filled style - background becomes link color, text becomes TOC bg color for contrast
		$css .= 'html body .anchorkit-toc-container.anchorkit-toc-custom-styling .anchorkit-toc-view-more-btn:hover{';
		$css .= 'background-color:var(--anchorkit-toc-link-color) !important;';
		$css .= 'color:var(--anchorkit-toc-bg) !important;';
		$css .= 'border-color:var(--anchorkit-toc-link-color) !important;';
		$css .= '}';

		// Pro: Advanced Typography - Apply to ALL TOC containers (presets and custom)


		// Pro: Box Shadow - Apply to ALL TOC containers (presets and custom)
		if (anchorkit_get_option('anchorkit_toc_box_shadow_enabled', false)) {
			// Build shadow from components
			$h_offset_light = (int) anchorkit_get_option('anchorkit_toc_shadow_h_offset_light', 0);
			$v_offset_light = (int) anchorkit_get_option('anchorkit_toc_shadow_v_offset_light', 4);
			$blur_light = (int) anchorkit_get_option('anchorkit_toc_shadow_blur_light', 6);
			$spread_light = (int) anchorkit_get_option('anchorkit_toc_shadow_spread_light', 0);
			$color_light = anchorkit_get_option('anchorkit_toc_shadow_color_light', '#000000');
			$opacity_light = (float) anchorkit_get_option('anchorkit_toc_shadow_opacity_light', 0.1);

			$h_offset_dark = (int) anchorkit_get_option('anchorkit_toc_shadow_h_offset_dark', 0);
			$v_offset_dark = (int) anchorkit_get_option('anchorkit_toc_shadow_v_offset_dark', 2);
			$blur_dark = (int) anchorkit_get_option('anchorkit_toc_shadow_blur_dark', 8);
			$spread_dark = (int) anchorkit_get_option('anchorkit_toc_shadow_spread_dark', 0);
			$color_dark = anchorkit_get_option('anchorkit_toc_shadow_color_dark', '#000000');
			$opacity_dark = (float) anchorkit_get_option('anchorkit_toc_shadow_opacity_dark', 0.3);

			// Convert hex to RGB
			list($r_light, $g_light, $b_light) = sscanf($color_light, '#%02x%02x%02x');
			list($r_dark, $g_dark, $b_dark) = sscanf($color_dark, '#%02x%02x%02x');

			// Build shadow strings
			$box_shadow_light = "{$h_offset_light}px {$v_offset_light}px {$blur_light}px {$spread_light}px rgba({$r_light}, {$g_light}, {$b_light}, {$opacity_light})";
			$box_shadow_dark = "{$h_offset_dark}px {$v_offset_dark}px {$blur_dark}px {$spread_dark}px rgba({$r_dark}, {$g_dark}, {$b_dark}, {$opacity_dark})";

			// Apply to ALL TOC containers - including presets (high specificity to override preset defaults)
			// IMPORTANT: Exclude instance TOCs (Elementor/Gutenberg) - they manage their own shadow via scoped style blocks
			$css .= '.anchorkit-toc-container:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-custom-styling.anchorkit-toc-container:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-preset-minimal.anchorkit-toc-container:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-preset-classic.anchorkit-toc-container:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-preset-modern.anchorkit-toc-container:not([data-anchorkit-instance]){' . "\n";
			$css .= 'box-shadow:' . esc_attr($box_shadow_light) . ' !important;' . "\n";
			$css .= '}' . "\n";

			// Dark theme
			// IMPORTANT: Exclude instance TOCs (Elementor/Gutenberg) - they manage their own shadow via scoped style blocks
			$css .= '.anchorkit-toc-container.anchorkit-toc-theme-dark:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-custom-styling.anchorkit-toc-container.anchorkit-toc-theme-dark:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-preset-minimal.anchorkit-toc-container.anchorkit-toc-theme-dark:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-preset-classic.anchorkit-toc-container.anchorkit-toc-theme-dark:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-preset-modern.anchorkit-toc-container.anchorkit-toc-theme-dark:not([data-anchorkit-instance]){' . "\n";
			$css .= 'box-shadow:' . esc_attr($box_shadow_dark) . ' !important;' . "\n";
			$css .= '}' . "\n";

			// Safari fix: Use class-based dark mode detection
			// IMPORTANT: Exclude instance TOCs (Elementor/Gutenberg) - they manage their own shadow via scoped style blocks
			$css .= 'html.anchorkit-prefers-dark .anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]),' . "\n";
			$css .= 'html.anchorkit-prefers-dark .anchorkit-toc-custom-styling.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]),' . "\n";
			$css .= 'html.anchorkit-prefers-dark .anchorkit-toc-preset-minimal.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]),' . "\n";
			$css .= 'html.anchorkit-prefers-dark .anchorkit-toc-preset-classic.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]),' . "\n";
			$css .= 'html.anchorkit-prefers-dark .anchorkit-toc-preset-modern.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]){' . "\n";
			$css .= 'box-shadow:' . esc_attr($box_shadow_dark) . ' !important;' . "\n";
			$css .= '}' . "\n";

			if ($theme === 'system') {
				// Fallback media query
				$css .= '@media (prefers-color-scheme: dark){' . "\n";
				$css .= '.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]),' . "\n";
				$css .= '.anchorkit-toc-custom-styling.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]),' . "\n";
				$css .= '.anchorkit-toc-preset-minimal.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]),' . "\n";
				$css .= '.anchorkit-toc-preset-classic.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]),' . "\n";
				$css .= '.anchorkit-toc-preset-modern.anchorkit-toc-container.anchorkit-toc-theme-system:not([data-anchorkit-instance]){' . "\n";
				$css .= 'box-shadow:' . esc_attr($box_shadow_dark) . ' !important;' . "\n";
				$css .= '}' . "\n";
				$css .= '}' . "\n";
			}
		} else {
			// Explicitly remove box-shadow when the toggle is off to prevent default shadows from persisting
			// IMPORTANT: Exclude instance TOCs (Elementor/Gutenberg) - they manage their own shadow via scoped style blocks
			$css .= '.anchorkit-toc-container:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-custom-styling.anchorkit-toc-container:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-preset-minimal.anchorkit-toc-container:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-preset-classic.anchorkit-toc-container:not([data-anchorkit-instance]),' . "\n";
			$css .= '.anchorkit-toc-preset-modern.anchorkit-toc-container:not([data-anchorkit-instance]){' . "\n";
			$css .= 'box-shadow: none !important;' . "\n";
			$css .= '}' . "\n";
		}

		// Mobile Responsiveness - Hide TOC on small screens if enabled
		$hide_on_mobile = (bool) anchorkit_get_option('anchorkit_toc_hide_on_mobile', false);
		$mobile_breakpoint = (int) anchorkit_get_option('anchorkit_toc_mobile_breakpoint', 782);

		if ($hide_on_mobile) {
			$css .= "\n/* Mobile Responsiveness - Hide on mobile enabled, breakpoint: " . $mobile_breakpoint . "px */\n";
			$css .= '@media (max-width: ' . $mobile_breakpoint . 'px) {' . "\n";
			$css .= '    .anchorkit-toc-container.anchorkit-toc-hide-mobile { display: none !important; }' . "\n";
			$css .= '}' . "\n";

			// Show the TOC above the breakpoint
			$css .= '@media (min-width: ' . ($mobile_breakpoint + 1) . 'px) {' . "\n";
			$css .= '    .anchorkit-toc-hide-mobile { display: block !important; }' . "\n";
			$css .= '}' . "\n";
		}

		return $css;
	}
}
