<?php

if (!defined('ABSPATH')) {
	exit;
}

// Render Table of Contents settings tabs
add_action(
	'anchorkit_settings_tabs',
	function ($active_tab) {
		// Handle all TOC-related tabs
		$toc_tabs = array('quick-start', 'appearance-behavior', 'structure-seo', 'advanced-accessibility', 'help', 'anchorkit-pro');
		if (!in_array($active_tab, $toc_tabs)) {
			return;
		}

		// Enqueue frontend CSS for preview (needed for preset styles)
		wp_enqueue_style(
			'anchorkit-toc-frontend-preview',
			plugins_url('css/anchorkit-frontend.css', ANCHORKIT_PLUGIN_FILE),
			array(),
			ANCHORKIT_PLUGIN_VERSION
		);
		wp_register_style('anchorkit-toc-preview-inline', false, array('anchorkit-toc-frontend-preview'), ANCHORKIT_PLUGIN_VERSION);
		wp_enqueue_style('anchorkit-toc-preview-inline');
		wp_add_inline_style('anchorkit-toc-preview-inline', '/* AnchorKit preview custom CSS placeholder */');

		// Enqueue the TOC preview script
		wp_enqueue_script(
			'anchorkit-toc-admin-preview',
			plugins_url('js/toc-admin-preview.js', ANCHORKIT_PLUGIN_FILE),
			array('jquery', 'anchorkit-admin-js'),
			ANCHORKIT_PLUGIN_VERSION,
			true
		);

		$post_types = get_post_types(array('public' => true), 'objects');

		// Get option values with defaults
		$toc_enabled = get_option('anchorkit_toc_enabled', false);
		$post_types_option = get_option('anchorkit_toc_post_types', array('post', 'page'));
		$automatic_insertion = get_option('anchorkit_toc_automatic_insertion', true);
		$position = get_option('anchorkit_toc_position', 'before_first_heading');
		$collapsible = get_option('anchorkit_toc_collapsible', true);
		$initial_state = get_option('anchorkit_toc_initial_state', 'expanded');
		$smooth_scroll = get_option('anchorkit_toc_smooth_scroll', true);
		// Load hierarchical view with proper fallback (default to true only if never saved)
		$hierarchical_view_option = get_option('anchorkit_toc_hierarchical_view', 'not_set');
		$hierarchical_view = ($hierarchical_view_option === 'not_set') ? true : (bool) $hierarchical_view_option;
		$show_numerals = get_option('anchorkit_toc_show_numerals', false);
		$live_preview_enabled = get_option('anchorkit_toc_live_preview_enabled', false);
		$toc_theme = get_option('anchorkit_toc_theme', 'system');

		// Load light and dark mode colors separately (sanitized)
		$bg_color_light = anchorkit_get_sanitized_color_option('anchorkit_toc_bg_color_light', '#FFFFFF');
		$text_color_light = anchorkit_get_sanitized_color_option('anchorkit_toc_text_color_light', '#333333');
		$link_color_light = anchorkit_get_sanitized_color_option('anchorkit_toc_link_color_light', '#0073AA');
		$link_hover_color_light = anchorkit_get_sanitized_color_option('anchorkit_toc_link_hover_color_light', '#005177');
		$border_color_light = anchorkit_get_sanitized_color_option('anchorkit_toc_border_color_light', '#DDDDDD');
		$bullet_color_light = anchorkit_get_sanitized_color_option('anchorkit_toc_bullet_color_light', '#0073AA');

		$bg_color_dark = anchorkit_get_sanitized_color_option('anchorkit_toc_bg_color_dark', '#1e1e1e');
		$text_color_dark = anchorkit_get_sanitized_color_option('anchorkit_toc_text_color_dark', '#e0e0e0');
		$link_color_dark = anchorkit_get_sanitized_color_option('anchorkit_toc_link_color_dark', '#7ec4ee');
		$link_hover_color_dark = anchorkit_get_sanitized_color_option('anchorkit_toc_link_hover_color_dark', '#a8d8f0');
		$border_color_dark = anchorkit_get_sanitized_color_option('anchorkit_toc_border_color_dark', '#404040');
		$bullet_color_dark = anchorkit_get_sanitized_color_option('anchorkit_toc_bullet_color_dark', '#7ec4ee');

		// Legacy single values (for backward compatibility - fallback to light mode)
		$bg_color = anchorkit_get_sanitized_color_option('anchorkit_toc_bg_color', $bg_color_light);
		$text_color = anchorkit_get_sanitized_color_option('anchorkit_toc_text_color', $text_color_light);
		$link_color = anchorkit_get_sanitized_color_option('anchorkit_toc_link_color', $link_color_light);
		$link_hover_color = anchorkit_get_sanitized_color_option('anchorkit_toc_link_hover_color', $link_hover_color_light);
		$border_color = anchorkit_get_sanitized_color_option('anchorkit_toc_border_color', $border_color_light);
		$font_family = get_option('anchorkit_toc_font_family', 'inherit');
		$font_size = get_option('anchorkit_toc_font_size', 14);
		$padding = get_option('anchorkit_toc_padding', 15);
		$border_width = get_option('anchorkit_toc_border_width', 1);
		$border_radius = get_option('anchorkit_toc_border_radius', 4);
		$include_headings = anchorkit_get_global_heading_levels();
		$min_headings = get_option('anchorkit_toc_min_headings', 2);
		$exclude_selectors = get_option('anchorkit_toc_exclude_selectors', '');
		$title_text = get_option('anchorkit_toc_title_text', 'Table of Contents');
		$show_label = get_option('anchorkit_toc_show_label', true);
		$style_preset = get_option('anchorkit_toc_style_preset', 'minimal');
		$hide_on_mobile = get_option('anchorkit_toc_hide_on_mobile', false);
		$aria_label = get_option('anchorkit_toc_aria_label', '');
		$back_to_top_link = get_option('anchorkit_toc_back_to_top_link', false);
		$bullet_style = get_option('anchorkit_toc_bullet_style', 'disc');
		$bullet_character = get_option('anchorkit_toc_bullet_character', '•');
		$numbering_style = get_option('anchorkit_toc_numbering_style', 'hierarchical');
		$numbering_format = get_option('anchorkit_toc_numbering_format', 'decimal');
		$numbering_sep = get_option('anchorkit_toc_numbering_separator', '.');
		$scroll_offset = get_option('anchorkit_toc_scroll_offset', 0);
		$toc_width = get_option('anchorkit_toc_width', '80');
		$toc_float = get_option('anchorkit_toc_float', 'none');
		$custom_styling = get_option('anchorkit_toc_custom_styling', false);

		// Pro features - Sticky TOC
		$sticky_enabled = get_option('anchorkit_toc_sticky_enabled', false);
		$sticky_position = get_option('anchorkit_toc_sticky_position', 'content'); // Default: sticky in content
		$sticky_offset = get_option('anchorkit_toc_sticky_offset', 20);

		// Pro features - Advanced Styling
		$box_shadow_enabled = get_option('anchorkit_toc_box_shadow_enabled', false);
		$box_shadow_custom = sanitize_text_field(get_option('anchorkit_toc_box_shadow_custom', '0 4px 6px rgba(0, 0, 0, 0.1)'));
		$box_shadow_light = sanitize_text_field(get_option('anchorkit_toc_box_shadow_light', $box_shadow_custom));
		$default_dark_shadow = '0 2px 8px rgba(0, 0, 0, 0.3)';
		$box_shadow_dark = sanitize_text_field(get_option('anchorkit_toc_box_shadow_dark', $box_shadow_custom ?: $default_dark_shadow));
		if (empty($box_shadow_dark)) {
			$box_shadow_dark = $default_dark_shadow;
		}

		// Box Shadow Components - Light Mode (Default: 0 4px 6px rgba(0,0,0,0.1))
		$shadow_h_offset_light = (int) get_option('anchorkit_toc_shadow_h_offset_light', 0);
		$shadow_v_offset_light = (int) get_option('anchorkit_toc_shadow_v_offset_light', 4);
		$shadow_blur_light = (int) get_option('anchorkit_toc_shadow_blur_light', 6);
		$shadow_spread_light = (int) get_option('anchorkit_toc_shadow_spread_light', 0);
		$shadow_color_light = sanitize_text_field(get_option('anchorkit_toc_shadow_color_light', '#000000'));
		$shadow_opacity_light = (float) get_option('anchorkit_toc_shadow_opacity_light', 0.1);

		// Box Shadow Components - Dark Mode (Default: 0 2px 8px rgba(0,0,0,0.3))
		$shadow_h_offset_dark = (int) get_option('anchorkit_toc_shadow_h_offset_dark', 0);
		$shadow_v_offset_dark = (int) get_option('anchorkit_toc_shadow_v_offset_dark', 2);
		$shadow_blur_dark = (int) get_option('anchorkit_toc_shadow_blur_dark', 8);
		$shadow_spread_dark = (int) get_option('anchorkit_toc_shadow_spread_dark', 0);
		$shadow_color_dark = sanitize_text_field(get_option('anchorkit_toc_shadow_color_dark', '#000000'));
		$shadow_opacity_dark = (float) get_option('anchorkit_toc_shadow_opacity_dark', 0.3);
		$entrance_animation = get_option('anchorkit_toc_entrance_animation', false);
		$animation_type = get_option('anchorkit_toc_animation_type', 'fade');
		$line_height = get_option('anchorkit_toc_line_height', 1.6);
		$letter_spacing = get_option('anchorkit_toc_letter_spacing', 0);
		$text_transform = get_option('anchorkit_toc_text_transform', 'none');
		$link_underline = get_option('anchorkit_toc_link_underline', 'none');
		$custom_id = get_option('anchorkit_toc_custom_id', '');
		$custom_class = get_option('anchorkit_toc_custom_class', '');
		$custom_css = get_option('anchorkit_toc_custom_css', '');
		// Pro features - View More
		$view_more_enabled = get_option('anchorkit_toc_view_more_enabled', false);
		$initial_count = get_option('anchorkit_toc_initial_count', 5);
		$view_more_text = get_option('anchorkit_toc_view_more_text', 'View More');
		$view_less_text = get_option('anchorkit_toc_view_less_text', 'View Less');
		$show_reading_time_global = get_option('anchorkit_toc_show_reading_time', false);
		$reading_speed_global = (int) get_option('anchorkit_toc_reading_speed', 200);
		$show_word_count_global = get_option('anchorkit_toc_show_word_count', false);
		$time_format_global = get_option('anchorkit_toc_time_format', 'min_read');

		$legacy_active_link_color = anchorkit_get_sanitized_color_option('anchorkit_toc_active_link_color', '#00A0D2');
		$dark_active_default = '#5ba3d0';
		if (!empty($legacy_active_link_color) && strtolower($legacy_active_link_color) !== '#00a0d2') {
			$dark_active_default = $legacy_active_link_color;
		}
		$active_link_color_light = anchorkit_get_sanitized_color_option('anchorkit_toc_active_link_color_light', $legacy_active_link_color);
		$active_link_color_dark = anchorkit_get_sanitized_color_option('anchorkit_toc_active_link_color_dark', $dark_active_default);
		?>
	<form method="post" action="options.php">
		<?php settings_fields('anchorkit_settings'); ?>
		<?php do_settings_sections('anchorkit_settings'); ?>

		<?php
			// Render content based on active tab
			switch ($active_tab) {
				case 'quick-start':
					anchorkit_render_quick_start_tab($toc_enabled, $post_types, $post_types_option, $automatic_insertion, $position, $min_headings, $show_label, $title_text, $include_headings, $style_preset);
					break;
				case 'appearance-behavior':
					anchorkit_render_appearance_behavior_tab($toc_enabled, $collapsible, $initial_state, $smooth_scroll, $hierarchical_view, $show_numerals, $live_preview_enabled, $toc_theme, $bg_color, $text_color, $link_color, $link_hover_color, $border_color, $font_family, $font_size, $padding, $border_width, $border_radius, $bullet_style, $bullet_character, $numbering_style, $numbering_format, $numbering_sep, $title_text, $scroll_offset, $toc_width, $toc_float, $custom_styling, $style_preset, $back_to_top_link, $sticky_enabled, $sticky_position, $sticky_offset, $box_shadow_enabled, $box_shadow_custom, $box_shadow_light, $box_shadow_dark, $entrance_animation, $animation_type, $line_height, $letter_spacing, $text_transform, $link_underline, $custom_id, $custom_class, $custom_css, $view_more_enabled, $initial_count, $view_more_text, $view_less_text, $show_reading_time_global, $reading_speed_global, $show_word_count_global, $time_format_global, $bg_color_light, $text_color_light, $link_color_light, $link_hover_color_light, $border_color_light, $bullet_color_light, $bg_color_dark, $text_color_dark, $link_color_dark, $link_hover_color_dark, $border_color_dark, $bullet_color_dark, $active_link_color_light, $active_link_color_dark, $legacy_active_link_color, $shadow_h_offset_light, $shadow_v_offset_light, $shadow_blur_light, $shadow_spread_light, $shadow_color_light, $shadow_opacity_light, $shadow_h_offset_dark, $shadow_v_offset_dark, $shadow_blur_dark, $shadow_spread_dark, $shadow_color_dark, $shadow_opacity_dark);
					break;
				case 'structure-seo':
					anchorkit_render_structure_seo_tab($toc_enabled, $include_headings, $exclude_selectors);
					break;
				case 'advanced-accessibility':
					anchorkit_render_advanced_accessibility_tab($toc_enabled, $hide_on_mobile, $aria_label, $back_to_top_link);
					break;
				case 'help':
					anchorkit_render_help_tab();
					break;
				case 'anchorkit-pro':
					anchorkit_render_pro_tab();
					break;
			}
			?>

		<input type="hidden" name="anchorkit_settings_tab" value="<?php echo esc_attr($active_tab); ?>">
	</form>
	<?php
	}
);
// Quick Start Tab
function anchorkit_render_quick_start_tab($toc_enabled, $post_types, $post_types_option, $automatic_insertion, $position, $min_headings, $show_label, $title_text, $include_headings, $style_preset)
{
	?>
	<!-- Card 1: Activation -->
	<div class="anchorkit-card">
		<h2>Activation</h2>
		<p class="anchorkit-card-description">Enable the Table of Contents feature on your site.</p>

		<div class="setting-wrapper">
			<label class="m3-switch">
				<input type="checkbox" name="anchorkit_toc_enabled" value="1" <?php checked(1, $toc_enabled, true); ?>>
				<span class="m3-track"><span class="m3-thumb"></span></span>
				<span class="m3-switch-label">
					Table of Contents is <span
						class="status-text"><?php echo esc_html($toc_enabled ? 'enabled' : 'disabled'); ?></span>
				</span>
			</label>
			<span class="helper-text">Enable or disable TOC site-wide.</span>
		</div>
	</div>

	<!-- Card 2: Where to Show -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<h2>Where to Show</h2>
		<p class="anchorkit-card-description">Choose where your Table of Contents appears.</p>

		<div class="setting-wrapper">
			<h3>Show on:</h3>
			<div class="post-types-grid">
				<?php foreach ($post_types as $post_type): ?>
					<label>
						<input type="checkbox" name="anchorkit_toc_post_types[]"
							value="<?php echo esc_attr($post_type->name); ?>" <?php checked(in_array($post_type->name, $post_types_option), true); ?>>
						<span class="anchorkit-option-label"><?php echo esc_html($post_type->label); ?></span>
					</label>
				<?php endforeach; ?>
			</div>
			<span class="helper-text">Select post types for TOC.</span>
		</div>

		<div class="form-group">
			<div class="anchorkit-input-container">
				<select class="anchorkit-select" name="anchorkit_toc_min_headings" id="anchorkit_toc_min_headings">
					<?php for ($i = 1; $i <= 10; $i++): ?>
						<option value="<?php echo esc_attr($i); ?>" <?php selected($min_headings, $i); ?>>
							<?php echo esc_html($i); ?> or more
							headings
						</option>
					<?php endfor; ?>
				</select>
				<label for="anchorkit_toc_min_headings">Show when</label>
			</div>
			<span class="helper-text">Minimum headings required to show TOC.</span>
		</div>

		<div class="setting-wrapper">
			<label class="m3-switch">
				<input type="checkbox" name="anchorkit_toc_automatic_insertion" value="1" <?php checked(1, $automatic_insertion, true); ?>>
				<span class="m3-track"><span class="m3-thumb"></span></span>
				<span class="m3-switch-label">
					Automatic Insertion
				</span>
			</label>
			<span class="helper-text">Auto-insert TOC or use Gutenberg block / <code>[anchorkit_toc]</code>
				shortcode.</span>
		</div>

		<div class="form-group conditional-field" data-condition="anchorkit_toc_automatic_insertion"
			data-condition-value="1">
			<div class="anchorkit-input-container">
				<select class="anchorkit-select" name="anchorkit_toc_position" id="anchorkit_toc_position">
					<option value="before_first_heading" <?php selected($position, 'before_first_heading'); ?>>Beginning
						of
						content (recommended)</option>
					<option value="after_first_paragraph" <?php selected($position, 'after_first_paragraph'); ?>>After
						first
						paragraph</option>
					<option value="top_of_content" <?php selected($position, 'top_of_content'); ?>>Very top (before
						everything)</option>
					<option value="bottom_of_content" <?php selected($position, 'bottom_of_content'); ?>>End of content
					</option>
				</select>
				<label for="anchorkit_toc_position">Position in content</label>
			</div>
			<span class="helper-text">Where TOC appears in content.</span>
		</div>
	</div>

	<!-- Card 3: What to Include -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<h2>What to Include</h2>
		<p class="anchorkit-card-description">Select which heading levels appear in your Table of Contents.</p>

		<div class="setting-wrapper">
			<h3>Include heading levels:</h3>
			<div class="post-types-grid">
				<?php foreach (array('h1', 'h2', 'h3', 'h4', 'h5', 'h6') as $heading): ?>
					<label>
						<input type="checkbox" name="anchorkit_toc_include_headings[]" value="<?php echo esc_attr($heading); ?>"
							<?php checked(in_array($heading, $include_headings), true); ?>>
						<span class="anchorkit-option-label"><?php echo esc_html(strtoupper($heading)); ?></span>
					</label>
				<?php endforeach; ?>
			</div>
			<span class="helper-text">Choose heading levels (H2-H4 recommended).</span>
		</div>
	</div>

	<!-- Card 4: Style Preset -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<h2>Style Presets</h2>
		<p class="anchorkit-card-description">Choose a beautiful preset that matches your site's design.</p>

		<div class="anchorkit-preset-picker">
			<?php
			$presets = array(
				'minimal' => array(
					'name' => 'Minimal',
					'description' => 'Clean and subtle',
					'preview_class' => 'preset-minimal',
					'pro' => false,
				),
			);

			// PRO PRESETS: This block is stripped by Freemius
			if (anchorkit_fs() && anchorkit_fs()->is__premium_only()) {
				$presets['classic'] = array(
					'name' => 'Classic',
					'description' => 'Traditional with borders',
					'preview_class' => 'preset-classic',
					'pro' => true,
				);
				$presets['modern'] = array(
					'name' => 'Modern',
					'description' => 'Material Design 3',
					'preview_class' => 'preset-modern',
					'pro' => true,
				);
			}
			foreach ($presets as $preset_key => $preset_data):
				// Skip pro presets entirely when user is not pro
				// This block is stripped by Freemius in free version
				if ($preset_data['pro']) {
					if (!(anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code())) {
						continue;
					}
				}
				?>
				<label class="anchorkit-preset-option <?php echo esc_attr($style_preset === $preset_key ? 'selected' : ''); ?>">
					<input type="radio" name="anchorkit_toc_style_preset" value="<?php echo esc_attr($preset_key); ?>" <?php checked($style_preset, $preset_key); ?>>
					<div class="preset-card">
						<div class="preset-preview <?php echo esc_attr($preset_data['preview_class']); ?>">
							<div class="preview-title"></div>
							<div class="preview-item"></div>
							<div class="preview-item preview-item-sub"></div>
							<div class="preview-item"></div>
						</div>
						<?php if ($preset_data['pro']): ?>
							<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
						<?php endif; ?>
						<div class="preset-info">
							<strong><?php echo esc_html($preset_data['name']); ?></strong>
							<span class="preset-description"><?php echo esc_html($preset_data['description']); ?></span>
						</div>
					</div>
					<span class="preset-checkmark">✓</span>
				</label>
			<?php endforeach; ?>
		</div>

		<?php
		// Check if Custom Styling Mode is active
		$custom_styling_active = (bool) get_option('anchorkit_toc_custom_styling', false);

		if ($custom_styling_active):
			?>
			<div class="anchorkit-notice anchorkit-notice-warning">
				<strong>⚠️ Custom Styling Active:</strong> Your preset selection won't be visible in the preview or frontend
				because <strong>Custom Styling Mode</strong> is currently enabled. To use presets, disable custom styling in the
				<strong><a href="?page=anchorkit-settings&tab=appearance-behavior" class="anchorkit-notice-link">Appearance &
						Behavior</a></strong> tab.
			</div>
		<?php endif; ?>

		<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
			<div class="anchorkit-notice anchorkit-notice-info">
				<strong>💡 Tip:</strong> All presets auto-adapt to light and dark mode. Need custom colors? Visit
				<strong>Appearance & Behavior</strong> tab.
			</div>
		<?php endif; ?>
	</div>

	<!-- Card 5: Header Label -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<h2>Title</h2>
		<p class="anchorkit-card-description">Customize the title shown above your Table of Contents.</p>

		<div class="setting-wrapper">
			<label class="m3-switch">
				<input type="checkbox" name="anchorkit_toc_show_label" value="1" <?php checked(1, $show_label, true); ?>>
				<span class="m3-track"><span class="m3-thumb"></span></span>
				<span class="m3-switch-label">
					Show title
				</span>
			</label>
			<span class="helper-text">Show title above TOC.</span>
		</div>

		<div class="form-group conditional-field" data-condition="anchorkit_toc_show_label" data-condition-value="1">
			<div class="anchorkit-input-container">
				<input type="text" class="anchorkit-input" name="anchorkit_toc_title_text" id="anchorkit_toc_title_text"
					value="<?php echo esc_attr($title_text); ?>" placeholder="Contents">
				<label for="anchorkit_toc_title_text">Title text</label>
			</div>
			<span class="helper-text">Title text for TOC header.</span>
		</div>
	</div>

	<!-- Card 6: Need More Control? -->
	<div class="anchorkit-card conditional-field anchorkit-info-card <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<h3 class="anchorkit-info-card-header">🎨 Need more control?</h3>
		<p class="anchorkit-info-card-description">Your TOC is ready to go! For advanced customization, explore:</p>
		<ul class="anchorkit-info-list">
			<li class="anchorkit-info-list-item">→ <strong>Appearance & Behavior</strong> — Colors, fonts, collapsible
				options,
				smooth scrolling</li>
			<li class="anchorkit-info-list-item">→ <strong>Structure & SEO</strong> — Exclude specific headings, custom
				labels
			</li>
			<li>→ <strong>Advanced & Accessibility</strong> — Mobile settings, ARIA labels, accessibility features</li>
		</ul>
	</div>
	<?php
}

// Appearance & Behavior Tab
function anchorkit_render_appearance_behavior_tab($toc_enabled, $collapsible, $initial_state, $smooth_scroll, $hierarchical_view, $show_numerals, $live_preview_enabled, $toc_theme, $bg_color, $text_color, $link_color, $link_hover_color, $border_color, $font_family, $font_size, $padding, $border_width, $border_radius, $bullet_style, $bullet_character, $numbering_style, $numbering_format, $numbering_sep, $title_text, $scroll_offset, $toc_width, $toc_float, $custom_styling, $style_preset, $back_to_top_link, $sticky_enabled, $sticky_position, $sticky_offset, $box_shadow_enabled, $box_shadow_custom, $box_shadow_light, $box_shadow_dark, $entrance_animation, $animation_type, $line_height, $letter_spacing, $text_transform, $link_underline, $custom_id, $custom_class, $custom_css, $view_more_enabled, $initial_count, $view_more_text, $view_less_text, $show_reading_time_global, $reading_speed_global, $show_word_count_global, $time_format_global, $bg_color_light, $text_color_light, $link_color_light, $link_hover_color_light, $border_color_light, $bullet_color_light, $bg_color_dark, $text_color_dark, $link_color_dark, $link_hover_color_dark, $border_color_dark, $bullet_color_dark, $active_link_color_light, $active_link_color_dark, $active_link_color_base, $shadow_h_offset_light, $shadow_v_offset_light, $shadow_blur_light, $shadow_spread_light, $shadow_color_light, $shadow_opacity_light, $shadow_h_offset_dark, $shadow_v_offset_dark, $shadow_blur_dark, $shadow_spread_dark, $shadow_color_dark, $shadow_opacity_dark)
{
	$preset_names = array(
		'minimal' => 'Minimal',
		'classic' => 'Classic',
		'modern' => 'Modern',
	);
	$box_shadow_light_display = $box_shadow_light ?: $box_shadow_custom;
	$box_shadow_dark_display = $box_shadow_dark ?: $box_shadow_custom;
	$active_link_color_display = $active_link_color_light ?: $active_link_color_base;
	$active_link_color_dark_display = $active_link_color_dark ?: $active_link_color_base;
	?>
	<!-- Custom Styling Mode Activation Card -->
	<div id="anchorkit-custom-styling-activation-card"
		class="anchorkit-card conditional-field anchorkit-custom-styling-info <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?> <?php echo esc_attr($custom_styling ? 'warning' : 'info'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<div class="anchorkit-custom-styling-inner">
			<div class="anchorkit-custom-styling-content">
				<h2 class="anchorkit-custom-styling-header">
					<span class="anchorkit-custom-styling-icon">
						<?php echo esc_html($custom_styling ? '🎨' : 'ℹ️'); ?>
					</span>
					<?php echo esc_html($custom_styling ? 'Custom Styling Mode Active' : 'Using Preset Styling'); ?>
				</h2>

				<?php if ($custom_styling): ?>
					<p class="anchorkit-custom-styling-description">
						<strong>You're in full control.</strong> Custom styling is enabled and overriding the
						<strong><?php echo esc_html($preset_names[$style_preset] ?? 'Minimal'); ?></strong> preset.
						All color, typography, and spacing settings below will be applied to your Table of Contents.
					</p>
					<p class="anchorkit-custom-styling-note">
						<strong>Note:</strong> When custom styling is active, your preset's design is replaced with your custom
						settings.
						Use the controls below for colors, typography, borders, and container spacing, and keep managing bullet
						styles and width in their dedicated cards.
					</p>
				<?php else: ?>
					<p class="anchorkit-custom-styling-description">
						<strong>Currently using the <?php echo esc_html($preset_names[$style_preset] ?? 'Minimal'); ?>
							preset.</strong>
						Enable custom styling below to unlock full control over style customization.
					</p>
				<?php endif; ?>

				<div class="setting-wrapper anchorkit-custom-styling-toggle-wrapper">
					<label class="m3-switch">
						<input type="checkbox" name="anchorkit_toc_custom_styling" value="1" <?php checked(1, $custom_styling, true); ?>>
						<span class="m3-track"><span class="m3-thumb"></span></span>
						<span class="m3-switch-label anchorkit-custom-styling-toggle-label">
							<?php echo esc_html($custom_styling ? 'Disable' : 'Enable'); ?> Custom Styling Mode
						</span>
					</label>
					<span class="helper-text anchorkit-custom-styling-helper">
						<?php
						echo esc_html(
							$custom_styling
							? 'Turn this off to return to your preset design.'
							: 'Turn this on to unlock full customization of colors, fonts, and spacing.'
						);
						?>
					</span>
				</div>
			</div>
		</div>
	</div>

	<!-- Card 1: Behavior & Interaction -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<h2>Behavior & Interaction</h2>
		<p class="anchorkit-card-description">Customize how users interact with the Table of Contents.</p>

		<div class="form-group">
			<div class="anchorkit-input-container">
				<select class="anchorkit-select" name="anchorkit_toc_theme" id="anchorkit_toc_theme">
					<option value="system" <?php selected($toc_theme, 'system'); ?>>System Default (Auto Light/Dark)
					</option>
					<option value="light" <?php selected($toc_theme, 'light'); ?>>Light</option>
					<option value="dark" <?php selected($toc_theme, 'dark'); ?>>Dark</option>
				</select>
				<label for="anchorkit_toc_theme">Theme</label>
			</div>
			<span class="helper-text">Choose theme adaptation. System matches user's OS preference.</span>
		</div>

		<div class="setting-wrapper">
			<label class="m3-switch">
				<input type="checkbox" name="anchorkit_toc_collapsible" value="1" <?php checked(1, $collapsible, true); ?>>
				<span class="m3-track"><span class="m3-thumb"></span></span>
				<span class="m3-switch-label">
					Allow TOC to be collapsed/expanded
				</span>
			</label>
			<span class="helper-text">Let users show/hide the TOC.</span>
		</div>

		<div class="form-group conditional-field" data-condition="anchorkit_toc_collapsible" data-condition-value="1">
			<div class="anchorkit-input-container">
				<select class="anchorkit-select" name="anchorkit_toc_initial_state" id="anchorkit_toc_initial_state">
					<option value="expanded" <?php selected($initial_state, 'expanded'); ?>>Expanded</option>
					<option value="collapsed" <?php selected($initial_state, 'collapsed'); ?>>Collapsed</option>
				</select>
				<label for="anchorkit_toc_initial_state">Initial TOC View</label>
			</div>
			<span class="helper-text">TOC state on page load.</span>
		</div>

		<div class="setting-wrapper">
			<label class="m3-switch">
				<input type="checkbox" name="anchorkit_toc_smooth_scroll" value="1" <?php checked(1, $smooth_scroll, true); ?>>
				<span class="m3-track"><span class="m3-thumb"></span></span>
				<span class="m3-switch-label">
					Smooth Scrolling to anchors
				</span>
			</label>
			<span class="helper-text">Animate scrolling to headings.</span>
		</div>

		<div class="form-group">
			<div class="anchorkit-label-with-tooltip">
				<label for="anchorkit_toc_scroll_offset">Scroll Offset (px)</label>
				<?php
				echo wp_kses(
					anchorkit_tooltip(
						'<strong>Compensate for sticky headers</strong><br><br>' .
						'When users click a TOC link, this adds extra space from the top of the viewport to prevent content from hiding behind sticky headers or navigation bars.<br><br>' .
						'<strong>Common values:</strong><br>' .
						'<code>0</code> - No offset (default)<br>' .
						'<code>80</code> - Standard header height<br>' .
						'<code>120</code> - Large header with admin bar<br><br>' .
						'<strong>Tip:</strong> Set this to your sticky header\'s height for perfect alignment.'
					),
					anchorkit_get_allowed_html_tags()
				);
				?>
			</div>
			<div class="anchorkit-input-container">
				<input type="number" class="anchorkit-input" name="anchorkit_toc_scroll_offset"
					id="anchorkit_toc_scroll_offset" value="<?php echo esc_attr($scroll_offset); ?>" min="0" max="200"
					placeholder="0">
				<label for="anchorkit_toc_scroll_offset" class="anchorkit-sr-only">Scroll Offset (px)</label>
			</div>
		</div>
	</div>

	<!-- Card 4: Spacing & Layout (Width Controls) -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1" data-card-group="anchorkit_toc_spacing_layout">
		<h2>Spacing & Layout</h2>
		<p class="anchorkit-card-description">Control the width and positioning of the Table of Contents.</p>

		<div class="form-group">
			<div class="anchorkit-input-container">
				<input type="number" class="anchorkit-input" name="anchorkit_toc_width" id="anchorkit_toc_width_qs"
					value="<?php echo esc_attr(get_option('anchorkit_toc_width', '80')); ?>" min="20" max="100"
					placeholder="80">
				<label for="anchorkit_toc_width_qs">Width (%)</label>
			</div>
			<span class="helper-text">Responsive width as percentage of container.</span>
		</div>

		<div class="form-group">
			<div class="anchorkit-input-container">
				<select class="anchorkit-select" name="anchorkit_toc_alignment" id="anchorkit_toc_alignment">
					<?php $toc_alignment = get_option('anchorkit_toc_alignment', 'center'); ?>
					<option value="left" <?php selected($toc_alignment, 'left'); ?>>Left Aligned</option>
					<option value="center" <?php selected($toc_alignment, 'center'); ?>>Centered</option>
					<option value="right" <?php selected($toc_alignment, 'right'); ?>>Right Aligned</option>
				</select>
				<label for="anchorkit_toc_alignment">Alignment</label>
			</div>
			<span class="helper-text">Horizontal alignment when TOC is narrower than content.</span>
		</div>
	</div>

	<!-- PRO FEATURE: Bullets & Numbering -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<!-- FULL WORKING VERSION (Pro Users) -->
		<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
			<h2>Bullets & Numbering</h2>
			<p class="anchorkit-card-description">Customize the visual markers for your Table of Contents items.</p>

			<div class="form-group">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_bullet_style" id="anchorkit_toc_bullet_style">
						<option value="none" <?php selected($bullet_style, 'none'); ?>>None (no bullets)</option>
						<option value="disc" <?php selected($bullet_style, 'disc'); ?>>Disc (•)</option>
						<option value="circle" <?php selected($bullet_style, 'circle'); ?>>Circle (○)</option>
						<option value="square" <?php selected($bullet_style, 'square'); ?>>Square (▪)</option>
						<option value="arrow" <?php selected($bullet_style, 'arrow'); ?>>Arrow (→)</option>
						<option value="chevron" <?php selected($bullet_style, 'chevron'); ?>>Chevron (›)</option>
						<option value="check" <?php selected($bullet_style, 'check'); ?>>Check (✓)</option>
						<option value="star" <?php selected($bullet_style, 'star'); ?>>Star (★)</option>
						<option value="custom_character" <?php selected($bullet_style, 'custom_character'); ?>>Custom
							Character
						</option>
					</select>
					<label for="anchorkit_toc_bullet_style">Bullet Style</label>
				</div>
				<span class="helper-text">Choose a bullet or numbering style for TOC items.</span>
			</div>

			<!-- Custom Bullet Character Input (conditional) -->
			<div class="form-group conditional-field <?php echo esc_attr($bullet_style === 'custom_character' ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_bullet_style" data-condition-value="custom_character">
				<div class="anchorkit-input-container">
					<input type="text" class="anchorkit-input" name="anchorkit_toc_bullet_character"
						id="anchorkit_toc_bullet_character" value="<?php echo esc_attr($bullet_character); ?>" maxlength="3"
						placeholder="•">
					<label for="anchorkit_toc_bullet_character">Custom Bullet Character</label>
				</div>
				<span class="helper-text">Enter your custom bullet character (1-3 characters, e.g., ►, ✦, ☞).</span>
			</div>

			<p class="helper-text anchorkit-helper-spacing">
				These bullet style controls apply in both preset and Custom Styling modes.
			</p>

			<!-- Hierarchical View Toggle -->
			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_hierarchical_view" value="1" <?php checked(1, $hierarchical_view, true); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">
						Hierarchical View
					</span>
				</label>
				<span class="helper-text">Indent sub-headings to show document hierarchy.</span>
			</div>

			<!-- Show Numerals Toggle -->
			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_show_numerals" value="1" <?php checked(1, $show_numerals, true); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">
						Show Numerals
					</span>
				</label>
				<span class="helper-text">Display numbers for each TOC item (works with hierarchical view).</span>
			</div>

			<!-- Numbering Style (conditional) -->
			<div class="form-group conditional-field <?php echo esc_attr($show_numerals ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_show_numerals" data-condition-value="1">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_numbering_style" id="anchorkit_toc_numbering_style">
						<option value="hierarchical" <?php selected($numbering_style, 'hierarchical'); ?>>Hierarchical (1.2.3)
						</option>
						<option value="flat" <?php selected($numbering_style, 'flat'); ?>>Flat (1, 2, 3, 4...)</option>
					</select>
					<label for="anchorkit_toc_numbering_style">Numbering Style</label>
				</div>
				<span class="helper-text">Choose between hierarchical (1.2.3) or flat (1, 2, 3) numbering.</span>
			</div>

			<!-- Numbering Format (conditional) -->
			<div class="form-group conditional-field <?php echo esc_attr($show_numerals ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_show_numerals" data-condition-value="1">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_numbering_format" id="anchorkit_toc_numbering_format">
						<option value="decimal" <?php selected($numbering_format, 'decimal'); ?>>Decimal (1, 2, 3)</option>
						<option value="decimal_leading_zero" <?php selected($numbering_format, 'decimal_leading_zero'); ?>>
							Decimal Leading Zero (01, 02, 03)</option>
						<option value="upper_roman" <?php selected($numbering_format, 'upper_roman'); ?>>Upper Roman (I, II,
							III)</option>
						<option value="lower_roman" <?php selected($numbering_format, 'lower_roman'); ?>>Lower Roman (i, ii,
							iii)</option>
						<option value="upper_alpha" <?php selected($numbering_format, 'upper_alpha'); ?>>Upper Alpha (A, B, C)
						</option>
						<option value="lower_alpha" <?php selected($numbering_format, 'lower_alpha'); ?>>Lower Alpha (a, b, c)
						</option>
					</select>
					<label for="anchorkit_toc_numbering_format">Numbering Format</label>
				</div>
				<span class="helper-text">Choose how numbers are displayed (decimal, roman numerals, or alphabetic).</span>
			</div>

			<!-- Numbering Separator (conditional) -->
			<div class="form-group conditional-field <?php echo esc_attr($show_numerals ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_show_numerals" data-condition-value="1">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_numbering_separator"
						id="anchorkit_toc_numbering_separator">
						<option value="." <?php selected($numbering_sep, '.'); ?>>Dot (.)</option>
						<option value="-" <?php selected($numbering_sep, '-'); ?>>Dash (-)</option>
						<option value="·" <?php selected($numbering_sep, '·'); ?>>Middle Dot (·)</option>
					</select>
					<label for="anchorkit_toc_numbering_separator">Separator</label>
				</div>
				<span class="helper-text">Character used between hierarchical levels and after the final number.</span>
			</div>

			<div class="anchorkit-notice anchorkit-notice-info-bordered">
				<strong>💡 Tip:</strong> Bullets work great for casual content, while hierarchical numbering (1.2.3) is perfect
				for documentation and tutorials.
			</div>
		</div>
	<?php endif; ?>

	<!-- PRO FEATURE: Reading Time -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
			<h2>Reading Time & Word Count</h2>
			<p class="anchorkit-card-description">Add lightweight metadata beside each heading to highlight reading effort.</p>

			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_show_reading_time" value="1" <?php checked($show_reading_time_global, 1); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">Show Reading Time</span>
				</label>
				<span class="helper-text">Displays labels such as “5 min read” next to each TOC item.</span>
			</div>

			<div class="form-group conditional-field" data-condition="anchorkit_toc_show_reading_time" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_reading_speed"
						id="anchorkit_toc_reading_speed" value="<?php echo esc_attr($reading_speed_global); ?>" min="100"
						max="600" step="10" placeholder="200">
					<label for="anchorkit_toc_reading_speed">Reading Speed (words per minute)</label>
				</div>
				<span class="helper-text">Words per minute for calculation (Default: 200).</span>
			</div>

			<div class="form-group conditional-field" data-condition="anchorkit_toc_show_reading_time" data-condition-value="1">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_time_format" id="anchorkit_toc_time_format">
						<option value="min_read" <?php selected($time_format_global, 'min_read'); ?>>“5 min read”</option>
						<option value="minutes" <?php selected($time_format_global, 'minutes'); ?>>“~5 minutes”</option>
						<option value="short" <?php selected($time_format_global, 'short'); ?>>“5m”</option>
					</select>
					<label for="anchorkit_toc_time_format">Time Format</label>
				</div>
				<span class="helper-text">Choose how the reading time text should appear.</span>
			</div>

			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_show_word_count" value="1" <?php checked($show_word_count_global, 1); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">Show Word Count</span>
				</label>
				<span class="helper-text">Display word count for each TOC section.</span>
			</div>
		</div>
	<?php endif; ?>

	<!-- PRO FEATURE: View More -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<!-- FULL WORKING VERSION (Pro Users) -->
		<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
			<h2>View More</h2>
			<p class="anchorkit-card-description">Show a limited number of content items initially with an expand/collapse
				button.</p>

			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_view_more_enabled" value="1" <?php checked($view_more_enabled, 1); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">Enable View More</span>
				</label>
				<span class="helper-text">Show "View More" button to expand additional items.</span>
			</div>

			<div class="form-group conditional-field" data-condition="anchorkit_toc_view_more_enabled" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_initial_count"
						id="anchorkit_toc_initial_count" value="<?php echo esc_attr($initial_count); ?>" min="1" max="50"
						step="1" placeholder="5">
					<label for="anchorkit_toc_initial_count">Initial Item Count</label>
				</div>
				<span class="helper-text">Number of TOC items to show before the "View More" button (1-50).</span>
			</div>

			<div class="form-group conditional-field" data-condition="anchorkit_toc_view_more_enabled" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="text" class="anchorkit-input" name="anchorkit_toc_view_more_text"
						id="anchorkit_toc_view_more_text" value="<?php echo esc_attr($view_more_text); ?>"
						placeholder="View More">
					<label for="anchorkit_toc_view_more_text">"View More" Button Text</label>
				</div>
				<span class="helper-text">Text displayed on the expand button.</span>
			</div>

			<div class="form-group conditional-field" data-condition="anchorkit_toc_view_more_enabled" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="text" class="anchorkit-input" name="anchorkit_toc_view_less_text"
						id="anchorkit_toc_view_less_text" value="<?php echo esc_attr($view_less_text); ?>"
						placeholder="View Less">
					<label for="anchorkit_toc_view_less_text">"View Less" Button Text</label>
				</div>
				<span class="helper-text">Text displayed on the collapse button.</span>
			</div>
		</div>
	<?php endif; ?>
	<!-- Card 2: Custom Styling Settings -->
	<div id="anchorkit-custom-styling-settings-card"
		class="anchorkit-card conditional-field anchorkit-lockable-card anchorkit-min-h-400 <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?><?php echo esc_attr($custom_styling ? '' : ' is-locked'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1" data-lock-mode="custom-off">
		<h2>Custom Styling Settings</h2>
		<div class="anchorkit-lock-overlay" role="presentation"
			aria-hidden="<?php echo esc_attr($custom_styling ? 'true' : 'false'); ?>">
			<div class="anchorkit-lock-overlay__content">
				<div class="anchorkit-lock-overlay__icon" aria-hidden="true">🔒</div>
				<h3 class="anchorkit-lock-overlay__title">Enable Custom Styling Mode</h3>
				<p class="anchorkit-lock-overlay__message">
					Turn on <strong>Custom Styling Mode</strong> in the activation card above to unlock advanced controls
					for colors, typography, borders, and container spacing.
				</p>
				<p class="anchorkit-lock-overlay__subtitle">
					Bullets &amp; Numbering and Spacing &amp; Layout remain available below for quick tweaks in any mode.
				</p>
			</div>
		</div>
		<p id="anchorkit-custom-styling-settings-description" class="anchorkit-card-description">
			<?php if ($custom_styling): ?>
				Customize colors, fonts, and spacing for your Table of Contents. These settings override the
				<?php echo esc_html($preset_names[$style_preset] ?? 'Minimal'); ?> preset.
			<?php else: ?>
				<em>Enable Custom Styling Mode above to unlock these settings.</em>
			<?php endif; ?>
		</p>

		<div class="conditional-field <?php echo esc_attr($custom_styling ? '' : 'anchorkit-conditional-hidden'); ?>"
			data-condition="anchorkit_toc_custom_styling" data-condition-value="1">

			<!-- Colors Section -->
			<h3>Colors</h3>

			<!-- Hidden inputs for light mode colors -->
			<input type="hidden" name="anchorkit_toc_bg_color_light" id="anchorkit_toc_bg_color_light"
				value="<?php echo esc_attr($bg_color_light); ?>">
			<input type="hidden" name="anchorkit_toc_text_color_light" id="anchorkit_toc_text_color_light"
				value="<?php echo esc_attr($text_color_light); ?>">
			<input type="hidden" name="anchorkit_toc_link_color_light" id="anchorkit_toc_link_color_light"
				value="<?php echo esc_attr($link_color_light); ?>">
			<input type="hidden" name="anchorkit_toc_link_hover_color_light" id="anchorkit_toc_link_hover_color_light"
				value="<?php echo esc_attr($link_hover_color_light); ?>">
			<input type="hidden" name="anchorkit_toc_border_color_light" id="anchorkit_toc_border_color_light"
				value="<?php echo esc_attr($border_color_light); ?>">
			<input type="hidden" name="anchorkit_toc_bullet_color_light" id="anchorkit_toc_bullet_color_light"
				value="<?php echo esc_attr($bullet_color_light); ?>">

			<!-- Hidden inputs for dark mode colors -->
			<input type="hidden" name="anchorkit_toc_bg_color_dark" id="anchorkit_toc_bg_color_dark"
				value="<?php echo esc_attr($bg_color_dark); ?>">
			<input type="hidden" name="anchorkit_toc_text_color_dark" id="anchorkit_toc_text_color_dark"
				value="<?php echo esc_attr($text_color_dark); ?>">
			<input type="hidden" name="anchorkit_toc_link_color_dark" id="anchorkit_toc_link_color_dark"
				value="<?php echo esc_attr($link_color_dark); ?>">
			<input type="hidden" name="anchorkit_toc_link_hover_color_dark" id="anchorkit_toc_link_hover_color_dark"
				value="<?php echo esc_attr($link_hover_color_dark); ?>">
			<input type="hidden" name="anchorkit_toc_border_color_dark" id="anchorkit_toc_border_color_dark"
				value="<?php echo esc_attr($border_color_dark); ?>">
			<input type="hidden" name="anchorkit_toc_bullet_color_dark" id="anchorkit_toc_bullet_color_dark"
				value="<?php echo esc_attr($bullet_color_dark); ?>">

			<!-- Row 1: Background + Text -->
			<div class="color-picker-row">
				<div class="input-container anchorkit-color-picker">
					<input type="color" id="anchorkit_toc_bg_color" class="theme-aware-color" data-color-type="bg_color"
						value="<?php echo esc_attr($bg_color_light); ?>">
					<label for="anchorkit_toc_bg_color">Background Color</label>
					<span class="color-value"><?php echo esc_html(strtoupper($bg_color_light)); ?></span>
				</div>
				<div class="input-container anchorkit-color-picker">
					<input type="color" id="anchorkit_toc_text_color" class="theme-aware-color" data-color-type="text_color"
						value="<?php echo esc_attr($text_color_light); ?>">
					<label for="anchorkit_toc_text_color">Text Color</label>
					<span class="color-value"><?php echo esc_html(strtoupper($text_color_light)); ?></span>
				</div>
			</div>

			<!-- Row 2: Link + Link Hover -->
			<div class="color-picker-row">
				<div class="input-container anchorkit-color-picker">
					<input type="color" id="anchorkit_toc_link_color" class="theme-aware-color" data-color-type="link_color"
						value="<?php echo esc_attr($link_color_light); ?>">
					<label for="anchorkit_toc_link_color">Link Color</label>
					<span class="color-value"><?php echo esc_html(strtoupper($link_color_light)); ?></span>
				</div>
				<div class="input-container anchorkit-color-picker">
					<input type="color" id="anchorkit_toc_link_hover_color" class="theme-aware-color"
						data-color-type="link_hover_color" value="<?php echo esc_attr($link_hover_color_light); ?>">
					<label for="anchorkit_toc_link_hover_color">Link Hover Color</label>
					<span class="color-value"><?php echo esc_html(strtoupper($link_hover_color_light)); ?></span>
				</div>
			</div>

			<!-- Row 3: Border Color -->
			<div class="color-picker-row">
				<div class="input-container anchorkit-color-picker">
					<input type="color" id="anchorkit_toc_border_color" class="theme-aware-color"
						data-color-type="border_color" value="<?php echo esc_attr($border_color_light); ?>">
					<label for="anchorkit_toc_border_color">Border Color</label>
					<span class="color-value"><?php echo esc_html(strtoupper($border_color_light)); ?></span>
				</div>
			</div>

			<!-- Row 4: Bullet Color -->
			<div class="color-picker-row">
				<div class="input-container anchorkit-color-picker">
					<input type="color" id="anchorkit_toc_bullet_color" class="theme-aware-color"
						data-color-type="bullet_color" value="<?php echo esc_attr($bullet_color_light); ?>">
					<label for="anchorkit_toc_bullet_color">Bullet Color</label>
					<span class="color-value"><?php echo esc_html(strtoupper($bullet_color_light)); ?></span>
				</div>
			</div>
			<span class="helper-text anchorkit-mt-neg-8 anchorkit-mb-16">Color for bullets and
				numbers. Default is the theme's link color.</span>

			<hr class="section-divider">

			<h3>Typography</h3>

			<div class="form-group">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_font_family" id="anchorkit_toc_font_family">
						<option value="inherit" <?php selected($font_family, 'inherit'); ?>>Inherit from Theme</option>
						<option value="sans-serif" <?php selected($font_family, 'sans-serif'); ?>>System Sans-Serif
						</option>
						<option value="serif" <?php selected($font_family, 'serif'); ?>>System Serif</option>
						<option value="monospace" <?php selected($font_family, 'monospace'); ?>>System Monospace</option>
					</select>
					<label for="anchorkit_toc_font_family">Font Family</label>
				</div>
				<span class="helper-text">Font family for all TOC text.</span>
			</div>

			<div class="form-group">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_font_size" id="anchorkit_toc_font_size"
						value="<?php echo esc_attr($font_size); ?>" min="10" max="24" placeholder="14">
					<label for="anchorkit_toc_font_size">Base Font Size (px)</label>
				</div>
				<span class="helper-text">Base font size. All heading levels scale from this. Default: 14px</span>
			</div>

			<hr class="section-divider">

			<h3>Spacing & Layout</h3>

			<!-- Compact row: Padding, Border Width, Border Radius -->
			<div class="compact-controls-row">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_padding" id="anchorkit_toc_padding"
						value="<?php echo esc_attr($padding); ?>" min="0" max="50" placeholder="15">
					<label for="anchorkit_toc_padding">Padding (px)</label>
				</div>
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_border_width"
						id="anchorkit_toc_border_width" value="<?php echo esc_attr($border_width); ?>" min="0" max="10"
						placeholder="1">
					<label for="anchorkit_toc_border_width">Border (px)</label>
				</div>
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_border_radius"
						id="anchorkit_toc_border_radius" value="<?php echo esc_attr($border_radius); ?>" min="0" max="30"
						placeholder="4">
					<label for="anchorkit_toc_border_radius">Radius (px)</label>
				</div>
			</div>
			<span class="compact-row-helper">Inner padding, border width (0 = none), and corner radius.</span>

		</div><!-- End conditional custom styling -->
	</div>
	<!-- PRO FEATURE: Advanced Typography -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<!-- FULL WORKING VERSION (Pro Users) -->
		<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
			<h2>Advanced Typography</h2>
			<p class="anchorkit-card-description">Enable granular font size control for each heading level. When enabled, these
				settings override the base font size.</p>

			<!-- Toggle for Advanced Typography Override -->
			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_advanced_typography_override" value="1" <?php checked(get_option('anchorkit_toc_advanced_typography_override', false), true); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">
						Override Base Font Size
					</span>
				</label>
				<span class="helper-text">Set custom font sizes per heading level.</span>
			</div>

			<!-- Font Size Fields (conditional) -->
			<div class="form-group conditional-field <?php echo esc_attr(get_option('anchorkit_toc_advanced_typography_override', false) ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_advanced_typography_override" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_title_font_size"
						id="anchorkit_toc_title_font_size"
						value="<?php echo esc_attr(get_option('anchorkit_toc_title_font_size', '20')); ?>" min="12" max="32"
						placeholder="20">
					<label for="anchorkit_toc_title_font_size">Title Font Size (px)</label>
				</div>
				<span class="helper-text">Font size for "Table of Contents" title. Default: 20px</span>
			</div>

			<div class="form-group conditional-field <?php echo esc_attr(get_option('anchorkit_toc_advanced_typography_override', false) ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_advanced_typography_override" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_h2_font_size"
						id="anchorkit_toc_h2_font_size"
						value="<?php echo esc_attr(get_option('anchorkit_toc_h2_font_size', '18')); ?>" min="10" max="24"
						placeholder="18">
					<label for="anchorkit_toc_h2_font_size">H2 Items Font Size (px)</label>
				</div>
				<span class="helper-text">Font size for H2 level items. Default: 18px</span>
			</div>

			<div class="form-group conditional-field <?php echo esc_attr(get_option('anchorkit_toc_advanced_typography_override', false) ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_advanced_typography_override" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_h3_font_size"
						id="anchorkit_toc_h3_font_size"
						value="<?php echo esc_attr(get_option('anchorkit_toc_h3_font_size', '16')); ?>" min="10" max="24"
						placeholder="16">
					<label for="anchorkit_toc_h3_font_size">H3 Items Font Size (px)</label>
				</div>
				<span class="helper-text">Font size for H3 level items. Default: 16px</span>
			</div>

			<div class="form-group conditional-field <?php echo esc_attr(get_option('anchorkit_toc_advanced_typography_override', false) ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_advanced_typography_override" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_h4_font_size"
						id="anchorkit_toc_h4_font_size"
						value="<?php echo esc_attr(get_option('anchorkit_toc_h4_font_size', '14')); ?>" min="10" max="24"
						placeholder="14">
					<label for="anchorkit_toc_h4_font_size">H4 Items Font Size (px)</label>
				</div>
				<span class="helper-text">Font size for H4 level items. Default: 14px</span>
			</div>

			<div class="form-group conditional-field <?php echo esc_attr(get_option('anchorkit_toc_advanced_typography_override', false) ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_advanced_typography_override" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_h5_font_size"
						id="anchorkit_toc_h5_font_size"
						value="<?php echo esc_attr(get_option('anchorkit_toc_h5_font_size', '13')); ?>" min="10" max="24"
						placeholder="13">
					<label for="anchorkit_toc_h5_font_size">H5 Items Font Size (px)</label>
				</div>
				<span class="helper-text">Font size for H5 level items. Default: 13px</span>
			</div>

			<div class="form-group conditional-field <?php echo esc_attr(get_option('anchorkit_toc_advanced_typography_override', false) ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_advanced_typography_override" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_h6_font_size"
						id="anchorkit_toc_h6_font_size"
						value="<?php echo esc_attr(get_option('anchorkit_toc_h6_font_size', '12')); ?>" min="10" max="24"
						placeholder="12">
					<label for="anchorkit_toc_h6_font_size">H6 Items Font Size (px)</label>
				</div>
				<span class="helper-text">Font size for H6 level items. Default: 12px</span>
			</div>

			<h3>Text Styling</h3>

			<div class="form-group">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_line_height" id="anchorkit_toc_line_height"
						value="<?php echo esc_attr(get_option('anchorkit_toc_line_height', '1.6')); ?>" min="1" max="3"
						step="0.1" placeholder="1.6">
					<label for="anchorkit_toc_line_height">Line Height</label>
				</div>
				<span class="helper-text">Line height for TOC items. Default: 1.6 (Min: 1, Max: 3)</span>
			</div>

			<div class="form-group">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_letter_spacing"
						id="anchorkit_toc_letter_spacing"
						value="<?php echo esc_attr(get_option('anchorkit_toc_letter_spacing', '0')); ?>" min="-2" max="5"
						step="0.1" placeholder="0">
					<label for="anchorkit_toc_letter_spacing">Letter Spacing (px)</label>
				</div>
				<span class="helper-text">Letter spacing for TOC text. Default: 0 (Min: -2, Max: 5)</span>
			</div>

			<div class="form-group">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_text_transform" id="anchorkit_toc_text_transform">
						<option value="none" <?php selected(get_option('anchorkit_toc_text_transform', 'none'), 'none'); ?>>
							None
						</option>
						<option value="uppercase" <?php selected(get_option('anchorkit_toc_text_transform', 'none'), 'uppercase'); ?>>UPPERCASE</option>
						<option value="lowercase" <?php selected(get_option('anchorkit_toc_text_transform', 'none'), 'lowercase'); ?>>lowercase</option>
						<option value="capitalize" <?php selected(get_option('anchorkit_toc_text_transform', 'none'), 'capitalize'); ?>>Capitalize Each Word</option>
					</select>
					<label for="anchorkit_toc_text_transform">Text Transform</label>
				</div>
				<span class="helper-text">Text transformation style for TOC items.</span>
			</div>

			<div class="form-group">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_link_underline" id="anchorkit_toc_link_underline">
						<option value="none" <?php selected($link_underline, 'none'); ?>>None</option>
						<option value="always" <?php selected($link_underline, 'always'); ?>>Always Underlined</option>
						<option value="hover" <?php selected($link_underline, 'hover'); ?>>Underline on Hover</option>
						<option value="hover_none" <?php selected($link_underline, 'hover_none'); ?>>No Underline on Hover
						</option>
					</select>
					<label for="anchorkit_toc_link_underline">Link Underline</label>
				</div>
				<span class="helper-text">Choose when to display underlines on TOC links.</span>
			</div>
		</div>
	<?php endif; ?>

	<!-- PRO FEATURE: Sticky/Fixed Position -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<!-- FULL WORKING VERSION (Pro Users) -->
		<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
			<h2>Sticky/Fixed Position</h2>
			<p class="anchorkit-card-description">Keep the Table of Contents visible as users scroll down the page.</p>

			<!-- Enable Sticky TOC -->
			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_sticky_enabled" value="1" <?php checked($sticky_enabled, 1); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">Enable Sticky TOC</span>
				</label>
				<span class="helper-text">TOC will stick to the screen while scrolling, keeping navigation always
					accessible.</span>
			</div>

			<!-- Sticky Position -->
			<div class="form-group conditional-field" data-condition="anchorkit_toc_sticky_enabled" data-condition-value="1">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_sticky_position" id="anchorkit_toc_sticky_position">
						<option value="content" <?php selected($sticky_position, 'content'); ?>>In Content (Default)</option>
						<option value="left" <?php selected($sticky_position, 'left'); ?>>Floating Left Sidebar</option>
						<option value="right" <?php selected($sticky_position, 'right'); ?>>Floating Right Sidebar</option>
					</select>
					<label for="anchorkit_toc_sticky_position">Sticky Position</label>
				</div>
				<span class="helper-text">Choose where the sticky TOC appears. "In Content" keeps it in its original position
					within your content.</span>
			</div>

			<!-- Offset from Top -->
			<div class="form-group conditional-field" data-condition="anchorkit_toc_sticky_enabled" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_sticky_offset"
						id="anchorkit_toc_sticky_offset" value="<?php echo esc_attr($sticky_offset); ?>" min="0" max="200"
						placeholder="20">
					<label for="anchorkit_toc_sticky_offset">Offset from Top (px)</label>
				</div>
				<span class="helper-text">Distance from top of viewport when TOC becomes sticky. Useful if you have a fixed
					header. Default: 20px (Min: 0, Max: 200).</span>
			</div>

			<!-- Scroll Spy (requires sticky) -->
			<div class="form-group conditional-field" data-condition="anchorkit_toc_sticky_enabled" data-condition-value="1">
				<input type="hidden" name="anchorkit_toc_active_link_color_light" id="anchorkit_toc_active_link_color_light"
					value="<?php echo esc_attr($active_link_color_display); ?>">
				<input type="hidden" name="anchorkit_toc_active_link_color_dark" id="anchorkit_toc_active_link_color_dark"
					value="<?php echo esc_attr($active_link_color_dark_display); ?>">
				<input type="hidden" name="anchorkit_toc_active_link_color" id="anchorkit_toc_active_link_color_base"
					value="<?php echo esc_attr($active_link_color_base); ?>">
				<div class="input-container anchorkit-color-picker">
					<input type="color" id="anchorkit_toc_active_link_color" class="theme-aware-color"
						data-color-type="active_link_color" value="<?php echo esc_attr($active_link_color_display); ?>">
					<label for="anchorkit_toc_active_link_color">Active Link Color (Scroll Spy)</label>
					<span class="color-value"><?php echo esc_html(strtoupper($active_link_color_display)); ?></span>
				</div>
				<span class="helper-text">Automatically highlights the TOC link for the currently visible section as users
					scroll. Dark mode currently uses
					<?php echo esc_html(strtoupper($active_link_color_dark_display)); ?>.</span>
			</div>

			<!-- Back to Top Link (requires sticky) -->
			<div class="setting-wrapper conditional-field" data-condition="anchorkit_toc_sticky_enabled"
				data-condition-value="1">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_back_to_top_link" value="1" <?php checked(1, $back_to_top_link, true); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">
						Add "Back to Top" link
					</span>
				</label>
				<span class="helper-text">Quick link to return to the top of the page.</span>
			</div>

			<div class="form-group conditional-field <?php echo esc_attr($back_to_top_link ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_back_to_top_link" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="text" class="anchorkit-input" name="anchorkit_toc_back_to_top_text"
						id="anchorkit_toc_back_to_top_text"
						value="<?php echo esc_attr(get_option('anchorkit_toc_back_to_top_text', 'Back to Top')); ?>"
						placeholder="Back to Top">
					<label for="anchorkit_toc_back_to_top_text">"Back to Top" Text</label>
				</div>
				<span class="helper-text">Text for back to top link.</span>
			</div>

			<div class="form-group conditional-field <?php echo esc_attr($back_to_top_link ? '' : 'anchorkit-conditional-hidden'); ?>"
				data-condition="anchorkit_toc_back_to_top_link" data-condition-value="1">
				<div class="anchorkit-input-container">
					<input type="number" class="anchorkit-input" name="anchorkit_toc_back_to_top_font_size"
						id="anchorkit_toc_back_to_top_font_size"
						value="<?php echo esc_attr(get_option('anchorkit_toc_back_to_top_font_size', 14)); ?>" min="10" max="24"
						placeholder="14">
					<label for="anchorkit_toc_back_to_top_font_size">"Back to Top" Font Size (px)</label>
				</div>
				<span class="helper-text">Font size for the back to top link. Default: 14px (Min: 10, Max: 24)</span>
			</div>
		</div>
	<?php endif; ?>
	<!-- PRO FEATURE: Advanced Styling -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<!-- FULL WORKING VERSION (Pro Users) -->
		<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
			<h2>Advanced Styling</h2>
			<p class="anchorkit-card-description">Unlock advanced design options for pixel-perfect customization.</p>

			<!-- Custom Box Shadow -->
			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_box_shadow_enabled" value="1" <?php checked($box_shadow_enabled, 1); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">Custom Box Shadow</span>
				</label>
				<span class="helper-text">Add depth and visual interest with a custom shadow effect.</span>
			</div>

			<!-- Box Shadow Components (Theme-Aware) -->
			<div class="form-group conditional-field" data-condition="anchorkit_toc_box_shadow_enabled"
				data-condition-value="1">
				<h3 class="anchorkit-shadow-section-title">Shadow Configuration
				</h3>
				<span class="helper-text anchorkit-shadow-section-description">Configure shadow properties for light
					and dark modes separately. The fields update based on the preview mode selected.</span>

				<!-- Hidden fields to store computed box-shadow strings -->
				<input type="hidden" name="anchorkit_toc_box_shadow_light" id="anchorkit_toc_box_shadow_light"
					value="<?php echo esc_attr($box_shadow_light_display); ?>">
				<input type="hidden" name="anchorkit_toc_box_shadow_dark" id="anchorkit_toc_box_shadow_dark"
					value="<?php echo esc_attr($box_shadow_dark_display); ?>">
				<input type="hidden" name="anchorkit_toc_box_shadow_custom" id="anchorkit_toc_box_shadow_custom_hidden"
					value="<?php echo esc_attr($box_shadow_light_display); ?>">

				<!-- Theme-aware hidden inputs for component values (Light Mode) -->
				<input type="hidden" name="anchorkit_toc_shadow_h_offset_light" id="anchorkit_toc_shadow_h_offset_light"
					value="<?php echo esc_attr($shadow_h_offset_light); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_v_offset_light" id="anchorkit_toc_shadow_v_offset_light"
					value="<?php echo esc_attr($shadow_v_offset_light); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_blur_light" id="anchorkit_toc_shadow_blur_light"
					value="<?php echo esc_attr($shadow_blur_light); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_spread_light" id="anchorkit_toc_shadow_spread_light"
					value="<?php echo esc_attr($shadow_spread_light); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_color_light" id="anchorkit_toc_shadow_color_light"
					value="<?php echo esc_attr($shadow_color_light); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_opacity_light" id="anchorkit_toc_shadow_opacity_light"
					value="<?php echo esc_attr($shadow_opacity_light); ?>">

				<!-- Theme-aware hidden inputs for component values (Dark Mode) -->
				<input type="hidden" name="anchorkit_toc_shadow_h_offset_dark" id="anchorkit_toc_shadow_h_offset_dark"
					value="<?php echo esc_attr($shadow_h_offset_dark); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_v_offset_dark" id="anchorkit_toc_shadow_v_offset_dark"
					value="<?php echo esc_attr($shadow_v_offset_dark); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_blur_dark" id="anchorkit_toc_shadow_blur_dark"
					value="<?php echo esc_attr($shadow_blur_dark); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_spread_dark" id="anchorkit_toc_shadow_spread_dark"
					value="<?php echo esc_attr($shadow_spread_dark); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_color_dark" id="anchorkit_toc_shadow_color_dark"
					value="<?php echo esc_attr($shadow_color_dark); ?>">
				<input type="hidden" name="anchorkit_toc_shadow_opacity_dark" id="anchorkit_toc_shadow_opacity_dark"
					value="<?php echo esc_attr($shadow_opacity_dark); ?>">

				<!-- Visible component inputs (theme-aware) -->
				<div class="shadow-components-grid anchorkit-grid-2col">
					<!-- Horizontal Offset -->
					<div class="anchorkit-input-container">
						<input type="number" class="anchorkit-input theme-aware-shadow-component" data-component="h_offset"
							id="shadow_h_offset_visible" value="<?php echo esc_attr($shadow_h_offset_light); ?>" min="-100"
							max="100" step="1">
						<label for="shadow_h_offset_visible">Horizontal Offset (px)</label>
					</div>

					<!-- Vertical Offset -->
					<div class="anchorkit-input-container">
						<input type="number" class="anchorkit-input theme-aware-shadow-component" data-component="v_offset"
							id="shadow_v_offset_visible" value="<?php echo esc_attr($shadow_v_offset_light); ?>" min="-100"
							max="100" step="1">
						<label for="shadow_v_offset_visible">Vertical Offset (px)</label>
					</div>
				</div>

				<div class="shadow-components-grid anchorkit-grid-2col">
					<!-- Blur Radius -->
					<div class="anchorkit-input-container">
						<input type="number" class="anchorkit-input theme-aware-shadow-component" data-component="blur"
							id="shadow_blur_visible" value="<?php echo esc_attr($shadow_blur_light); ?>" min="0" max="100"
							step="1">
						<label for="shadow_blur_visible">Blur Radius (px)</label>
					</div>

					<!-- Spread Radius -->
					<div class="anchorkit-input-container">
						<input type="number" class="anchorkit-input theme-aware-shadow-component" data-component="spread"
							id="shadow_spread_visible" value="<?php echo esc_attr($shadow_spread_light); ?>" min="-50" max="50"
							step="1">
						<label for="shadow_spread_visible">Spread Radius (px)</label>
					</div>
				</div>

				<div class="shadow-components-grid anchorkit-grid-2col-last">
					<!-- Shadow Color -->
					<div class="input-container anchorkit-color-picker">
						<input type="color" class="theme-aware-shadow-component" data-component="color"
							id="shadow_color_visible" value="<?php echo esc_attr($shadow_color_light); ?>">
						<label for="shadow_color_visible">Shadow Color</label>
						<span class="color-value"><?php echo esc_html(strtoupper($shadow_color_light)); ?></span>
					</div>

					<!-- Opacity -->
					<div class="anchorkit-input-container">
						<input type="number" class="anchorkit-input theme-aware-shadow-component" data-component="opacity"
							id="shadow_opacity_visible" value="<?php echo esc_attr($shadow_opacity_light); ?>" min="0" max="1"
							step="0.01" placeholder="0.1">
						<label for="shadow_opacity_visible">Opacity</label>
					</div>
				</div>

				<span class="helper-text anchorkit-helper-bottom">Changes are reflected on the preview TOC.
					Switch between light/dark modes to configure each theme separately.</span>
			</div>

			<!-- Entrance Animations -->
			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_entrance_animation" value="1" <?php checked($entrance_animation, 1); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">Entrance Animations</span>
				</label>
				<span class="helper-text">Animate the TOC when it first appears on the page.</span>
			</div>

			<!-- Animation Type -->
			<div class="form-group conditional-field" data-condition="anchorkit_toc_entrance_animation"
				data-condition-value="1">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_animation_type" id="anchorkit_toc_animation_type">
						<option value="fade" <?php selected($animation_type, 'fade'); ?>>Fade In</option>
						<option value="slide" <?php selected($animation_type, 'slide'); ?>>Slide Up</option>
						<option value="zoom" <?php selected($animation_type, 'zoom'); ?>>Zoom In</option>
					</select>
					<label for="anchorkit_toc_animation_type">Animation Type</label>
				</div>
				<span class="helper-text">Choose how the TOC appears on the page.</span>
			</div>

			<!-- Custom TOC ID -->
			<div class="form-group">
				<div class="anchorkit-input-container">
					<input type="text" class="anchorkit-input" name="anchorkit_toc_custom_id" id="anchorkit_toc_custom_id"
						value="<?php echo esc_attr($custom_id); ?>" placeholder="toc-container">
					<label for="anchorkit_toc_custom_id">TOC Container ID</label>
				</div>
				<span class="helper-text">Custom ID for TOC container (for CSS/JavaScript targeting).</span>
			</div>

			<!-- CSS Classes -->
			<div class="form-group">
				<label style="display: block; margin-bottom: 8px; font-weight: 500;">CSS Classes</label>
				<!-- Hidden field to store space-separated values -->
				<input type="hidden" name="anchorkit_toc_custom_class" id="anchorkit_toc_custom_class"
					value="<?php echo esc_attr($custom_class); ?>">

				<!-- Visual interface for CSS classes -->
				<div id="anchorkit_css_classes_container" class="anchorkit-css-classes-container">
					<!-- JavaScript will populate this -->
				</div>

				<button type="button" class="anchorkit-add-class-btn" id="anchorkit_add_css_class">
					<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16">
						<path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z" />
					</svg>
					Add CSS Class
				</button>

				<span class="helper-text" style="margin-top: 12px; display: block;">Add CSS class names (with or without
					<code>.</code> prefix).</span>
			</div>

			<!-- Custom CSS Field -->
			<div class="form-group">
				<div class="anchorkit-input-container textarea-container">
					<textarea class="anchorkit-input anchorkit-textarea" name="anchorkit_toc_custom_css"
						id="anchorkit_toc_custom_css" rows="5"
						placeholder=".my-toc { border: 2px solid #1a73e8; }"><?php echo esc_textarea($custom_css); ?></textarea>
					<label for="anchorkit_toc_custom_css">Additional CSS</label>
				</div>
				<span class="helper-text">Custom CSS rules for TOC styling. Example:
					<code>.my-toc { border: 2px solid red; }</code></span>
			</div>
		</div>
	<?php endif; ?>
<?php
}
// Content & SEO Tab (Simplified from Structure & SEO)
function anchorkit_render_structure_seo_tab($toc_enabled, $include_headings, $exclude_selectors)
{
	// Get SEO settings
	$anchor_format = get_option('anchorkit_toc_anchor_format', 'auto');
	$anchor_prefix = get_option('anchorkit_toc_anchor_prefix', 'section');

	// Pro features - Schema.org & Rich Snippets
	$schema_enabled = get_option('anchorkit_toc_schema_enabled', false);
	$schema_type = get_option('anchorkit_toc_schema_type', 'Article');

	// Pro features - ACF Integration
	$acf_enabled = (bool) get_option('anchorkit_toc_acf_enabled', false);
	$acf_field_names = get_option('anchorkit_toc_acf_field_names', '');
	$acf_merge_mode = get_option('anchorkit_toc_acf_merge_mode', 'after');
	$valid_merge_modes = array('before', 'after', 'replace');
	if (!in_array($acf_merge_mode, $valid_merge_modes, true)) {
		$acf_merge_mode = 'after';
	}
	?>
	<!-- Card 1: Content Exclusion -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<h2>Content Exclusion</h2>
		<p class="anchorkit-card-description">Exclude specific headings from appearing in your Table of Contents.</p>

		<div class="form-group">
			<div class="anchorkit-label-with-tooltip">
				<label for="anchorkit_toc_exclude_selectors">Exclude Headings (CSS Selectors)</label>
				<?php
				echo wp_kses(
					anchorkit_tooltip(
						'<strong>Exclude specific headings from TOC</strong><br><br>' .
						'Enter comma-separated CSS selectors:<br>' .
						'<code>.toc-exclude</code><br>' .
						'<code>#sidebar h2</code><br>' .
						'<code>.wp-block-heading.no-toc</code><br><br>' .
						'Perfect for excluding headings in sidebars, footers, or promotional content.'
					),
					anchorkit_get_allowed_html_tags()
				);
				?>
			</div>
			<div class="anchorkit-input-container">
				<input type="text" class="anchorkit-input" name="anchorkit_toc_exclude_selectors"
					id="anchorkit_toc_exclude_selectors" value="<?php echo esc_attr($exclude_selectors); ?>"
					placeholder=".toc-exclude, .no-toc">
				<label for="anchorkit_toc_exclude_selectors" class="anchorkit-sr-only">Exclude Headings (CSS
					Selectors)</label>
			</div>
		</div>
	</div>

	<!-- Card 2: SEO & Anchor IDs -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<div class="anchorkit-card anchorkit-pro-feature conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<h2>SEO & Anchor IDs</h2>
			<p class="anchorkit-card-description">Optimize anchor links for better SEO and shareable URLs.</p>

			<div class="form-group">
				<div class="anchorkit-label-with-tooltip">
					<label for="anchorkit_toc_anchor_format">Anchor ID Format</label>
					<?php
					echo wp_kses(
						anchorkit_tooltip(
							'<strong>How anchor IDs are created</strong><br><br>' .
							'<strong>Auto-generated (recommended):</strong><br>' .
							'Creates clean IDs from heading text<br>' .
							'<code>#introduction</code>, <code>#getting-started</code><br><br>' .
							'<strong>Sequential (Pro):</strong><br>' .
							'Simple numbered IDs<br>' .
							'<code>#toc-1</code>, <code>#toc-2</code>, <code>#toc-3</code><br><br>' .
							'<strong>Prefixed (Pro):</strong><br>' .
							'Custom prefix + heading text<br>' .
							'<code>#section-introduction</code><br><br>' .
							'SEO-friendly IDs improve search ranking and social sharing.'
						),
						anchorkit_get_allowed_html_tags()
					);
					?>
				</div>
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_anchor_format" id="anchorkit_toc_anchor_format">
						<option value="auto" <?php selected($anchor_format, 'auto'); ?>>Auto-generated from heading text
						</option>
						<option value="sequential" <?php selected($anchor_format, 'sequential'); ?>>Sequential numbers (toc-1,
							toc-2, toc-3)</option>
						<option value="prefixed" <?php selected($anchor_format, 'prefixed'); ?>>Custom prefix + heading text
						</option>
					</select>
					<label for="anchorkit_toc_anchor_format" class="anchorkit-sr-only">Anchor ID Format</label>
				</div>
			</div>

			<div class="form-group conditional-field" data-condition="anchorkit_toc_anchor_format"
				data-condition-value="prefixed">
				<div class="anchorkit-input-container">
					<input type="text" class="anchorkit-input" name="anchorkit_toc_anchor_prefix"
						id="anchorkit_toc_anchor_prefix" value="<?php echo esc_attr($anchor_prefix); ?>" placeholder="section">
					<label for="anchorkit_toc_anchor_prefix">Custom Anchor Prefix</label>
				</div>
			</div>

			<div class="anchorkit-notice anchorkit-notice-success">
				<strong>✅ SEO Benefit:</strong> Clean, descriptive anchor IDs improve search engine indexing and make your links
				more shareable on social media.
			</div>
		</div>
	<?php endif; ?>

	<!-- PRO FEATURE: Advanced SEO -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<div class="anchorkit-card anchorkit-pro-feature conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
			<h2>Advanced SEO</h2>
			<p class="anchorkit-card-description">Boost search visibility with Schema.org markup and rich snippets.</p>

			<!-- Enable Schema.org -->
			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_schema_enabled" value="1" <?php checked($schema_enabled, 1); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">Enable Schema.org markup</span>
				</label>
				<span class="helper-text">Add structured data for search engines.</span>
			</div>

			<!-- Schema Type -->
			<div class="form-group conditional-field" data-condition="anchorkit_toc_schema_enabled" data-condition-value="1">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_schema_type" id="anchorkit_toc_schema_type">
						<option value="Article" <?php selected($schema_type, 'Article'); ?>>Article</option>
						<option value="BlogPosting" <?php selected($schema_type, 'BlogPosting'); ?>>Blog Posting</option>
						<option value="WebPage" <?php selected($schema_type, 'WebPage'); ?>>Web Page</option>
						<option value="HowTo" <?php selected($schema_type, 'HowTo'); ?>>How-To Guide</option>
						<option value="FAQPage" <?php selected($schema_type, 'FAQPage'); ?>>FAQ Page</option>
					</select>
					<label for="anchorkit_toc_schema_type">Schema Type</label>
				</div>
				<span class="helper-text">Match schema type to content.</span>
			</div>

			<div class="setting-wrapper">
				<h3>Per-Post Type Schema Settings</h3>
				<p class="anchorkit-card-description" style="margin-bottom: 15px;">Override the global schema type for specific
					post types.</p>
				<?php
				$toc_post_types = get_option('anchorkit_toc_post_types', array('post', 'page'));
				$per_post_type_schema = get_option('anchorkit_toc_schema_type_per_post_type', array());

				foreach ($toc_post_types as $post_type):
					$post_type_obj = get_post_type_object($post_type);
					if (!$post_type_obj) {
						continue;
					}

					$current_schema = isset($per_post_type_schema[$post_type]) ? $per_post_type_schema[$post_type] : '';
					?>
					<div class="form-group conditional-field" data-condition="anchorkit_toc_schema_enabled"
						data-condition-value="1">
						<div class="anchorkit-input-container">
							<select class="anchorkit-select"
								name="anchorkit_toc_schema_type_per_post_type[<?php echo esc_attr($post_type); ?>]"
								id="anchorkit_toc_schema_type_<?php echo esc_attr($post_type); ?>">
								<option value="" <?php selected($current_schema, ''); ?>>Use Global Default
									(<?php echo esc_html($schema_type); ?>)</option>
								<option value="Article" <?php selected($current_schema, 'Article'); ?>>Article</option>
								<option value="BlogPosting" <?php selected($current_schema, 'BlogPosting'); ?>>Blog Posting
								</option>
								<option value="WebPage" <?php selected($current_schema, 'WebPage'); ?>>Web Page</option>
								<option value="HowTo" <?php selected($current_schema, 'HowTo'); ?>>How-To Guide</option>
								<option value="FAQPage" <?php selected($current_schema, 'FAQPage'); ?>>FAQ Page</option>
							</select>
							<label
								for="anchorkit_toc_schema_type_<?php echo esc_attr($post_type); ?>"><?php echo esc_html($post_type_obj->labels->singular_name); ?></label>
						</div>
					</div>
				<?php endforeach; ?>
			</div>

			<div class="anchorkit-notice anchorkit-notice-success">
				<strong>✅ SEO Boost:</strong> Schema markup helps search engines understand your content structure and may
				display jump links in search results.
			</div>
		</div>
	<?php endif; ?>
<?php
}

// Advanced Settings Tab (Enhanced from Advanced & Accessibility)
function anchorkit_render_advanced_accessibility_tab($toc_enabled, $hide_on_mobile, $aria_label, $back_to_top_link)
{
	// Get custom labels setting (moved from Content & SEO tab)
	$custom_labels = get_option('anchorkit_toc_custom_labels', '');

	// Pro features - Enhanced Navigation
	$scroll_easing = get_option('anchorkit_toc_scroll_easing', 'ease-in-out');
	$scroll_duration = get_option('anchorkit_toc_scroll_duration', 500);
	?>
	<!-- Card 1: Custom TOC Labels -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<?php
		$acf_enabled = get_option('anchorkit_toc_acf_enabled', false);
		$acf_field_names = get_option('anchorkit_toc_acf_field_names', '');
		$acf_merge_mode = get_option('anchorkit_toc_acf_merge_mode', 'before');
		?>
		<h2>Custom TOC Labels</h2>
		<p class="anchorkit-card-description">Replace specific heading text in your Table of Contents with custom labels.
		</p>

		<div class="form-group">
			<!-- Hidden field to store JSON -->
			<input type="hidden" name="anchorkit_toc_custom_labels" id="anchorkit_toc_custom_labels"
				value="<?php echo esc_attr(wp_unslash($custom_labels)); ?>">

			<!-- Visual key-value interface -->
			<div id="anchorkit_custom_labels_container" class="anchorkit-custom-labels-container">
				<!-- JavaScript will populate this -->
			</div>

			<button type="button" class="anchorkit-add-label-btn" id="anchorkit_add_custom_label">
				<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16">
					<path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z" />
				</svg>
				Add Custom Label
			</button>

			<div class="anchorkit-label-with-tooltip" style="margin-top: 12px;">
				<label for="anchorkit_custom_labels_container">Label Replacements</label>
				<?php
				echo wp_kses(
					anchorkit_tooltip(
						'<strong>Customize heading labels</strong><br><br>' .
						'Replace specific heading text that appears in your TOC with custom labels.<br><br>' .
						'<strong>Example:</strong><br>' .
						'"Introduction" → "Getting Started"<br>' .
						'"FAQ" → "Questions & Answers"<br><br>' .
						'The actual headings on your page remain unchanged - only the TOC labels are replaced.'
					),
					anchorkit_get_allowed_html_tags()
				);
				?>
			</div>

			<div id="anchorkit_toc_custom_labels_error" class="field-error-message anchorkit-error-hidden"></div>
		</div>
	</div>

	<!-- Card 2: Mobile & Responsiveness -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<h2>Mobile & Responsiveness</h2>
		<p class="anchorkit-card-description">Optimize your Table of Contents for mobile devices.</p>

		<div class="setting-wrapper">
			<label class="m3-switch">
				<input type="checkbox" name="anchorkit_toc_hide_on_mobile" value="1" <?php checked(1, $hide_on_mobile, true); ?>>
				<span class="m3-track"><span class="m3-thumb"></span></span>
				<span class="m3-switch-label">
					Hide TOC on small screens
				</span>
			</label>
			<span class="helper-text">Hide TOC on mobile devices.</span>
		</div>

		<div class="form-group conditional-field" data-condition="anchorkit_toc_hide_on_mobile" data-condition-value="1">
			<div class="anchorkit-input-container">
				<input type="number" class="anchorkit-input" name="anchorkit_toc_mobile_breakpoint"
					id="anchorkit_toc_mobile_breakpoint"
					value="<?php echo esc_attr(get_option('anchorkit_toc_mobile_breakpoint', 782)); ?>" min="320" max="1200"
					placeholder="782">
				<label for="anchorkit_toc_mobile_breakpoint">Mobile Breakpoint (px)</label>
			</div>
			<span class="helper-text">Screen width threshold for hiding.</span>
		</div>
	</div>

	<!-- Card 3: Accessibility -->
	<div class="anchorkit-card conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
		data-condition="anchorkit_toc_enabled" data-condition-value="1">
		<h2>Accessibility</h2>
		<p class="anchorkit-card-description">Ensure your TOC is accessible to all users.</p>

		<div class="form-group">
			<div class="anchorkit-label-with-tooltip">
				<label for="anchorkit_toc_aria_label">ARIA Label for TOC Container</label>
				<?php
				echo wp_kses(
					anchorkit_tooltip(
						'<strong>Accessible name for screen readers</strong><br><br>' .
						'Provides an accessible name for the TOC container that screen readers announce.<br><br>' .
						'<strong>Examples:</strong><br>' .
						'<code>Table of Contents</code> (default)<br>' .
						'<code>Article sections</code><br>' .
						'<code>Page navigation</code><br><br>' .
						'If left empty, uses your TOC title or "Table of Contents" by default.'
					),
					anchorkit_get_allowed_html_tags()
				);
				?>
			</div>
			<div class="anchorkit-input-container">
				<input type="text" class="anchorkit-input" name="anchorkit_toc_aria_label" id="anchorkit_toc_aria_label"
					value="<?php echo esc_attr($aria_label); ?>" placeholder="Table of Contents">
				<label for="anchorkit_toc_aria_label" class="anchorkit-sr-only">ARIA Label for TOC Container</label>
			</div>
		</div>
	</div>

	<!-- PRO FEATURE: ACF Integration -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<div class="anchorkit-card anchorkit-pro-feature conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
			<h2>ACF Integration</h2>
			<p class="anchorkit-card-description">Merge headings stored inside Advanced Custom Fields with your main content
				TOC.
			</p>

			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_acf_enabled" value="1" <?php checked($acf_enabled, true); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">
						Enable ACF integration
					</span>
				</label>
				<span class="helper-text">Include headings from flexible content, repeaters, or custom WYSIWYG fields.</span>
			</div>

			<div class="form-group conditional-field" data-condition="anchorkit_toc_acf_enabled" data-condition-value="1">
				<!-- Hidden field to store comma-separated values -->
				<input type="hidden" name="anchorkit_toc_acf_field_names" id="anchorkit_toc_acf_field_names"
					value="<?php echo esc_attr($acf_field_names); ?>">

				<!-- Visual interface for ACF field names -->
				<div id="anchorkit_acf_fields_container" class="anchorkit-acf-fields-container">
					<!-- JavaScript will populate this -->
				</div>

				<button type="button" class="anchorkit-add-field-btn" id="anchorkit_add_acf_field">
					<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16">
						<path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z" />
					</svg>
					Add ACF Field
				</button>

				<span class="helper-text" style="margin-top: 12px; display: block;">Add ACF field names to scan for headings.
					Leave empty to scan all fields automatically.</span>
			</div>

			<div class="form-group conditional-field" data-condition="anchorkit_toc_acf_enabled" data-condition-value="1">
				<div class="anchorkit-input-container">
					<select class="anchorkit-select" name="anchorkit_toc_acf_merge_mode" id="anchorkit_toc_acf_merge_mode">
						<option value="before" <?php selected($acf_merge_mode, 'before'); ?>>Before main content</option>
						<option value="after" <?php selected($acf_merge_mode, 'after'); ?>>After main content</option>
						<option value="replace" <?php selected($acf_merge_mode, 'replace'); ?>>Replace main content</option>
					</select>
					<label for="anchorkit_toc_acf_merge_mode">Merge Mode</label>
				</div>
				<span class="helper-text">Decide how headings from ACF fields are combined with the standard content.</span>
			</div>

			<div class="anchorkit-notice anchorkit-notice-success">
				<strong>✅ Workflow Tip:</strong> Use this when hero sections or custom modules store headings in separate fields
				so the TOC never misses them.
			</div>
		</div>
	<?php endif; ?>

	<!-- PRO FEATURE: AMP Compatibility -->
	<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
		<?php $amp_enabled = (bool) get_option('anchorkit_toc_amp_enabled', false); ?>
		<div class="anchorkit-card anchorkit-pro-feature conditional-field <?php echo esc_attr($toc_enabled ? '' : 'disabled-card'); ?>"
			data-condition="anchorkit_toc_enabled" data-condition-value="1">
			<div class="anchorkit-pro-badge" aria-hidden="true">AnchorKit Pro</div>
			<h2>AMP Compatibility</h2>
			<p class="anchorkit-card-description">Optimized TOC for AMP (Accelerated Mobile Pages) with no JavaScript and inline
				CSS.</p>

			<div class="setting-wrapper">
				<label class="m3-switch">
					<input type="checkbox" name="anchorkit_toc_amp_enabled" value="1" <?php checked(1, $amp_enabled, true); ?>>
					<span class="m3-track"><span class="m3-thumb"></span></span>
					<span class="m3-switch-label">
						Enable AMP-compatible TOC
					</span>
				</label>
				<span class="helper-text">Automatically render simplified TOC on AMP pages.</span>
			</div>

			<div class="anchorkit-notice anchorkit-notice-success">
				<strong>⚡ AMP Mode:</strong> When enabled, the plugin automatically detects AMP pages and renders a simplified,
				AMP-valid TOC without JavaScript or external stylesheets.
			</div>
		</div>
	<?php endif; ?>
<?php
}

// Help Tab
function anchorkit_render_help_tab()
{
	// Set support subject based on premium status
	$support_subject = __('AnchorKit WordPress Plugin Support', 'anchorkit-table-of-contents');
	// Premium override - this block is stripped from free version
	if (anchorkit_fs() && anchorkit_fs()->is__premium_only()) {
		if (anchorkit_fs()->can_use_premium_code()) {
			$support_subject = __('AnchorKit Pro WordPress Plugin Support', 'anchorkit-table-of-contents');
		}
	}
	$support_mailto = 'mailto:support@smelter.studio?subject=' . rawurlencode($support_subject);
	$forum_url = 'https://wordpress.org/support/plugin/anchorkit/';
	?>
	<div class="anchorkit-card">
		<h2>Documentation & Support</h2>
		<p class="anchorkit-card-description">Find helpful resources and documentation for AnchorKit.</p>

		<h3>Getting Started</h3>
		<p>AnchorKit automatically generates a Table of Contents based on the headings in your content. Simply enable the
			feature in the Quick Start tab and configure your preferences.</p>

		<h3>Manual Placement Options</h3>
		<p>You can manually insert a Table of Contents anywhere in your content using either method:</p>

		<h4>Gutenberg Block (Recommended)</h4>
		<p>In the block editor, click the "+" icon and search for "Table of Contents" or "AnchorKit". Insert the block
			wherever you want the TOC to appear.</p>

		<h4>Shortcode</h4>
		<p>You can also use the shortcode in any post, page, or widget:</p>

		<h5>Basic Usage</h5>
		<code>[anchorkit_toc]</code>

		<h5>Advanced Usage with Attributes</h5>
		<p>AnchorKit supports 20+ shortcode attributes for complete customization:</p>

		<div class="anchorkit-shortcode-examples">
			<h6>Display Control</h6>
			<code>[anchorkit_toc header_label="Contents" display_header_label="yes" toggle_view="no" initial_view="hide"]</code>

			<h6>Content Filtering</h6>
			<code>[anchorkit_toc heading_levels="2,3,4" exclude="Test" post_types="post,page"]</code>

			<h6>Appearance & Behavior</h6>
			<code>[anchorkit_toc display_counter="yes" theme="dark" preset="modern" hierarchical="yes"]</code>

			<h6>PRO Features</h6>
			<code>[anchorkit_toc sticky="yes" show_reading_time="yes" view_more="5"]</code>
		</div>

		<h5>All Supported Attributes</h5>
		<div class="anchorkit-attributes-grid">
			<div class="attribute-category">
				<strong>Display Control:</strong>
				<code>header_label</code>, <code>display_header_label</code>, <code>toggle_view</code>,
				<code>initial_view</code>
			</div>
			<div class="attribute-category">
				<strong>Content Filtering:</strong>
				<code>heading_levels</code>, <code>exclude</code>, <code>post_types</code>, <code>post_in</code>,
				<code>post_not_in</code>, <code>min_headings</code>
			</div>
			<div class="attribute-category">
				<strong>Appearance:</strong>
				<code>display_counter</code>, <code>class</code>, <code>theme</code>, <code>preset</code>
			</div>
			<div class="attribute-category">
				<strong>Behavior:</strong>
				<code>hierarchical</code>, <code>smooth_scroll</code>, <code>device_target</code>
			</div>
			<div class="attribute-category">
				<strong>PRO Features:</strong>
				<code>view_more</code>, <code>sticky</code>, <code>show_reading_time</code>, <code>show_word_count</code>
			</div>
		</div>

		<h3>Need Help?</h3>
		<p>If you need assistance with AnchorKit, please start with these resources:</p>
		<ul>
			<li><a href="<?php echo esc_url('https://www.getanchorkit.com/docs'); ?>" target="_blank"
					rel="noopener noreferrer">Documentation</a></li>
			<li>
				<a href="<?php echo esc_url($forum_url); ?>" target="_blank" rel="noopener noreferrer">
					<?php esc_html_e('WordPress.org Support Forum (Free)', 'anchorkit-table-of-contents'); ?>
				</a>
			</li>
			<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
				<li>
					<a href="<?php echo esc_url($support_mailto); ?>">
						<?php esc_html_e('Priority Email Support (AnchorKit Pro)', 'anchorkit-table-of-contents'); ?>
					</a>
				</li>
			<?php else: ?>
				<li>
					<?php esc_html_e('Priority email support is available for AnchorKit Pro customers.', 'anchorkit-table-of-contents'); ?>
				</li>
			<?php endif; ?>
			<li><a href="<?php echo esc_url('https://www.getanchorkit.com'); ?>" target="_blank"
					rel="noopener noreferrer">AnchorKit Website</a></li>
		</ul>

		<h3>Developer Hooks & Filters</h3>
		<p>AnchorKit includes 15+ WordPress-style hooks and filters for complete customization. All hooks receive contextual
			data for intelligent modifications:</p>

		<div class="anchorkit-hooks-grid">
			<div class="hook-category">
				<h4>🔥 High Priority Hooks</h4>
				<ul class="anchorkit-hook-list">
					<li><code>anchorkit_should_display_toc</code> &mdash; Master filter to control TOC visibility</li>
					<li><code>anchorkit_anchor_id</code> &mdash; Modify generated anchor IDs</li>
					<li><code>anchorkit_min_headings</code> &mdash; Override minimum headings required</li>
					<li><code>anchorkit_post_types</code> &mdash; Override allowed post types</li>
					<li><code>anchorkit_exclude_post</code> &mdash; Exclude specific posts</li>
				</ul>
			</div>

			<div class="hook-category">
				<h4>⚡ Medium Priority Hooks</h4>
				<ul class="anchorkit-hook-list">
					<li><code>anchorkit_title_text</code> &mdash; Modify TOC title dynamically</li>
					<li><code>anchorkit_heading_text</code> &mdash; Modify individual heading text in TOC</li>
					<li><code>anchorkit_content_to_parse</code> &mdash; Filter content before heading extraction</li>
					<li><code>anchorkit_smooth_scroll_offset</code> &mdash; Adjust scroll offset for fixed headers</li>
				</ul>
			</div>

			<div class="hook-category">
				<h4>🎯 Rendering Hooks</h4>
				<ul class="anchorkit-hook-list">
					<li><code>anchorkit_toc_headings</code> &mdash; Modify or reorder parsed headings</li>
					<li><code>anchorkit_toc_container_classes</code> &mdash; Add/remove wrapper classes</li>
					<li><code>anchorkit_toc_html</code> &mdash; Filter final TOC HTML output</li>
					<li><code>anchorkit_toc_before</code> / <code>anchorkit_toc_after</code> &mdash; Inject markup
						before/after TOC</li>
					<li><code>anchorkit_toc_link_attributes</code> &mdash; Modify link attributes (rel, target, etc.)</li>
				</ul>
			</div>
		</div>

		<h4>Usage Examples</h4>
		<p>Add these examples to your theme's <code>functions.php</code> to customize AnchorKit:</p>

		<div class="anchorkit-code-examples">
			<h5>Prefix Anchor IDs</h5>
			<pre><code>// Add custom prefix to all anchor links
								add_filter('anchorkit_anchor_id', function($anchor, $text, $format) {
									return 'section-' . $anchor;
								}, 10, 3);</code></pre>

			<h5>Customize TOC Container</h5>
			<pre><code>// Add custom CSS classes to TOC container
								add_filter('anchorkit_toc_container_classes', function($classes, $context) {
									$classes[] = 'my-theme-toc';
									$classes[] = 'sidebar-navigation';
									return $classes;
								}, 10, 2);</code></pre>

			<h5>Modify Heading Display</h5>
			<pre><code>// Truncate long headings in TOC
								add_filter('anchorkit_heading_text', function($text, $heading, $context) {
									if (strlen($text) > 50) {
										return substr($text, 0, 47) . '...';
									}
									return $text;
								}, 10, 3);</code></pre>

			<h5>Add Content Before TOC</h5>
			<pre><code>// Inject custom content before TOC
								add_action('anchorkit_toc_before', function($context) {
									echo '<div class="toc-intro">Quick navigation:</div>';
								}, 10, 1);</code></pre>

			<h5>Control TOC Visibility</h5>
			<pre><code>// Hide TOC on specific post types
								add_filter('anchorkit_should_display_toc', function($should_display, $post_id, $headings) {
									if (get_post_type($post_id) === 'product') {
										return false; // Don't show on WooCommerce products
									}
									return $should_display;
								}, 10, 3);</code></pre>
		</div>
	</div>

	<!-- FAQ Section -->
	<div class="anchorkit-card">
		<h2 class="anchorkit-faq-title">Frequently Asked Questions</h2>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">Why doesn't the Table of Contents appear on some posts?</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					AnchorKit only shows the TOC when several conditions are met. Make sure the TOC is enabled in the Quick
					Start
					tab, the current post type is selected under <strong>Show on</strong>, and the page has at least the
					configured
					minimum number of headings. Automatic insertion is also skipped when a manual TOC is present on the page
					(AnchorKit block, shortcode, Elementor widget, or sidebar widget) so you never get duplicate TOCs.
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">How does AnchorKit work with the Gutenberg block and Elementor widget?</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					The Gutenberg block and Elementor widget let you place a TOC exactly where you want it and override
					global
					settings per instance. By default, their controls fall back to the global AnchorKit settings, so you get
					a
					consistent starting point. When either is used on a page, AnchorKit detects the manual TOC and disables
					automatic insertion for that request so only one TOC is rendered.
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">Can I customize the TOC output using code?</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					Yes. AnchorKit exposes several actions and filters so developers can modify headings, add custom
					classes,
					change link attributes, or wrap the TOC in additional markup. Start with the hooks listed above
					(<code>anchorkit_toc_headings</code>, <code>anchorkit_toc_container_classes</code>,
					<code>anchorkit_toc_link_attributes</code>, and <code>anchorkit_toc_html</code>) and see the examples in
					the
					online documentation for more advanced recipes.
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">Will AnchorKit conflict with other TOC plugins like Easy TOC?</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					AnchorKit doesn't modify or disable other plugins, so if another TOC plugin is active you may see
					multiple TOCs
					on the same page. To avoid confusion, we recommend deactivating other TOC plugins or disabling their
					automatic
					insertion so AnchorKit can be the only TOC on the page.
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">Where can I get help if something isn't working?</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					Start by checking the documentation at <a
						href="<?php echo esc_url('https://www.getanchorkit.com/docs'); ?>" target="_blank"
						rel="noopener noreferrer">getanchorkit.com/docs</a>, which covers setup, integrations, and developer
					hooks.
					<?php if (anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()): ?>
						<?php esc_html_e('If you still need help, email our priority support team at', 'anchorkit-table-of-contents'); ?>
						<a href="<?php echo esc_url($support_mailto); ?>">support@smelter.studio</a>
						<?php esc_html_e('and include as much detail as possible (site URL, theme, other TOC plugins, and steps to reproduce the issue).', 'anchorkit-table-of-contents'); ?>
					<?php else: ?>
						<?php esc_html_e('If you still need help, please open a support topic on the WordPress.org forum for AnchorKit and include as much detail as possible (site URL, theme, other TOC plugins, and steps to reproduce the issue).', 'anchorkit-table-of-contents'); ?>
					<?php endif; ?>
				</div>
			</div>
		</div>
	</div>
	<?php
}

// AnchorKit Pro Tab
function anchorkit_render_pro_tab()
{
	$plan_cards = function_exists('anchorkit_get_pro_plan_cards') ? anchorkit_get_pro_plan_cards() : array();
	$default_upgrade_url = !empty($plan_cards) ? $plan_cards[0]['checkout_url'] : anchorkit_get_checkout_url();
	?>
	<!-- Hero Card -->
	<div class="anchorkit-card anchorkit-pro-hero">
		<h1 class="anchorkit-pro-title"><?php esc_html_e('AnchorKit Pro', 'anchorkit-table-of-contents'); ?></h1>
		<p class="anchorkit-pro-description">
			<?php esc_html_e('Transform your Table of Contents with premium features, advanced customization, and powerful SEO tools.', 'anchorkit-table-of-contents'); ?>
		</p>
		<?php if (!(anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code())): ?>
		<?php else: ?>
			<div class="anchorkit-pro-hero-badge">
				<strong
					class="anchorkit-pro-hero-badge-text"><?php esc_html_e('✨ You\'re a Pro User!', 'anchorkit-table-of-contents'); ?></strong>
			</div>
		<?php endif; ?>
	</div>

	<?php if (!(anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code()) && !empty($plan_cards)): ?>
		<!-- Pricing Table -->
		<div class="anchorkit-card">
			<h2><?php esc_html_e('Simple, Transparent Pricing', 'anchorkit-table-of-contents'); ?></h2>
			<p class="anchorkit-card-description">
				<?php esc_html_e('Choose the license plan that works best for your needs.', 'anchorkit-table-of-contents'); ?>
			</p>

			<?php
			$anchorkit_blogger_test_price_enabled = defined('ANCHORKIT_BLOGGER_TEST_PRICE') && ANCHORKIT_BLOGGER_TEST_PRICE;
			$anchorkit_blogger_price = $anchorkit_blogger_test_price_enabled ? 59 : 59;

			// Get checkout URLs from plan cards
			$blogger_url = isset($plan_cards[0]) ? $plan_cards[0]['checkout_url'] : $default_upgrade_url;
			$freelancer_url = isset($plan_cards[1]) ? $plan_cards[1]['checkout_url'] : $default_upgrade_url;
			$agency_url = isset($plan_cards[2]) ? $plan_cards[2]['checkout_url'] : $default_upgrade_url;
			?>

			<div class="pricing-grid">
				<!-- Blogger Plan -->
				<div class="pricing-card">
					<div class="pricing-tier"><?php esc_html_e('Blogger', 'anchorkit-table-of-contents'); ?></div>
					<div class="pricing-amount">
						<?php echo esc_html(sprintf('$%d/yr', $anchorkit_blogger_price)); ?>
					</div>
					<div class="pricing-savings">
						<?php esc_html_e('Perfect for a single site', 'anchorkit-table-of-contents'); ?>
					</div>
					<ul class="pricing-features">
						<li><?php esc_html_e('1 Site', 'anchorkit-table-of-contents'); ?></li>
						<li><?php esc_html_e('All Pro Features', 'anchorkit-table-of-contents'); ?></li>
						<li><?php esc_html_e('Standard support', 'anchorkit-table-of-contents'); ?></li>
					</ul>
					<a href="<?php echo esc_url($blogger_url); ?>" target="_blank" rel="noopener noreferrer"
						class="pricing-button">
						<?php esc_html_e('Upgrade to Pro', 'anchorkit-table-of-contents'); ?>
					</a>
				</div>

				<!-- Freelancer Plan (Most Popular) -->
				<div class="pricing-card popular">
					<div class="pricing-tier"><?php esc_html_e('Freelancer', 'anchorkit-table-of-contents'); ?></div>
					<div class="pricing-amount">
						<?php echo esc_html('$79/yr'); ?>
					</div>
					<div class="pricing-savings">
						<?php esc_html_e('Best for freelancers & small teams', 'anchorkit-table-of-contents'); ?>
					</div>
					<ul class="pricing-features">
						<li><?php esc_html_e('Up to 10 Sites', 'anchorkit-table-of-contents'); ?></li>
						<li><?php esc_html_e('All Pro Features', 'anchorkit-table-of-contents'); ?></li>
						<li><?php esc_html_e('Priority email support', 'anchorkit-table-of-contents'); ?></li>
					</ul>
					<a href="<?php echo esc_url($freelancer_url); ?>" target="_blank" rel="noopener noreferrer"
						class="pricing-button">
						<?php esc_html_e('Upgrade to Pro', 'anchorkit-table-of-contents'); ?>
					</a>
				</div>

				<!-- Agency Plan -->
				<div class="pricing-card agency-full">
					<div class="pricing-tier"><?php esc_html_e('Agency', 'anchorkit-table-of-contents'); ?></div>
					<div class="pricing-amount">
						<?php echo esc_html('$199/yr'); ?>
					</div>
					<div class="pricing-savings">
						<?php esc_html_e('Unlimited client and network sites', 'anchorkit-table-of-contents'); ?>
					</div>
					<ul class="pricing-features">
						<li><?php esc_html_e('Unlimited Sites', 'anchorkit-table-of-contents'); ?></li>
						<li><?php esc_html_e('All Pro Features', 'anchorkit-table-of-contents'); ?></li>
						<li><?php esc_html_e('Priority email support', 'anchorkit-table-of-contents'); ?></li>
					</ul>
					<a href="<?php echo esc_url($agency_url); ?>" target="_blank" rel="noopener noreferrer"
						class="pricing-button">
						<?php esc_html_e('Upgrade to Pro', 'anchorkit-table-of-contents'); ?>
					</a>
				</div>
			</div>

			<p class="anchorkit-card-description is-centered-text">
				<?php esc_html_e('🔒 Secure payments • 🛠️ 1-year updates & support • 🔁 Optional auto-renewal • 🚫 Cancel anytime • 🛡️ 14-day money-back guarantee', 'anchorkit-table-of-contents'); ?>
			</p>
		</div>
	<?php endif; ?>

	<!-- Pro vs Free Comparison Table -->
	<div class="anchorkit-card">
		<table class="anchorkit-comparison-table">
			<thead>
				<tr>
					<th><?php esc_html_e('Feature', 'anchorkit-table-of-contents'); ?></th>
					<th class="text-center"><?php esc_html_e('Free', 'anchorkit-table-of-contents'); ?></th>
					<th class="text-center"><?php esc_html_e('Pro', 'anchorkit-table-of-contents'); ?></th>
				</tr>
			</thead>
			<tbody>
				<!-- CORE FEATURES -->
				<tr>
					<td colspan="3" class="feature-category">
						<?php esc_html_e('📋 Core Features', 'anchorkit-table-of-contents'); ?>
					</td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Automatic TOC Generation', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Auto-generate TOCs from your content headings', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Gutenberg Block & Shortcode', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Insert TOC anywhere using block or shortcode', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Post Type Selection', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Choose where TOCs appear (posts, pages, etc.)', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Collapsible TOC', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Let users expand/collapse the TOC', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Smooth Scrolling', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Animated scroll to sections', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>

				<!-- DESIGN & STYLING -->
				<tr>
					<td colspan="3" class="feature-category">
						<?php esc_html_e('🎨 Design & Styling', 'anchorkit-table-of-contents'); ?>
					</td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Preset Styles', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Beautiful pre-designed TOC styles', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Custom Bullet Points & Numbering', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Choose from multiple bullet styles (disc, circle, square, arrow, chevron) and numbering formats', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Basic Custom Styling', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Colors, fonts, spacing, borders', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Advanced Styling', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Box shadows, entrance animations, typography controls, custom CSS class', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Light & Dark Mode', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Auto-adapts to user\'s theme preference', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>

				<!-- NAVIGATION & UX -->
				<tr>
					<td colspan="3" class="feature-category">
						<?php esc_html_e('🧭 Navigation & UX', 'anchorkit-table-of-contents'); ?>
					</td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Scroll Spy (Active Highlighting)', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Highlight current section as users scroll', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Sticky/Fixed Position', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Keep TOC visible while scrolling (in-content, left, or right)', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Back to Top Link', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Convenient link to scroll back to page top', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Enhanced Scroll Animations', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Custom easing curves and animation durations', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('View More/Less', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Show initial items with expand/collapse button for long TOCs', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>

				<!-- PAGE BUILDER INTEGRATION -->
				<tr>
					<td colspan="3" class="feature-category">
						<?php esc_html_e('🔌 Page Builder Integration', 'anchorkit-table-of-contents'); ?>
					</td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Elementor Widget', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Drag-and-drop TOC widget for Elementor with live preview', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Gutenberg Block', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Native Gutenberg block with settings panel (limited features in free, full features in pro)', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span><span
							class="feature-label"><?php esc_html_e('Limited', 'anchorkit-table-of-contents'); ?></span>
					</td>
					<td><span class="check-icon">✓</span><span
							class="feature-label"><?php esc_html_e('Full Features', 'anchorkit-table-of-contents'); ?></span>
					</td>
				</tr>

				<!-- SEO & STRUCTURE -->
				<tr>
					<td colspan="3" class="feature-category">
						<?php esc_html_e('🔍 SEO & Structure', 'anchorkit-table-of-contents'); ?>
					</td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Custom Anchor ID Formats', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Control how anchor links are generated', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Heading Exclusion', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Exclude headings by CSS selector', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Advanced Content Filtering', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Exclude by keyword, trim patterns from headings', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Smart Heading Detection', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Min/max heading depth control', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Reading Time & Word Count', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Show reading time and word count per section', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Schema.org & Rich Snippets', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('JSON-LD structured data for better SEO', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('ACF Integration', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Seamless integration with Advanced Custom Fields', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('AMP Compatibility', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Optimized for AMP (Accelerated Mobile Pages)', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>

				<!-- ACCESSIBILITY -->
				<tr>
					<td colspan="3" class="feature-category">
						<?php esc_html_e('♿ Accessibility', 'anchorkit-table-of-contents'); ?>
					</td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('ARIA Labels & Keyboard Navigation', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Screen reader support and keyboard controls', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Hide on Mobile', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Optionally hide TOC on small screens', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Reduced Motion Support', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Respects user\'s motion preferences', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>

				<!-- SUPPORT -->
				<tr>
					<td colspan="3" class="feature-category">
						<?php esc_html_e('💬 Support', 'anchorkit-table-of-contents'); ?>
					</td>
				</tr>
				<tr>
					<td>
						<div class="feature-name"><?php esc_html_e('Community Support', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Forums and documentation', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="check-icon">✓</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
				<tr>
					<td>
						<div class="feature-name">
							<?php esc_html_e('Priority Email Support', 'anchorkit-table-of-contents'); ?>
						</div>
						<div class="feature-description">
							<?php esc_html_e('Direct support with faster response times', 'anchorkit-table-of-contents'); ?>
						</div>
					</td>
					<td><span class="cross-icon">✗</span></td>
					<td><span class="check-icon">✓</span></td>
				</tr>
			</tbody>
		</table>
	</div>
	<!-- FAQ Section -->
	<div class="anchorkit-card">
		<h2><?php esc_html_e('Frequently Asked Questions', 'anchorkit-table-of-contents'); ?></h2>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">
						<?php esc_html_e('What\'s the difference between Free and Pro?', 'anchorkit-table-of-contents'); ?>
					</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					<?php esc_html_e('The free version includes all core TOC features like automatic generation, Gutenberg block, shortcode, and flexible styling. Pro adds advanced features like sticky positioning, scroll spy, custom bullet styles, reading time, schema markup, entrance animations, and priority support.', 'anchorkit-table-of-contents'); ?>
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">
						<?php esc_html_e('Can I upgrade from Free to Pro anytime?', 'anchorkit-table-of-contents'); ?>
					</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					<?php esc_html_e('Yes! You can upgrade to Pro at any time. Your existing settings and configurations will be preserved, and you\'ll instantly unlock all Pro features.', 'anchorkit-table-of-contents'); ?>
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">
						<?php esc_html_e('What payment methods do you accept?', 'anchorkit-table-of-contents'); ?>
					</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					<?php esc_html_e('We accept all major credit cards (Visa, MasterCard, American Express, Discover) and PayPal through secure Stripe payments.', 'anchorkit-table-of-contents'); ?>
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">
						<?php esc_html_e('Is there a money-back guarantee?', 'anchorkit-table-of-contents'); ?>
					</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					<?php esc_html_e('Absolutely! All plans come with a 14-day money-back guarantee. If you\'re not satisfied for any reason, contact us within 14 days for a full refund, no questions asked.', 'anchorkit-table-of-contents'); ?>
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">
						<?php esc_html_e('Can I cancel auto-renewal?', 'anchorkit-table-of-contents'); ?>
					</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					<?php esc_html_e('Yes. You can turn off auto-renewal at any time from your billing area. Your license will remain active until the end of the paid term, and you can renew manually later if you want to resume updates and support.', 'anchorkit-table-of-contents'); ?>
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">
						<?php esc_html_e('Can I use Pro on client websites?', 'anchorkit-table-of-contents'); ?>
					</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					<?php esc_html_e('Yes! The Freelancer and Agency plans are designed for this. Freelancer allows up to 10 sites (perfect for client projects), while Agency provides unlimited activations for web design agencies and developers.', 'anchorkit-table-of-contents'); ?>
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">
						<?php esc_html_e('What happens if my license expires?', 'anchorkit-table-of-contents'); ?>
					</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					<?php esc_html_e('If an annual license expires, Pro features will continue to work on your site, but you won\'t receive updates or support until you renew. Renewing your license reactivates updates and support for another year.', 'anchorkit-table-of-contents'); ?>
				</div>
			</div>
		</div>

		<div class="faq-item">
			<div class="faq-header" data-toggle="faq">
				<div class="faq-question-wrapper">
					<div class="faq-question">
						<?php esc_html_e('Do you offer discounts for non-profits or educational institutions?', 'anchorkit-table-of-contents'); ?>
					</div>
				</div>
				<div class="faq-toggle"></div>
			</div>
			<div class="faq-answer">
				<div class="faq-answer-text">
					<?php esc_html_e('Yes! We offer special pricing for registered non-profit organizations and educational institutions. Please contact us with verification documents to discuss available discounts.', 'anchorkit-table-of-contents'); ?>
				</div>
			</div>
		</div>
	</div>
	<?php
}
// Frontend functionality for Table of Contents
// Placeholder - Add actual TOC generation logic here

/*
=========================================================
 *  TABLE OF CONTENTS – FRONT-END FEATURE IMPLEMENTATION
 *  -----------------------------------------------------
 *  1. Heading parsing + anchor generation (PHP)
 *  2. TOC HTML builder (PHP)
 *  3. Automatic insertion + shortcode (PHP)
 *  4. Script / style enqueuing (PHP)
 *  =======================================================*/
