<?php
/**
 * Admin styles and scripts.
 *
 * @package AnchorKit_TOC
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Enqueue custom admin styles.
 *
 * @param string $hook The current admin page hook.
 * @return void
 */
function anchorkit_enqueue_admin_styles( $hook ) {
	if ( 'settings_page_anchorkit-settings' !== $hook ) {
		return;
	}

	// Enqueue frontend CSS for preview (provides base TOC and preset styles).
	wp_enqueue_style(
		'anchorkit-frontend-styles',
		plugin_dir_url( __FILE__ ) . '../css/anchorkit-frontend.css',
		array(),
		ANCHORKIT_PLUGIN_VERSION
	);

	// Then enqueue admin styles (admin-specific overrides and UI).
	wp_enqueue_style(
		'anchorkit-admin-styles',
		plugin_dir_url( __FILE__ ) . '../css/anchorkit-admin.css',
		array( 'anchorkit-frontend-styles' ),
		ANCHORKIT_PLUGIN_VERSION
	);

	// Add pro-specific CSS only for premium users - stripped from free version by Freemius.
	if ( anchorkit_fs() && anchorkit_fs()->is__premium_only() && anchorkit_fs()->can_use_premium_code() ) {
		wp_add_inline_style(
			'anchorkit-admin-styles',
			'
            .anchorkit-pro-feature {
                position: relative;
                background: linear-gradient(135deg, #f8f9ff 0%, #fef8ff 100%);
                border: 2px solid #e8d9f6;
            }

            .anchorkit-pro-badge {
                display: inline-flex;
                align-items: center;
                gap: 6px;
                background: linear-gradient(135deg, #3f51b5 0%, #5c6bc0 100%);
                color: white;
                padding: 6px 14px;
                border-radius: 20px;
                font-size: 13px;
                font-weight: 600;
                box-shadow: 0 2px 8px rgba(63, 81, 181, 0.3);
                position: absolute;
                top: 16px;
                right: 16px;
                margin-bottom: 0;
                animation: gradientShift 3s ease infinite;
                background-size: 200% 200%;
            }

            .pro-icon {
                font-size: 14px;
                animation: sparkle 2s ease-in-out infinite;
            }

            @keyframes sparkle {
                0%, 100% {
                    transform: scale(1);
                }
                50% {
                    transform: scale(1.2);
                }
            }

            .pro-feature-preview {
                position: relative;
                margin: 20px 0;
            }

            .pro-feature-preview.disabled {
                pointer-events: none;
                opacity: 0.5;
                filter: grayscale(30%);
            }

            .pro-feature-preview.disabled .m3-switch {
                opacity: 0.6;
            }

            .pro-feature-preview.disabled input,
            .pro-feature-preview.disabled select,
            .pro-feature-preview.disabled textarea {
                cursor: not-allowed;
                background: #f5f5f5;
            }

            .pro-upgrade-prompt {
                background: linear-gradient(135deg, #f0f6ff 0%, #fef0ff 100%);
                border: 2px dashed #9f7aea;
                border-radius: 12px;
                padding: 24px;
                text-align: center;
                margin-top: 20px;
            }

            .pro-upgrade-prompt p {
                font-size: 15px;
                color: #4a5568;
                margin-bottom: 16px;
                line-height: 1.6;
            }

            .pro-upgrade-prompt strong {
                color: #2d3748;
            }

            .preset-card .anchorkit-pro-badge {
                top: 8px;
                right: 8px;
            }

            @keyframes gradientShift {
                0% {
                    background-position: 0% 50%;
                }
                50% {
                    background-position: 100% 50%;
                }
                100% {
                    background-position: 0% 50%;
                }
            }

            .pricing-grid {
                display: grid;
                gap: 24px;
                margin: 40px 0;
            }

            @media (min-width: 768px) {
                .pricing-grid {
                    grid-template-columns: repeat(3, 1fr);
                }
            }

            @media (min-width: 1024px) {
                .pricing-grid {
                    grid-template-columns: repeat(3, 1fr);
                }
            }

            .pricing-card {
                background: white;
                border: 2px solid #e2e8f0;
                border-radius: 12px;
                padding: 32px 24px;
                text-align: center;
                transition: all 0.3s ease;
            }

            .pricing-card:hover {
                border-color: #667eea;
                box-shadow: 0 10px 30px rgba(102, 126, 234, 0.2);
            }

            .pricing-card.popular {
                border-color: #667eea;
                box-shadow: 0 10px 30px rgba(102, 126, 234, 0.2);
            }

            .pricing-card.popular::before {
                content: "Popular";
                position: absolute;
                top: -12px;
                left: 50%;
                transform: translateX(-50%);
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 4px 12px;
                border-radius: 20px;
                font-size: 12px;
                font-weight: 600;
            }
        '
		);
	}

	// Enqueue admin JS for real-time updates.
	wp_enqueue_script(
		'anchorkit-admin-js',
		plugin_dir_url( __FILE__ ) . '../js/anchorkit-admin.js',
		array( 'jquery' ),
		ANCHORKIT_PLUGIN_VERSION,
		true
	);

	// Pass the ajax URL, nonce, and current settings to the JS file.
	wp_localize_script(
		'anchorkit-admin-js',
		'anchorkit_ajax',
		array(
			'ajax_url'              => admin_url( 'admin-ajax.php' ),
			'nonce'                 => wp_create_nonce( 'anchorkit_ajax_nonce' ),
			// Current TOC settings for preview (needed when fields are on different tabs).
			'current_preset'        => anchorkit_get_option( 'anchorkit_toc_style_preset', 'minimal' ),
			'custom_styling'        => anchorkit_get_option( 'anchorkit_toc_custom_styling', false ),
			'show_label'            => anchorkit_get_option( 'anchorkit_toc_show_label', true ),
			'title_text'            => anchorkit_get_option( 'anchorkit_toc_title_text', 'Table of Contents' ),
			'collapsible'           => anchorkit_get_option( 'anchorkit_toc_collapsible', true ),
			'hierarchical_view'     => anchorkit_get_option( 'anchorkit_toc_hierarchical_view', true ),
			'bullet_style'          => anchorkit_get_option( 'anchorkit_toc_bullet_style', 'disc' ),
			'bullet_character'      => anchorkit_get_option( 'anchorkit_toc_bullet_character', '•' ),
			'alignment'             => anchorkit_get_option( 'anchorkit_toc_alignment', 'center' ),
			// View More feature defaults so sidebar preview can reflect settings across tabs.
			'view_more_enabled'     => anchorkit_get_option( 'anchorkit_toc_view_more_enabled', false ),
			'initial_count'         => anchorkit_get_option( 'anchorkit_toc_initial_count', 5 ),
			'view_more_text'        => anchorkit_get_option( 'anchorkit_toc_view_more_text', 'View More' ),
			'view_less_text'        => anchorkit_get_option( 'anchorkit_toc_view_less_text', 'View Less' ),
			// Back to Top feature defaults for cross-tab preview consistency.
			'back_to_top_link'      => anchorkit_get_option( 'anchorkit_toc_back_to_top_link', false ),
			'back_to_top_text'      => anchorkit_get_option( 'anchorkit_toc_back_to_top_text', 'Back to Top' ),
			'back_to_top_font_size' => anchorkit_get_option( 'anchorkit_toc_back_to_top_font_size', 14 ),
		)
	);
}
add_action( 'admin_enqueue_scripts', 'anchorkit_enqueue_admin_styles' );

/**
 * Add reset confirmation dialog.
 *
 * @return void
 */
function anchorkit_add_reset_dialog() {
	$screen = get_current_screen();
	if ( 'settings_page_anchorkit-settings' !== $screen->id ) {
		return;
	}
	?>
	<div id="anchorkit-reset-dialog" class="anchorkit-modal">
		<div class="anchorkit-modal-content">
			<div class="anchorkit-modal-header">
				<h3><?php esc_html_e( 'Reset to Default Values', 'anchorkit-table-of-contents' ); ?></h3>
				<button class="anchorkit-modal-close">
					<svg xmlns="http://www.w3.org/2000/svg" height="24" viewBox="0 -960 960 960" width="24">
						<path
							d="m256-200-56-56 224-224-224-224 56-56 224 224 224-224 56 56-224 224 224 224-56 56-224-224-224 224Z"
							fill="currentColor" />
					</svg>
				</button>
			</div>
			<div class="anchorkit-modal-body">
				<p><?php esc_html_e( 'Are you sure you want to reset these settings to their default values? This action cannot be undone.', 'anchorkit-table-of-contents' ); ?>
				</p>
			</div>
			<div class="anchorkit-modal-footer">
				<button class="anchorkit-modal-cancel"><?php esc_html_e( 'Cancel', 'anchorkit-table-of-contents' ); ?></button>
				<button class="anchorkit-modal-confirm"><?php esc_html_e( 'Reset to Default', 'anchorkit-table-of-contents' ); ?></button>
			</div>
		</div>
	</div>
	<?php
}
add_action( 'admin_footer', 'anchorkit_add_reset_dialog' );
