<?php
/**
 * Plugin Name:  AmwalPay for BA Booking
 * Description: Amwal Credit Card Payment gateway for BA Booking.
 * Version: 1.0.4
 * Text Domain: amwalpay-for-ba-booking
 * Domain Path: /languages
 * Requires PHP: 7.4
 * Requires at least: 5.0
 * Requires Plugins: ba-book-everything
 * Tested up to: 6.9
 * Author: AmwalPay
 * Author URI: https://amwalpay.om/
 * License: GNU General Public License v3.0
 * License URI: http://www.gnu.org/licenses/gpl-3.0.html
 * Copyright: © 2025 AmwalPay
 */

if (!defined('ABSPATH')) {
    exit;
}
// Define constant
if (!defined('AMWALPBA_VERSION')) {
    define('AMWALPBA_VERSION', '1.0.4');
}
if (!defined('AMWALPBA_PLUGIN_NAME')) {
    define('AMWALPBA_PLUGIN_NAME', dirname(plugin_basename(__FILE__)));
}
if (!defined('AMWALPBA_PLUGIN_PATH')) {
    define('AMWALPBA_PLUGIN_PATH', plugin_dir_path(__FILE__));
}

add_action('plugins_loaded', function () {
    if (class_exists('BABE_Settings')) {
        AMWALPBA_Amwal_Pay::init();
        AmwalpBa::CreateBaAmwalpayCardsTable();
    } else {
        add_action('admin_notices', function () {
            echo '<div class="notice notice-error"><p><strong>AmwalPay for BA Booking</strong> requires the <strong>BA Book Everything</strong> plugin to be installed and active.</p></div>';
        });
    }
});

/**
 * AMWALPBA_Amwal_Pay Class.
 * Get general settings
 * @class 		AMWALPBA_Amwal_Pay
 * @version		1.0.0
 * @author 		AmwalPay
 */
require_once AMWALPBA_PLUGIN_PATH . 'includes/amwalpba.php';
class AMWALPBA_Amwal_Pay
{

    // payment method name
    private static $payment_method = 'ba_amwal';

    private static $tab_title = '';
    private static $description = '';
    private static $live = '';
    private static $merchant_id = '';
    private static $terminal_id = '';
    private static $secret_key = '';
    private static $payment_view = '';
    private static $contact_info_type = '';
    private static $apple_cert = '';
    private static $log_path = WP_CONTENT_DIR . '/uploads/ba_amwalpay.log';
    private static $debug = '1';
    private static $has_items = '';

    //////////////////////////////
    /**
     * Hook in tabs.
     */
    public static function init()
    {

        add_action('babe_settings_payment_method_' . self::$payment_method, array(__CLASS__, 'amwalpba_add_settings'), 10, 3);

        if (class_exists('BABE_Settings')) {
            add_filter('babe_sanitize_' . BABE_Settings::$option_name, array(__CLASS__, 'amwalpba_sanitize_settings'), 10, 2);
        }

        add_action('init', array(__CLASS__, 'amwalpba_init_settings'), 20);

        add_filter('babe_checkout_payment_title_' . self::$payment_method, array(__CLASS__, 'amwalpba_payment_method_title'), 10, 3);

        if (version_compare(BABE_VERSION, '1.8.0', '>=')) {

            add_filter('babe_checkout_payment_description_' . self::$payment_method, array(__CLASS__, 'amwalpba_payment_method_fields_html'), 10, 3);

            add_action('babe_init_payment_methods', array(__CLASS__, 'amwalpba_init_payment_method'));

            add_action('babe_order_start_paying_with_' . self::$payment_method, array(__CLASS__, 'amwalpba_order_to_pay'), 10, 4);
        } else {

            add_filter('babe_checkout_payment_fields_' . self::$payment_method, array(__CLASS__, 'amwalpba_payment_method_fields_html'), 10, 3);

            add_action('babe_payment_methods_init', array(__CLASS__, 'amwalpba_init_payment_method'));

            add_action('babe_order_to_pay_by_' . self::$payment_method, array(__CLASS__, 'amwalpba_order_to_pay'), 10, 4);
        }


        add_action('wp_enqueue_scripts', array(__CLASS__, 'amwalpba_scripts'));
    }

    public static function amwalpba_init_settings()
    {

        if (!class_exists('BABE_Settings')) {
            return;
        }

        self::$description = BABE_Settings::$settings[self::$payment_method . '_description'] ?? __('Amwal Payment Gateway for Oman  and supports all card and wallet payment!', 'amwalpay-for-ba-booking');

        BABE_Settings::$settings[self::$payment_method . '_description'] = self::$description;

        self::$tab_title = BABE_Settings::$settings[self::$payment_method . '_tab_title'] ?? __('Amwal Pay', 'amwalpay-for-ba-booking');

        BABE_Settings::$settings[self::$payment_method . '_tab_title'] = self::$tab_title;

        self::$live = BABE_Settings::$settings[self::$payment_method . '_live'] ?? 'uat';
        self::$merchant_id = BABE_Settings::$settings[self::$payment_method . '_merchant_id'] ?? '';
        self::$terminal_id = BABE_Settings::$settings[self::$payment_method . '_terminal_id'] ?? '';
        self::$secret_key = BABE_Settings::$settings[self::$payment_method . '_secret_key'] ?? '';
        self::$payment_view = BABE_Settings::$settings[self::$payment_method . '_payment_view'] ?? '1';
        self::$contact_info_type = BABE_Settings::$settings[self::$payment_method . '_contact_info_type'] ?? '1';
        self::$apple_cert = BABE_Settings::$settings[self::$payment_method . '_apple_cert'] ?? '';
        self::$has_items = BABE_Settings::$settings[self::$payment_method . '_has_items'] ?? '1';

        BABE_Settings::$settings[self::$payment_method . '_live'] = self::$live;
        BABE_Settings::$settings[self::$payment_method . '_merchant_id'] = self::$merchant_id;
        BABE_Settings::$settings[self::$payment_method . '_terminal_id'] = self::$terminal_id;
        BABE_Settings::$settings[self::$payment_method . '_secret_key'] = self::$secret_key;
        BABE_Settings::$settings[self::$payment_method . '_payment_view'] = self::$payment_view;
        BABE_Settings::$settings[self::$payment_method . '_contact_info_type'] = self::$contact_info_type;
        BABE_Settings::$settings[self::$payment_method . '_apple_cert'] = self::$apple_cert;
        BABE_Settings::$settings[self::$payment_method . '_has_items'] = self::$has_items;

        self::amwalpba_create_apple_association_file_on_save(self::$apple_cert);
        if (AmwalpBa::sanitizeVar('ba_amwalpay_callback') && AmwalpBa::sanitizeVar('ba_amwalpay_callback') === 'true') {
            self::callback();
        }
        if (AmwalpBa::sanitizeVar('ba_amwalpay_cloud_notification') && AmwalpBa::sanitizeVar('ba_amwalpay_cloud_notification') === 'true') {
            self::call_cloud_notification();
        }
    }

    /**
     * Add settings
     *
     * @param string $section_id
     * @param string $option_menu_slug
     * @param string $option_name
     */
    public static function amwalpba_add_settings($section_id, $option_menu_slug, $option_name)
    {

        add_settings_field(
            self::$payment_method . '_tab_title', // ID
            __('Payment tab title', 'amwalpay-for-ba-booking'), // Title
            array('BABE_Settings_admin', 'text_field_callback'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_tab_title', 'settings_name' => $option_name) // Args array
        );

        add_settings_field(
            self::$payment_method . '_description', // ID
            __('Payment description', 'amwalpay-for-ba-booking'), // Title
            array('BABE_Settings_admin', 'textarea_callback'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_description', 'settings_name' => $option_name) // Args array
        );

        add_settings_field(
            self::$payment_method . '_live', // ID
            __('Environment', 'amwalpay-for-ba-booking'), // Title
            array(__CLASS__, 'env_setting'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_live', 'settings_name' => $option_name) // Args array
        );

        add_settings_field(
            self::$payment_method . '_merchant_id', // ID
            __('Merchant id', 'amwalpay-for-ba-booking'), // Title
            array('BABE_Settings_admin', 'text_field_callback'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_merchant_id', 'settings_name' => $option_name) // Args array
        );
        add_settings_field(
            self::$payment_method . '_terminal_id', // ID
            __('Terminal id', 'amwalpay-for-ba-booking'), // Title
            array('BABE_Settings_admin', 'text_field_callback'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_terminal_id', 'settings_name' => $option_name) // Args array
        );
        add_settings_field(
            self::$payment_method . '_secret_key', // ID
            __('Secret key', 'amwalpay-for-ba-booking'), // Title
            array('BABE_Settings_admin', 'text_field_callback'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_secret_key', 'settings_name' => $option_name) // Args array
        );
        add_settings_field(
            self::$payment_method . '_payment_view', // ID
            __('Payment Page View', 'amwalpay-for-ba-booking'), // Title
            array(__CLASS__, 'payment_view_setting'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_payment_view', 'settings_name' => $option_name) // Args array
        );
        add_settings_field(
            self::$payment_method . '_contact_info_type', // ID
            __('Contact Info Type', 'amwalpay-for-ba-booking'), // Title
            array(__CLASS__, 'contact_info_type_setting'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_contact_info_type', 'settings_name' => $option_name) // Args array
        );
         add_settings_field(
            self::$payment_method . '_has_items', // ID
            __("Show Product Details on Amwal's Checkout", 'amwalpay-for-ba-booking'), // Title
            array(__CLASS__, 'setting_has_items'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_has_items', 'settings_name' => $option_name) // Args array
        );
       
        add_settings_field(
            self::$payment_method . '_apple_cert', // ID
            __('Apple Pay Certificate', 'amwalpay-for-ba-booking'), // Title
            array('BABE_Settings_admin', 'textarea_callback'), // Callback
            $option_menu_slug, // Page
            $section_id,  // Section
            array('option' => self::$payment_method . '_apple_cert', 'settings_name' => $option_name) // Args array
        );

    }
    public static function setting_has_items($args)
    {
        $selected_page = isset(BABE_Settings::$settings[$args['option']]) ? BABE_Settings::$settings[$args['option']] : '1';
        $post_options = '<option value="1" ' . selected($selected_page, '1', false) . '>' . esc_html__('Yes', 'amwalpay-for-ba-booking') . '</option>';
        $post_options .= '<option value="2" ' . selected($selected_page, '2', false) . '>' . esc_html__('No', 'amwalpay-for-ba-booking') . '</option>';

        if (!empty($post_options)) {
            printf(
                '<select id="%s" name="%s">%s</select>',
                esc_attr($args['option']),
                esc_attr($args['settings_name'] . '[' . $args['option'] . ']'),
                wp_kses($post_options, ['option' => ['value' => [], 'selected' => []]]) // Allow only <option> tags with specific attributes
            );
        }

    }
    public static function env_setting($args)
    {
        // 
        $selected_page = isset(BABE_Settings::$settings[$args['option']]) ? BABE_Settings::$settings[$args['option']] : 'uat';

        $post_options = '<option value="prod" ' . selected($selected_page, 'prod', false) . '>' . esc_html__('Production', 'amwalpay-for-ba-booking') . '</option>';
        $post_options .= '<option value="uat" ' . selected($selected_page, 'uat', false) . '>' . esc_html__('UAT', 'amwalpay-for-ba-booking') . '</option>';
        $post_options .= '<option value="sit" ' . selected($selected_page, 'sit', false) . '>' . esc_html__('SIT', 'amwalpay-for-ba-booking') . '</option>';

        if (!empty($post_options)) {
            printf(
                '<select id="%s" name="%s">%s</select>',
                esc_attr($args['option']),
                esc_attr($args['settings_name'] . '[' . $args['option'] . ']'),
                wp_kses($post_options, ['option' => ['value' => [], 'selected' => []]]) // Allow only <option> tags with specific attributes
            );
        }
    }
    public static function payment_view_setting($args)
    {
        // 
        $selected_page = isset(BABE_Settings::$settings[$args['option']]) ? BABE_Settings::$settings[$args['option']] : '1';

        $post_options = '<option value="1" ' . selected($selected_page, '1', false) . '>' . esc_html__('Onsite Pop Up', 'amwalpay-for-ba-booking') . '</option>';
        $post_options .= '<option value="2" ' . selected($selected_page, '2', false) . '>' . esc_html__('Onsite Full Page', 'amwalpay-for-ba-booking') . '</option>';
        $post_options .= '<option value="3" ' . selected($selected_page, '3', false) . '>' . esc_html__('Offsite Redirection', 'amwalpay-for-ba-booking') . '</option>';

        if (!empty($post_options)) {
            printf(
                '<select id="%s" name="%s">%s</select>',
                esc_attr($args['option']),
                esc_attr($args['settings_name'] . '[' . $args['option'] . ']'),
                wp_kses($post_options, ['option' => ['value' => [], 'selected' => []]]) // Allow only <option> tags with specific attributes
            );
        }
    }

    public static function contact_info_type_setting($args)
    {
        // 
        $selected_page = isset(BABE_Settings::$settings[$args['option']]) ? BABE_Settings::$settings[$args['option']] : '1';

        $post_options = '<option value="1" ' . selected($selected_page, '1', false) . '>' . esc_html__('All', 'amwalpay-for-ba-booking') . '</option>';
        $post_options .= '<option value="2" ' . selected($selected_page, '2', false) . '>' . esc_html__('Email Only', 'amwalpay-for-ba-booking') . '</option>';
        $post_options .= '<option value="3" ' . selected($selected_page, '3', false) . '>' . esc_html__('Phone Only', 'amwalpay-for-ba-booking') . '</option>';
        $post_options .= '<option value="4" ' . selected($selected_page, '4', false) . '>' . esc_html__('None', 'amwalpay-for-ba-booking') . '</option>';

        if (!empty($post_options)) {
            printf(
                '<select id="%s" name="%s">%s</select>',
                esc_attr($args['option']),
                esc_attr($args['settings_name'] . '[' . $args['option'] . ']'),
                wp_kses($post_options, ['option' => ['value' => [], 'selected' => []]]) // Allow only <option> tags with specific attributes
            );
        }
    }

    /**
     * Sanitize settings
     *
     * @param array $new_input
     * @param array $input
     * @return array
     */
    public static function amwalpba_sanitize_settings($new_input, $input)
    {

        $new_input[self::$payment_method . '_tab_title'] = isset($input[self::$payment_method . '_tab_title']) ? sanitize_text_field($input[self::$payment_method . '_tab_title']) : '';
        $new_input[self::$payment_method . '_description'] = isset($input[self::$payment_method . '_description']) ? sanitize_textarea_field($input[self::$payment_method . '_description']) : '';
        $new_input[self::$payment_method . '_live'] = isset($input[self::$payment_method . '_live']) ? sanitize_textarea_field($input[self::$payment_method . '_live']) : '';
        $new_input[self::$payment_method . '_merchant_id'] = isset($input[self::$payment_method . '_merchant_id']) ? sanitize_textarea_field($input[self::$payment_method . '_merchant_id']) : '';
        $new_input[self::$payment_method . '_terminal_id'] = isset($input[self::$payment_method . '_terminal_id']) ? sanitize_textarea_field($input[self::$payment_method . '_terminal_id']) : '';
        $new_input[self::$payment_method . '_secret_key'] = isset($input[self::$payment_method . '_secret_key']) ? sanitize_textarea_field($input[self::$payment_method . '_secret_key']) : '';
        $new_input[self::$payment_method . '_payment_view'] = isset($input[self::$payment_method . '_payment_view']) ? sanitize_textarea_field($input[self::$payment_method . '_payment_view']) : '';
        $new_input[self::$payment_method . '_contact_info_type'] = isset($input[self::$payment_method . '_contact_info_type']) ? sanitize_textarea_field($input[self::$payment_method . '_contact_info_type']) : '';
        $new_input[self::$payment_method . '_apple_cert'] = isset($input[self::$payment_method . '_apple_cert']) ? sanitize_textarea_field($input[self::$payment_method . '_apple_cert']) : '';
        $new_input[self::$payment_method . '_has_items'] = isset($input[self::$payment_method . '_has_items']) ? sanitize_textarea_field($input[self::$payment_method . '_has_items']) : '';
        return $new_input;
    }

    ////////////////////////
    /**
     * Init payment method
     * @param array $payment_methods
     * @return void
     */
    public static function amwalpba_init_payment_method($payment_methods)
    {

        if (!isset($payment_methods[self::$payment_method])) {
            BABE_Payments::add_payment_method(self::$payment_method, __('Amwal Pay', 'amwalpay-for-ba-booking'));
        }
    }

    ////////////////////////
    /**
     * Output payment method title for checkout form
     * @param string $method_title
     * @param array $args
     * @param string $input_fields_name
     * @return string
     */
    public static function amwalpba_payment_method_title($method_title, $args, $input_fields_name)
    {

        return self::$tab_title;
    }

    ////////////////////////
    /**
     * Output payment method fields html for checkout form
     * @param string $fields
     * @param array $args
     * @param string $input_fields_name
     * @return string
     */
    public static function amwalpba_payment_method_fields_html($fields, $args, $input_fields_name)
    {
        return self::$description;
    }

    ////////////////////////
    /**
     * Init payment method
     * @param int $order_id
     * @param array $args
     * @param string $current_url
     * @param string $success_url
     * @return void
     */
    public static function amwalpba_order_to_pay($order_id, $args, $current_url, $success_url)
    {
        BABE_Order::update_order_status($order_id, 'payment_expected');

        do_action('babe_order_before_to_pay', $order_id, $args['payment']['payment_method'], $args);
        wp_safe_redirect(add_query_arg(['ba_smartbox' => 'true'], $current_url));
    }
    public static function excuse_hook_javascript($amount, $refNumber)
    {
        $locale = get_locale(); // Get the current language locale
        $currentDate = new DateTime();
        $datetime = $currentDate->format('YmdHis');
        // if $locale content en make $locale = "en"
        if (strpos($locale, 'en') !== false) {
            $locale = "en";
        } else {
            $locale = "ar";
        }
        $sessionToken = AmwalpBa::getUserTokens(self::$live, self::$merchant_id, self::$secret_key);
        // Generate secure hash
        $secret_key = AmwalpBa::generateString(
            $amount,
            512,
            self::$merchant_id,
            $refNumber
            ,
            self::$terminal_id,
            self::$secret_key,
            $datetime,
            $sessionToken
        );

        $urls = (isset(self::$payment_view) && in_array(self::$payment_view, [1, 2]))
            ? ['', '']
            : [add_query_arg(['ba_amwalpay_callback' => 'true'], home_url()), AmwalpBa::getCurrentUrlWithoutSmartbox()];

        list($returnUrl, $cancelUrl) = $urls;

        $data = (object) [
            'AmountTrxn' => "$amount",
            'MerchantReference' => "$refNumber",
            'MID' => self::$merchant_id,
            'TID' => self::$terminal_id,
            'CurrencyId' => 512,
            'LanguageId' => $locale,
            'SecureHash' => $secret_key,
            'TrxDateTime' => $datetime,
            'PaymentViewType' => (isset(self::$payment_view) && in_array(self::$payment_view, [1, 2])) ? self::$payment_view : 1,
            'RequestSource' => 'BA_Booking',
            'SessionToken' => $sessionToken,
            'ReturnUrl' => $returnUrl,
            'CancelUrl' => $cancelUrl,
            'ContactInfoType' => self::$contact_info_type,
            'OrderItems' => self::get_invoice_items($amount),
        ];
        AmwalpBa::addLogs(self::$debug, self::$log_path, esc_html__('Payment Request: ', 'amwalpay-for-ba-booking'), print_r($data, 1));
        $jsonData = wp_json_encode($data);
        return $jsonData;
    }
    public static function get_invoice_items($amount)
    {
        $Items = array();
        if (self::$has_items == '1') {
            $order_items_arr = BABE_Order::get_order_items(AmwalpBa::sanitizeVar('order_id'));
            $item = reset($order_items_arr);

            $Items = array();

            $Items[] = array(
                'Name' => esc_html(mb_strimwidth($item['order_item_name'], 0, 45, '...')),
                'Price' => 'OMR ' . $amount,
            );

        }
        return $Items;
    }
    public static function callback()
    {

        $order_num = AmwalpBa::sanitizeVar('merchantReference');
        $order_id = BABE_Order::get_order_id_by_order_number($order_num);
        $order_hash = BABE_Order::get_order_hash($order_id);
        $args = array(
            'order_id' => $order_id,
            'order_num' => $order_num,
            'order_hash' => $order_hash,
            'current_action' => 'to_confirm'
        );

        $success_url = BABE_Settings::get_confirmation_page_url($args);
        self::save_card_token(AmwalpBa::sanitizeVar('customerId'));
        $isPaymentApproved = false;

        $integrityParameters = [
            "amount" => AmwalpBa::sanitizeVar('amount'),
            "currencyId" => AmwalpBa::sanitizeVar('currencyId'),
            "customerId" => AmwalpBa::sanitizeVar('customerId'),
            "customerTokenId" => AmwalpBa::sanitizeVar('customerTokenId'),
            "merchantId" => BABE_Settings::$settings[self::$payment_method . '_merchant_id'],
            "merchantReference" => AmwalpBa::sanitizeVar('merchantReference'),
            "responseCode" => AmwalpBa::sanitizeVar('responseCode'),
            "terminalId" => BABE_Settings::$settings[self::$payment_method . '_terminal_id'],
            "transactionId" => AmwalpBa::sanitizeVar('transactionId'),
            "transactionTime" => AmwalpBa::sanitizeVar('transactionTime')
        ];

        $secureHashValue = AmwalpBa::generateStringForFilter($integrityParameters, BABE_Settings::$settings[self::$payment_method . '_secret_key']);
        $integrityParameters['secureHashValue'] = $secureHashValue;
        $integrityParameters['secureHashValueOld'] = AmwalpBa::sanitizeVar('secureHashValue');
        AmwalpBa::addLogs(self::$debug, self::$log_path, esc_html__('Callback Response: ', 'amwalpay-for-ba-booking'), print_r($integrityParameters, 1));
        if ((AmwalpBa::sanitizeVar('responseCode') === '00' && $secureHashValue == AmwalpBa::sanitizeVar('secureHashValue'))) {
            $isPaymentApproved = true;
        }

        if ($isPaymentApproved) {

            BABE_Order::update_order_status($order_id, 'payment_received');
            do_action('babe_order_completed', $order_id);
            wp_safe_redirect($success_url);
        } else {

            BABE_Order::update_order_status($order_id, 'canceled');
            wp_safe_redirect(home_url());
        }
        exit;
    }
    public static function call_cloud_notification()
    {
        $post_data = file_get_contents('php://input');
        $json_data = json_decode($post_data, true);

        AmwalpBa::addLogs(self::$debug, self::$log_path, esc_html__('In Cloud Notification Response: ', 'amwalpay-for-ba-booking'), print_r($json_data, 1));
        if (empty($json_data)) {
            AmwalpBa::addLogs(self::$debug, self::$log_path, esc_html__('Ops. you are accessing wrong order.', 'amwalpay-for-ba-booking'));
            wp_die(esc_html(__('Ops. you are accessing wrong order.', 'amwalpay-for-ba-booking')));
        }

        if ($json_data['MerchantId'] != self::$merchant_id || $json_data['TerminalId'] != self::$terminal_id) {
            AmwalpBa::addLogs(self::$debug, self::$log_path, esc_html__('Ops. your configuration is mismatched.', 'amwalpay-for-ba-booking'));
            wp_die(esc_html(__('Ops. your configuration is mismatched.', 'amwalpay-for-ba-booking')));
        }

        $order_num = $json_data['MerchantReference'];
        $order_id = BABE_Order::get_order_id_by_order_number($order_num);
        if (empty($order_id) || is_null($order_id) || false === $order_id || '' === $order_id) {
            AmwalpBa::addLogs(self::$debug, self::$log_path, esc_html__('Ops. your configuration is mismatched.', 'amwalpay-for-ba-booking'));
            wp_die(esc_html(__('Ops. your configuration is mismatched.', 'amwalpay-for-ba-booking')));
        }

        $integrityParameters = [
            "Amount" => $json_data['Amount'],
            "AuthorizationDateTime" => $json_data['AuthorizationDateTime'],
            "CurrencyId" => $json_data['CurrencyId'],
            "DateTimeLocalTrxn" => $json_data['DateTimeLocalTrxn'],
            "MerchantId" => self::$merchant_id,
            "MerchantReference" => $json_data['MerchantReference'],
            "Message" => $json_data['Message'],
            "PaidThrough" => $json_data['PaidThrough'],
            "ResponseCode" => $json_data['ResponseCode'],
            "SystemReference" => $json_data['SystemReference'],
            "TerminalId" => self::$terminal_id,
            "TxnType" => $json_data['TxnType']
        ];

        $secureHashValue = AmwalpBa::generateStringForFilter($integrityParameters, self::$secret_key);
        $integrityParameters['secureHashValue'] = $secureHashValue;
        $integrityParameters['secureHashValueOld'] = $json_data['SecureHash'];

        AmwalpBa::addLogs(self::$debug, self::$log_path, esc_html__('Calculated Hash: ', 'amwalpay-for-ba-booking'), print_r($integrityParameters, 1));

        $msg = __('In Cloud Notification for Order #', 'amwalpay-for-ba-booking') . $order_num;

        if (
            $json_data['ResponseCode'] === '00' && $secureHashValue == $json_data['SecureHash']
        ) {
            BABE_Order::update_order_status($order_id, 'payment_received');
            do_action('babe_order_completed', $order_id);
            $note = __('AmwalPay Payment has Successful', 'amwalpay-for-ba-booking');
            $msg = $msg . ' ' . $note;
            AmwalpBa::addLogs(self::$debug, self::$log_path, $msg);
        } else {
            BABE_Order::update_order_status($order_id, 'canceled');
            $note = __('AmwalPay Payment has Failed', 'amwalpay-for-ba-booking');
            $msg = $msg . ' ' . $note;
            AmwalpBa::addLogs(self::$debug, self::$log_path, $msg);
        }

        die(esc_html("Order updated: $order_num"));

    }
    public static function save_card_token($customerTokenId)
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ba_amwalpay_cards_token';
        if (is_user_logged_in()) {
            if (!empty($customerTokenId) && $customerTokenId !== 'null') {
                $current_user = wp_get_current_user();
                $user_id = $current_user->ID;
                $user_email = $current_user->user_email;
                AmwalpBa::addLogs(self::$debug, self::$log_path, ' Customer save Card Token , for User -- ' . $user_email, wp_json_encode($customerTokenId));

                $token = $wpdb->get_results(
                    $wpdb->prepare(
                        "SELECT * FROM {$wpdb->prefix}ba_amwalpay_cards_token WHERE user_id = %d and merchant_id = %s and environment = %s",
                        $user_id,
                        self::$merchant_id,
                        self::$live
                    ),
                    OBJECT
                );
                if (!$token) {
                    $wpdb->insert(
                        $table_name,
                        array(
                            'user_id' => $user_id,
                            'token' => $customerTokenId,
                            'merchant_id' => self::$merchant_id,
                            'environment' => self::$live
                        )
                    );
                } else {
                    $wpdb->update(
                        $table_name,
                        array(
                            'token' => $customerTokenId,
                        ),
                        array(
                            'user_id' => $user_id,
                            'merchant_id' => self::$merchant_id,
                            'environment' => self::$live
                        )
                    );
                }
            }
        }
    }
    public static function amwalpba_create_apple_association_file_on_save($apple_cert)
    {
        if (!is_admin()) {
            return;
        }

        $root_path = ABSPATH;
        $wellknown_path = $root_path . '.well-known/';

        if (!file_exists($wellknown_path)) {
            wp_mkdir_p($wellknown_path);
        }

        $apple_file = $wellknown_path . 'apple-developer-merchantid-domain-association.txt';

        if (empty($apple_cert)) {
            if (file_exists($apple_file)) {
                wp_delete_file($apple_file);
            }
        } else {
            $existing_content = file_exists($apple_file) ? file_get_contents($apple_file) : '';
            if (trim($existing_content) !== trim($apple_cert)) {
                file_put_contents($apple_file, $apple_cert);
            }
        }
    }
    public static function amwalpba_scripts()
    {
        if (AmwalpBa::sanitizeVar('ba_smartbox') == 'true') {
            $api_url = AmwalpBa::getApiUrl(self::$live);
            wp_enqueue_script('ba-smartbox-url', $api_url['smartbox'], array(), AMWALPBA_VERSION, true);
            wp_enqueue_script('ba-smart-box', plugins_url(AMWALPBA_PLUGIN_NAME) . '/assets/js/smart_box.js', array('jquery'), AMWALPBA_VERSION, true);

            wp_localize_script(
                'ba-smart-box',
                'ba',
                array(
                    'jsonData' => self::excuse_hook_javascript(BABE_Order::get_order_total_amount(AmwalpBa::sanitizeVar('order_id')), AmwalpBa::sanitizeVar('order_num')),
                    'callback' => add_query_arg(['ba_amwalpay_callback' => 'true'], home_url()),
                    'cancel_url' => AmwalpBa::getCurrentUrlWithoutSmartbox(),
                    'payment_view' => self::$payment_view
                )
            );
        }
    }
}