const { addFilter } = wp.hooks;
const { createHigherOrderComponent } = wp.compose;
const { Fragment } = wp.element;
const { BlockControls, InspectorControls } = wp.blockEditor;
const { ToolbarGroup, ToolbarButton, Dropdown, PanelBody, SelectControl } = wp.components;
const { createElement } = wp.element;
const { useState, useEffect } = wp.element;
const { apiFetch } = wp;

// Debug log
console.log('Altomatic format selector loaded');

// Helper to get current image size
const getCurrentImageSize = (url, sizeOptions) => {
    if (!url) {
        console.log('No URL provided, defaulting to full');
        return 'full';
    }

    console.log('Getting size for URL:', url);

    // Log what we received for debugging
    if (!sizeOptions || Object.keys(sizeOptions).length === 0) {
        console.log('No size options provided, using URL pattern only');
    } else {
        console.log('Size options:', sizeOptions);
    }

    // Extract dimensions from URL (look for patterns like -300x133.webp or -300x133.jpg)
    const dimensionsMatch = url.match(/-(\d+x\d+)\.(webp|jpeg|jpg|png|gif|avif)$/);
    if (dimensionsMatch) {
        const dimensions = dimensionsMatch[1];
        console.log('Found dimensions in URL:', dimensions);

        // Common size dimension patterns
        if (dimensions === '150x150') {
            console.log('Matched thumbnail dimensions (150x150)');
            return 'thumbnail';
        }

        if (dimensions.startsWith('300x')) {
            console.log('Matched medium dimensions (300x)');
            return 'medium';
        }

        if (dimensions.startsWith('768x')) {
            console.log('Matched medium_large dimensions (768x)');
            return 'medium_large';
        }

        if (dimensions.startsWith('1024x')) {
            console.log('Matched large dimensions (1024x)');
            return 'large';
        }

        if (dimensions.startsWith('1536x')) {
            console.log('Matched 1536x1536 dimensions');
            return '1536x1536';
        }

        if (dimensions.startsWith('2048x')) {
            console.log('Matched 2048x2048 dimensions');
            return '2048x2048';
        }

        // Try to match dimensions with size options if available
        if (sizeOptions) {
            for (const [sizeName, sizeData] of Object.entries(sizeOptions)) {
                if (sizeData?.url?.includes(dimensions)) {
                    console.log(`Found size match by URL pattern: ${sizeName}`);
                    return sizeName;
                }
            }
        }
    }

    // Special pattern for "large" which is common to be 1024px width
    if (url.includes('-1024x')) {
        console.log('Matched large size by 1024px width pattern');
        return 'large';
    }

    // Special pattern for "medium" which is common to be 300px width
    if (url.includes('-300x')) {
        console.log('Matched medium size by 300px width pattern');
        return 'medium';
    }

    // Special pattern for "medium_large" which is common to be 768px width
    if (url.includes('-768x')) {
        console.log('Matched medium_large size by 768px width pattern');
        return 'medium_large';
    }

    // Check for thumbnail size which is usually 150x150
    if (url.includes('-150x150')) {
        console.log('Matched thumbnail size by 150x150 pattern');
        return 'thumbnail';
    }

    // Full/original size usually doesn't have dimensions in the URL or is explicitly marked as original
    if (!url.match(/-\d+x\d+\./)) {
        console.log('No dimensions pattern in URL, matching as full size');
        return 'full';
    }

    console.log('Could not determine size from URL pattern, defaulting to full');
    return 'full';
};

// Helper to safely convert value to string and uppercase
const safeToUpperCase = (value) => {
    return String(value || 'original').toUpperCase();
};

// Helper to process formats from WordPress media API
const processFormats = (mediaData) => {
    if (!mediaData || !mediaData.media_details || !mediaData.media_details._altomatic_optimized_sizes) {
        console.log('No _altomatic_optimized_sizes data found');
        return {};
    }

    const optimizedSizes = mediaData.media_details._altomatic_optimized_sizes;
    const result = {};

    // Get all sizes from media_details
    const wpSizes = mediaData.media_details.sizes || {};

    // Process each size
    Object.keys(wpSizes).forEach(size => {
        result[size] = {};

        // Add original format (from WordPress)
        result[size]['original'] = {
            url: wpSizes[size].source_url,
            mime_type: wpSizes[size].mime_type
        };

        // Add optimized formats if available
        if (optimizedSizes) {
            // For each optimized format (webp, avif, etc.)
            Object.keys(optimizedSizes).forEach(format => {
                // Check if this format has this size
                if (optimizedSizes[format] && optimizedSizes[format][size]) {
                    const formatData = optimizedSizes[format][size];

                    // Create URL from the file path
                    // The path is like: /var/www/html/wp-content/uploads/2025/03/kids_2560-150x150.webp
                    // We need to extract just the uploads part and create a proper URL

                    let fileUrl = '';
                    if (formatData.file) {
                        // Extract path after uploads directory
                        const uploadsPath = formatData.file.match(/\/uploads\/(.+)$/);
                        if (uploadsPath && uploadsPath[1]) {
                            // Base URL can be extracted from the original size URL
                            const baseUrl = wpSizes[size].source_url.split('/wp-content/uploads/')[0];
                            fileUrl = `${baseUrl}/wp-content/uploads/${uploadsPath[1]}`;
                        }
                    }

                    // If we couldn't extract the path, use a pattern-based approach
                    if (!fileUrl) {
                        // Create URL by changing file extension in the original URL
                        const originalUrl = wpSizes[size].source_url;
                        fileUrl = originalUrl.replace(/\.(jpeg|jpg|png|gif)$/, `.${format}`);
                    }

                    result[size][format] = {
                        url: fileUrl,
                        mime_type: formatData['mime-type'] || `image/${format}`
                    };
                }
            });
        }
    });

    console.log('Processed formats:', result);
    return result;
};

// Helper to get URL for a specific size and format
const getUrlForSizeAndFormat = (attributes, imageFormats, size, format) => {
    console.log('Getting URL for size:', size, 'format:', format);
    console.log('Available formats for size:', imageFormats?.[size]);

    if (!size || !format) {
        console.log('Missing size or format');
        return null;
    }

    // Check if the format is available for this size
    const sizeFormats = imageFormats?.[size];
    if (!sizeFormats) {
        console.log(`No formats available for size ${size}`);
        return null;
    }

    if (!sizeFormats[format]) {
        console.log(`Format ${format} not available for size ${size}`);
        return null;
    }

    console.log(`Using URL for ${size}/${format}:`, sizeFormats[format].url);
    return sizeFormats[format].url;
};

// Add format attribute to image block
addFilter(
    'blocks.getBlockAttributes',
    'altomatic/add-format-attribute',
    (attributes, blockType) => {
        if (blockType.name !== 'core/image') {
            return attributes;
        }

        return {
            ...attributes,
            imageFormat: {
                type: 'string',
                default: 'original'
            }
        };
    }
);

// Format selector component
const FormatSelector = function(props) {
    const [formats, setFormats] = useState({});
    const [isLoading, setIsLoading] = useState(true);
    const [detectedSize, setDetectedSize] = useState('full');

    // Fetch formats when component mounts or image ID changes
    useEffect(() => {
        if (!props.imageId) return;

        setIsLoading(true);
        apiFetch({ path: `/wp/v2/media/${props.imageId}?context=view&_locale=user` })
            .then(response => {
                console.log('Fetched media data:', response);
                const processedFormats = processFormats(response);
                setFormats(processedFormats);
                setIsLoading(false);
            })
            .catch(error => {
                console.error('Error fetching media data:', error);
                setIsLoading(false);
            });
    }, [props.imageId]);

    // Function to get current size - prioritize WordPress size info
    const getCurrentSize = () => {
        // First check if WordPress size info is available
        if (props.sizeSlug) {
            console.log('FormatSelector using provided sizeSlug:', props.sizeSlug);
            return props.sizeSlug;
        }

        // Fall back to URL pattern detection
        const size = getCurrentImageSize(props.currentUrl, props.sizeOptions);
        console.log('FormatSelector detected size from URL:', size);
        return size;
    };

    // Get current size on URL or sizeSlug change
    useEffect(() => {
        if (props.currentUrl || props.sizeSlug) {
            const size = getCurrentSize();
            console.log('FormatSelector current size:', size);
            setDetectedSize(size);
        }
    }, [props.currentUrl, props.sizeOptions, props.sizeSlug]);

    // Only show formats available for the current size
    const availableFormats = formats?.[detectedSize] || {};
    console.log('Available formats for size:', detectedSize, availableFormats);

    // Build format options - without duplicating Original
    const formatOptions = [];

    // Add Original as the first option
    formatOptions.push({ label: 'Original', value: 'original' });

    // Add other formats (excluding original which we've already added)
    if (availableFormats) {
        Object.keys(availableFormats).forEach(format => {
            if (format !== 'original' && availableFormats[format]?.url) {
                formatOptions.push({
                    label: safeToUpperCase(format),
                    value: format
                });
            }
        });
    }

    console.log('Format options for size', detectedSize, ':', formatOptions);

    if (isLoading) {
        return createElement('div', {}, 'Loading formats...');
    }

    return createElement(SelectControl, {
        label: 'Altomatic Image Format',
        value: props.value || 'original',
        options: formatOptions,
        onChange: props.onChange,
        style: { minWidth: '200px' }
    });
};

// Add format selector to image block controls
const withFormatSelector = createHigherOrderComponent((BlockEdit) => {
    return (props) => {
        if (props.name !== 'core/image') {
            return createElement(BlockEdit, props);
        }

        // Debug current attributes
        console.log('Image block attributes:', props.attributes);

        const { attributes, setAttributes } = props;
        const [formats, setFormats] = useState({});
        const [detectedSize, setDetectedSize] = useState('full');
        const [isFormatLoading, setIsFormatLoading] = useState(true);

        // Function to get the current size using multiple methods
        const getCurrentSizeFromProps = () => {
            // First, check for direct WordPress size information
            if (attributes.sizeSlug) {
                console.log('Using WordPress sizeSlug:', attributes.sizeSlug);
                return attributes.sizeSlug;
            }

            // Second, try from URL pattern
            return getCurrentImageSize(attributes.url, attributes.sizes);
        };

        // Fetch formats when image ID changes
        useEffect(() => {
            if (!attributes.id) return;

            setIsFormatLoading(true);
            apiFetch({ path: `/wp/v2/media/${attributes.id}?context=view&_locale=user` })
                .then(response => {
                    console.log('Main component fetched media data:', response);
                    const processedFormats = processFormats(response);
                    setFormats(processedFormats);
                    setIsFormatLoading(false);
                })
                .catch(error => {
                    console.error('Error fetching media data:', error);
                    setIsFormatLoading(false);
                });
        }, [attributes.id]);

        // Initialize image format if not set
        useEffect(() => {
            if (attributes.url && !attributes.imageFormat) {
                setAttributes({
                    imageFormat: 'original'
                });
            }
        }, [attributes.url]);

        // Get current size on URL or sizeSlug change
        useEffect(() => {
            if (attributes.url) {
                const newSize = getCurrentSizeFromProps();
                console.log('Current size determined by multiple methods:', newSize);

                // Only update if size actually changed
                if (newSize !== detectedSize) {
                    console.log('Size changed from', detectedSize, 'to:', newSize);
                    setDetectedSize(newSize);

                    // Check if current format is available in new size
                    const format = attributes.imageFormat || 'original';
                    console.log('Current format is:', format);

                    // If format is not original, we need to check if it's available in the new size
                    // and apply it if available
                    if (format !== 'original' && !isFormatLoading && formats) {
                        console.log('Checking if', format, 'is available for size', newSize);
                        const formatUrl = getUrlForSizeAndFormat(attributes, formats, newSize, format);

                        if (formatUrl) {
                            console.log('Format', format, 'is available for size', newSize, ', applying it');
                            // Format is available in the new size, apply it
                            if (formatUrl !== attributes.url) {
                                console.log('Setting URL to format URL:', formatUrl);
                                setAttributes({
                                    url: formatUrl
                                });
                            } else {
                                console.log('URL already matches format URL, no change needed');
                            }
                        } else {
                            console.log('Format', format, 'is not available for size', newSize, ', reverting to original');
                            // Format is not available in the new size, revert to original
                            const originalUrl = getUrlForSizeAndFormat(attributes, formats, newSize, 'original');
                            if (originalUrl) {
                                console.log('Setting URL to original format:', originalUrl);
                                setAttributes({
                                    url: originalUrl,
                                    imageFormat: 'original' // Important: update the format attribute too
                                });
                            }
                        }
                    }
                }
            }
        }, [attributes.url, attributes.sizes, attributes.sizeSlug, formats, isFormatLoading]);

        // Handle format changes
        const handleFormatChange = (format) => {
            const size = getCurrentSizeFromProps();
            console.log('Format change requested - Size:', size, 'Format:', format);

            // Always update the format attribute, even if the URL doesn't change
            // This ensures the UI reflects the user's choice
            if (format !== attributes.imageFormat) {
                console.log('Updating format attribute from', attributes.imageFormat, 'to', format);

                // Get the URL for the selected format
                const newUrl = getUrlForSizeAndFormat(attributes, formats, size, format);
                console.log('Format change - New URL:', newUrl, 'Current URL:', attributes.url);

                if (newUrl) {
                    console.log('Changing URL and format');
                    setAttributes({
                        url: newUrl,
                        imageFormat: format
                    });
                } else {
                    console.log('Format not available, reverting to original');
                    const originalUrl = getUrlForSizeAndFormat(attributes, formats, size, 'original');
                    if (originalUrl) {
                        console.log('Setting URL to original format:', originalUrl);
                        setAttributes({
                            url: originalUrl,
                            imageFormat: 'original'
                        });
                    } else {
                        console.log('Could not determine original URL');
                    }
                }
            } else {
                console.log('Format already set to', format, '- no change needed');
            }
        };

        // Create toolbar dropdown content
        const dropdownContent = () => {
            return createElement('div',
                {
                    style: {
                        padding: '16px',
                        minWidth: '200px'
                    }
                },
                createElement(FormatSelector, {
                    value: attributes.imageFormat || 'original',
                    formats: formats,
                    onChange: handleFormatChange,
                    currentUrl: attributes.url,
                    sizeOptions: attributes.sizes,
                    imageId: attributes.id,
                    sizeSlug: attributes.sizeSlug
                })
            );
        };

        // Create toolbar button
        const toolbarButton = ({ isOpen, onToggle }) => {
            // Only show format that is currently selected
            const buttonText = safeToUpperCase(attributes.imageFormat || 'original');

            // Show disabled state if format is not available for current size
            const currentSize = getCurrentSizeFromProps();
            const formatAvailable =
                attributes.imageFormat === 'original' ||
                formats[currentSize]?.[attributes.imageFormat]?.url;

            return createElement(
                ToolbarButton,
                {
                    onClick: onToggle,
                    'aria-expanded': isOpen,
                    style: { minWidth: '100px' },
                    disabled: !formatAvailable && attributes.imageFormat !== 'original'
                },
                buttonText
            );
        };

        return createElement(
            Fragment,
            null,
            createElement(BlockEdit, props),
            createElement(
                BlockControls,
                null,
                createElement(
                    ToolbarGroup,
                    null,
                    createElement(Dropdown, {
                        className: 'altomatic-format-dropdown',
                        renderToggle: toolbarButton,
                        renderContent: dropdownContent,
                        position: 'bottom right'
                    })
                )
            ),
            createElement(
                InspectorControls,
                null,
                createElement(
                    PanelBody,
                    {
                        title: 'Altomatic Image Format',
                        initialOpen: true
                    },
                    createElement(FormatSelector, {
                        value: attributes.imageFormat || 'original',
                        formats: formats,
                        onChange: handleFormatChange,
                        currentUrl: attributes.url,
                        sizeOptions: attributes.sizes,
                        imageId: attributes.id,
                        sizeSlug: attributes.sizeSlug
                    })
                )
            )
        );
    };
}, 'withFormatSelector');

// Add custom styles
const style = document.createElement('style');
style.textContent = `
    .altomatic-format-dropdown .components-dropdown-menu__menu {
        min-width: 200px;
        padding: 8px;
    }
    .components-button.has-icon.has-text {
        min-width: 100px;
    }
`;
document.head.appendChild(style);

addFilter(
    'editor.BlockEdit',
    'altomatic/format-selector',
    withFormatSelector
);
